#!/usr/bin/env bash

# OMPi OpenMP Compiler
# == Copyright since 2001 the OMPi Team
# == Dept. of Computer Science & Engineering, University of Ioannina
# 
# This file is part of OMPi.
# 
# OMPi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# OMPi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with OMPi; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
#

# remote_offload_local_setup.sh
# Single node setup script. Must be executed by the root setup 
# script with: 
#   ssh <host> 'bash -s -- < <dir>/remote_offload_local_setup.sh <args>'

ISPRIMARY=false
USE_TEMPDIR=false
BUILD_ONLY=false
TEMPDIR=
REDIRECT_TO="&1"
BUILD_FOLDER="./build"

ConfigFlags=()
PREVINFO_MSG=

_done() { 
	if [[ "$USE_TEMPDIR" = "true" ]]; then
		>&2 printf "$PREVINFO_MSG [done]\n"
	else
		>&2 printf " done.\n"
	fi
}

_info() {
	PREVINFO_MSG="\t($MYHOSTNAME) $@"
	>&2 printf "$PREVINFO_MSG"
	if [[ "$USE_TEMPDIR" = "true" ]]; then
		>&2 printf "\n"
	fi
}

_success() {
	>&2 printf "Configuration completed for $MYHOSTNAME.\n"
	exit 0
}

_fail() {
	if [[ "$USE_TEMPDIR" = "true" ]]; then
		>&2 printf "$PREVINFO_MSG [error: $@].\n"
	else
		>&2 printf " error: $@.\n"
	fi
	exit 1
}

_check_for_error() {
	if [[ "$1" != "0" ]]; then
		shift
		_fail "$@"
	fi
	_done
}

for arg in "$@"; do
	# Check if the argument starts with "-"
	if [[ $arg == -* ]]; then
		# Extract the argument name and value
		arg_name="${arg%%=*}"
		arg_value="${arg#*=}"

		if [[ "$arg_name" == "$arg_value" ]]; then
			arg_str="$arg_name"
		else
			arg_str="$arg_name=$arg_value"
		fi

		# Convert argument name to lowercase for case-insensitive matching
		lowercase_name=${arg_name,,}

		# Check if the argument should be filtered out
		case $lowercase_name in
			--hostname )
				MYHOSTNAME="${arg_value}"
				;;
			--modules )
				MODULES="${arg_value}"
				;;
			--primary )
				ISPRIMARY=true
				REDIRECT_TO="/dev/null"
				;;
			--build-only )
				BUILD_ONLY=true
				;;
			--tempdir )
				USE_TEMPDIR=true
				TEMPDIR="${arg_value}"
				;;
			*)
				ConfigFlags+=("$arg_str")
				;;
		esac

	fi
done

# If no hostname was given, set to system hostname
[[ -z ${MYHOSTNAME} ]] && MYHOSTNAME=$HOSTNAME

# Add OMPi installation directory to PATH
if [[ ! -z $INSTALLDIR ]]; then
	PATH="$INSTALLDIR/bin:$PATH"
else
	_info "warning: OMPi installation directory was not given.\n"
fi

if [[ "$USE_TEMPDIR" = "true" ]]; then
	PATH="$TEMPDIR/$MYHOSTNAME/bin:$PATH"
fi

# Throw a warning if '--modules' was not given
[[ -z ${MODULES} ]] && [[ "$ISPRIMARY" = false ]] && _info "warning: missing \"--modules=<moddevs>\" argument\n"

# Extracts module names from --modules="..."
parse_module_names() {
	local _modules=($MODULES)
	local i=0
	local modules=$1
	
	for item in "${_modules[@]}"
	do
		if [[ "$((i % 2))" == "0" ]]; then
			if [[ "$item" != "cpu" ]]; then
				modules="$modules $item"
			fi
		fi
		i=$((i + 1))
	done
	
	# Remove leading white space from modstring
	printf "${modules:1}"
}

_sort_string () {
    arr=($@)
    IFS=$'\n' sorted=($(sort <<<"${arr[*]}")); unset IFS
    printf "%s " "${sorted[@]}"
}

# Configures OMPi with given flags and checks that installed modules are
# those declared in the configuration file.
ompi_configure() {
	local expected_modules=$(parse_module_names)
	local installed_modules=""
	local res=
	
	_info ">> Configuring OMPi..."
	if [[ "$ISPRIMARY" = true ]]; then
		meson setup $BUILD_FOLDER --native-file remote-offloading.txt "${ConfigFlags[@]}" -Droff-worker-address=""
	else
		if [[ "$USE_TEMPDIR" = "true" ]]; then
			meson setup $BUILD_FOLDER-$MYHOSTNAME --native-file remote-offloading.txt "${ConfigFlags[@]}"  --prefix=$TEMPDIR/$MYHOSTNAME -Droff-worker-address=$MYHOSTNAME >&1 2>&1
		else
			meson setup $BUILD_FOLDER --native-file remote-offloading.txt "${ConfigFlags[@]}"  -Droff-worker-address=$MYHOSTNAME >&1 2>&1
		fi

		installed_modules=$(cat .ompi_configured_modules_$MYHOSTNAME)
		if [[ $(_sort_string $installed_modules) != $(_sort_string $expected_modules) ]]; then
			_fail "module mismatch (Requested '$expected_modules' and found '$installed_modules')"
		fi
	fi
	
	res=$?
	_check_for_error $res "meson setup (exit code $res)"
}

# Builds OMPi
ompi_build() {
	local res=
	local _build_dir=
	
	_info ">> Building OMPi..."
	if [[ "$USE_TEMPDIR" = "true" ]]; then
		_build_dir="$BUILD_FOLDER-$MYHOSTNAME"
	else
		_build_dir="$BUILD_FOLDER"
	fi

	cd $_build_dir
	meson compile >&1 2>&1
	res=$?
	cd ..
	
	_check_for_error $res "meson compile (exit code $res)"
}

# Installs OMPi
ompi_install() {
	local res=
	local _build_dir=
	
	_info ">> Installing OMPi..."
	if [[ "$USE_TEMPDIR" = "true" ]]; then
		_build_dir="$BUILD_FOLDER-$MYHOSTNAME"
	else
		_build_dir="$BUILD_FOLDER"
	fi

	cd $_build_dir
	meson install >&1 2>&1
	res=$?
	cd ..
	
	# Only worker nodes use tempdirs; remove their build directories
	if [[ "$USE_TEMPDIR" = "true" ]]; then
		rm -rf $_build_dir
	fi

	_check_for_error $res "meson install (exit code $res)"
}

# Verifies that installed modules have the correct number of devices
ompi_check_modules() {
	local res=
	
	# (1) Check that ompiconf exists
	_info ">> Checking that ompiconf exists..."
	which ompiconf >> /dev/null 2>&1
	res=$?
	_check_for_error $res "ompiconf executable missing (check your \$PATH)"
	
	# (2) Proceed with verification
	_info ">> Checking installed modules & devices..."
	ompiconf __internal__ --check="${MODULES}" >> /dev/null 2>&1
	res=$?
	
	_check_for_error $res "ompiconf: check for modules/devices failed (exit code $res)"
}

echo "${ConfigFlags[@]}"
# All configuration steps
if [[ "$BUILD_ONLY" = false ]]; then
	ompi_configure
fi
# ompi_build
ompi_install
if [[ "$ISPRIMARY" = false ]]; then
	ompi_check_modules
	rm -rf $BUILD_FOLDER
fi

_success