/*
  OMPi OpenMP Compiler
  == Copyright since 2001, the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ompiconf.c
 * Provides information about the OMPi installation (currently device-only info)
 */
 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include "config.h"
#include "modules.h"
#include "stddefs.h"
#include "assorted.h"
#include "str.h"

#ifdef PORTABLE_BUILD
	char InstallPath[PATHSIZE], LibDir[PATHSIZE], ModuleDir[PATHSIZE];
#endif


#define ompi_info() \
	fprintf(stderr, \
	 "OMPi-conf, the %s module helper:\n  >>  config. devices: %s\n",\
	  PKG_NAME, MODULES_CONFIG)

static void ompi_help(char *binary_name)
{
	fprintf(stderr, "\nUsage: %s [ompiconf options]\n",
			binary_name);
	fprintf(stderr, "\n"
			"   Useful OMPi-conf options:\n"
			"                         --help: display this information\n"
			"          --devinfo[=<devices>]: show short info about configured devices\n"
			"         --devvinfo[=<devices>]: show verbose log about configured devices\n"
			"                 -v / --verbose: enable verbose output (show the actual steps)\n"
			"\n");
}

bool checkmodule = false;
bool showdevinfo = false;
bool verbose = false;
bool internal = false;
char *module_query = NULL;
char *roff_query = NULL;
char *reqmodules = NULL;   /* Requested modules by the user */
char *orig_reqmodules = NULL;

/* Remote device query flags */
bool getnprocs = false;
bool getmpihosts = false;
bool creatempirunscript = false;

#define OMPI_OPTNAME(opt) "-" #opt
#define OPTNAME(opt)      "--" #opt
#define OPTNAME_V(opt)    "V--" #opt "="
#define OPTION(opt)       OPT_##opt
#define INTERNAL          "__internal__"

typedef enum {
	OPTION(unknown) = -1, /* unknown option */
	
	OPTION(dummy1),   /* Info/help options */
	OPTION(help),     OPTION(verbose),   OPTION(internal),
	OPTION(devinfo),  OPTION(devvinfo),  OPTION(check),    
	OPTION(roffaction), OPTION(bundlekernels), OPTION(complete),
	OPTION(devinfo_v),  OPTION(devvinfo_v),

	OPTION(lastoption)    /* marker */
} option_t;

char *optnames[] = {
	NULL,

	OPTNAME(help),     OPTNAME(verbose),   INTERNAL,
	OPTNAME(devinfo),  OPTNAME(devvinfo),  OPTNAME_V(check),  
	OPTNAME_V(roff-action), OPTNAME(bundle-kernels), OPTNAME_V(complete),
	OPTNAME_V(devinfo), OPTNAME_V(devvinfo),

	NULL
};


option_t optid(char *arg, char **val)
{
	int i;

	for (i = 0; i < OPTION(lastoption); i++)
	{
		if (!optnames[i])   /* Skip dummy options */
			continue;
		if (optnames[i][0] == 'V')     /* Option with value */
		{
			if (strncmp(optnames[i]+1, arg, strlen(optnames[i])-1) == 0)
			{
				*val = arg + strlen(optnames[i]) - 1;
				return ((option_t) i);
			}
		}
		else
			if (strcmp(optnames[i], arg) == 0)
				return ((option_t) i);
	}
	return (OPTION(unknown));
}


#ifdef OMPI_REMOTE_OFFLOADING

/* Handle string given after "--roff-action" argument.
 * Currently actions are hard-coded...
 */
static void handle_roff_action(char *action)
{
	/* --roff-action="get-num-mpiprocs" */
	if (strcmp(action, "get-num-mpiprocs") == 0)
	{
		getnprocs = true;
		return;
	}
	
	/* --roff-action="get-mpi-hosts" */
	if (strcmp(action, "get-mpi-hosts") == 0)
	{
		getmpihosts = true;
		return;
	}
	
	/* --roff-action="create-mpirun-script" */
	if (strcmp(action, "create-mpirun-script") == 0)
	{
		creatempirunscript = true;
		return;
	}
	
	fprintf(stderr, "Unknown remote devices action '%s'.\n", action);
	exit(0);
}

#endif


void parse_args(int argc, char **argv)
{
	int  oid, complete_oid;
	char *parameter, *val, *compval, *ompiconf_bin = argv[0];

	argv++;
	argc--;

	while (argc)
	{
		switch (oid = optid(parameter = argv[0], &val))
		{
			case OPTION(complete):
				ompiconf_modules_employ(NULL, &oc_modulenames, &oc_nmodules);
				int i, j;
				switch ( complete_oid = optid(val, &compval) )
				{
					case OPTION(roffaction):
					case OPTION(bundlekernels):
					case OPTION(complete):
					case OPTION(check):
						break;

					default:
					{
						bool valempty = false;
						char *letteropts = "v";
						char *skip[] = {
							"__internal__", "--check=", "--complete=", "--bundle-kernels",
							NULL
						};

						for (i = 0; i < strlen(val); i++)
						{
							if (isspace(val[i])) 
							{
								valempty = true;
								break;
							}
						}
						if (val[0] == '-' && strlen(val) <= 2)
						{
							if (strlen(val) == 1)
							{
								for (i = 0; i < strlen(letteropts); i++)
									fprintf(stderr, "-%c\n", letteropts[i]);
							}
							else if (strchr(letteropts, val[1]) && strcmp(val, "--"))
							{
								fprintf(stderr, "%s\n", val);
								break;
							}
						}
						
						for (i = 0; i != OPTION(lastoption); i++)
						{
							bool _skip = false;
							if (!optnames[i])
								continue;

							char *optname = optnames[i]+(optnames[i][0] == 'V' ? 1 : 0);

							for (j = 0; skip[j] != NULL; j++)
							{
								if (!strncmp(optname, skip[j], strlen(skip[j])))
								{
									_skip = true;
									break;
								}
							}

							if (_skip)
								continue;
								
							if (!strncmp(optname, val, strlen(val)) || valempty)
								fprintf(stderr, "%s\n", optname);
						}
						
						break;
					}
				}
				exit(0);
				break;
			case OPTION(help):
				ompi_info();
				ompi_help(ompiconf_bin);
				exit(0);
				break;
			case OPTION(verbose):
				goto BEVERB;
			case OPTION(internal):  
				internal = true; 
				break;
			case OPTION(devvinfo):  
				verbose = true; 
			case OPTION(devinfo):   
				showdevinfo = true; 
				break;
			case OPTION(devvinfo_v):  
				verbose = true; 
			case OPTION(devinfo_v):   
				reqmodules = val;
#ifdef OMPI_REMOTE_OFFLOADING
				if (strcmp(reqmodules, "all") != 0)
					fprintf(stderr, "[ompiconf warning]: remote offloading enabled; information for "
					                "all modules will be shown.\n");
				reqmodules = NULL;
#endif
				showdevinfo = true; 
				break;
			case OPTION(check):     
				if (!internal) goto UNSUPPORTED;
				module_query = strndup(val, strlen(val)); 
				checkmodule = true; 
				break;
#ifdef OMPI_REMOTE_OFFLOADING
			case OPTION(roffaction): 
				if (!internal) goto UNSUPPORTED;
				handle_roff_action(val);
				break;
#endif
			default:
				if (parameter[0] == '-')
				{
					switch (parameter[1])
					{
						case 'v':
							BEVERB:
							verbose = true;
							break;
						default:
							goto UNSUPPORTED;
					}
				} else {
					UNSUPPORTED:
					fprintf(stderr, "Command not supported, run %s with no arguments for help.\n", 
									 ompiconf_bin);
					exit(0);
				}
		}
		
		argc = argc - 1;
		argv = argv + 1;
	}
}

int main(int argc, char *argv[])
{
	int check_failed;

#ifdef PORTABLE_BUILD
	char argv0[PATHSIZE], path[PATHSIZE];
	bool argv0_haspath = false;

	strncpy(argv0, argv[0], PATHSIZE - 1);

	if (is_path(argv0))
		argv0_haspath = true;

#if defined(HAVE_REALPATH)
	char  *res;
	
	res = realpath(argv[0], argv0);
	if (res != NULL) /* ompiconf is in PATH */
#else
	if (argv0_haspath)
#endif
		get_path(argv0, path, PATHSIZE);

	str LibDir_tmp = Strnew();
	/* Code for making ompi portable in Linux. This wont work on other OSes.
	 */
	char buffer[PATHSIZE];

	if (argv0_haspath)
		strncpy(buffer, path, PATHSIZE - 1);
	else
	{
	#ifdef HAVE_READLINK
		ssize_t len = readlink("/proc/self/exe", buffer, PATHSIZE - 1);
		if (len == -1)
		{
			fprintf(stderr, "[ompiconf error]: couldn't retrieve installation path using readlink.\n");
			exit(EXIT_FAILURE);
		}
		else
		{
			if (len == PATHSIZE - 1)
			{
				fprintf(stderr, "path to %s too long.\n", PKG_NAME);
				exit(EXIT_FAILURE);
			}
			else
				buffer[len] = '\0';
		}
	#else
		fprintf(stderr, "[ompiconf error]: please provide the full path to %s.\n", argv[0]);
		exit(EXIT_FAILURE);
	#endif
	}

	/* Get the path and get rid of the trailing "bin/" */
	get_path(buffer, InstallPath, PATHSIZE);
	if (strcmp(InstallPath + strlen(InstallPath) - 4, "bin/"))
	{
		fprintf(stderr, "[ompiconf error]: invalid installation path for a portable build.\n");
		exit(EXIT_FAILURE);
	}
	InstallPath[strlen(InstallPath) - 4] = 0;

	/* Construct LibDir */
	if (strlen(InstallPath) + 8 + strlen(PKG_NAME) + 1 > PATHSIZE)
	{
		fprintf(stderr, "[ompiconf error]: path to %s too long.\n", PKG_NAME);
		exit(EXIT_FAILURE);
	}
	str_printf(LibDir_tmp, "%s%s", InstallPath, PKG_LIBDIR);
	strncpy(LibDir, str_string(LibDir_tmp), PATHSIZE - 1);

	/* Construct ModuleDir (<LibDir>/devices) */
	if (strlen(LibDir) + strlen("/devices") + 1 > PATHSIZE)
	{
		fprintf(stderr, "[ompiconf error]: path to %s too long.\n", PKG_NAME);
		exit(EXIT_FAILURE);
	}
	str_printf(LibDir_tmp, "/devices");
	strncpy(ModuleDir, str_string(LibDir_tmp), PATHSIZE - 1);

	str_free(LibDir_tmp);
#endif

	parse_args(argc, argv);

	if (reqmodules)
		orig_reqmodules = strdup(reqmodules);
		
	ompiconf_modules_employ(NULL, &all_modulenames, &all_nmodules);
	ompiconf_modules_employ(reqmodules, &oc_modulenames, &oc_nmodules);
	
#ifdef OMPI_REMOTE_OFFLOADING
	ompiconf_remote_modules_employ();
#endif

	if (argc == 1)
	{
		ompi_info();
		ompi_help(argv[0]);
		exit(0);
	}

	if (showdevinfo)
	{
		ompiconf_modules_show_info(verbose);
#ifdef OMPI_REMOTE_OFFLOADING
		ompiconf_remote_modules_finalize();
#endif
		if (orig_reqmodules)
			free(orig_reqmodules);
		exit(0);
	}

	if (internal)
	{
#ifdef OMPI_REMOTE_OFFLOADING
		if (getnprocs) /* remote offloading query */
		{
			printf("%d\n", ompiconf_get_num_mpiprocs() + 1); /* master + workers */
			exit(0);
		}
		
		if (getmpihosts)
		{
			printf("%s\n", ompiconf_get_node_names());
			exit(0);
		}
		
		if (creatempirunscript)
		{
			ompiconf_create_mpirun_script();
			exit(0);
		}
#endif
		
		if (checkmodule)
		{
			check_failed = ompiconf_modules_query(module_query);
			free(module_query);
			exit(check_failed);
		}
	}

#ifdef OMPI_REMOTE_OFFLOADING
	ompiconf_remote_modules_finalize();
#endif
	
}
