/*
  OMPi OpenMP Compiler
  == Copyright since 2001, the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* OMPICONF_MODULES.C
 * Module support for ompiconf
 */

/* 
 * December 2022:
 *   Created out of code in ompicc/kernels.c.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "config.h"
#include "modules.h"
#include "stddefs.h"
#include "str.h"
#ifdef OMPI_REMOTE_OFFLOADING
	#include "roff_config.h"
#endif

char **oc_modulenames, **all_modulenames;  /* Pointers to module names */
int  oc_nmodules, all_nmodules;


void ompiconf_modules_employ(char *modstring, char ***modnames, int *nmodules)
{
	char *s;
	int  i;

	/* Pass 1: find # modules */
#ifndef OMPI_REMOTE_OFFLOADING
	if ((!modstring) || (modstring && strcmp(modstring, "all") == 0))
		modstring = strdup(MODULES_CONFIG);
#else
	modstring = strdup(MODULES_CONFIG);
#endif
	for (; isspace(*modstring) || *modstring == ','; modstring++)
		;
	if (*modstring == 0)
		return;
	for ((*nmodules) = 1, s = modstring; *s; s++)
	{
		if (isspace(*s) || *s == ',')
		{
			for (*s = 0, s++; isspace(*s) || *s == ','; s++)
				*s = ',';  /* all spaces become commas */
			if (*s)
				(*nmodules)++;
			s--;
		}
	}

	/* Pass 2: fix pointers */
	if (((*modnames) = (char **) malloc((*nmodules)*sizeof(char *))) == NULL)
	{
		fprintf(stderr, "cannot allocate memory");
		exit (1);
	}
	for (i = 0, s = modstring; i < (*nmodules); i++)
	{
		for ((*modnames)[i] = s++; *s; s++)
			;
		if (i == (*nmodules)-1)
			break;
		for (; *s == 0 || *s == ','; s++)
			;
	}
}

#ifdef OMPI_REMOTE_OFFLOADING

void ompiconf_remote_modules_finalize(void)
{
	roff_config_finalize();
}


static void add_suffix_to_modulename(roff_config_module_t *module, void *data)
{
	str tmp = Str(module->name);
	size_t newsize = ROFFCONF_MAXNAME + 8;
	module->name = realloc(module->name, newsize * sizeof(char));
	if (module->name == NULL)
	{
		perror("add_suffix_to_modulename():");
		exit(EXIT_FAILURE);
	}
	snprintf(module->name, newsize - 1, "%s_node%d", str_string(tmp), module->node->id);
	str_free(tmp);
}


void ompiconf_remote_modules_employ(void)
{
	roff_config_initialize(DONT_IGNORE_DISABLED_MODULES, false); 
	
	/* Append "_nodeX" suffix to each remote module */
	roff_config_iterate(add_suffix_to_modulename, NULL);
}


static void roff_print_information(roff_config_module_t *mod, int devoffset)
{
	int k;
	
	fprintf(stderr, "OMPi %s remote device module.\n", mod->name);
	fprintf(stderr, "Available devices : %d\n\n", mod->num_devices);
	for (k = 0; k < mod->num_devices; k++)
	{
		fprintf(stderr, "  device id < %d > { \n", devoffset+k);
		fprintf(stderr, "    node: %s\n", mod->node->name);
		fprintf(stderr, "  }\n");
	}
}

#if 0
static int modstr_contains_module_name(const char* modstr, const char* name)
{
	int i, j = 0;
	int found = 0, last = 0;
	int sim = 0;
	int target = strlen(name);

	for (i = 0; i < strlen(modstr) + 1; i++) 
	{
		if (modstr[i] == ' ' || modstr[i] == '\0') 
		{
			if (sim == target) 
			{ 
				found = 1; 
				break; 
			}
			else 
				sim = j = 0;
			continue;
		}

		if (modstr[i] == name[j]) 
			sim++; 

		j++;
	}

	return found;
}
#endif

void ompiconf_remote_modules_show_info(int devoffset, int verbose)
{
	int   i, j, k, numdevs;
	roff_config_node_t *node;
	roff_config_module_t *module;

	for (i = 0; i < roff_config.num_nodes; i++)
	{
		node = &(roff_config.nodes[i]);
		for (j = 0; j < node->num_modules; j++)
		{
			module = &(node->modules[j]);
			numdevs = module->num_devices;

			if (!module->is_disabled)
			{
				str portstr = Str("");
				// if (node->port != 22) 
				// 	str_printf(portstr, ":%d", node->port);

				if (verbose)
					fprintf(stderr, "MODULE [%s] (%s):\n------\n", 
									module->name, node->name);
				else
					fprintf(stderr, "  MODULE [%s] (remote at %s%s) provides device(s) : ", 
					                module->name, node->name, str_string(portstr));
					
				if (verbose)
				{
					roff_print_information(module, devoffset);
				}
				else
				{
					for (k = 0; k < numdevs; k++)
						fprintf(stderr, "%d ", devoffset+k);
					fprintf(stderr, "\n");
				}
				
				devoffset += numdevs;
				if (verbose)
					fprintf(stderr, "------\n\n");
			}
		}
	}

}


int ompiconf_get_num_mpiprocs(void)
{
	return roff_config.num_nodes; /* for now */
}

char *ompiconf_get_node_names(void)
{
	return roff_config_get_node_names(); 
}

/* Outputs an ompi-mpirun.sh script for executing a
 * remote offloading application. 
 * Note: Used when ROFF_USE_STATIC_MPI_PROCS is defined.
 */
void ompiconf_create_mpirun_script(void)
{
	FILE *fp;
	char *scriptname = PKG_NAME "-mpirun.sh";
	char command[128];
	
	snprintf(command, 127, "chmod +x %s", scriptname);
	
	fp = fopen(scriptname, "w");
	if (fp == NULL)
	{
		fprintf(stderr, "could not open %s\n", scriptname);
		exit(EXIT_FAILURE);
	}

	fprintf(fp, "#!/bin/bash\n\n");
	fprintf(fp, "# Custom mpirun script\n");
	fprintf(fp, "# Generated automatically by ompiconf\n#\n");
	fprintf(fp, "# Usage: ./%s [mpirun-options] <executable>\n\n", scriptname);
	fprintf(fp, "NPROCS=%d\n", ompiconf_get_num_mpiprocs() + 1);
	fprintf(fp, "HOSTS=%s\n", ompiconf_get_node_names());
	fprintf(fp, "mpiexec -n $NPROCS --host $HOSTNAME,$HOSTS \"$@\"\n");
	fclose(fp);
	
	if (system(command) != 0)
	{
		fprintf(stderr, "error chaning %s permissions\n", scriptname);
		exit(EXIT_FAILURE);
	}
}

#endif


#ifdef HAVE_DLOPEN

#include <dlfcn.h>

static void *module_open(char *name)
{
	void *handle;
	str tmp = Strnew();

	/* Check current folder */
	str_printf(tmp, "./%s.so", name);
	handle = dlopen(str_string(tmp), RTLD_LAZY);
	if (handle)
	{
		str_free(tmp);
		return handle;
	}

	/* Check ompi's library folder */
	str_truncate(tmp);
	str_printf(tmp, "%s/%s/hostpart.so", ModuleDir, name);
	handle = dlopen(str_string(tmp), RTLD_LAZY);
	if (handle)
	{
		str_free(tmp);
		return handle;
	}

	/* Finally check system's library folder */
	str_truncate(tmp);
	str_printf(tmp, "%s.so", name);
	handle = dlopen(str_string(tmp), RTLD_LAZY);

	str_free(tmp);
	return handle;
}


void ompiconf_modules_show_info(int verbose)
{
	int   i, j, md, ndev, (*get_num_devices)(), (*finalize)();
	void  *modh, (*print_info)(int, str);
	char  *error;
	void (*register_str_printf)(int  (*str_printf_in)(str, char *, ...));
	str info = Strnew();
	
#ifndef OMPI_REMOTE_OFFLOADING
	fprintf(stderr, "%d configured device module(s)%s %s\n",
		            all_nmodules, all_nmodules > 0 ? ":" : "", MODULES_CONFIG);
#else
	fprintf(stderr, "%d configured local device module(s)%s %s\n",
		            all_nmodules, all_nmodules > 0 ? ":" : "", MODULES_CONFIG);
	fprintf(stderr, "%d configured remote device module(s): %s%s\n", 
	                roff_config.nuniquemodules + (roff_config.has_cpu_module), 
	                roff_config.has_cpu_module ? "cpu " : "", 
	                str_string(roff_config.modstr) ? str_string(roff_config.modstr) : "");
#endif
	if (!orig_reqmodules || (orig_reqmodules && !strcmp(orig_reqmodules, "all")))
		fprintf(stderr, "\n");
	if (orig_reqmodules && strcmp(orig_reqmodules, "all") != 0)
		fprintf(stderr, "showing information for %d device module(s): %s\n\n",
		                oc_nmodules, orig_reqmodules);

	for (i = ndev = 0; i < oc_nmodules; i++)
	{
		if (verbose)
			fprintf(stderr, "MODULE [%s]:\n------\n", oc_modulenames[i]);
		else
			fprintf(stderr, "  MODULE [%s] provides device(s) : ", oc_modulenames[i]);
		
		modh = module_open(oc_modulenames[i]);
		if (!modh)
		{
			fprintf(stderr, "module failed to open.\n");
			if (verbose)
				fprintf(stderr, "  [ reported error: %s ]\n", dlerror());
		}
		else
		{
			if (verbose)
			{
				register_str_printf = dlsym(modh, "hm_register_str_printf");
				if ((error = dlerror()) != NULL)
					fprintf(stderr, "%s\n", error);
				else
				{
					register_str_printf(str_printf);
					print_info = dlsym(modh, "hm_print_information");
					if ((error = dlerror()) != NULL)
						fprintf(stderr, "%s\n", error);
					else
					{
						str_truncate(info);
						print_info(ndev, info);
						fprintf(stderr, "%s", str_string(info));
					}
				}
			}

			get_num_devices = dlsym(modh, "hm_get_num_devices");
			finalize = dlsym(modh, "hm_finalize");
			if ((error = dlerror()) != NULL)
				fprintf(stderr, "%s\n", error);
			else
			{
				md = get_num_devices();
				finalize();
			}
			
			if (!verbose)
			{
				for (j = 0; j < md; j++)
					fprintf(stderr, "%d ", ndev+j);
				fprintf(stderr, "\n");
			}
			ndev += md;
			dlclose(modh);
		}
		if (verbose)
			fprintf(stderr, "------\n\n");
	}
	
#ifdef OMPI_REMOTE_OFFLOADING
	ompiconf_remote_modules_show_info(ndev, verbose);
	ndev += roff_config.num_devices;
#endif

	if (verbose)
		fprintf(stderr, "Total number of available devices: %d\n", ndev);

	if (ndev == 0)
		fprintf(stderr, "  No queried devices found.\n");
		
	str_free(info);
}

#else

void ompiconf_modules_show_info(int verbose)
{
	fprintf(stderr, "Unfortunately, there is no support for modules.\n");
#ifdef OMPI_REMOTE_OFFLOADING
	ompiconf_remote_modules_show_info(verbose, 0);
#endif
}

#endif

static int get_device_count(char *name)
{
	int count = -1;
	void *modh;
	int (*func_handle)(), (*fin_handle)();

	modh = module_open(name);
	if (modh)
	{
		func_handle = dlsym(modh, "hm_get_num_devices");
		fin_handle = dlsym(modh, "hm_finalize");
		count = func_handle();
		fin_handle();
		dlclose(modh);
	}
	return count;
}

int ompiconf_modules_query(char *modstr)
{
	int count, actual_count, failed_check = 0;
	char *modname, *devs, *ref = modstr;

	if (!modstr || strlen(modstr) == 0) 
	{
		fprintf(stderr, "Option '--check' must be followed by a valid, "
		                "non-empty string containing module names.\n");
		exit(0);
	}

	fprintf(stderr, "Running query for module(s): %s\n", modstr);

	while ((modname = strtok_r(ref, " ", &ref)))
	{
		devs = strtok_r(ref, " ", &ref);
		if (!modname || !devs)
		{
			fprintf(stderr, "Invalid query provided, exiting.\n");
			exit(1);
		}

		/* Skip check for host (CPU) module */
		if (!strncmp(modname, "cpu", 3))
			continue;
			
		if (!isdigit(modname[0]) && (count = atoi(devs)) > 0)
		{
			if ((actual_count = get_device_count(modname)) == count)
				printf ("Query [%s %s]: CHECK.\n", modname, devs);
			else
			{
				printf ("Query [%s %s]: FAIL (got \"%d\").\n", modname, devs, actual_count);
				failed_check = 1;
			}
		}
		else 
		{
			fprintf(stderr, "Invalid query provided, exiting.\n");
			exit(1);
		}
	}

	return failed_check;
}
