/*
  OMPi OpenMP Compiler
  == Copyright since 2001, the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* optioncs.c
 * Handle arguments/options to ompicc and _ompi
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/stat.h>
#include "ompicc.h"
#include "kernels.h"
#include "help.h"
#include "options.h"
#include "assorted.h"


/*
 * General options
 */
 
 
#define OMPI_OPTNAME(opt) "-" #opt
#define OPTNAME(opt)      "--" #opt
#define OPTNAME_V(opt)    "V--" #opt "="
#define OPTION(opt)       OPT_##opt

typedef enum {
	OPTION(unknown) = -1, /* unknown option */
	
	OPTION(dummy1),  /* Info/help options */
	OPTION(version),     OPTION(help),        OPTION(options),     
	OPTION(info),
	
	OPTION(dummy2),  /* Code generation and runtime options */
	OPTION(ort),         OPTION(devs),        OPTION(devopt),
	OPTION(taskopt),     OPTION(reduction),   OPTION(comm),
	OPTION(portable),    OPTION(initort), 
	OPTION(bubins),      OPTION(nobubins),

	OPTION(dummy3),  /* Debug/inspect options */
	OPTION(verbose),     OPTION(keep),        OPTION(keepall),
	OPTION(nolineno),    OPTION(dbg),
	
	OPTION(dummy4),  /* Feature options */
	OPTION(scopecheck),  OPTION(carstats),
	
	OPTION(dummy5),  /* Developer options */
	OPTION(nomp),        OPTION(nox),         OPTION(nomakefile),
	OPTION(gdb),         OPTION(valgrind),    OPTION(complete),
	OPTION(kernel),
	
	OPTION(lastoption)    /* marker */
} option_t;

char *optnames[] = {
	NULL, /* NULL signifies section change */
	OPTNAME(version),     OPTNAME(help),      OPTNAME(options),     
	OPTNAME(info),

	NULL,
	OPTNAME_V(ort),       OPTNAME_V(devs),    OPTNAME(devopt),
	OPTNAME_V(taskopt),   OPTNAME_V(reduction), OPTNAME_V(comm),
	OPTNAME(portable),    OPTNAME(initort),
	OPTNAME(bundle-kernels),
	OPTNAME(no-bundle-kernels),

	NULL,
	OPTNAME(verbose),     OPTNAME(keep),        OPTNAME(keepall),
	OPTNAME(nolineno),    OPTNAME(dbg),
	
	NULL,
	OPTNAME(scopecheck),  OPTNAME(carstats),
	
	NULL,
	OPTNAME(nomp),        OPTNAME(nox),         OPTNAME(nomakefile),
	OPTNAME(gdb),         OPTNAME(valgrind),    OPTNAME_V(complete),
	OPTNAME_V(kernel),    
	
	NULL
};

char *taskargnames[] = { "size", "speed", "speed+", NULL };
char *redargnames[]  = { "old", "default", "rtlib", NULL };
char *infoargnames[]  = { "env", "directive", "clause", "find", NULL };

char *optinfo[][2] = {   /* possible values, option description */
	{ "Info/help options" },  /* Used as title in section changes */
		{ "", "show ompicc version" },
		{ "", "show short OMPi options info" },
		{ "", "show all OMPi options" },
		{ " [args]", "show OpenMP-related help" },
	
	{ "Code generation and runtime options" },
		{ "<eelib>", "threading library to use" },
		{ "<module,module,...>", "target only the named device modules" },
		{ " [option]", "give device option" },
		{ "[size|speed|speed+]", "default: speed" },
		{ "[default|rtlib|old]", "" },
		{ "<commlib>", "communication library to use for remote offloading" },
		{ "", "enable portable compilation" },
		{ "", "force libort initialization" },
		{ "", "enable bundling of kernel files" },
		{ "", "disable bundling of kernel files" },
  
	{ "Debug/inspect options" },
		{ "", "detail every step of the compilation process" },
		{ "", "keep main intermediate file" },
		{ "", "keep all intermediate files" },
		{ "", "disable # <line> in intermediate files" },
		{ "", "force compiler to display some debug info" }, 
	
	{ "Feature options" },
		{ "", "enable autoscoping analysis" }, 
		{ "", "turn on C.A.R.S. kernel analysis" },
	
	{ "Developer options" },
		{ "", "disable OpenMP" }, 
		{ "", "disable OMPi extensions" },
		{ "", "" }, 
		{ "", "runs _ompi from within gdb" }, 
		{ "", "runs _ompi from within valgrind" }, 
		{ "", "" },
		{ "", "" },
	{ NULL }
};


static option_t optid(char *arg, char **val)
{
	int i;

	for (i = 0; i < OPTION(lastoption); i++)
	{
		if (!optnames[i])   /* Skip dummy options */
			continue;
		if (optnames[i][0] == 'V')     /* Option with value */
		{
			if (strncmp(optnames[i]+1, arg, strlen(optnames[i])-1) == 0)
			{
				*val = arg + strlen(optnames[i]) - 1;
				return ((option_t) i);
			}
		}
		else
			if (strcmp(optnames[i], arg) == 0)
				return ((option_t) i);
	}
	return ( OPTION(unknown) );
}

#define INITWIDTH 21

static void showallopts()
{
	int i, space;
	char *realopt;
	
	ompi_shortinfo();
	fprintf(stderr, "\n[ AVAILABLE OPTIONS ]\n");
	for (i = 0; i < OPTION(lastoption); i++)
		if (optnames[i])
		{
			space = 0;
			realopt = optnames[i]+(optnames[i][0]=='V' ? 1 : 0);
			if (strcmp(realopt, OPTNAME(verbose)) == 0)
				{ fprintf(stderr, "  -v | "); space += 7; }
			else if (strcmp(realopt, OPTNAME(keepall)) == 0)
				{ fprintf(stderr, "  -K | "); space += 7; }
			else if (strcmp(realopt, OPTNAME(keep)) == 0)
				{ fprintf(stderr, "  -k | "); space += 7; }
			else if (strcmp(realopt, OPTNAME_V(complete)+1) == 0)
				continue;
			else if (strcmp(realopt, OPTNAME_V(kernel)+1) == 0)
				continue;
			else
				{ fprintf(stderr, "  "); space += 2; }
			
			space += fprintf(stderr, "%s%s", realopt, optinfo[i][0]);
			if (space > INITWIDTH && strlen(optinfo[i][1]))
			{
				fprintf(stderr, "\n");
				space = 0;
			}
			if (strlen(optinfo[i][1]))
				fprintf(stderr, "%*s (%s)", INITWIDTH-space+1, "", optinfo[i][1]);
			fprintf(stderr, "\n");
		}
		else
			fprintf(stderr, "\n%s\n%.*s\n", optinfo[i][0], (int)strlen(optinfo[i][0]), 
			       "------------------------------------------------------");
}


/* 
 * Device options 
 */


#define DEVOPTNAME(opt)   "--" #opt
#define DEVOPTNAME_V(opt) "V--" #opt "="
#define DEVOPTNAME_C(opt) "V--" #opt "-"
#define DEVOPTION(opt)    DEVOPT_##opt

static bool devoptsused = false;

typedef enum {
	DEVOPTION(unknown) = -1, /* unknown option */

	DEVOPTION(dummy1),  /* Jobs */
	DEVOPTION(jobs), DEVOPTION(jobs_V),

	DEVOPTION(dummy2),  /* Code target options*/
	DEVOPTION(cuda), DEVOPTION(opencl), DEVOPTION(vulkan),

	DEVOPTION(lastoption)    /* marker */
} devoption_t;

char *devoptnames[] = {
	NULL,
	DEVOPTNAME(jobs), DEVOPTNAME_V(jobs),

	NULL,
	DEVOPTNAME_C(cuda), DEVOPTNAME_C(opencl), DEVOPTNAME_C(vulkan),

	NULL
};

typedef struct devoptlist_
{
	char *name;
	char **opts;
	unsigned int numopts;
} devoptlist_t;

#define _DEVOPTLIST(name) devoptlist_t name ## _devoptlist = { #name, NULL, 0 }

_DEVOPTLIST(cuda);
_DEVOPTLIST(opencl);
_DEVOPTLIST(vulkan);

/* Allocate/grow a devoptlist, and add an option to it. 
 */
static void _devoptlist_add(devoptlist_t *list, char *opt)
{
	if ((list->numopts % 16) == 0)
		list->opts = (list->opts == NULL) ? 
		                smalloc(16 * sizeof(char *)) : 
		                srealloc(list->opts, (list->numopts + 16) * sizeof(char *));

	list->opts[list->numopts++] = opt;
}


/* Print all list options to a str (should be initialized).
 */
static void _devoptlist_print_args(str s, devoptlist_t *list)
{
	int i;
	for (i = 0; i < list->numopts; i++)
		str_printf(s, "%s-%s-%s", i ? " -" : "-", list->name, list->opts[i]);
	if (list->numopts) 
		str_printf(s, " "); /* safety */
}


static
int clamp(int x, int min, int max)
{
	return ((x > max) ? (max) : ((x < min) ? (min) : (x)));
}


static
bool is_number(char *str)
{
	int i;
	for (i = 0; i < strlen(str); i++)
		if (!isdigit(str[i]))
			return false;
	return true;
}


option_t devoptid(char *arg, char **val, bool *valexists)
{
	int i;
	*valexists = false;

	for (i = 0; i < DEVOPTION(lastoption); i++)
	{
		if (!devoptnames[i])   /* Skip dummy options */
			continue;
		if (devoptnames[i][0] == 'V')     /* Option with value */
		{
			if (strncmp(devoptnames[i]+1, arg, strlen(devoptnames[i])-1) == 0)
			{
				*val = arg + strlen(devoptnames[i]) - 1;
				*valexists = true;
				return ((devoption_t) i);
			}
		}
		else
			if (strcmp(devoptnames[i], arg) == 0)
				return ((devoption_t) i);
	}
	return ( DEVOPTION(unknown) );
}


static void show_alldevopts()
{
	ompi_shortinfo();
	fprintf(stderr, "\nSyntax: --devopt <option> [--devopt <option> ...]\n");
	fprintf(stderr, "\n[ Available device options ]\n");
	fprintf(stderr, "  -j[N] | --jobs[=N]   "
		      "(deploy (max N) parallel make jobs for building kernels)\n");
	fprintf(stderr, "  --vulkan-require=int8|16|64\n"
	                "                       "
	        "(enables explicit integer types in Vulkan devices)\n");
}


void handle_devopt(int argc, char **argv)
{
	char *val, *optval;
	bool valexists = false;

	if (argc == 1)
		goto UNSUPPORTED_DEVOPT;

	/* Skip "--devopt" */
	argv++;
	val = strdup(argv[0]);

	switch ( devoptid(val, &optval, &valexists) )
	{
		case DEVOPTION(jobs):
			reqjobs = 0;
			break;
		case DEVOPTION(jobs_V):
		{
			if (strlen(optval) == 0 || !is_number(optval)) 
				goto UNSUPPORTED_DEVOPT;

			reqjobs = clamp(atoi(optval), 0, 99);
			break;
		}
		case DEVOPTION(cuda):
			_devoptlist_add(&cuda_devoptlist, strdup(optval));
			break;
		case DEVOPTION(opencl):
			_devoptlist_add(&opencl_devoptlist, strdup(optval));
			break;
		case DEVOPTION(vulkan):
			_devoptlist_add(&vulkan_devoptlist, strdup(optval));
			break;
		default:
		{
			if (val[0] == '-')
			{
				switch (val[1])
				{
					case 'j':
					{
						if (strlen(val) == 2)
						{
							reqjobs = 0;
							break;
						}

						if (!is_number(val = val+2)) 
							goto UNSUPPORTED_DEVOPT;

						reqjobs = clamp(atoi(val), 0, 99);
						break;
					}
					default:
						goto UNSUPPORTED_DEVOPT;
						break;
				}
			}
			else 
			{
				UNSUPPORTED_DEVOPT:
				show_alldevopts();
				_exit(1);
			}	
			break;
		}
	}	

	devoptsused = true;
}


/*
 * _ompi options
 */
 
 
typedef struct _ompi_opt_
{
	option_t id;
	char    *name;
	bool     used;
} _ompi_opt_t;


/* ompicc <-> _ompi option mapping */
_ompi_opt_t _ompi_opts[] = {
	{ OPTION(taskopt),    OMPI_OPTNAME(taskopt),     false },
	{ OPTION(reduction),  "",                        false },
	{ OPTION(portable),   OMPI_OPTNAME(portable),    false },
	{ OPTION(nolineno),   OMPI_OPTNAME(nolineno),    false },
	{ OPTION(initort),    OMPI_OPTNAME(initort),     false },
	{ OPTION(dbg),        OMPI_OPTNAME(showdbginfo), false },
	{ OPTION(scopecheck), OMPI_OPTNAME(autoscope),   false },
	{ OPTION(carstats),   OMPI_OPTNAME(drivecar),    false },
	{ OPTION(nomp),       OMPI_OPTNAME(nomp),        false },
	{ OPTION(nox),        OMPI_OPTNAME(nox),         false },
	{ OPTION(bubins),     OMPI_OPTNAME(bundlebins),  false },
	{ OPTION(nobubins),   OMPI_OPTNAME(dummy),       false },
	{ OPTION(lastoption), NULL,                      false }
};

/* Marks an _ompi option as `used' */
static void _ompi_opt_setused(option_t oid)
{
	int i;
	for (i = 0; _ompi_opts[i].id != OPTION(lastoption); i++)
		if (_ompi_opts[i].id == oid)
		{
			_ompi_opts[i].used = true;
			return;
		}
}


/* Builds the _ompi options string, which is passed when running _ompi 
 */
void build_ompi_optstr(str oopts, char *fname)
{
	int i;
	
	/* (1) Reset */
	str_truncate(oopts);
	
	/* (2) Check OMPi CFLAGS and append _ompi options, accordingly */
	if (strstr(CFLAGS, "OMPI_MAIN=LIB"))
		str_printf(oopts, " -nomain ");
    
	if (strstr(CFLAGS, "OMPI_MEMMODEL=PROC+THR"))
		str_printf(oopts, " -procs -threads ");   
	else 
		if (strstr(CFLAGS, "OMPI_MEMMODEL=PROC"))
			str_printf(oopts, " -procs ");
		else 
			if (strstr(CFLAGS, "OMPI_MEMMODEL=THR"))
				str_printf(oopts, " -threads ");
	
	/* (3) Iterate through all used ompicc options and append
	 * the corresponding _ompi ones */
	for (i = 0; _ompi_opts[i].id != OPTION(lastoption); i++)
	{
		if (_ompi_opts[i].used && _ompi_opts[i].name) 
		{
			switch (_ompi_opts[i].id)
			{
				/* Append the taskopt level */
				case OPTION(taskopt):
					str_printf(oopts, " %s%d ", _ompi_opts[i].name, taskoptlevel);
					break;
				/* Requires manual handling */
				case OPTION(reduction):
					if (reductionOld)
						str_printf(oopts, " -oldred ");
					else if (reductionRTL)
						str_printf(oopts, " -rtlibred ");
					break;
				/* Every other argument */
				default:
					str_printf(oopts, " %s ", _ompi_opts[i].name);
					break;
			}
		}
	}
	
	/* (4) Always include the `--usemod' options */
	str_printf(oopts, " %s ", modules_argfor_ompi());

	/* (5) Append all `--<codetarg>=<value>' options */
	_devoptlist_print_args(oopts, &cuda_devoptlist);
	_devoptlist_print_args(oopts, &opencl_devoptlist);
	_devoptlist_print_args(oopts, &vulkan_devoptlist);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 *    GET OMPICC COMMAND-LINE OPTIONS                                *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


static char **avail_eelibs;
static int    num_eelibs;

/* Get all configured EELibs, for auto-completion purposes */
static void _get_available_eelibs()
{
	int i;
	char *ortstr, *s;
	ortstr = strdup("default " ORTLIBS_CONFIG);
	
	for (; isspace(*ortstr) || *ortstr == ','; ortstr++)
		;
	if (*ortstr == 0)
		return;
	for (num_eelibs = 1, s = ortstr; *s; s++)
	{
		if (isspace(*s) || *s == ',')
		{
			for (*s = 0, s++; isspace(*s) || *s == ','; s++)
				*s = ',';  /* all spaces become commas */
			if (*s)
				num_eelibs++;
			s--;
		}
	}

	if ((avail_eelibs = (char **) malloc(num_eelibs*sizeof(char *))) == NULL)
		ompicc_error(1, "cannot allocate auto-completion memory.\n");

	for (i = 0, s = ortstr; i < num_eelibs; i++)
	{
		for (avail_eelibs[i] = s++; *s; s++)
			;
		if (i == num_eelibs-1)
			break;
		for (; *s == 0 || *s == ','; s++)
			;
	}
}


static int fok(char *fname)
{
	struct stat buf;
	return (stat(fname, &buf) == 0);
}


static arg_t *new_arg(char opt, char *val)
{
	arg_t *p;

	if ((p = (arg_t *) malloc(sizeof(arg_t))) == NULL)
		ompicc_error(-1, "malloc() failed\n");
	p->opt = opt;
	if (val != NULL)
		strcpy(p->val, val);
	else p->val[0] = 0;
	p->next = NULL;
	return p;
}


void arglist_add(arglist_t *l, arg_t *arg)
{
	if (l->head == NULL)
		l->head = l->tail = arg;
	else
	{
		l->tail->next = arg;
		l->tail = arg;
	}
}


int append_arg(arglist_t *l, int argc, char **argv, int proceed)
{
	char opt, val[SLEN];
	arg_t *p;

	val[0] = 0;
	if (argv[0][0] == 0)
		return 0;
	if (argv[0][0] != '-')
	{
		p = new_arg(0, *argv);
		arglist_add(l, p);
		return 0;
	}
	opt = argv[0][1];

	if (argv[0][2] != 0)
	{
		strcpy(val, &argv[0][2]);
		p = new_arg(opt, val);
		arglist_add(l, p);
		return 0;
	}
	else
	{
		if (proceed && argc > 1)
			strcpy(val, &argv[1][0]);
		p = new_arg(opt, val);
		arglist_add(l, p);
		return proceed && argc > 1;
	}
}


void get_user_options(int argc, char **argv)
{
	int  d, ortlib = 0, commlib = 0, oid, dev_oid;
	char *parameter, *val, *compval, *ompibin = argv[0];
	extern void get_ort_flags(), get_comm_flags();

	_get_available_eelibs();     /* Get all configured EELibs */

	argv++;
	argc--;

	while (argc)
	{
		d = 0;
		switch ( oid = optid(parameter = argv[0], &val) )
		{
			case OPTION(complete):
			{
				int i;
				char *tmpmods = strdup(MODULES_CONFIG);
				modules_employ(tmpmods);
				switch ( dev_oid = optid(val, &compval) )
				{
					/* Autocomplete --devs=... */
					case OPTION(devs):
					{
						for (i = 0; i < nmodules; i++)
							if (strncmp(modulenames[i], compval, strlen(compval)) == 0)
								fprintf(stderr, "%s%s\n", optnames[dev_oid]+1, modulenames[i]);
						break;
					}
					/* Autocomplete --taskopt=... */
					case OPTION(taskopt):
					{
						for (i = 0; taskargnames[i] != NULL; i++)
							if (strncmp(taskargnames[i], compval, strlen(compval)) == 0)
								fprintf(stderr, "%s%s\n", optnames[dev_oid]+1, taskargnames[i]);
						break;
					}
					/* Autocomplete --reduction=... */
					case OPTION(reduction):
					{
						for (i = 0; redargnames[i] != NULL; i++)
							if (strncmp(redargnames[i], compval, strlen(compval)) == 0)
								fprintf(stderr, "%s%s\n", optnames[dev_oid]+1, redargnames[i]);
						break; 
					}
					/* Autocomplete --ort=... */
					case OPTION(ort):
					{
						for (i = 0; i < num_eelibs; i++)
							if (strncmp(avail_eelibs[i], compval, strlen(compval)) == 0)
								fprintf(stderr, "%s%s\n", optnames[dev_oid]+1, avail_eelibs[i]);
						break;
					}
					default:
					{
						bool valempty = false;
						char *letteropts = "clLIDUokKv";
						
						for (i = 0; i < strlen(val); i++)
						{
							if (isspace(val[i])) 
							{
								valempty = true;
								break;
							}
						}
						if (val[0] == '-' && strlen(val) <= 2)
						{
							if (strlen(val) == 1)
							{
								for (i = 0; i < strlen(letteropts); i++)
									fprintf(stderr, "-%c\n", letteropts[i]);
							}
							else if (strchr(letteropts, val[1]) && strcmp(val, "--"))
							{
								fprintf(stderr, "%s\n", val);
								break;
							}
						}
						for (i = 0; i != OPTION(lastoption); i++)
						{
							if (!optnames[i])
								continue;

							if (!strncmp(optnames[i]+(optnames[i][0] == 'V' ? 1 : 0), val, strlen(val)) 
							|| valempty)
								fprintf(stderr, "%s\n", 
									optnames[i][0] == 'V' ? optnames[i]+1 : optnames[i]);
						}
						
						break;
					}
				}
				free(tmpmods);
				_exit(0);
				break;
			}
			case OPTION(help):
				strcpy(ortlibname, "default");
				strcpy(commlibname, "default_comm");
				get_ort_flags();
#ifdef OMPI_REMOTE_OFFLOADING
				get_comm_flags();
#endif
				ompi_longinfo();
				help_ompicc(ompibin);
				_exit(0);
				break;
			case OPTION(version):
				fprintf(stderr, "%s\n", PKG_VERSION);
				_exit(0);
				break;
			case OPTION(options):
				showallopts();
				_exit(0);
				break;
			case OPTION(devopt):
				handle_devopt(argc, argv);
				argc--; argv++;
				break;
			case OPTION(info):
				help_OpenMP(argc, argv);
				_exit(0);
				break;
			case OPTION(ort):
				strncpy(ortlibname, val, 511);
				ortlib = 1;
				break;
			case OPTION(comm):
				strncpy(commlibname, val, 511);
				commlib = 1;
				break;
			case OPTION(portable):
				portable_userprog = true;
				break;
			case OPTION(kernel):     
				onlykernelfile = true;
				_kernid = atoi(val);
				break;
			case OPTION(bubins):
				bundleKernels = BUNDLE_BINS;   /* someday we may also do sources */
				break;
			case OPTION(nobubins):
				bundleKernels = BUNDLE_NONE;
				break;
			case OPTION(keep):
				goto KEEP_MAIN;
			case OPTION(keepall):
				goto KEEP_ALL;
			case OPTION(verbose):
				goto BEVERB;
			case OPTION(nomakefile): makefile = false; break;
			case OPTION(nolineno):   cppLineNo = true; break;
			case OPTION(gdb):        usegdb = DBG_GDB; break;
			case OPTION(valgrind):   usegdb = DBG_VALGRIND; break;
			case OPTION(carstats):   usecarstats = true; break;
			case OPTION(taskopt):
				if (strcmp(val, "size") == 0)
					taskoptlevel = 0;
				else
					if (strcmp(val, "speed") == 0)
						taskoptlevel = 1;
					else
						if (strcmp(val, "speed+") == 0)
							taskoptlevel = 2;
						else
							ompicc_error(1,"unknown taskopt option (try 'size'/'speed'/'speed+').\n");
				break;
			case OPTION(devs):
				reqmodules = val;
				break;
			case OPTION(reduction):
				if (strcmp(val, "old") == 0)
					reductionOld = true;
				else
					if (strcmp(val, "rtlib") == 0)
					{
						reductionOld = false;
						reductionRTL = true;
					}
					else 
						if (strcmp(val, "default") == 0)
						{
							reductionOld = false;
							reductionRTL = false;
						}
						else
						  ompicc_error(1,"unknown reduction option (try 'old'/'rtlib').\n");
						  
				break;
			default:
				if (oid != OPTION(unknown))          /* option passed simply to _ompi */
					break;
				if (parameter[0] == '-')              /* option */
					switch (parameter[1])
					{
						case 'c':
							mustlink = false;
							break;
						case 'l':
							d = append_arg(&user_link_args, argc, argv, 1);
							break;
						case 'L':
							d = append_arg(&user_link_args, argc, argv, 1);
							break;
						case 'I':
							d = append_arg(&user_prep_args, argc, argv, 1);
							break;
						case 'D':
							d = append_arg(&user_prep_args, argc, argv, 1);
							break;
						case 'U':
							d = append_arg(&user_prep_args, argc, argv, 1);
							break;
						case 'o':
							d = append_arg(&user_outfile, argc, argv, 1);
							break;
						case 'k':
							KEEP_MAIN:
							keep = 1; d = 0;
							break;
						case 'K':
							KEEP_ALL:
							keep = 2; d = 0;
							break;
						case 'v':
							BEVERB:
							verbose = true; d = 0;
							break;
						default:
							d = append_arg(&user_scc_flags, argc, argv, 0);
							break;
					}
				else
				{
					d = append_arg(&user_files, argc, argv, 0);
					if (!fok(user_files.tail->val))
						ompicc_error(1, "file %s does not exist\n", user_files.tail->val);
				};
		}
		
		/* Mark option as "used" (for (no)bubins we need to do it later) */
		if ((oid > OPTION(unknown)) && (oid < OPTION(lastoption)) && 
		    (oid != OPTION(bubins)) && (oid != OPTION(nobubins)))
			_ompi_opt_setused(oid);
			
		argc = argc - 1 - d;
		argv = argv + 1 + d;
	}
	
	if (!ortlib)
		strcpy(ortlibname, "default");          /* Default lib to use */
	if (!commlib)
		strcpy(commlibname, "default_comm");    /* Default lib to use */
	if (bundleKernels)                        /* Kernel bundling */
		_ompi_opt_setused( OPTION(bubins) );
}
