/*
  OMPi OpenMP Compiler
  == Copyright since 2001, the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* OMPICC
 * A driver for the OMPi compiler.
 */

/*
 * May 2019:
 *   Code related to modules/devices and kernels moved to kernels.c.
 * July 2016:
 *   Small refactoring and more documentation.
 * Oct 2010:
 *   Improvements; correct argument quoting.
 * Apr 2010:
 *   Some improvements
 * Aug 2007:
 *   Major simplifications due to new threadprivate implementation.
 *   Removed POMP support.
 * May 2007:
 *   Fixed some definitions for compiling/linking/preprocessing.
 *   -lrt now determined at configuration time.
 *   Some less important bug fixes.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#if defined(HAVE_REALPATH)
	#include <limits.h>
#endif

#include "config.h"
#include "stddefs.h"
#include "str.h"
#include "keyval.h"
#include "ompicc.h"
#include "kernels.h"
#include "help.h"
#include "options.h"
#include "assorted.h"
#ifdef OMPI_REMOTE_OFFLOADING
	#include "roff_config.h"
#endif


/* How does ompicc decide on the flags to use when processing user programs?
 *
 * First, ompicc knows the flags used to build OMPi. These are hard-coded
 * at the time ompicc is built and include all the flags required to build
 * the last threading library.
 *
 * Second, ompicc reads the OMPI_XXX environmental variables. If any of
 * them exists, it overrides the corresponding hard-coded flags above.
 * In the end, the values of PREPROCESSOR, COMPILER, CPPFLAGS, CFLAGS,
 * LDFLAGS and ORTINFO variables below are either set to the hard-coded flags
 * or the corresponding environmental variable. These are the ones used to
 * build the user program.
 *
 * Finally, the user may pass additional flags when running ompicc. These
 * are stored in the variables named "user_xxx" (a bit below) and are
 * used as *additional* flags.
 *
 * Esoteric note:
 *   Because the hard-coded flags are the ones used to build
 *   the last threading library, there may be linker flags needed if the user
 *   utilizes another library with --ort.
 *   Should fix some day :-(
 */

/* Definitions (macros) provided extrernally (see meson.build).
 * These are hard-coded when making ompicc and are the defaults.
 * They can be user-overriden through the OMPI_XXX environmental variables.
 * The last two are overriden if a portable build of OMPi is in effect.
 *
 * CPPcmd
 * CCcmd
 * PreprocFlags
 * CompileFlags
 * LinkFlags
 */

#ifdef PORTABLE_BUILD
	char InstallPath[PATHSIZE], LibDir[PATHSIZE], IncludeDir[PATHSIZE];
#endif

/* Configuration defaults */
#ifdef ENABLE_KERNEL_BUNDLING
bundling_e bundleKernels = BUNDLE_BINS;
#else
bundling_e bundleKernels = BUNDLE_NONE;
#endif

/* Flags collected from the OMPI_CPP, OMPI_CC, OMPI_CPPFLAGS,
 * OMPI_CFLAGS and OMPI_LDFLAGS, the above had-coded values and the 
 * library-specific configuration file.
 */
char PREPROCESSOR[PATHSIZE], COMPILER[PATHSIZE], ORTINFO[FLAGSIZE],
     CPPFLAGS[FLAGSIZE], CFLAGS[FLAGSIZE], LDFLAGS[FLAGSIZE];
#ifdef OMPI_REMOTE_OFFLOADING
	char COMMINFO[FLAGSIZE];
#endif
char ortlibname[PATHSIZE],           /* The runtime library needed */
     commlibname[PATHSIZE],
     RealOmpiName[PATHSIZE];         /* The full _ompi path */

bool cppLineNo = false;
bool mustlink = true;     /* Becomes 0 if -c parameter is specified */
bool makefile = true;     /* Becomes 0 if --nomakefile parameter is specified */
int  keep = 0;            /* Becomes 1/2 if -k/K parameter is specified */
bool verbose = false;
bool showdevinfo = false; /* Display info about configured devices */
bool longdevinfo = false; /* short by default */
int  usegdb = DBG_NONE;   /* If > 0, _ompi will be run through a debugger */
bool usevalgrind = false; /* If true _ompi will be run through gdb. */
bool usecarstats = false;
bool reductionOld = false; /* Make _ompi produce newer style reduction code */
bool reductionRTL = false; /* New reduction code style selector */
bool portable_userprog = false;   /* Portable user application */
int  taskoptlevel = 1;     /* default; never used by ompicc, though */
char *reqmodules = NULL;   /* Requested modules by the user */
int  reqjobs = -1;

/* For bundling kernel files */
bool onlykernelfile = false; /* Compile only a single kernel file */
int  _kernid = 0;

arglist_t user_files     = { NULL, NULL };  /* Files to be compiled/linked */
arglist_t user_outfile   = { NULL, NULL };  /* Output, -o XXX */
arglist_t user_prep_args = { NULL, NULL };  /* Preprocessor args */
arglist_t user_link_args = { NULL, NULL };  /* Linker args given by the user */
arglist_t user_scc_flags = { NULL, NULL };  /* Remaining system compiler args */


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *        UTILIITIES                                         *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* This function uses system(3) to execute the shell command pointed by 'cmd'.
 * On success, the termination status of the child process which is retrieved
 * using WIFEXITED and WEXITSTATUS as described in waitpid(2) is returned.
 * If the child process couldn't be created or its status couldn't be retrieved
 * a) if 'exit_flag' is NULL -1 is returned; 
 * b) if 'exit_flag' is not NULL, exit_flag is set to -1. If system(3) is 
 *    successful and 'exit_flag' is not NULL, exit_flag is set to 0.
 */
int sysexec(char *cmd, int *exit_flag)
{
	int ret;

	/* If the child process could not be created or its status
	 * could not be retrieved set exit_flag to 1.
	 */
	if ((ret = system(cmd)) == -1)
	{
		if (exit_flag)
			return (*exit_flag = -1);
		_exit(-1);
	}
	if (exit_flag) 
		*exit_flag = 0;
	return (WIFEXITED(ret) ? WEXITSTATUS(ret) : ret);
}


/* Removes files given a list of space-separated file names 
 */
void removefiles(char *names)
{
	char *file, *tmp = names ? strdup(names) : NULL;
	if (tmp)
	{
		for (file = strtok(tmp, " \t\n"); file; file = strtok(NULL, " \t\n"))
			unlink(file);
		free(tmp);
	}
}


void ompi_longinfo()
{
#ifdef OMPI_REMOTE_OFFLOADING
	fprintf(stderr,
		"%scc %s using:\n  >> system compiler: %s\n  >> runtime library: %s\n"
		"  >> communication library: %s\n"
		"  >> config. devices: %s\n",
		PKG_NAME, PKG_VERSION, COMPILER, *ORTINFO ? ORTINFO : ortlibname,
		*COMMINFO ? COMMINFO : ortlibname, MODULES_CONFIG);
#else
	fprintf(stderr,
		"%scc %s using:\n  >> system compiler: %s\n  >> runtime library: %s\n"
							"  >> config. devices: %s\n",
		PKG_NAME, PKG_VERSION, COMPILER, *ORTINFO ? ORTINFO : ortlibname,
		MODULES_CONFIG);
#endif
}


char *get_basename(char *path)
{
	char *s;

	if (path == NULL || *path == 0)
		return ".";
	else
		if (path[0] == '/' && path[1] == 0)
			return path;
		else
		{
			s = path;
			while (*s)
				s++;
			s--;
			while (*s == '/' && s != path)   /* destructive */
				*s-- = 0;
			if (s == path)
				return path;
			while (*s != '/' && s != path)
				s--;
			if (*s == '/')
				return s + 1;
			else return path;
		}
}


static int quotedlen(char *s)
{
	int len;
	for (len = 0; *s != 0; s++)
		len += (*s == '"' ? 2 : 1);
	return (len);
}


/* Prepare a string from an argument list; all args are quoted */
static void strarglist(char *dest, arglist_t *l, int maxlen)
{
	arg_t *p;
	char  *c, *d = dest;

	for (*d = 0, p = l->head; p != NULL; p = p->next)
	{
		if ((d - dest) + quotedlen(p->val) + 6 >= maxlen)
			ompicc_error(1, "argument(s) too long; rebuild OMPi with larger LEN.\n");
		if (p->opt)
		{
			snprintf(dest, maxlen, "-%c ", p->opt);
			dest += ((p->opt == 'o') ? 3 : 2);
		}
		*(dest++) = '"';
		for (c = p->val; *c != 0; c++)
		{
			if (*c == '"') *(dest++) = '\\';
			*(dest++) = *c;
		}
		*(dest++) = '"';
		*(dest++) = ' ';
	}
	*dest = 0;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *        COMPILING                                          *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

str _ompi_optstr; /* Contains all _ompi arguments */

char *compile_prefix()
{
	static char prefix[LEN+64];
	
	if (usegdb != DBG_NONE)
	{
		if (usegdb == DBG_GDB)
			sprintf(prefix, "gdb %s -ex 'set arg", RealOmpiName);
		else
			sprintf(prefix, "valgrind %s", RealOmpiName);
		return prefix;
	}
	else
		return RealOmpiName;
}

char *compile_suffix()
{
	if (usegdb != DBG_NONE)
	{
		if (usegdb == DBG_GDB)
			return "'";
		else
			return "";
	}
	else
		return "";
}


void ompicc_compile(char *fname)
{
	char *s, *compfmt, preflags[SLEN], noext[PATHSIZE];
	char strscc_flags[LEN], strgoutfile[LEN];
	str cmd = Strnew();
	str outfile = Strnew();
	int  res, eflag;
	int  nkernels = 0;  /* # kernels in user code */
	int  nmodstocheck = 0;
	
#ifdef OMPI_REMOTE_OFFLOADING
	nmodstocheck = roff_config.nuniquemodules;
#endif

	if ((s = strrchr(fname, '.')) != NULL)
		if (strcmp(s, ".o") == 0) return;     /* skip .o files */

	strcpy(noext, fname);
	if ((s = strrchr(noext, '.')) != NULL) *s = 0; /* remove ext */
	str_printf(outfile, "%s_ompi.c", noext);

	/* Preprocess
	 */
	strarglist(preflags, &user_prep_args, SLEN);
#if defined(__SYSOS_cygwin) && defined(__SYSCOMPILER_cygwin)
	/* Hack for CYGWIN gcc */
	str_printf(cmd, "%s -U__CYGWIN__ -D__extension__=  -U__GNUC__ "
#ifdef OMPI_OMP_EXT
                    " -D_OMPI_OMP_EXT "
#endif
#ifdef OMPI_REMOTE_OFFLOADING
                     " -D_OMPI_REMOTE_OFFLOADING "
#endif
	                   " %s -I%s %s \"%s\" > \"%s.pc\"",
	        PREPROCESSOR, CPPFLAGS, IncludeDir, preflags, fname, noext);
#else
	/* We need to explicitly #define _OMPI_OMP_EXT because it is checked inside
	 * <omp.h> (and possibly other user-facing files, too)
	 */
	str_printf(cmd, "%s -U__GNUC__ %s "
#ifdef OMPI_OMP_EXT
                     " -D_OMPI_OMP_EXT "
#endif
#ifdef OMPI_REMOTE_OFFLOADING
                     "-D_OMPI_REMOTE_OFFLOADING "
#endif
	                "-I%s %s \"%s\" > \"%s.pc\"",
	        PREPROCESSOR, CPPFLAGS, IncludeDir, preflags, fname, noext);
#endif
	if (verbose)
		fprintf(stderr, "====> Preprocessing file (%s.c)\n  [ %s ]\n", noext, 
		        str_string(cmd));
	if ((res = sysexec(str_string(cmd), NULL)) != 0)
		_exit(res);

	/* Transform
	 */

	build_ompi_optstr(_ompi_optstr, fname);
		
#ifdef PORTABLE_BUILD
	compfmt = "%s \"%s.pc\" __ompi__ \"%s\" \"%s\"%s > \"%s\"%s";
#else
	compfmt = "%s \"%s.pc\" __ompi__%s > \"%s\"%s";
#endif

	str_truncate(cmd);
	str_printf(cmd, compfmt,
	           compile_prefix(),
	           noext,
#ifdef PORTABLE_BUILD
	           LibDir,             /* Use as the first argument */
	           InstallPath,        /* Second argument */
#endif
	           str_string(_ompi_optstr),
	           str_string(outfile),
	           compile_suffix());
	if (verbose)
		fprintf(stderr, "====> Transforming file (%s.c)\n  [ %s ]\n", noext, 
		        str_string(cmd));
	res = sysexec(str_string(cmd), NULL);
	if (keep < 2)
	{
		str_truncate(cmd);
		str_printf(cmd, "%s.pc", noext);               /* remove preprocessed file */
		unlink(str_string(cmd));
	}
	if (res == 33)                                /* no pragma omp directives */
	{
		FILE *of = fopen(str_string(outfile), "w");
		if (of == NULL)
		{
			fprintf(stderr, "Cannot write to intermediate file.\n");
			_exit(1);
		}
		if (cppLineNo)
			fprintf(of, "# 1 \"%s.c\"\n", noext); /* Identify the original file */
		fclose(of);
		str_truncate(cmd);
		str_printf(cmd, "cat \"%s.c\" >> \"%s\"", noext, str_string(outfile));
		if (sysexec(str_string(cmd), &eflag))
		{
			unlink(str_string(outfile));
			_exit(res);
		}
	}
	else
	{
		if (res != 0)
		{
			if (!keep)
				unlink(str_string(outfile));
			_exit(res);
		}
		else   /* All OK - parse the 2nd line to get needed info */
		{
			FILE *of = fopen(str_string(outfile), "r");
			char ch = 0;
			int  nk;

			if (of != NULL)
			{
				for (ch = fgetc(of); ch!=EOF && ch!='\n'; ch = fgetc(of))
					;
				if (ch == '\n' && fscanf(of, "$OMPi__nk:%d", &nk) == 1)
					nkernels = nk;
				fclose(of);
			}
		}
	}

	/* Kernels must be created within the directory of the source file.
	 * If all source files are in the same directory we are doing
	 * redundant copies here.
	 */
	if ((bundleKernels != BUNDLE_SRCS) && nkernels && (nmodules || nmodstocheck))
	{
		if (verbose)
			fprintf(stderr, "====> Generating kernel makefiles for %d module(s)\n",
			                nmodules + nmodstocheck);
		/* 99.9% get_basename didn't touch it */
		if ((res = kernel_makefiles(fname, nkernels)) != 0)
		{
			kernel_binaries_remove(fname, nkernels);
			if (!keep)
				unlink(str_string(outfile)); /* remove _ompi.c */
			ompicc_error(1, "kernel makefile execution failed; exiting.\n");
		}
	}

	/* Compile the transformed file
	 */
	strarglist(strscc_flags, &user_scc_flags, LEN);
	str_truncate(cmd);
	str_printf(cmd, "%s \"%s\" -c %s %s -I%s %s %s",
	   COMPILER, str_string(outfile), CFLAGS, (bundleKernels) ? NoLTOopt : LTOopt,
	   IncludeDir, preflags, strscc_flags);

	if (verbose)
		fprintf(stderr, "====> Compiling file (%s):\n  [ %s ]\n", 
		        str_string(outfile), str_string(cmd));
	res = sysexec(str_string(cmd), &eflag);
	if (!keep)
	{
		if (bundleKernels == BUNDLE_SRCS)
			kernel_sources_remove(fname, nkernels);
		else if (bundleKernels == BUNDLE_BINS)
			kernel_binaries_remove(fname, nkernels);
		unlink(str_string(outfile));
	}
	if (eflag || res)
		_exit(res);

	/* Settle the output file name
	 */
	strarglist(strgoutfile, &user_outfile, LEN);
	strcpy(noext, get_basename(fname));
	if ((s = strrchr(noext, '.')) != NULL) * s = 0; /* remove ext */
	str_truncate(outfile);
	if (user_outfile.head != NULL && !mustlink)
		str_printf(outfile, "%s", user_outfile.head->val);
	else
		str_printf(outfile, "%s.o", noext);
	strcat(noext, "_ompi.o");
	if (verbose)
		fprintf(stderr, "====> Renaming file \"%s\" to \"%s\"\n",
		        noext, str_string(outfile));
	rename(noext, str_string(outfile));

#if defined(OMPI_REMOTE_OFFLOADING) && defined(ROFF_USE_STATIC_MPI_PROCS)
	if (verbose)
		fprintf(stderr, "====> Generating %s-mpirun.sh script\n", PKG_NAME);
	str_truncate(cmd);
	str_printf(cmd, "%sconf __internal__ --roff-action=\"create-mpirun-script\"",
	                PKG_NAME);
	res = sysexec(str_string(cmd), &eflag);
	if (res)
	{
		fprintf(stderr, "could not create mpirun script\n");
		_exit(res);
	}
#endif
	
	str_free(cmd);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *        LINKING                                            *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


void ompicc_link()
{
	arg_t *p;
	char cur_obj[PATHSIZE], tmp[PATHSIZE];
	str cmd = Strnew();
	str objects = Strnew();
	char strsccargs[LEN], strlinkargs[LEN], strgoutfile[LEN], strprepargs[LEN];
	int len, is_tmp, eflag;
	str tmpfile_names = Str(NULL);

	for (p = user_files.head; p != NULL; p = p->next)
	{
		strcpy(cur_obj, p->val);
		is_tmp = 0;
		len = strlen(cur_obj);
		if (cur_obj[len - 1] == 'c')
			is_tmp = 1;
		if (is_tmp)
		{
			cur_obj[len - 2] = 0;
			strcpy(tmp, cur_obj);
			strcpy(cur_obj, get_basename(tmp));
			strcat(cur_obj, ".o");
			str_printf(tmpfile_names, "%s ", cur_obj);
		}
		str_printf(objects,"\"%s\" ", cur_obj);
	}
	strarglist(strsccargs,  &user_scc_flags, LEN);
	strarglist(strlinkargs, &user_link_args, LEN);
	strarglist(strgoutfile, &user_outfile, LEN);
	strarglist(strprepargs, &user_prep_args, LEN);

	/* We have to include -lort 2 times due to circular dependencies
	 * with the threading libraries.
	 */
#ifdef OMPI_REMOTE_OFFLOADING
	str_printf(cmd,
	  "%s %s %s %s -I%s %s %s %s -L%s -L%s/%s -L%s/%s -lort -lcomm %s %s "
	  "-lort -lcomm",
	  COMPILER, str_string(objects), CFLAGS, (bundleKernels) ? NoLTOopt : LTOopt,
	  IncludeDir, strprepargs, strsccargs, strgoutfile, LibDir, LibDir, 
	  ortlibname, LibDir, commlibname, LDFLAGS, strlinkargs);
#else
	str_printf(cmd,
	  "%s %s %s %s -I%s %s %s %s -L%s -L%s/%s -lort %s %s -lort",
	  COMPILER, str_string(objects), CFLAGS, (bundleKernels) ? NoLTOopt : LTOopt,
	  IncludeDir, strprepargs, strsccargs, strgoutfile, LibDir, LibDir, 
	  ortlibname, LDFLAGS, strlinkargs);
#endif

	if (verbose)
		fprintf(stderr, "====> Linking:\n  [ %s ]\n", str_string(cmd));
	if (sysexec(str_string(cmd), &eflag))
		fprintf(stderr, "Error: could not perform linking.\n");
		
	/* Remove unnecessary files */
	removefiles( str_string(tmpfile_names) );
	str_free(tmpfile_names);
	str_free(cmd);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *                  Generate Makefile                        *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


void replace_variable(char *line, char *variable, char *replace_with)
{
	char *p, tmp[SLEN];

	while ((p = strstr(line, variable)) != NULL)
	{
		snprintf(tmp, SLEN, "%s%s", replace_with, p + strlen(variable));
		//Should check if p - line + strlen(tmp) > sizeof(line)
		strcpy(p, tmp);
	}
}


/* Prepare a string from an argument list; all args are quoted;
 * copied from strarglilst()
 */
static void getargs(char *dest, arglist_t *l, int maxlen)
{
	arg_t *p;
	char  *c, *d = dest;

	for (*d = 0, p = l->head; p != NULL; p = p->next)
	{
		if ((d - dest) + quotedlen(p->val) + 6 >= maxlen)
			ompicc_error(1, "argument(s) too long; rebuild OMPi with larger LEN.\n");
		for (c = p->val; *c != 0; c++)
		{
			if (*c == '"') *(dest++) = '\\';
			*(dest++) = *c;
		}
		if (p->next)
			*(dest++) = ' ';
	}
	*dest = 0;
}


void ompicc_makefile(char *compilerPath)
{
	char strfiles[LEN], strgoutfile[LEN];
	char line[SLEN];
	str infile = Strnew();
	FILE *fp, *infp;

	/* Check if the library has a template Makefile
	 */
	str_printf(infile, "%s/%s/MakefileTemplate", LibDir, ortlibname);
	if (!(infp = fopen(str_string(infile), "r")))
	{
		makefile = false;
		str_free(infile);
		return;
	}
	str_free(infile);

	getargs(strfiles, &user_files, LEN);

	//The file ompi will output
	if (user_outfile.head && strlen(user_outfile.head->val) != 0)
		snprintf(strgoutfile, LEN, "%s", user_outfile.head->val);
	else
		snprintf(strgoutfile, LEN, "a.out");

	if (verbose)
		fprintf(stderr, "====> Outputing Makefile\n");
	if ((fp = fopen("Makefile", "w")) == NULL)
		fprintf(stderr, "Error: could not generate Makefile\n");
	else
	{
		fprintf(fp, "# Makefile generated by %s\n", PKG_NAME);

		while (fgets(line, sizeof line, infp) != NULL)
		{
			replace_variable(line, "@OMPICC@", compilerPath);
			replace_variable(line, "@OMPI_INPUT@", strfiles);
			replace_variable(line, "@OMPI_OUTPUT@", strgoutfile);
			replace_variable(line, "@OMPI_ORTLIB@", ortlibname);
			fputs(line, fp);  /* write the line */
		}

		fclose(fp);
	}
}


void ompicc_makefile_compile()
{
	char cmd[LEN];
	int res;

	/* Run the compile target */
	snprintf(cmd, LEN, "make compile");  /* make should be on the user PATH */

	if (verbose)
		fprintf(stderr, "====> Running target compile on generated Makefile\n");

	if ((res = sysexec(cmd, NULL)) != 0)
		_exit(res);
}


void ompicc_makefile_link()
{
	char cmd[LEN];
	int res;

	/* Run the link target */
	snprintf(cmd, LEN, "make link");  /* make should be on the user PATH */

	if (verbose)
		fprintf(stderr, "====> Running target link on generated Makefile\n");

	if ((res = sysexec(cmd, NULL)) != 0)
		_exit(res);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *        THE MAIN() PART                                    *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* Get values from environmental variables if they exist,
 * otherwise grab the default configuration values (as hard-coded
 * from Makefile.am)
 */
void ompicc_get_envvars()
{
	char *t;

	if ((t = getenv("OMPI_CPP")) == NULL)
		strncpy(PREPROCESSOR, CPPcmd, 511);
	else
		strncpy(PREPROCESSOR, t, 511);

	if ((t = getenv("OMPI_CPPFLAGS")) == NULL)
		strncpy(CPPFLAGS, PreprocFlags, 511);
	else
		strncpy(CPPFLAGS, t, 511);

	if ((t = getenv("OMPI_CC")) == NULL)
		strncpy(COMPILER, CCcmd, 511);
	else
		strncpy(COMPILER, t, 511);

	if ((t = getenv("OMPI_CFLAGS")) == NULL)
		strncpy(CFLAGS, CompileFlags, 511);
	else
		strncpy(CFLAGS, t, 511);

	if ((t = getenv("OMPI_LDFLAGS")) == NULL)
		strncpy(LDFLAGS, LinkFlags, 511);
	else
		strncpy(LDFLAGS, t, 511);
}


/* Read the runtime library's configuration file
 */
void get_ort_flags()
{
	str confpath = Strnew();
	FILE *fp;
	void setflag(char *, char *, void *);

	str_printf(confpath, "%s/%s/ortconf.%s", LibDir, ortlibname, ortlibname);
	if ((fp = fopen(str_string(confpath), "r")) == NULL)
		ompicc_error(1, "library `%s' cannot be found\n  (%s is missing)\n",
		             ortlibname, str_string(confpath));
	keyval_read(fp, setflag, NULL);
	fclose(fp);
	str_free(confpath);
}


#ifdef OMPI_REMOTE_OFFLOADING
/* Read the communication library's configuration file
 */
void get_comm_flags()
{
	str confpath = Strnew();
	FILE *fp;
	void setflag(char *, char *, void *);

	str_printf(confpath, "%s/%s/commconf.%s", LibDir, commlibname, commlibname);
	if ((fp = fopen(str_string(confpath), "r")) == NULL)
		ompicc_error(1, "library `%s' cannot be found\n  (%s is missing)\n",
		             commlibname, str_string(confpath));
	keyval_read(fp, setflag, NULL);
	fclose(fp);
	str_free(confpath);
}
#endif


void setflag(char *key, char *value, void *ignore)
{
	if (strcmp(key, "ORTINFO") == 0 && strlen(value) + strlen(ORTINFO) < FLAGSIZE)
		strcat(*ORTINFO ? strcat(ORTINFO, " ") : ORTINFO, value);
#ifdef OMPI_REMOTE_OFFLOADING
	if (strcmp(key, "COMMINFO")==0 && strlen(value) + strlen(COMMINFO) < FLAGSIZE)
		strcat(*COMMINFO ? strcat(COMMINFO, " ") : COMMINFO, value);
#endif
	if (strcmp(key, "CPPFLAGS")==0 && strlen(value) + strlen(CPPFLAGS) < FLAGSIZE)
		strcat(strcat(CPPFLAGS, " "), value);
	if (strcmp(key, "CFLAGS") == 0 && strlen(value) + strlen(CFLAGS) < FLAGSIZE)
		strcat(strcat(CFLAGS, " "), value);
	if (strcmp(key, "LDFLAGS") == 0 && strlen(value) + strlen(LDFLAGS) < FLAGSIZE)
		strcat(strcat(LDFLAGS, " "), value);
	if (strcmp(key, "CC") == 0 && strlen(COMPILER) < FLAGSIZE)
		strcpy(COMPILER, value);
	if (strcmp(key, "CPP") == 0 && strlen(PREPROCESSOR) < FLAGSIZE)
		strcpy(PREPROCESSOR, value);
}


int main(int argc, char **argv)
{
	arg_t *p;
	char  argv0[PATHSIZE];
	char  path[PATHSIZE];
#if !defined(HAVE_REALPATH) || defined(PORTABLE_BUILD)
	bool  argv0_haspath = false;
#endif
	str RealOmpiName_tmp = Strnew();

	strncpy(argv0, argv[0], PATHSIZE - 1);

#if !defined(HAVE_REALPATH) || defined(PORTABLE_BUILD)
	if (is_path(argv0))
		argv0_haspath = true;
#endif

#if defined(HAVE_REALPATH)
	char  *res;
	
	res = realpath(argv[0], argv0);
	if (res == NULL) /* ompicc is in PATH */
		str_printf(RealOmpiName_tmp, "_%s", PKG_NAME);
	else
	{
		get_path(argv0, path, PATHSIZE); /* path before ompicc */
		str_printf(RealOmpiName_tmp, "%s_%s", path, PKG_NAME);
	}
#else
	if (argv0_haspath)
	{
		get_path(argv0, path, PATHSIZE);
		str_printf(RealOmpiName_tmp, "%s", path);
	}
	str_printf(RealOmpiName_tmp, "_%s", PKG_NAME);
#endif

	_ompi_optstr = Str("");

#ifdef PORTABLE_BUILD
	/* Code for making ompi portable in Linux. This wont work on other OSes.
	 */
	str IncludeDir_tmp = Strnew(), LibDir_tmp = Strnew();
	char buffer[PATHSIZE];

	if (argv0_haspath)
		strncpy(buffer, path, PATHSIZE - 1);
	else
	{
	#ifdef HAVE_READLINK
		ssize_t len = readlink("/proc/self/exe", buffer, PATHSIZE - 1);
		if (len == -1)
			ompicc_error(1, "couldn't retrieve installation path using readlink.\n");
		else
			if (len == PATHSIZE - 1)
				ompicc_error(1, "path to %s too long.\n", PKG_NAME);
			else
				buffer[len] = '\0';
	#else
		ompicc_error(1, "please provide the full path to %s.\n", argv[0]);
	#endif
	}

	/* Get the path and get rid of the trailing "bin/" */
	get_path(buffer, InstallPath, PATHSIZE);
	if (strcmp(InstallPath + strlen(InstallPath) - 4, "bin/"))
		ompicc_error(1, "invalid installation path for a portable build.\n");
	InstallPath[strlen(InstallPath) - 4] = 0;
	
	if (strlen(InstallPath) + 8 + strlen(PKG_NAME) + 1 > PATHSIZE)
		ompicc_error(1, "path to %s too long.\n", PKG_NAME);

	str_printf(LibDir_tmp, "%s%s", InstallPath, PKG_LIBDIR);
	str_printf(IncludeDir_tmp, "%s%s", InstallPath, PKG_INCLUDEDIR);

	strncpy(LibDir, str_string(LibDir_tmp), PATHSIZE - 1);
	strncpy(IncludeDir, str_string(IncludeDir_tmp), PATHSIZE - 1);

	str_free(LibDir_tmp);
	str_free(IncludeDir_tmp);
#endif

	strncpy(RealOmpiName, str_string(RealOmpiName_tmp), PATHSIZE - 1);
	ompicc_get_envvars();
	get_user_options(argc, argv);

#if defined(OMPI_REMOTE_OFFLOADING)
	if (reqmodules && strcmp(reqmodules, "all") != 0 &&
		  (bundleKernels != BUNDLE_NONE || !onlykernelfile))
		fprintf(stderr,
			"[ompicc warning]: remote offloading enabled; "
			"all modules will be employed.\n");

	/* Do not fix requested modules to "all" when bundling is 
	 * enabled and single makefile execution is requested (--kernel=...)
	 */
	if (bundleKernels != BUNDLE_NONE || !onlykernelfile)
		reqmodules = strdup(MODULES_CONFIG);
#endif

	get_ort_flags();
#ifdef OMPI_REMOTE_OFFLOADING
	get_comm_flags();
#endif
	modules_employ(reqmodules);
	
#ifdef OMPI_REMOTE_OFFLOADING
	remote_modules_employ();
#endif

	if (argc == 1)
	{
		ompi_longinfo();
		help_ompicc(argv[0]);
		exit(0);
	}

	if (verbose)
	{
		ompi_longinfo();
		fprintf(stderr, "----\n");
	}

	if (user_files.head == NULL)
	{
		fprintf(stderr, "No input file specified; "
		                "run %s with no arguments for help.\n", argv[0]);
		exit(0);
	}

	if (bundleKernels && onlykernelfile)
	{
		if (kernel_makefile_single(user_files.head->val) != 0)
			ompicc_error(1, "kernel makefile execution failed; exiting.\n");
		exit(0);
	}

	if (makefile)
		ompicc_makefile(argv[0]);

	/* If makefile exists
	 */
	if (makefile)
	{
		ompicc_makefile_compile();
		if (mustlink)
			ompicc_makefile_link();
	}
	else
	{
		for (p = user_files.head; p != NULL; p = p->next)
			ompicc_compile(p->val);
		if (mustlink)
			ompicc_link();
	}
	
#ifdef OMPI_REMOTE_OFFLOADING
	remote_modules_finalize();
#endif

	str_free(_ompi_optstr);
	str_free(RealOmpiName_tmp);

	return (0);
}
