/*
  OMPi OpenMP Compiler
  == Copyright since 2001, the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* help.c
 * Info and help functions
 */

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include "str.h"
#include "ompicc.h"


typedef enum { 
		EV_OMP, EV_ORT, EV_EELIB, EV_MODULE, EV_COMPILER, EV_OTHER 
	} evspace_e;
typedef struct {
		evspace_e space;
		char     *name;   /* Up to 26 chars shown */
		char    **values; /* Max 45 chars per line shown; optimal: up to 28 chars */
		char     *defval; /* Default value; optimal: up to 7 */
		char    **info;   /* Max 72 chars per line, used only at high verbosity 
		                     levels; the first line may be used at lower verbosity 
		                     levels, too */
	} envvarinfo_t;

#define ONEVAL(v)  (char *[]) { #v, NULL }
static envvarinfo_t envvars[] = {
		{ EV_OMP, "OMP_DYNAMIC",           ONEVAL(boolean) },
		{ EV_OMP, "OMP_NESTED",            ONEVAL(boolean) },
		{ EV_OMP, "OMP_SCHEDULE",          (char *[])
		          { "policy[,int]", "policy=static|dynamic|guided|auto", NULL }
		}, 
		{ EV_OMP, "OMP_STACKSIZE",         ONEVAL(int[C]    C=B|K|M|G), "K" },
		{ EV_OMP, "OMP_THREAD_LIMIT",      ONEVAL(int) },
		{ EV_OMP, "OMP_MAX_ACTIVE_LEVELS", ONEVAL(int) },
		{ EV_OMP, "OMP_WAIT_POLICY",       ONEVAL(active|passive) },
		{ EV_OMP, "OMP_NUM_THREADS",       (char *[]) 
		          { "int[,int[,int ...]]", NULL } },
		{ EV_OMP, "OMP_PROC_BIND",         (char *[])
		          { "true|false|<list of types>",
		            "types=master|close|spread", NULL } 
		},
		{ EV_OMP, "OMP_CANCELLATION",      ONEVAL(boolean) },
		{ EV_OMP, "OMP_DISPLAY_ENV",       ONEVAL(true|false|verbose) },
		{ EV_OMP, "OMP_PLACES",            (char *[])
		          { "symbolic[(int)]|<list of places>", 
		            "symbolic=threads|cores|sockets", NULL } 
		},
		{ EV_OMP, "OMP_DEFAULT_DEVICE",    ONEVAL(int) },
		{ EV_OMP, "OMP_MAX_TASK_PRIORITY", ONEVAL(int) },
		{ EV_OMP, "OMP_DISPLAY_AFFINITY",  ONEVAL(boolean) },
		{ EV_OMP, "OMP_AFFINITY_FORMAT",   ONEVAL(string) },
		{ EV_OMP, "OMP_TARGET_OFFLOAD",    ONEVAL(mandatory|disabled|default) },
		
		{ EV_ORT, "OMPI_DYNAMIC_TASKQUEUESIZE", ONEVAL(boolean), "false" },
		{ EV_ORT, "OMPI_TASK_DEQUEUE_POLICY",   ONEVAL(fifo|lifo), "fifo" },
		{ EV_ORT, "OMPI_PAR2TASK_POLICY",  ONEVAL(true|false|auto), "auto" },
		{ EV_ORT, "OMPI_HOSTTARGET_SHARE", ONEVAL(boolean), "true" },
		{ EV_ORT, "OMPI_SCHED_AUTO",       ONEVAL(string), "static" },
		{ EV_ORT, "OMPI_TAG_SCHED_<TAG>",  ONEVAL(string), "static" },
		
		{ EV_EELIB, "OMPI_PTHREADS_POLICY",  ONEVAL(kill|pool|prepool), "pool" },
		{ EV_EELIB, "OMPI_PTHREADS_MAXTHR",  ONEVAL(int), "no limit",
		            ONEVAL((max #persistent threads)) },
		{ EV_EELIB, "OMPI_PTHREADS_OVERSUB_THR", ONEVAL(int),
	    "110\% of the number of processors",
		  (char *[]) {
		    "# threads above which oversubscription occurs",
		    "(default: 110\% of the number of processors)",
		    NULL
		  }
		},
		{ EV_EELIB, "OMPI_PTHREADS_TRANSI_THR", ONEVAL(int), 
	    "200\% of the number of processors",
		  (char *[]) {
		    "# threads above which to create transient threads",
		    "(default: 200\% of the number of processors)",
		    NULL
		  }
		},
		
		{ EV_MODULE, "OMPI_CUDA_ARCH_TARGET", ONEVAL(string) },
		{ EV_MODULE, "OMPI_MODULE_OPENCL_XLIMIT", 
		             ONEVAL(warn|nowarn|exit|exitsilent), "warn"
		},
		{ EV_MODULE, "OMPI_MODULE_OPENCL_ZEROCOPY", ONEVAL(boolean), "false", 
		  (char *[]) {
		    "Try to use zero-copy argument transfers to OpenCL kernels.",
		    "Use this with great care as it may actually cause performance drops",
		    "in GPUs that do not support it well.",
		    NULL
		  }
		},
		{ EV_OMP, NULL, NULL }
	};


void _display_one_envvar(envvarinfo_t *e)
{
	int i;
	
	fprintf(stderr, "  %-26.26s ", e->name);
	for (i = 0; e->values[i]; i++)
	{
		if (i) 
			fprintf(stderr, "%29s", " ");
			
		if (e->values[i+1]) /* not last line */
			fprintf(stderr, "%-45.45s\n", e->values[i]);
		else                /* last line */
		{
			if (strlen(e->values[i]) <= 28 && e->defval && strlen(e->defval) <= 7)
				fprintf(stderr, "%-28.28s  ", e->values[i]);
			else
				fprintf(stderr, "%-45.45s\n", e->values[i]);
		}
	}
	if (e->defval)
	{
		if (strlen(e->defval) <= 7)
			fprintf(stderr, "%.*s(default:%s)\n", (int) (75-strlen(e->defval)-10),
			                " ", e->defval);
		else 
			fprintf(stderr, "%*s(default: %s)\n", 29, " ", e->defval);
	}
	/* Ignore description for now */
	if (0 && e->info)
		for (i = 0; e->info[i]; i++)
			fprintf(stderr, "    - %-70.70s\n", e->info[i]);
}


void display_envvars(int verbosity)
{
	envvarinfo_t *e = envvars;
	evspace_e space = -1;
	int has;
	
	fprintf(stderr, "\nAll runtime env. variables\n--------------------------\n");
	for (space = EV_OMP; space <= EV_OTHER; space++)
		for (e = envvars, has = 0; e->name; e++)
		{
			if (space != e->space) continue;
			if (!has)
			{
				has = 1;
				fprintf(stderr, "\n[ %s variables ]\n", 
				                e->space==EV_OMP ? "OpenMP" :
				                e->space==EV_COMPILER ? "OMPI compiler" :
				                e->space==EV_ORT ? "OMPi runtime (ORT)" :
				                e->space==EV_EELIB ? "OMPi EELib" :
				                e->space==EV_MODULE ? "OMPi devices" : "Other");
			}
			_display_one_envvar(e);
		};
}


void search_envvars(char *what)
{
	envvarinfo_t *e = envvars;
	int i, found = 0;
	
	/* Search once to see if there exists any */
	for (e = envvars; e->name; e++)
	{
		if (strstr(e->name, what)) { found = 1; break; }
		for (i = 0; e->values[i]; i++)
			if (strstr(e->values[i], what)) { found = 1; break; }
		if (e->values[i]) break;
	}

	if (!found) return;
	
	fprintf(stderr, "--- Environmental variables ---\n");
	for (e = envvars; e->name; e++)
		if (strstr(e->name, what))
			_display_one_envvar(e);
		else 
			for (i = 0; e->values[i]; i++)
				if (strstr(e->values[i], what))
				{
					_display_one_envvar(e);
					break;
				}
}


static void helpOpenMPshow()
{
	fprintf(stderr, "\n[ Arguments for OpenMP-related help ]\n");
	fprintf(stderr, "  --info env              "
	       " (shows all OpenMP and OMPi environmental variables)\n");
	fprintf(stderr, "  --info directive <name> "
	       " (shows all clauses allowed in the given directive)\n");
	fprintf(stderr, "  --info clause <name>    "
	       " (shows all directives accepting the given clause)\n");
	fprintf(stderr, "  --info find <string>    "
	       " (shows OpenMP entities containing the given string)\n");
}


void help_OpenMP(int argc, char **argv)
{
	str cmd = Strnew();
	
	ompi_shortinfo();
	if (argc == 1)
	{
		HELPERROR:
		helpOpenMPshow();
		return;
	}
	argc--; argv++;
	if (strcmp(*argv, "env") == 0)
	{
		display_envvars(0);
		return;
	}
	if (argc == 1)
		goto HELPERROR;
	
	if (strncmp(*argv, "dir", 3) == 0)
		str_printf(cmd, "%s __internal__ dirclauses %s", RealOmpiName, argv[1]);
	else if (strncmp(*argv, "clau", 4) == 0)
		str_printf(cmd, "%s __internal__ clausedirs %s", RealOmpiName, argv[1]);
	else if (strncmp(*argv, "find", 4) == 0)
		str_printf(cmd, "%s __internal__ searchomp %s", RealOmpiName, argv[1]);
	else
		goto HELPERROR;
	
	fprintf(stderr, "[ OpenMP help/information about %s ]\n\n", argv[1]);
	if (strncmp(*argv, "find", 4) == 0)
		search_envvars(argv[1]);
	sysexec(str_string(cmd), NULL);
	str_free(cmd);
}


void help_ompicc(char *ompibin)
{
	fprintf(stderr, 
		"\nUsage: %s [ompi options] [system compiler options] programfile(s)\n",
		ompibin);
	fprintf(stderr, "\n"
		"   Useful OMPi options:\n"
		"           --options: show all OMPi options\n"
		"       --info [what]: OpenMP-related help\n"
		"                  -k: keep intermediate file\n"
		"                  -v: be verbose (show the actual steps)\n"
#ifdef OMPI_REMOTE_OFFLOADING
		"       --comm=<name>: use a specific remote offloading comm library\n"
#endif
		"    --devs=<devices>: target the given devices\n"
		"   --devopt [option]: give device option\n"
		"\n"
		"Use environmental variables OMPI_CPP, OMPI_CC, OMPI_CPPFLAGS,\n"
		"OMPI_CFLAGS, OMPI_LDFLAGS to have OMPi use a particular base\n"
		"preprocessor and compiler, along with specific flags.\n");
}


void ompicc_error(int exitcode, char *format, ...)
{
	va_list ap;

	va_start(ap, format);
	fprintf(stderr, "[ompicc error]: ");
	vfprintf(stderr, format, ap);
	va_end(ap);

	exit(exitcode);
}
