/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* teams.c -- OMPi RunTime library; teams construct */

#include "ort_prive.h"



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * TEAMS                                                             *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* Setup the very initial league */
void league_initial()
{
	static int initleague_cgsize = 1;
	
	ort->league.numteams = 1;
	ort->league.threadlimit = ort->icvs.threadlimit;
	ort->league.func = NULL;
	ort->league.funcarg = NULL;
	ort->league.cg_size = &initleague_cgsize;
	ort->league.cg_inithr = NULL;
}


/* Install a new league, with all the initial threads */
static 
void league_start(int numteams, int thrlimit, void *(*func)(void *), void *arg)
{
	int i;

	ort->league.numteams = numteams;
	ort->league.threadlimit = thrlimit;
	ort->league.func = func;
	ort->league.funcarg = arg;
	ort->league.cg_size = ort_alloc(numteams * sizeof(int));
	ort->league.cg_inithr = ort_alloc(numteams * sizeof(ort_eecb_t *));
	for (i = 0; i < numteams; i++)
	{
		ort->league.cg_size[i] = 1;
		ort->league.cg_inithr[i] = NULL;
	}
}


/* Destory league and re-install the initial one */
static void league_finish()
{
	if (ISINITLEAGUE())
		return;
	ort_free((void *) ort->league.cg_size);
	ort_free(ort->league.cg_inithr);
	league_initial();
}


/* This is the function which initial threads (execution entities)
 * call to execute the teams user code.
 */
void teams_initial_thread_work(int teamid)
{
	ort_eecb_t *curr = __MYCB, *newcb = eecb_initial_prepare(NULL);
	
#if ORT_DEBUG & DBG_ORT
	ort_debug_thread("in initial_ee_dowork(); about to execute func.");
#endif

	newcb->cgid = teamid;
	ort->league.cg_inithr[teamid] = newcb;  /* Do we need this??? */
	__CURRIMPLTASK(newcb)->icvs.threadlimit = ort->league.threadlimit;
	__SETMYCB(newcb);

	/* OMPevent: initial-task-begin */

	if (ort->icvs.proc_bind != omp_proc_bind_false && 
		  ort->eecaps.supports_proc_binding)
	{
#define INITEECB ort->initial_eecb
    /* Assign initial threads to places using spread policy */
		spread_subpartition(teamid, ort->league.numteams, 
		                    INITEECB->currplace, INITEECB->pfrom, INITEECB->pto, 
												&newcb->pfrom, &newcb->pto);
		/* Avoid re-binding thread 0 if # places is enough for all */
		if (ort->league.numteams > INITEECB->pto - INITEECB->pfrom + 1 || teamid!=0)
			newcb->currplace = ee_bindme(ort->place_partition, newcb->pfrom);
#undef INITEECB
	}

#ifdef EE_TYPE_PROCESS
 /* From its ID, get and call the actual function */
	(*thrfunc_get_ptr(ort->league.func))(ort->league.funcarg);
#else
	(*(ort->league.func))(ort->league.funcarg); /* Execute the actual function */
#endif

	/* OMPevent: initial-task-end */

	__SETMYCB(curr);
	free_initial_eecb(newcb);
}


void _ort_execute_teams(void *(*func)(void *), void *shared, int min_teams,
                        int max_teams, int thr_limit)
{
	ort_eecb_t *me = __MYCB;

	if (__CURRTASK(me) != __CURRIMPLTASK(me) || me->level != 0 || !ISINITLEAGUE())
		ort_error(1, "'teams' request from an invalid region\n");

	/* 1. Fix number of teams (= initial threads) and the thread limit,
	 * according to OpenMP 5.2 rules
	 */
	if (max_teams < 0)     /* No num_teams specified */
	{
		if (ort->icvs.nteams > 0)  
			max_teams = ort->icvs.nteams;   /* Actually <= ort->icvs.nteams */
		else
			max_teams = 1;                  /* Implementation defined */
	}

	if (min_teams < 0)                  /* If no minimum given, min = max */
		min_teams = max_teams;
	if (min_teams > max_teams) 
		min_teams = max_teams;

	if (thr_limit <= 0)
	{
		if (ort->icvs.teams_thread_limit > 0)
			thr_limit = ort->icvs.teams_thread_limit;/* Actually between 1 and this */
		else
		{
			thr_limit = ort->icvs.ncpus / max_teams; /* Implementation defined */
			if (thr_limit < 1) thr_limit = 1;
		}
	}

	/* OMPevent: teams-begin */

	/* 2. Create the initial threads and setup the league 
	 */
	if ((max_teams = ee_request(min_teams-1, 0, 0)) != min_teams-1)
		ort_warning("[%s]: cannot create %d teams; creating %d teams instead\n",
		            __func__, min_teams, max_teams+1);

	league_start(max_teams+1, thr_limit, func, shared);  /* A league of teams */
	SFENCE;
	if (max_teams > 0)                          /* Create the initial threads */
		ee_create(max_teams, 0, NULL /*no parent*/, &me->ee_info);

	/* 3. Participate as team #0 
	 */
	teams_initial_thread_work(0);

	/* 4. Wait for teams to finish and demolish the league
	 */
	if (max_teams > 0)
		ee_waitall(&me->ee_info);    /* Wait till all initial threads finish */

	league_finish();

	/* OMPevent: teams-end */
}
