/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/


#ifndef __ROFF_COMMON_H__
#define __ROFF_COMMON_H__

#include "stddefs.h"

#ifdef OMPI_REMOTE_OFFLOADING

#include <stdlib.h>
#include "ort.h"
#include "rt_common.h"

#define OFFLOAD_OK 11 
#define FINALIZE_OK 12
#define PRIMARY_NODE 0
#define WAKE_UP_MESSAGE 99

/* length of num_args array */
#define NUM_ARGS_SIZE 3

typedef struct {
	void *data; /* The allocated memory */
} roff_allocitem_t;

typedef struct roff_alloctab_
{
	roff_allocitem_t *mem;     /* Per-device memory */
	int capacity;              /* Total allocated space */
	int global_vars_index;     /* Where next global var should be stored */
	int next_available_index;  /* To reduce searching time when adding */
	int device_id;
} roff_alloctab_t;

typedef struct roff_devinfo_
{
	int device_id_in_node;
	int device_id_in_module;
	int global_device_id;
	int node_id;
	char *modulename;
	void *devinfo;
	roff_alloctab_t *alloc_table;
	int sharedspace;
	devicestatus_e status;
} roff_devinfo_t;

extern roff_devinfo_t *devinfotab;
extern int devinfotab_size;

typedef unsigned long long roff_datatype_t;

extern roff_devinfo_t *devinfotab_get_from_gldevid(int global_device_id);

/* 
 * Communication layer
 */

/* Helper macros */
#define _Comm_Send_1int(num, to, tag) \
	Comm_Send(comminfo, to, TYPE(INT), num, 1, tag, NULL)
#define _Comm_Send_1ull(num, to, tag) \
	Comm_Send(comminfo, to, TYPE(UNSIGNED_LONG_LONG), num, 1, tag, NULL)
#define _Comm_Send_bytes(buf, size, to, tag) \
	Comm_Send(comminfo, to, TYPE(BYTE), buf, size, tag, NULL)

#define _Comm_Recv_1int(num, from, tag) \
	Comm_Recv(comminfo, from, TYPE(INT), num, 1, tag, NULL)
#define _Comm_Recv_1ull(num, from, tag) \
	Comm_Recv(comminfo, from, TYPE(UNSIGNED_LONG_LONG), num, 1, tag, NULL)
#define _Comm_Recv_bytes(buf, size, from, tag) \
	Comm_Recv(comminfo, from, TYPE(BYTE), buf, size, tag, NULL)

#define COMM_ANY_TAG -1

static inline int encode_tag(int device_id, int thread_id) 
{
    return (device_id << 16) | thread_id;
}

static inline void decode_tag(int encoded_tag, int *device_id, int *thread_id) 
{
    *device_id = encoded_tag >> 16;
    *thread_id = encoded_tag & 0xFFFF;
}

#define TYPE(t) COMM_##t
typedef enum Comm_Datatype_
{
	TYPE(CHAR) = 0, 
    TYPE(SIGNED_CHAR),   TYPE(UNSIGNED_CHAR),    TYPE(BYTE),      
	TYPE(WCHAR),         TYPE(SHORT),            TYPE(UNSIGNED_SHORT), 
	TYPE(INT),           TYPE(UNSIGNED),         TYPE(LONG),       
	TYPE(UNSIGNED_LONG), TYPE(FLOAT),            TYPE(DOUBLE),
    TYPE(LONG_DOUBLE),   TYPE(LONG_LONG),        TYPE(UNSIGNED_LONG_LONG),
    TYPE(PACKED),        TYPE(LB),               TYPE(UB),  
	TYPE(FLOAT_INT),     TYPE(DOUBLE_INT),       TYPE(LONG_INT),    
	TYPE(SHORT_INT),     TYPE(2INT),             TYPE(LONG_DOUBLE_INT),
	TYPE(COMPLEX),       TYPE(DOUBLE_COMPLEX),   TYPE(LOGICAL),
	TYPE(REAL),          TYPE(DOUBLE_PRECISION), TYPE(INTEGER),
    TYPE(2INTEGER),      TYPE(2REAL),            TYPE(2DOUBLE_PRECISION), 
	TYPE(CHARACTER),     TYPE(REAL4),            TYPE(REAL8),             
	TYPE(REAL16),        TYPE(COMPLEX8),         TYPE(COMPLEX16), 
	TYPE(COMPLEX32),     TYPE(INTEGER1),         TYPE(INTEGER2), 
	TYPE(INTEGER4),      TYPE(INTEGER8),         TYPE(INTEGER16), 
	TYPE(LAST)
} Comm_Datatype;

typedef struct Comm_Status_
{
	int   Int;
	char *Str;
} Comm_Status;

/** @brief Initializes the communication layer
 *  @param argc The user program argc
 *  @param argv The user program argv
 *  @return The communication layer info struct, otherwise NULL
 */
extern void *Comm_Init(int *argc, char ***argv);


/** @brief Initializes the worker side of the communication layer
 *  @param info The communication layer info struct
 */
extern void  Comm_Init_worker(void *info);


/** @brief Returns a string containing information for the running process
 *  @param info The communication layer info struct
 *  @return The information string
 */
extern char *Comm_Get_info(void *info);


/** @brief Finalizes the communication layer
 *  @param info The communication layer info struct
 */
extern void  Comm_Finalize(void *info);


/** @brief Spawns remote processes
 *  @param info The communication layer info struct
 *  @param num_nodes The number of remote nodes participating
 *  @param num_procs The number of remote processes to be spawned per node
 */
extern void Comm_Spawn(void *info, int num_nodes, int num_procs);


/** @brief Sends a message to a specific destination 
 *  @param info The communication layer info struct
 *  @param dst The receiver's ID
 *  @param type The type of data being sent
 *  @param buf The buffer containing the data
 *  @param size The sent data size
 *  @param tag The sent message tag
 *  @param status (ret) Information regarding sent message 
 */
extern void Comm_Send(void *info, int dst, Comm_Datatype type, void *buf, int size, int tag, 
                      Comm_Status *status);


/** @brief Receives a message from a specific source
 *  @param info The communication layer info struct
 *  @param src The sender's ID
 *  @param type The type of data being received
 *  @param buf The buffer which will store the received data
 *  @param size The received data size
 *  @param tag The received message tag
 *  @param status (ret) Information regarding received message 
 */
extern void Comm_Recv(void *info, int src, Comm_Datatype type, void *buf, int size, int tag, 
                      Comm_Status *status);


/** @brief Check for an incoming message without actually receiving it
 *  @param info The communication layer info struct
 *  @param src The sender's ID
 *  @param tag The incoming message tag
 *  @param status (ret) Information regarding the incoming message 
 */
extern int Comm_Probe(void *info, int src, int tag, Comm_Status *status);


/** @brief Returns the ID of the process participating in the layer
 *  @param info The communication layer info struct
 *  @return The process ID
 */
extern int Comm_Get_id(void *info);


/** @brief Performs a barrier call
 *  @param info The communication layer info struct
 */
extern void Comm_Barrier(void *info);

/** @brief Returns parent node id
 *  @param info The communication layer info struct
 */
int Comm_Get_node_id(void *info);

#endif /* OMPI_REMOTE_OFFLOADING */
#endif /* __ROFF_COMMON_H__ */
