/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <limits.h>
#include "omp.h"
#include "str.h"
#include "config.h"
#include "assorted.h"
#include "ort_prive.h"
#ifdef OMPI_REMOTE_OFFLOADING
	#include "roff_config.h"
	#include "remote/roff.h"
#endif

static volatile ee_lock_t mod_lock; /* Lock for functions in this file */

#ifdef PORTABLE_BUILD
	char *ModuleDir = "./";     /* dummy initialization */
	void _ort_set_moduledir(char *moduledir, int portable)
	{
		ModuleDir = strdup(moduledir);
		is_userprog_portable = portable;
	}
#endif
int  is_userprog_portable = 0;


/* All discovered devices are stored in a table (ort->ort_devices); 
 * the HOST device is stored first (at position 0). The position in the 
 * table is thde device index (devidx); this has nothing to do with the
 * OpenMP device ID (ompdevid), which is a number given by/to the user
 * program.
 * ort_devidx(ompdevid) gives the device index from a given OpenMP device ID
 * ort_ompdevid(devidx) gives the OpenMP device ID from a given device index
 */
static ort_device_t *add_device(ort_module_t *module, int idx_in_module)
{
#ifdef OMPI_REMOTE_OFFLOADING
	static int prevnode = -1, node_devs = 0;
#endif
	ort_device_t *dev = &(ort->ort_devices[ort->num_devices]);
	dev->idx           = ort->num_devices++;  /* advance the counter too */
	dev->idx_in_module = idx_in_module;
	dev->module        = module;
	dev->initialized   = false;
	dev->device_info   = NULL;
	dev->sharedspace   = 0;

	if (idx_in_module == 0)
		module->first_global_devidx = dev->idx;

#ifdef OMPI_REMOTE_OFFLOADING
	dev->is_cpudev = false;

	if (module)
	{
		/* Store the intranode device ID */
		if (module->remote)
		{
			if (prevnode == module->nodeid)
				node_devs++;
			else
			{
				node_devs = 0;
				prevnode = module->nodeid;
			}
			dev->id_in_node = node_devs;
		}
	}
	else /* When HAVE_DLOPEN is not defined, module == NULL */
		dev->is_cpudev = true;
	
#endif
	return (dev);
}


/* If remote offloading is enabled and I'm a worker, then I
 * need to check the configuration and add the appropriate # devices
 */
static int get_num_hostdevs(void)
{
	int numdevs = 1;

#ifdef OMPI_REMOTE_OFFLOADING
	int i;
	
	if (node_role == ROLE_WORKER)
	{
		roff_config_node_t *me = &(roff_config.nodes[roff_man_get_my_id() - 1]);
		for (i = 0; i < me->num_modules; i++)
		{
			if (IS_CPU_MODULE(me->modules[i].name))
			{
				numdevs = me->modules[i].num_devices;
				break;
			}
		}
	}
#endif

	return numdevs;
}

/* 
 * Add and initialize the host "device"
 *
 * ATTENTION:
 * OMPI_HOSTTARGET_SHARE is used to set hostdevs[i]->sharedspace to true/false;
 * It is true by default and it is made available only for experimentation.
 * If false, the host device tries to emulate non-shared address space devices,
 * e.g. a mapped item is actually malloc()'ed.
 * Be sure that this will BREAK existing user code, so set it to false only
 * if you know what you are doing!
 * For one, when the host becomes a fallback device (e.g. due to an if clause)
 * it won't behave as it should ;-)
 */
static void setup_host_moddev(int numdevs)
{
	ort_device_t *d;
	int i;

	ort->hostdevs = ort_alloc(numdevs * sizeof(ort_device_t*));
	ort->module_host.number_of_devices = numdevs;
	
	for (i = 0; i < numdevs; i++)
	{
		d = add_device(hostdev_get_module(), i);
		d->initialized = true;
		d->sharedspace = 1; /* 1 by default, changed by env.c */
#ifdef OMPI_REMOTE_OFFLOADING
		d->is_cpudev = true;
#endif
		/* Initialize device lock */
		d->lock = (volatile void *)ort_alloc(sizeof(ee_lock_t));
		ee_init_lock((ee_lock_t *) d->lock, ORT_LOCK_NORMAL);
		ort->hostdevs[i] = d; /* Keep the host devices */
	}
}

#ifdef OMPI_REMOTE_OFFLOADING

/* Assigns roff_XXX functions to a remote module and initializes it */
static bool load_and_init_remote_module(ort_module_t *m)
{
	m->unified_medaddr  = 0; /* should be retrieved by the remote module */
	
	m->initialize       = roff_initialize;
	m->finalize         = roff_finalize;
	m->dev_init         = roff_dev_init;
	m->dev_end          = roff_dev_end;
	m->offload          = roff_offload;
	m->dev_alloc        = roff_dev_alloc;
	m->dev_init_alloc_global = roff_dev_init_alloc_global;
	m->dev_free         = roff_dev_free;
	m->dev_free_global  = roff_dev_free_global;
	m->todev            = roff_todev;
	m->fromdev          = roff_fromdev;
	m->imed2umed_addr   = roff_imed2umed_addr;
	m->umed2imed_addr   = roff_umed2imed_addr;
/* new-hostpart-func.sh:roffload */

	m->is_cpumodule     = IS_CPU_MODULE(m->name);

	return MODULE_CALL(m, initialize, (m->name, m->first_global_devidx,
	                   ort_prepare_omp_lock, omp_set_lock, omp_unset_lock,
	                   sched_yield, ort->argc, ort->argv)) != 0;
}


/* This function discovers all remote devices and readjusts
 * allocated memory for the ort modules and devices array. 
 */
static void discover_remote_modules(int *nModules, int *nDevices)
{
#ifdef ROFF_IGNORE_CONFIGURATION_SNAPSHOT
	roff_config_initialize(DONT_IGNORE_DISABLED_MODULES, is_userprog_portable);
#else
	roff_config_initialize_from_hex(ompi_remote_devices, ompi_remote_devices_size, 
	                          DONT_IGNORE_DISABLED_MODULES, is_userprog_portable);
#endif

	(*nDevices) += roff_config.num_devices;
	(*nModules) += roff_config.num_modules;
	
	ort->modules = ort_realloc(ort->modules, (*nModules) * sizeof(ort_module_t));
	ort->ort_devices = ort_realloc(ort->ort_devices, 
	                               (*nDevices) * sizeof(ort_device_t));
	free((int*) ort->device_status);
	ort->device_status = ort_calloc((*nDevices) * 4);
	ort->device_status[0] = INT_MAX;

	ort->num_remote_modules = roff_config.num_modules;
	ort->num_remote_devices = roff_config.num_devices;
}


/* This function initializes all discovered remote modules
 */
static void setup_remote_modules()
{
	roff_config_node_t *node;
	roff_config_module_t *mod;
	int i = 0, j;
	int index = ort->num_local_modules;
	
	for (i = 0; i < roff_config.num_nodes; i++)
	{
		node = &(roff_config.nodes[i]);
		for (j = 0; j < node->num_modules; j++, index++)
		{
			mod = &(node->modules[j]);
			ort->modules[index].name = strdup(mod->name);
			ort->modules[index].name_for_roff = smalloc(128 * sizeof(char));
			snprintf(ort->modules[index].name_for_roff, 127, "%s_node%d",mod->name,i);
			
			ort->modules[index].handle = NULL; 
			ort->modules[index].nodeid = i + 1;
			ort->modules[index].node_name = strdup(node->name);
			ort->modules[index].initialized = true;
			ort->modules[index].initialized_successful = 
				load_and_init_remote_module(&(ort->modules[index]));
			ort->modules[index].number_of_devices = mod->num_devices;
			ort->modules[index].remote = true;
		}
	}
}

#endif /* OMPI_REMOTE_OFFLOADING */


/**
 * Get the device descriptor given a device index
 * @param devidx the device index
 * @return       the device descriptor (ort_device_t)
 */
ort_device_t *ort_get_device(int devidx)
{
	if (devidx >= 0 && devidx < ort->num_devices)
	{
		ee_set_lock((ee_lock_t *) &mod_lock);
		if (!ort->ort_devices[devidx].initialized)
			ort_init_device(devidx);
		ee_unset_lock((ee_lock_t *) &mod_lock);

		/* If the device has failed to initialize correctly fall back to host */
		if (ort->ort_devices[devidx].device_info == NULL)
			devidx = HOSTDEV_IDX;
	}
	else
		devidx = ort_illegal_device("device index", devidx);

	return &(ort->ort_devices[devidx]);
}


/**
 * Gives a one-time warning on illegal device id and returns a fallback id.
 * Used for uniform handling of the situation across the runtime.
 * @param reason   a message describing the failing device
 * @param ompdevid the failing device id
 * @return         the fallback device id
 */
int ort_illegal_device(char *reason, int ompdevid)
{
	static bool warned = false;
	
	if (ort->icvs.targetoffload == OFFLOAD_MANDATORY)
		ort_error(1, "Invalid %s value (%d); mandatory exiting.\n",reason,ompdevid);

	if (!warned)
	{
		ee_set_lock((ee_lock_t *) &mod_lock);
		if (!warned)
		{
			ort_warning("Invalid %s value (%d); falling back to host.\n",
			            reason, ompdevid);
			warned = true; FENCE;
		}
		ee_unset_lock((ee_lock_t *) &mod_lock);
	}
	return (HOSTDEV_IDX);
}


#ifdef HAVE_DLOPEN

#include <dlfcn.h>


static void *open_module(char *name, int type)
{
	void *handle;
	str tmp = Strnew();

	/* Check current folder */
	str_printf(tmp, "./%s.so", name);
	handle = dlopen(str_string(tmp), type);
	if (handle)
	{
		str_free(tmp);
		return handle;
	}

	/* Check ompi's library folder */
	str_truncate(tmp);
	str_printf(tmp, "%s/%s/hostpart.so", ModuleDir, name);
	handle = dlopen(str_string(tmp), type);
	if (handle)
	{
		str_free(tmp);
		return handle;
	}

	/* Finally check system's library folder */
	str_truncate(tmp);
	str_printf(tmp, "%s.so", name);
	handle = dlopen(str_string(tmp), type);

	str_free(tmp);
	return handle;
}


static 
void *load_symbol(void *module, char *moduleName, char *sym, int show_warn)
{
	char *error;
	void *temp;

	temp = dlsym(module, sym);

	if (((error = dlerror()) != NULL) && show_warn)
	{
		ort_warning("module: %s, symbol: %s, %s\n", moduleName, sym, error);
		return NULL;
	}

	return temp;
}


void ort_discover_modules(int nModules, char **modnames)
{
	int  i = 0, j, modidx = 0, nhostdevs = get_num_hostdevs(),
	     nDevices = nhostdevs;  /* we also count the hostdevs in nDevices */
	void *module;
	
	dlerror();  /* Clear dlerror */

	ort->modules = ort_alloc(nModules * sizeof(ort_module_t));
	for (i = 0; i < nModules; i++)
	{
		ort->modules[i].name = modnames[modidx++];
		ort->modules[i].handle = NULL;

#ifdef OMPI_REMOTE_OFFLOADING
		ort->modules[i].name_for_roff = ort->modules[i].node_name = NULL; /* nonremote */
		ort->modules[i].nodeid = -1; /* ditto */
		ort->modules[i].remote = false;
		
		/* If it's a remote module, do not insert it to the module list.
		 * This is a temporary fix; _ompi should not include remote modules to the
		 * variadic arguments of `_ort_init', only local ones. For this reason, we 
		 * should somehow separate remote modules from local ones, e.g. by passing 
		 * them through a new _ompi argument, instead of --usemod.
		 */
		if ((node_role == ROLE_PRIMARY) && 
		    (!contains_word(MODULES_CONFIG, ort->modules[i].name)))
			goto INITIALIZATION_FAIL;
#endif

		module = open_module(ort->modules[i].name, RTLD_LAZY);
		if (!module)
			ort_warning("Failed to open module \"%s\"\n", ort->modules[i].name);
		else
		{
			int  (*moddevcount)(void), (*modfin)();
			
			ort->modules[i].initialized = false;
			ort->modules[i].initialized_successful = false;
			moddevcount = load_symbol(module, ort->modules[i].name,
			                          "hm_get_num_devices", 1);
			modfin = load_symbol(module, ort->modules[i].name, "hm_finalize", 1);
			if (moddevcount != NULL && modfin != NULL)
			{
				ort->modules[i].number_of_devices = moddevcount();
				nDevices += ort->modules[i].number_of_devices;
				modfin();
				dlclose(module);
				continue;
			}
			dlclose(module);
		}

#ifdef OMPI_REMOTE_OFFLOADING
INITIALIZATION_FAIL:
#endif
		/* If we reached here we failed to get the number of devices */
		ort->modules[i].initialized = true;
		ort->modules[i].initialized_successful = false;
		ort->modules[i].number_of_devices = 0;
	}

	ort->ort_devices = ort_alloc(nDevices * sizeof(ort_device_t));
	ort->device_status = ort_calloc(nDevices * 4);
	ort->device_status[0] = INT_MAX;

	/* The host "module" and "device" 0; call it here to get id 0 */
	setup_host_moddev(nhostdevs);
	
	ort->num_local_modules = nModules;
	ort->num_local_devices = nDevices;
	
#ifdef OMPI_REMOTE_OFFLOADING
	if (!ort->embedmode && node_role == ROLE_PRIMARY)
	{
		discover_remote_modules(&nModules, &nDevices);
		setup_remote_modules();
	}
#endif
	for (i = 0; i < nModules; i++)
		for (j = 0; j < ort->modules[i].number_of_devices; j++)
			add_device(&(ort->modules[i]), j);
	ort->num_modules = nModules;

	ee_init_lock((ee_lock_t *) &mod_lock, ORT_LOCK_NORMAL);
}


#define LOADANDCHECK(val, sym) \
	if ((val = load_symbol(m->handle, m->name, "hm_" #sym, 1)) == NULL) \
		return false

static bool load_and_init_module(ort_module_t *m)
{
	int *x, num_devices = 0;
	int glompid;
	void (*register_str_printf)(int  (*str_printf_in)(str, char *, ...));

	dlerror(); /* Clear dlerror */

	LOADANDCHECK(x, unified_medaddr);
	m->unified_medaddr = *x;	
	LOADANDCHECK(m->initialize, initialize);
	LOADANDCHECK(m->finalize, finalize);
	LOADANDCHECK(m->dev_init, dev_init);
	LOADANDCHECK(m->dev_end, dev_end);
	LOADANDCHECK(m->offload, offload);
	LOADANDCHECK(m->dev_alloc, dev_alloc);
	LOADANDCHECK(m->dev_init_alloc_global, dev_init_alloc_global);
	LOADANDCHECK(m->dev_free, dev_free);
	LOADANDCHECK(m->dev_free_global, dev_free_global);
	LOADANDCHECK(m->todev, todev);
	LOADANDCHECK(m->fromdev, fromdev);
	LOADANDCHECK(m->imed2umed_addr, imed2umed_addr);
	LOADANDCHECK(m->umed2imed_addr, umed2imed_addr);
/* new-hostpart-func.sh:load */

	glompid = ort_ompdevid(m->first_global_devidx);
#ifdef OMPI_REMOTE_OFFLOADING
	if (node_role == ROLE_WORKER)
	{
		roff_config_node_t *me = &(roff_config.nodes[roff_man_get_my_id() - 1]);
		glompid += num_primary_devs + me->first_remote_devid;
	}
#endif

	num_devices = MODULE_CALL(m, initialize, (m->name, glompid,
	                          ort_prepare_omp_lock, omp_set_lock, omp_unset_lock,
	                          sched_yield, ort->argc, ort->argv));

	if (num_devices == 0)
		return false;

	LOADANDCHECK(register_str_printf, register_str_printf);
	register_str_printf(str_printf);

	return true;
}

#undef LOADANDCHECK


static void initialize_module(ort_module_t *m)
{
	m->initialized = true;
	m->initialized_successful = false;

	m->handle = open_module(m->name, RTLD_NOW);
	if (!m->handle)
	{
		ort_warning("Failed to initialize module \"%s\"\n", m->name);
		return;
	}
	m->is_cpumodule = false;
	m->initialized_successful = load_and_init_module(m);
	if (!m->initialized_successful)
	{
		ort_warning("Failed to initialize module \"%s\" functions\n", m->name);
		dlclose(m->handle);
	}
}


void ort_init_device(int devidx)
{
	ort_device_t *d = &(ort->ort_devices[devidx]);
	int idx;
	
	if (d->initialized)
		return;

	d->initialized = true;

	/* Check if module is initialized */
	if (!d->module->initialized)
		initialize_module(d->module);

	if (!d->module->initialized_successful)
		d->device_info = NULL;
	else
	{
		idx = d->idx_in_module;
		
#ifdef OMPI_REMOTE_OFFLOADING
		/* If the device belongs to a remote module, encode in-module device ID
		 * and node ID in `idx`
		 */
		if (d->module->remote)
		{
			if ((d->module->nodeid <= USHRT_MAX) && (d->id_in_node <= USHRT_MAX)
				&& (d->module->nodeid >= 0) && (d->id_in_node >= 0))
			{
				unsigned int enc_devid = _uint_encode2(d->id_in_node,d->module->nodeid);
				if (enc_devid <= INT_MAX) 
					idx = (int) enc_devid;
				else
					goto ENCDEVIDWARNING;
			}
			else
			{
				ENCDEVIDWARNING:
				fprintf(stderr, "[ORT warning] either devid or nodeid are off range;"
				                "using default devid.");
			}
		}
#endif /* OMPI_REMOTE_OFFLOADING */

		/* Call initialize function for device */
		d->device_info = MODULE_CALL(d->module, dev_init, (idx, &(ort->icvs),
		                             &(d->sharedspace)));

		/* Initialize device lock */
		d->lock = (volatile void *) ort_alloc(sizeof(ee_lock_t));
		ee_init_lock((ee_lock_t *) d->lock, ORT_LOCK_NORMAL);
		SFENCE; /* 100% initialized, before been assigned to "lock" */
	}
}


void ort_finalize_devices(void)
{
	int i;

	for (i = 0; i < ort->num_devices; i++)
		if (ort->ort_devices[i].initialized)
		{
			if (ort->ort_devices[i].device_info)
				MODULE_CALL(ort->ort_devices[i].module, 
				            dev_end, (ort->ort_devices[i].device_info)
				);

			/* Deinitialize and free device lock */
			ee_destroy_lock((ee_lock_t *) ort->ort_devices[i].lock);
			free((ee_lock_t *) ort->ort_devices[i].lock);
		};

	for (i = 0; i < ort->num_modules; i++)
	{
		if (ort->modules[i].initialized && ort->modules[i].initialized_successful)
			MODULE_CALL(&(ort->modules[i]), finalize, ());
#ifdef OMPI_REMOTE_OFFLOADING
		if (ort->modules[i].remote) 
		{
			free(ort->modules[i].name);	
			free(ort->modules[i].name_for_roff);
			free(ort->modules[i].node_name);
		}
		else
#endif
		{
			if (ort->modules[i].initialized && ort->modules[i].initialized_successful)
				dlclose(ort->modules[i].handle);
		}
	}

	free(ort->hostdevs);

#ifdef PORTABLE_BUILD
	free(ModuleDir);
#endif

	ort_kernfunc_cleanup();
}


/**
 * @brief Translates an OpenMP device ID to the corresponding device index
 *        used internally by ORT. This is clearly called by user- or compiler-
 *        facing functions and understands the DFLTDEV_ALIAS; as such, 
 *        user-facing functions should check for legal ompdevid.
 * @param ompdevid The OpenMP device ID 
 * @return         The corresponding ORT device index or -1 on error.
 */
int ort_devidx(int ompdevid)
{
	if (ompdevid == DFLTDEV_ALIAS)
		ompdevid = omp_get_default_device();
	if (ompdevid == HOSTDEV_ALIAS || ompdevid == ort->num_devices-1)
		return HOSTDEV_IDX;   /* Host device (v5.1) */
	return IS_OMPDEVID(ompdevid) ? ompdevid + 1 : -123;
}


/**
 * @brief Translates a device index used internally by ORT to the corresponding
 *        OpenMP device ID; the host is not returned as -1 so as to keep 
 *        negative numbers as errors.
 * @param ortdevidx The ORT device index 
 * @return          The corresponding OpenMP device ID or -2 on error.
 */
int ort_ompdevid(int ortdevidx)
{
	if (ortdevidx == HOSTDEV_IDX) 
		return ort->num_devices - 1;    /* Host device (v5.1) */
	if (ortdevidx > 0 && ortdevidx < ort->num_devices)
		return ortdevidx - 1;
	return -2;
}


#define ILLEGAL_INDEX_CHECK(ID,HIGH_BOUND,RET) \
	if((ID) < 0 || (ID) >= (HIGH_BOUND)) return (RET);

/**
 * @brief Given a global device ID returns the index of the node containing it. 
 *        Optional parameter for getting node-local device index.
 * 
 * @param global_device_id Global device ID
 * @param local_device_id  (ret) stores node-local device index or NULL
 * @return the index of node containing given device ID, or -1 if requested ID
 *         is out of bounds.
 */
int ompx_device_get_node(int global_device_id, int *local_device_id)
{
	ILLEGAL_INDEX_CHECK(global_device_id, ort->num_devices, -1)

#ifdef OMPI_REMOTE_OFFLOADING
	if (global_device_id >= ort->num_local_devices)
	{
		if (local_device_id != NULL)
			*local_device_id = ort->ort_devices[global_device_id].id_in_node;
		return ort->ort_devices[global_device_id].module->nodeid;
	}
#endif
	if (local_device_id != NULL)
		*local_device_id = global_device_id;
	return 0;
}


/**
 * @brief Given a global device ID, returns the name of its module. When the ID 
 *        is illegal returns "invalid device". Optional parameter for getting 
 *        module relative device index.
 * 
 * @param global_device_id Global device ID
 * @param modrel_device_id (ret) Module relative device index or passed as NULL 
 *                         to skip
 * @return the name of the module containg the given device ID, or "invalid 
 *         device" if requested ID is out of bounds.
 */
char *ompx_device_get_module(int global_device_id, int *modrel_device_id)
{
	ILLEGAL_INDEX_CHECK(global_device_id, ort->num_devices, "invalid device")

	if (modrel_device_id != NULL)
		*modrel_device_id = ort->ort_devices[global_device_id].idx_in_module;

	if (IS_HOST_DEVID(global_device_id))
		return "host";
		
	return ort->ort_devices[global_device_id].module->name;
}


/**
 * @brief Given a node index and the local device ID on that node, returns the  
 *        gloobal evice ID. Node 0 is host and local IDs start from 0.
 * 
 * @param node_index      Node which contains the device. Node 0 is host
 * @param local_device_id The local device ID on the node
 * @return the global device ID for the given node and local device ID, or -1
 *         for illegal arguments.
 */
int ompx_get_device_of_node(int node_index, int local_device_id)
{
	if (local_device_id < 0) 
		return -1;

	if (node_index == 0)
		return (local_device_id <= ort->num_local_devices) ? local_device_id+1 : -1;

#ifdef OMPI_REMOTE_OFFLOADING
	ILLEGAL_INDEX_CHECK(node_index, roff_config.num_nodes + 1, -1)
	roff_config_node_t *node = &(roff_config.nodes[node_index - 1]);

	if (local_device_id >= 0 && local_device_id < node->total_num_devices)
		return node->first_remote_devid + ort->num_local_devices + local_device_id;
#endif
	return -1; 
}


/**
 * @brief Given a module name and the module relative index of a device, returns
 *        the global device ID.
 * 
 * @param module_name      The name of the module the device belongs to
 * @param modrel_device_id The module-relative device index across all nodes
 * @return the global device ID for the given module and relative index, or -1
 *         for illegal arguments.
 */
int ompx_get_device_of_module(char *module_name, int modrel_device_id)
{
	int i, total_devices = ompx_get_module_num_devices(module_name);
	ILLEGAL_INDEX_CHECK(modrel_device_id, total_devices, -1)

	if (total_devices > 0)
	{
		for (i = 0; i < ort->num_modules; i++)
			if (strcmp(module_name, ort->modules[i].name) == 0)
			{
				if (modrel_device_id < ort->modules[i].number_of_devices)
					return ort->modules[i].first_global_devidx + modrel_device_id;
				modrel_device_id -= ort->modules[i].number_of_devices;
			}
	}
	return -1;
}


/**
 * Provides the number of devices that belong to a given node. Node 0 is host.
 * 
 * @param node_index Node which contains the device. Node 0 is host
 * @return the number of devices on the given node or -1 for illegal node index.
 */
int ompx_get_node_num_devices(int node_index)
{
	if (node_index == 0)
		return ort->num_local_devices - 1;
#ifdef OMPI_REMOTE_OFFLOADING
	else 
	{
		ILLEGAL_INDEX_CHECK(node_index, roff_config.num_nodes + 1, -1)
		return roff_config.nodes[node_index - 1].total_num_devices;
	}
#endif
	return -1;
}


/**
 * @brief Given a module, return the total number of its devices.
 * 
 * @param module_name The module name
 * @return the number of its devices, or -1 if the module is not found
 *         or has 0 devices.
 */
int ompx_get_module_num_devices(char *module_name)
{
	int i, total_devs = 0;

	for (i = 0; i < ort->num_modules; i++)
		if (strcmp(module_name, ort->modules[i].name) == 0)
			total_devs += ort->modules[i].number_of_devices;

	return (total_devs > 0) ? total_devs : -1;
}

/**
 * @brief Finds a module at a specific node and returns the number of its
 *        devices and (optionally) its first usable device ID.
 * 
 * @param node_name        The hostname of the node
 * @param module_name      The module to search for in the given node
 * @param global_device_id (ret) The first usable device ID of the module
 * @return > 0 if module is found, 0 if not and -1 if one of the arguments 
 *             are NULL    
 */
int ompx_get_module_num_devices_in_node(char *module_name, int node_index, 
                                        int *global_device_id)
{
	int i;

	if (module_name == NULL)
		return -1;

	if (node_index==0)
	{
		for (i = 0; i < ort->num_local_modules; i++)
			if (strcmp(module_name, ort->modules[i].name) == 0)
			{
				if (global_device_id != NULL)
					*global_device_id = ort->modules[i].first_global_devidx;
				return ort->modules[i].number_of_devices;
			}
	}
#ifdef OMPI_REMOTE_OFFLOADING
	else
	{
		ILLEGAL_INDEX_CHECK(node_index, roff_config.num_nodes + 1, -1)
		roff_config_node_t *node = &(roff_config.nodes[node_index - 1]);
		int mod_count = 0;

		for (i = 0; i < node->num_modules; i++){
			if ((strcmp(module_name, node->modules[i].name)) == 0)
			{
				if (global_device_id != NULL)
					*global_device_id = node->first_remote_devid + 
					                    mod_count + ort->num_local_devices;
				return node->modules[i].num_devices;
			} 
			else 
				mod_count += node->modules[i].num_devices;
		}
	}
#endif
	return -1;
}

/**
 * @brief Returns the ID of the most available device of a specific module.
 * 
 * @param module_name the name of the module
 * @return the ID of the first available device, or 0 (host) if no devices
 *         are available
 */
int ompx_get_available_device(char *module_name) 
{
	int i, dev_choice = 0;

	if (ort->device_status == NULL) 
	{
		ort_warning("Problem during device initialization; returning device 0 (host)\n");
		return 0;
	}

	ee_set_lock((ee_lock_t *) &mod_lock);

	for (i = 1; i < ort->num_devices; ++i) 
	{
		if (strcmp(module_name, ort->ort_devices[i].module->name) == 0) 
		{
			if (!ort->device_status[i]) 
			{
				dev_choice = i;
				break;
			}
			if (ort->device_status[i] < ort->device_status[dev_choice])
				dev_choice = i;
		}
	}

	if (dev_choice == 0)
		ort_warning("Device type %s not available; returning device 0 (host).\n",
		            module_name);
	else
		ort->device_status[dev_choice]++;

	ee_unset_lock((ee_lock_t *) &mod_lock);

	return dev_choice;
}


void reduce_device_status(int gID) 
{
	ee_set_lock((ee_lock_t *) &mod_lock);
	if (ort->device_status[gID] > 0)
		ort->device_status[gID]--;
	ee_unset_lock((ee_lock_t *) &mod_lock);
}


#ifdef OMPI_REMOTE_OFFLOADING
// These are currently available only internally
int ompx_devid_to_node_id(int device_id)
{
	return ort->ort_devices[device_id].module->nodeid;
}
int ompx_get_node_first_devid(int node_id)
{
	if (node_id < 1)
		return 0;
	return roff_config.nodes[node_id-1].first_remote_devid+ort->num_local_devices;
}
#endif

#else   /* not HAVE_DLOPEN */

void ort_discover_modules(int nmods, char **modnames)
{
	ort_device_t *d;
	ort->ort_devices = ort_alloc(1 * sizeof(ort_device_t));
	ort->hostdevs = ort_alloc(1 * sizeof(ort_device_t*))
	d = add_device(NULL, 0);   /* Host */
	d->initialized = true;
	ort->hostdevs[0] = d;
	ee_init_lock((ee_lock_t *) &mod_lock, ORT_LOCK_NORMAL);
}

void ort_init_device(int devidx) {}

void ort_finalize_devices() {}


#endif  /* HAVE_DLOPEN */
