/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* hostdev.c -- module interface for the host "device" */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include "ort_prive.h"


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * THE MODULE FUNCTIONS FOR THE HOST DEVICE                          *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


extern int avoid_optimizer;


/**
 * Offloads and executes a kernel file.
 *
 * @param device_info        the device
 * @param host_func pointer  to offload function on host address space
 * @param dev_data pointer   to a struct containing kernel variables
 * @param decl_data pointer  to a struct containing globally declared variables
 * @param kernel_filename_prefix filename of the kernel (without the suffix)
 * @param num_teams          num_teams clause from "teams" construct
 * @param num_threads        num_threads clause from combined parallel construct
 * @param thread_limit       thread_limit clause from "teams" construct
 * @param teamdims           an unsigned long long that contains the dimensions
 *                           of the launched league, encoded as follows:
 *                           x: bits 0-20, y: bits 21-41, z: bits 42-62 
 * @param thrdims            an unsigned long long that contains the
 *                           dimensions of each thread team, encoded as follows:
 *                           x: bits 0-20, y: bits 21-41, z: bits 42-62 
 * @param argptr             The addresses of all target data and target
 *                           declare variables (only used in OpenCL devices)
 *
 * NOTE: `teamdims' and `thrdims' can be decoded using the _ull_decode3 function
 */
static
int host_offload(void *device_info, void *(*host_func)(void *), void *dev_data,
                 void *decl_data, char *kernel_filename_prefix, int num_teams,
                 int num_threads, int thread_limit, unsigned long long teamdims, 
                 unsigned long long thrdims, int *num_args, void **args)
{
	execute_kernel_on_host(host_func, dev_data, num_teams, thread_limit);
	avoid_optimizer++;       /* gcc seems to inline and mess up with the args */
	return 0;
}


/**
 * Allocates memory on the device
 *
 * @param device_info the device
 * @param size        the number of bytes to allocate
 * @param map_memory  used in OpenCL, when set to 1 additionaly to the memory
 *                    allocation in shared virtual address space, the memory
 *                    is mapped with read/write permissions so the host cpu
 *                    can utilize it.
 * @param hostaddr    used in MPI to allocate #declare target link variables;
 *                    when we encounter such variables instead of
 *                    allocating new space, we should return a pointer to
 *                    their original address.
 * @param map_type    the mapping type that triggered this allocation 
                      (to/from/tofrom/alloc)
 * @return hostaddr a pointer to the allocated space
 */
static
void *host_dev_alloc(void *device_info, size_t size, int map_memory, 
                     void *hostaddr, int map_type)
{
	return ( ort_alloc(size) );
}


/**
 * Frees data allocated with host_dev_alloc
 *
 * @param device_info  the device
 * @param imedaddr     pointer to the memory that will be released
 * @param unmap_memory used in OpenCL, when set to 1 prior to the memory
 *                     deallocation, the memory is unmapped.
 */
static
void host_dev_free(void *device_info, void *imedaddr, int unmap_memory)
{
	free(imedaddr);
}


/**
 * Transfers data from the host to a device
 *
 * @param device_info the device
 * @param hostaddr    the source memory
 * @param hostoffset  offset from hostaddr
 * @param imedaddr    the target memory
 * @param size        the size of the memory block
 */
static
void host_todev(void *device_info, void *hostaddr, size_t hostoffset,
                void *imedaddr, size_t devoffset, size_t size)
{
	memcpy(imedaddr + devoffset, hostaddr + hostoffset, size);
}



/**
 * Allocates & initializes memory "on the device" for a global variable
 *
 * @param device_info the pointer returned by hm_dev_init()
 * @param global_id   the ID of the global variable
 * @param size        the number of bytes to allocate

 * @return            pointer to the allocated space (internal mediary address)
 */
static
void *host_dev_init_alloc_global(void *device_info, void *initfrom, size_t size, 
                                 int global_id, void *hostaddr)
{
	void *addr = host_dev_alloc(device_info, size, 0, hostaddr, MAP_TYPE_IGNORE);
	host_todev(device_info, initfrom, 0L, addr, 0L, size);

	return addr;
}


/**
 * Frees data allocated with host_dev_alloc
 *
 * @param device_info  the device
 * @param imedaddr     pointer to the memory that will be released
 * @param unmap_memory used in OpenCL, when set to 1 prior to the memory
 *                     deallocation, the memory is unmapped.
 */
static
void host_dev_free_global(void *device_info, void *imedaddr, int global_id)
{
	host_dev_free(device_info, imedaddr, 0);
}


/**
 * Transfers data from a device to the host
 *
 * @param device_info the source device
 * @param hostaddr    the target memory
 * @param hostoffset  offset from hostaddr
 * @param imedaddr    the source memory
 * @param size        the size of the memory block
 */
static
void host_fromdev(void *device_info, void *hostaddr, size_t hostoffset,
                  void *imedaddr, size_t devoffset, size_t size)
{
	memcpy(hostaddr + hostoffset, imedaddr + devoffset, size);
}


/**
 * Given an internal mediary address, it returns a usable mediary address
 *
 * @param device_info the device
 * @param iaddr       allocated memory from host_dev_alloc
 *
 * @return usable mediary address to pass to a kernel
 */
static
void *host_imed2umed_addr(void *device_info, void *iaddr)
{
  return iaddr;
}


/**
 * Given a usable mediary address, it returns the internal mediary address
 *
 * @param device_info the device
 * @param umedaddr    allocated memory from hm_dev_alloc
 *
 * @return internal mediary address to be used by ORT
 */
static
void *host_umed2imed_addr(void *device_info, void *umedaddr)
{
	return umedaddr;
}


/* new-hostpart-func.sh:hostfuncdef */


ort_module_t *hostdev_get_module()
{
	ort_module_t *m = &(ort->module_host);

	/* Fields:
	m->initialize       =
	m->finalize         =
	m->unified_medaddr  = set to 1 by default by ort_init
	*/
	m->offload          = host_offload;
	m->dev_alloc        = host_dev_alloc;
	m->dev_init_alloc_global = host_dev_init_alloc_global;
	m->dev_free         = host_dev_free;
	m->dev_free_global  = host_dev_free_global;
	m->todev            = host_todev;
	m->fromdev          = host_fromdev;
	m->imed2umed_addr   = host_imed2umed_addr;
	m->umed2imed_addr   = host_umed2imed_addr;
/* new-hostpart-func.sh:hostload */

	m->initialized      = 1;
	m->initialized_successful = 1;
#ifdef OMPI_REMOTE_OFFLOADING
	m->remote           = false;
#endif
	m->is_cpumodule     = false;
	m->name             = "cpu";
	
	return m;
}


int avoid_optimizer = 11;



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * DEVICE-PART FUNCTIONS FOR THE HOST DEVICE                         *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


static char *hostdev__dev_med2dev_addr(void *medaddr, unsigned long size)
{
	return medaddr;
}


/* you may want to override the default behavior */
char *(*actual__dev_med2dev_addr)(void *, u_long) = hostdev__dev_med2dev_addr;


char *_dev_med2dev_addr(void *medaddr, u_long size)
{
	return actual__dev_med2dev_addr(medaddr, size);
}
