/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* env.c -- OpenMP environmental variables */

#include "ort_prive.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <setjmp.h>

#ifdef HAVE_HWLOC
	#include <hwloc.h>
#endif

#ifdef OMPI_REMOTE_OFFLOADING
	#include "remote/roff.h"
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * GLOBAL VARIABLES / DEFINITIONS / MACROS                           *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#define ISTRUE(s) \
	(strncasecmp(s, "true", 4) == 0 || strncmp(s, "1", 1) == 0)
#define ISFALSE(s) \
	(strncasecmp(s, "false", 5) == 0 || strncmp(s, "0", 1) == 0)


#ifndef HAVE_STRNCASECMP
static int strncasecmp(char *s, char *t, int len)
{
	for (; *s && *t && len; len--, s++, t++)
		if (tolower((int) *s) != tolower((int) *t))
			break;
	return ( len ? tolower((int) *s) - tolower((int) *t) : 0 );
}
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * OMP_STACKSIZE                                                     *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* size | sizeB | sizeK | sizeM | sizeG */
static
long int _size_bytes(char *str)
{
	long int size;
	char *s, mod;

	for (s = str; isspace(*s) || isdigit(*s); s++)
		;
	switch (*s)  /* specifier */
	{
		case   0:
		case 'k': case 'K': mod = 10; break;  /* default */
		case 'b': case 'B': mod = 0;  break;
		case 'm': case 'M': mod = 20; break;
		case 'g': case 'G': mod = 30; break;
		default:
			goto WRONG_STSIZE;
	};
	if (*s)      /* past the specifier */
	{
		for (*s = 0, s++; isspace(*s); s++) /* */ ;
		if (*s) goto WRONG_STSIZE;
	}

	if ((size = atoi(str)) < 0) goto WRONG_STSIZE;
	if (mod) size <<= mod;
	if (size < 128)
	{
		ort_warning("illegal OMP_STACKSIZE value (%ld); using 256K instead\n", size);
		size = KBytes(256);
	}
	return (size);

WRONG_STSIZE:
	ort_warning("incorrect OMP_STACKSIZE specification; using system default.\n");
	return (-1);
}


static
void _show_size_bytes(FILE *fp, size_t n)
{
	if (n < 1024)
		fprintf(fp, "%ldB", n);
	else 
		if (n < 1024*1024)
		{
			if (n & 1023)
				fprintf(fp, "%ldB", n);
			else 
				fprintf(fp, "%ldK", n >> 10);
		}
		else
			if (n < 1024*1024*1024)
			{
				if (n & (1024*1024 - 1))
					fprintf(fp, "%ldB", n);
				else 
					fprintf(fp, "%ldM", n >> 20);
			}
			else
			{
				if (n & (1024*1024*1024 - 1))
					fprintf(fp, "%ldB", n);
				else 
					fprintf(fp, "%ldG", n >> 30);
			}
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * RETRIEVE AND DISPLAY ENVIRONMENTAL VARIABLES                      *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


// Return value is a check for giving an error or a default value
static int extract_integer_value(char *s, char tag, int* location)
{
	char *t;
	if ((t = strchr(s, tag)) != NULL)
		if ((t = strchr(t, '=')) != NULL)
		{
			sscanf(t+1, "%d", location);
			return 1;
		}
	return 0;
}

#ifdef OMPI_XTRA_LOOPSCHEDS
static int extract_double_value(char *s, char tag, double* location)
{
	char *t;
	if ((t = strchr(s, tag)) != NULL)
		if ((t = strchr(t, '=')) != NULL)
		{
			sscanf(t+1, "%lf", location);
			return 1;
		}
	return 0;
}
#endif

void basic_scheduling_method_chunksize(char* s, int* n, xsched_data_t *sched) {
	if (*s == ',') // OpenMP syntax
		sscanf(s + 1, "%d", n);
	else if (*s == '(') { // Label syntax
		if (!extract_integer_value(s, 'c', n))
			return;
	} else // Wrong syntax or no chunksize
		return;

	if (*n>0) 
		sched->chunksize = (u_long) *n;
	else
		ort_warning("illegal chunksize in environment "
			"variable OMP_SCHED_AUTO; using default value\n");
}


void env_read_auto_schedule(char* env_variable, xsched_data_t *sched)
{
	char *s;
	int  n;
	
	if ((s = getenv(env_variable)) != NULL)
	{
		sched->chunksize = 1;
#ifdef OMPI_XTRA_LOOPSCHEDS
    char *t;
    
		if (strncasecmp(s, "Trapezoid", 9) == 0)
		{
			if ((t = strchr(s, '(')) != NULL)
				s = t;
			if (!extract_integer_value(s, 'f', &sched->first_chunk))
				sched->first_chunk = 0;
			if (extract_integer_value(s, 'l', &n))
			{
				if (n>0) 
					sched->chunksize = (u_long) n;
				else 
					ort_warning("illegal chunksize in environment "
						"variable OMPI_SCHED_AUTO; using default value\n");
			}
			sched->schedule = omp_sched_trapezoid;
		}
		else if (strncasecmp(s, "Taper", 5) == 0)
		{
			if ((t = strchr(s, '(')) != NULL)
				s = t;
			else
				TAPER_ERROR:
				ort_error(1, "The Taper policy has the following syntax:\n"
							"\t'Taper(m=<mean>, s=<standard deviation>)'\n"
							"\t(optional inputs:a=<scaling factor>, c=<chunksize>)\n");
			if (extract_integer_value(s, 'm', &sched->mean)){
				if (sched->mean <= 0){
					ort_warning("illegal mean value for Taper method"
						"; using value 1\n");
					sched->mean = 1;
				}
			}
			else
				goto TAPER_ERROR;
			if(!extract_double_value(s, 's', &sched->sigma))
				goto TAPER_ERROR;
			if(!extract_double_value(s, 'a', &sched->alpha))
				sched->alpha = 1;
			if (extract_integer_value(s, 'c', &n)){
				if (n>0) 
					sched->chunksize = (u_long) n;
				else
					ort_warning("illegal chunksize in environment "
						"variable OMP_SCHED_AUTO; using default value\n");
			}
			sched->schedule = omp_sched_taper;
		}
		else if (strncasecmp(s, "FSC", 3) == 0) 
		{
			if ((t = strchr(s, '(')) != NULL)
				s = t;
			else
				FSC_ERROR:
				ort_error(1,"The Fixed Size Chunking policy has the following syntax:\n"
							"\t'FSC(h=<overhead>, s=<standard deviation>)'\n");
			if(extract_double_value(s, 's', &sched->sigma)){
				if (sched->sigma <= 0){
					ort_warning("illegal s value for FSC method"
						"; using value 1\n");
					sched->sigma = 1;
				}
			}
			else 
				goto FSC_ERROR;
			if (!extract_integer_value(s, 'h', &sched->first_chunk))
				goto FSC_ERROR;
			sched->schedule = omp_sched_fsc;
		}
		else if (strncasecmp(s, "Factoring2", 10) == 0){
			sched->schedule = omp_sched_factoring2;
		}
		else if (strncasecmp(s, "Factoring", 9) == 0)
		{
			if ((t = strchr(s, '(')) != NULL)
				s = t;
			else
				FACTORING_ERROR:
				ort_error(1, "The Factoring policy has the following syntax:\n"
							"\t'Factoring(m=<mean>, s=<standard deviation>)'\n");
			if(!extract_double_value(s, 's', &sched->sigma))
				goto FACTORING_ERROR;
			if (extract_integer_value(s, 'm', &sched->mean)){
				if (sched->mean <= 0){
					ort_warning("illegal mean value for Factoring method"
						"; using value 1\n");
					sched->mean = 1;
				}
			}
			else
				goto FACTORING_ERROR;
			sched->first_chunk=0;
			sched->schedule = omp_sched_factoring;
		}
		else if (strncasecmp(s, "Profiling", 9) == 0){
			sched->schedule = omp_sched_profiling;
		}
#endif    /* OMPI_XTRA_LOOPSCHEDS */
		if (strncasecmp(s, "static", 6) == 0)
		{
			sched -> chunksize = 0;
			basic_scheduling_method_chunksize(s+6, &n, sched);
			sched->schedule = omp_sched_static;
		}
		else if (strncasecmp(s, "dynamic", 7) == 0)
		{
			basic_scheduling_method_chunksize(s+7, &n, sched);
			sched->schedule = omp_sched_dynamic;
		}
		else if (strncasecmp(s, "guided", 6) == 0)
		{
			basic_scheduling_method_chunksize(s+6, &n,sched);
			sched->schedule = omp_sched_guided;
		}
	}
}

void env_read_tag_threads(char* env_name, int nthreads[], int* set_levels)
{
	char *s;
	int n, l=0;

	if ((s = getenv(env_name)) != NULL){
		char *t = s-1;
		do {
			if (sscanf(t+1, "%d", &n) == 1 && n > 0){
				if (l < MAX_NUMTHR_LEVELS)
					nthreads[l++] = n;
				else
					break;
			}
			else
				break;
		} while ((t = strchr(t + 1, ',')) != NULL);
	}

	*set_levels = l;
}

/* Environmental variables */
void env_get_environment(void)
{
	char *s, *t;
	int  n, i;

	if ((s = getenv("OMP_DYNAMIC")) != NULL)
	{
		if (ISTRUE(s))
			ort->icvs.dynamic = 1;
		else
			ort->icvs.dynamic = 0;
	}

	if ((s = getenv("OMP_NESTED")) != NULL)
	{
		if (ISTRUE(s))
			ort->icvs.nested = 1;
		else
			ort->icvs.nested = 0;
	}

	if ((s = getenv("OMP_SCHEDULE")) != NULL)
	{
		if ((t = strchr(s, ',')) != NULL)   /* Chunksize specified. */
		{
			sscanf(t + 1, "%d", &n);
			if (n > 0)
				ort->icvs.rtchunk = (u_long) n;
			else
				ort_warning("illegal chunksize in environment "
				            "variable OMP_SCHEDULE; using default value\n");
		}
		if ((t = strchr(s, ':')) != NULL) /* Schedule modifiers */
		{	// for now just skips them
			s = t+1;
		}
		if (strncasecmp(s, "static", 6) == 0)
			ort->icvs.rtschedule = omp_sched_static;
		else
			if (strncasecmp(s, "dynamic", 7) == 0)
				ort->icvs.rtschedule = omp_sched_dynamic;
			else
				if (strncasecmp(s, "guided", 6) == 0)
					ort->icvs.rtschedule = omp_sched_guided;
				else
					if (strncasecmp(s, "auto", 4) == 0)
						ort->icvs.rtschedule = omp_sched_auto;
					else
					{
						ort_warning("incorrect schedule type of environment "
						            "variable OMP_SCHEDULE; 'auto' assumed\n");
						ort->icvs.rtschedule = omp_sched_auto;
					}
	}

#ifdef OMPI_XTRA_LOOPSCHEDS
	env_read_auto_schedule("OMPI_SCHED_AUTO", &ort->icvs.xsched);
#endif

	/* OpenMP 3.0 */
	if ((s = getenv("OMP_STACKSIZE")) != NULL)
		if ((n = _size_bytes(s)) > 0)
			ort->icvs.stacksize = n;
	/* OpenMP 3.0 */
	if ((s = getenv("OMP_THREAD_LIMIT")) != NULL)
		if (sscanf(s, "%d", &n) == 1 && n > 0)
			ort->icvs.threadlimit = n;
	/* OpenMP 3.0 */
	if ((s = getenv("OMP_MAX_ACTIVE_LEVELS")) != NULL)
		if (sscanf(s, "%d", &n) == 1 && n >= 0)
			ort->icvs.levellimit = n;
	/* OpenMP 3.0 */
	if ((s = getenv("OMP_WAIT_POLICY")) != NULL)
	{
		if (strncasecmp(s, "active", 6) == 0)
			ort->icvs.waitpolicy = _OMP_ACTIVE;
		else
			if (strncasecmp(s, "passive", 7) == 0)
				ort->icvs.waitpolicy = _OMP_PASSIVE;
			else
				ort_warning("incorrect value of environment "
				            "variable OMP_WAIT_POLICY; ignoring\n");
	}

	/* Modified in OpenMP 3.1 */
	if ((s = getenv("OMP_NUM_THREADS")) != NULL)
	{
		int  l = 0;
		char *t = s - 1;

		do
		{
			if (sscanf(t + 1, "%d", &n) == 1 && n > 0)
			{
				if (l < MAX_NUMTHR_LEVELS)
					ort->nthr_per_level[l++] = n;
				else
				{
					ort_warning("too many levels in OMP_NUM_THREADS environment "
					            "variable;\n  using the first %d.\n", MAX_NUMTHR_LEVELS);
					break;
				}
			}
			else
			{
				ort_warning("illegal value in environment variable "
				            "OMP_NUM_THREADS; using default values\n");
				l = 0;
				break;
			}
		}
		while ((t = strchr(t + 1, ',')) != NULL);   /* One more level */

		ort->set_nthrlevs = l;
		if (l > 0)
			ort->icvs.nthreads = ort->nthr_per_level[0];  /* Force this initialy */
	}

	/* Modified in OpenMP 4.0 */
	if ((s = getenv("OMP_PROC_BIND")) != NULL)
	{
		int  l = 0;
		char *t = s - 1;
		char bind[256];
		omp_proc_bind_t bind_choice;

		do
		{
			if (sscanf(t + 1, "%s", bind) == 1)
			{
				if (ISFALSE(bind))
				{
					if (l != 0)
					{
						ort_warning("illegal value in environment variable "
						            "OMP_PROC_BIND; using default values\n");
						l = 0;
					}
					else
						ort->icvs.proc_bind = omp_proc_bind_false;

					break;
				}
				else
					if (ISTRUE(bind))
					{
						if (l != 0)
						{
							ort_warning("illegal value in environment variable "
							            "OMP_PROC_BIND; using default values\n");
							l = 0;
						}
						else
							ort->icvs.proc_bind = omp_proc_bind_true;

						break;
					}
					else
					{
						if (strncasecmp(bind, "primary", 7) == 0)
							bind_choice = omp_proc_bind_primary;
						else if (strncasecmp(bind, "master", 6) == 0)
							bind_choice = omp_proc_bind_master;
						else if (strncasecmp(bind, "close", 5)  == 0)
							bind_choice = omp_proc_bind_close;
						else if (strncasecmp(bind, "spread", 6) == 0)
							bind_choice = omp_proc_bind_spread;
						else
						{
							ort_warning("illegal value in environment variable "
							            "OMP_PROC_BIND; using default values\n");
							l = 0;
							break;
						}
					}
				if (l < MAX_BIND_LEVELS)
					ort->bind_per_level[l++] = bind_choice;
				else
				{
					ort_warning("too many levels in OMP_PROC_BIND environment "
					            "variable;\n  using the first %d.\n", MAX_BIND_LEVELS);
					break;
				}
			}
			else
			{
				ort_warning("illegal value in environment variable "
				            "OMP_PROC_BIND; using default values\n");
				l = 0;
				break;
			}
		}
		while ((t = strchr(t + 1, ',')) != NULL);   /* One more level */

		ort->set_bindlevs = l;
		if (l > 0)
			ort->icvs.proc_bind = ort->bind_per_level[0];  /* Force this initialy */
	}

	/* OpenMP 4.0 */
	if ((s = getenv("OMP_CANCELLATION")) != NULL)
	{
		if (ISTRUE(s))
			ort->icvs.cancel = 1;
	}

	/* OpenMP 4.0 */
	if ((s = getenv("OMP_PLACES")) != NULL)
		places_getenv(s);
	places_get_default(); /* Fallback to default places if required */

	/* OpenMP 4.0 */
	if (((s = getenv("OMP_DEFAULT_DEVICE")) != NULL) && (sscanf(s, "%d", &n)==1))
	{
		if ((!__ort_mods_required) || IS_OMPDEVID(n))
			ort->icvs.def_ompdevid = n;
		else
#ifdef OMPI_REMOTE_OFFLOADING
			if (!ort->embedmode && node_role == ROLE_PRIMARY)
#else
			if (!ort->embedmode)
#endif
				ort->icvs.def_ompdevid = ort_illegal_device("OMP_DEFAULT_DEVICE", n);
	}

	/* OpenMP 4.5, section 4.14, p. 303 */
	if ((s = getenv("OMP_MAX_TASK_PRIORITY")) != NULL)
		if (sscanf(s, "%d", &n) == 1 && n >= 0)
			ort->icvs.max_task_prio = n;

	/* OpenMP 5.0 */
	if ((s = getenv("OMP_DISPLAY_AFFINITY")) != NULL)
	{
		if (ISTRUE(s))
			ort_warning("OMP_DISPLAY_AFFINITY: currently ignoring any non-FALSE value\n");
			// ort->icvs.display_affinity = 1;
	}

	/* OpenMP 5.0 */
	if ((s = getenv("OMP_AFFINITY_FORMAT")) != NULL)
		ort_set_affinity_format(s);

	/* OpenMP 5.0 */
	if ((s = getenv("OMP_TARGET_OFFLOAD")) != NULL)     /* Ignored for now */
	{
		if (strncasecmp(s, "mandatory", 9) == 0)
			ort->icvs.targetoffload = OFFLOAD_MANDATORY;
		else
			if (strncasecmp(s, "disabled", 8) == 0)
				ort->icvs.targetoffload = OFFLOAD_DISABLED;
			else
				if (strncasecmp(s, "default", 7) == 0)
					ort->icvs.targetoffload = OFFLOAD_DEFAULT;
				else
					ort_warning("incorrect value of environment "
					            "variable OMP_TARGET_OFFLOAD; ignoring\n");
	}

	/* OpenMP 5.1 */
	if (((s = getenv("OMP_NUM_TEAMS")) != NULL) && (sscanf(s, "%d", &n)==1))
		if (n >= 0)
			ort->icvs.nteams = n;
	if (((s = getenv("OMP_TEAMS_THREAD_LIMIT"))!=NULL) && (sscanf(s,"%d",&n)==1))
		if (n >= 0)
			ort->icvs.teams_thread_limit = n;

	/* 
	 * OMPi-specific below
	 */
	
	if (((s = getenv("OMPI_TASKQUEUESIZE")) != NULL) && (sscanf(s, "%d", &n) == 1)
	    && n > 0)
		ort->taskqueuesize = n;
	else
		ort->taskqueuesize = TASK_QUEUE_SIZE;

	if (((s = getenv("OMPI_DYNAMIC_TASKQUEUESIZE")) != NULL) && ISTRUE(s))
		ort->dynamic_taskqueuesize = 1;
	else
		ort->dynamic_taskqueuesize = 0;

	if ((s = getenv("OMPI_TASK_DEQUEUE_POLICY")) != NULL)
	{
		if (strncasecmp(s, "FIFO", 4) == 0)
			ort->task_dequeue_policy = FIFO;
		else
			if (strncasecmp(s, "LIFO", 4) == 0)
				ort->task_dequeue_policy = LIFO;
	}
	else
		ort->task_dequeue_policy = FIFO;

	if ((s = getenv("OMPI_PAR2TASK_POLICY")) != NULL)
	{
		if (ISFALSE(s))
			ort->partotask_policy = FALSE;
		else if (ISTRUE(s))
			ort->partotask_policy = TRUE;
		else if (strncasecmp(s, "AUTO", 4) == 0)
			ort->partotask_policy = AUTO;
	}
	else
		ort->partotask_policy = AUTO;
	
	if (((s = getenv("OMPI_HOSTTARGET_SHARE")) != NULL) && ISFALSE(s))
	{
		for (i = 0; i < ort->module_host.number_of_devices; i++)
			ort->hostdevs[i]->sharedspace = 0;
	}
}


static
void show_places(int beautify)
{
	int i, j, k, ub, plen;
	int **places = ort->place_partition;

	printf("\t[host] OMP_PLACES='");
	if (numplaces(places) <= 0)
		printf("' (no places defined)\n");
	else
	{
		for (i = 0; i < numplaces(places); i++)
		{
			printf("{");
			plen = placelen(places, i);
			for (j = 0; j < plen; j++)
			{
				for (ub = -1, k = j; beautify && k+1 < plen; k++)
				{
					if (places[i+1][k+1] + 1 == places[i+1][k+2])
						ub = k + 1;
					else
						break;
				}
				/*
				 * No need to check the value of beautify here.
				 * When beautify is false, ub equals -1 due to
				 * the for-loop initialization above.
				 */
				if (ub == -1 || ub == j + 1)
				{
					printf("%d%c", places[i + 1][j + 1],
							(j == plen - 1) ? '}' : ',');
				}
				else
				{
					printf("%d-%d%c", places[i + 1][j + 1],
							places[i + 1][ub + 1],
							(ub == plen - 1) ? '}' : ',');
					j = ub;
				}
			}
			printf("%s", (i == numplaces(places) - 1) ? "'" : ",");
		}
		printf(" (%d places defined)\n", numplaces(places));
	}
}


#define DISPVAR(v,val) printf("\t[host] " #v "='" #val "'\n")
#define DISPTFVAR(v,flag,t,f) \
          printf("\t[host] " #v "='%s'\n", (flag) ? #t : #f)

/* OpenMP 4.0 */
/* If verbose is 1, extra OMPi-specific env vars are displayed;
 * If verbose is 0, nothing extra is displayed;
 * If verbose is -1, then we do what OMP_DISPLAY_ENV says.
 */
void env_display_vars(int verbose)
{
	char *bind_choice[5] = {"FALSE", "TRUE", "PRIMARY", "CLOSE", "SPREAD"};
	int  i;

	if (verbose == -1)       /* Check if user asked for something */
	{
		char *s = getenv("OMP_DISPLAY_ENV");    /* since OpenMP 4.0 */
		if (!s) return;
		if (ISTRUE(s))
			verbose = 0;
		else
			if (strcasecmp(s, "VERBOSE") == 0)
				verbose = 1;
			else
				return;
	}
	
	printf("OPENMP DISPLAY ENVIRONMENT BEGIN\n");
	printf("\t_OPENMP='%d'\n", _OPENMP);

	/* 
	 * OpenMP 1.0 - OpenMP 2.5 
	 */
	
	DISPTFVAR(OMP_DYNAMIC, ort->icvs.dynamic, TRUE, FALSE);
	DISPTFVAR(OMP_NESTED, ort->icvs.nested, TRUE, FALSE);
	if (ort->icvs.rtchunk == 0)
		printf("\t[host] OMP_SCHEDULE='AUTO'\n");
	else
		printf("\t[host] OMP_SCHEDULE='%s, %d'\n", 
		       ort->icvs.rtschedule == omp_sched_static ? "STATIC" :
		       ort->icvs.rtschedule == omp_sched_dynamic ? "DYNAMIC" :
		       ort->icvs.rtschedule == omp_sched_guided ? "GUIDED" :
		       ort->icvs.rtschedule == omp_sched_auto ? "AUTO" : "STATIC",
		       ort->icvs.rtchunk);

	/* 
	 * OpenMP 3.0
	 */
	
	printf("\t[host] OMP_STACKSIZE='");
	if (ort->icvs.stacksize == -1)
	{
		pthread_attr_t tattr;
		size_t         size;
		
		pthread_attr_init(&tattr);
		pthread_attr_getstacksize(&tattr, &size);
		_show_size_bytes(stdout, size);
		printf("' (system default)\n");
	}
	else
	{
		_show_size_bytes(stdout, ort->icvs.stacksize);
		printf("'\n");
	}
	printf("\t[host] OMP_THREAD_LIMIT='%d'\n", ort->icvs.threadlimit);
	printf("\t[host] OMP_MAX_ACTIVE_LEVELS='%d'\n", ort->icvs.levellimit);
	DISPTFVAR(OMP_WAIT_POLICY,ort->icvs.waitpolicy==_OMP_PASSIVE,PASSIVE,ACTVE);


	/* 
	 * OpenMP 3.1
	 */
	
	printf("\t[host] OMP_NUM_THREADS='");
	if (ort->set_nthrlevs > 0)
		for (i = 0; i < ort->set_nthrlevs; i++)
			printf("%d%s", ort->nthr_per_level[i], 
			               (i != ort->set_nthrlevs-1) ? "," : "'\n");
	else
		printf("%d'\n", (ort->icvs.nthreads == -1) ? 
		                ort->icvs.ncpus : ort->icvs.nthreads);

	/* 
	 * OpenMP 4.0
	 */
	
	printf("\t[host] OMP_PROC_BIND='");
	if (ort->set_bindlevs > 0)
		for (i = 0; i < ort->set_bindlevs; i++)
			printf("%s%s,", bind_choice[ort->bind_per_level[i]],
			                (i != ort->set_bindlevs-1) ? "," : "'\n");
	else
		if (ort->icvs.proc_bind == omp_proc_bind_true)
			printf("TRUE'\n");
		else
			if (ort->icvs.proc_bind == omp_proc_bind_false)
				printf("FALSE'\n");

	DISPTFVAR(OMP_CANCELLATION, ort->icvs.cancel, ENABLED, DISABLED);
	printf("\t[host] OMP_DEFAULT_DEVICE='%d'\n", ort->icvs.def_ompdevid);

	show_places(1);

	/* 
	 * OpenMP 4.5
	 */
	
	printf("\t[host] OMP_MAX_TASK_PRIORITY='%d'\n", ort->icvs.max_task_prio);

	/*
	 * OpenMP 5.0
	 */
	DISPTFVAR(OMP_DISPLAY_AFFINITY, ort->icvs.display_affinity, TRUE, FALSE);
	printf("\t[host] OMP_AFFINITY_FORMAT='%s'\n", ort->icvs.affinity_format);
	
	printf("\t[host] OMP_TARGET_OFFLOAD='%s'\n", 
	      ort->icvs.targetoffload == OFFLOAD_DEFAULT ? "DEFAULT" :
	      ort->icvs.targetoffload == OFFLOAD_DISABLED ? "DISABLED" : "MANDATORY");

	/* 
	 * OMPi-specific stuff
	 */
	
	if (verbose)
	{
		printf("\n");
		printf("\t[host] OMPI_TASKQUEUESIZE='%d'\n", ort->taskqueuesize);

		DISPTFVAR(OMPI_DYNAMIC_TASKQUEUESIZE, ort->dynamic_taskqueuesize, 
		          ENABLED, DISABLED);
		DISPTFVAR(OMPI_TASK_DEQUEUE_POLICY, ort->task_dequeue_policy == FIFO, 
		          FIFO, LIFO);

		if (ort->partotask_policy == FALSE)
			DISPVAR(OMPI_PAR2TASK_POLICY, FALSE);
		else if (ort->partotask_policy == TRUE)
			DISPVAR(OMPI_PAR2TASK_POLICY, TRUE);
		else
			DISPVAR(OMPI_PAR2TASK_POLICY, AUTO);
		
		if (__ort_mods_required)
			DISPTFVAR(OMPI_HOSTTARGET_SHARE,ort->hostdevs[0]->sharedspace,TRUE,FALSE);
		
#ifdef OMPI_XTRA_LOOPSCHEDS
		xsched_data_t *sched = &(ort->icvs.xsched);
		printf("\t[host] OMPI_SCHED_AUTO=");
		switch (sched->schedule)
		{
			case omp_sched_trapezoid:
				printf("'TRAPEZOID(f=%d, l=%ld)'\n",sched->first_chunk,sched->chunksize);
				break;
			case omp_sched_fsc:
				printf("'FSC(s=%f, h=%d)'\n", sched->sigma, sched->first_chunk);
				break;
			case omp_sched_taper:
				printf("'TAPER(m=%d, s=%f, a=%f, c=%ld)'\n", sched->mean, sched->sigma, 
				       sched->alpha, sched->chunksize);
				break;
			case omp_sched_factoring:
				printf("'FACTORING(m=%d, s=%f)'\n", sched->mean, sched->sigma);
				break;
			case omp_sched_dynamic:
				printf("'DYNAMIC(c=%ld)'\n", sched->chunksize);
				break;
			case omp_sched_guided:
				printf("'GUIDED(c=%ld)'\n", sched->chunksize);
				break;
			default:
				printf("'STATIC(c=%ld)'\n", sched->chunksize);
		}
#endif

		/* TODO: display eelib and devices env */
	}

	printf("OPENMP DISPLAY ENVIRONMENT END\n");
}
