/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* This is a template for the header file of a threading library for OMPi
 */

#ifndef __EE_H__
#define __EE_H__

#include "config.h"

#include <unistd.h>
#include <pthread.h>
#include "../sysdeps.h"

/* The threading library should provide the following types
 * (the sample here comes from ee_pthreads):
 *    othr_key_t
 *    othr_lock_t
 * 
 */
typedef struct                /* For nested locks */
{
	pthread_mutex_t lock;     /* The lock in question */
	pthread_mutex_t ilock;    /* Lock to access the whole struct */
	pthread_cond_t  cond;     /* For waiting until lock is available */
	int             count;    /* # times locked by the same thread */
	void           *owner;    /* The owner (task) of the nestable lock */
} othr_nestlock_t;

typedef union
{
	struct
	{
		int   type;                 /* normal/spin/nested */
		union
		{
			pthread_mutex_t normal;   /* normal lock */
			othr_nestlock_t nest;     /* nest lock */
#ifdef HAVE_SPINLOCKS           /* spin lock */
			pthread_spinlock_t spin;
#else
			struct
			{
				int             rndelay;     /* Used for initial spin delays */
				pthread_mutex_t mutex;
			} spin;
#endif
		} data;
	} lock;
} othr_lock_t;


/*
 * The threading library should provide the functions that follow.
 */

/* Base functions */
extern int   othr_initialize(int *argc, char ***argv,
                             ort_icvs_t *icv, ort_caps_t *cap);
extern void  othr_finalize(int exitvalue);
extern int   othr_request(int numthreads, int level, int oversubscribe);
extern void  othr_create(int numthreads, int level, void *arg, void **info);
extern void  othr_waitall(void **info);
#define othr_yield() sched_yield()

/* Additional functions */
	/* Bind to given place index */
extern int   othr_bindme(int **places, int pindex);
	/* Native execution entity descriptor (opaque) and the size it occupies */
extern void *othr_getself(unsigned int *size);  
	/* Native execution entity id */
extern int   othr_getselfid(void);

/* Thread-specifics */
#define othr_key_t            pthread_key_t
#define othr_key_create(a,b)  pthread_key_create(a,b)
#define othr_getspecific(a)   pthread_getspecific(a)
#define othr_setspecific(a,b) pthread_setspecific(a,b)

/* Locks */
extern int  othr_init_lock(othr_lock_t *lock, int type);
extern int  othr_destroy_lock(othr_lock_t *lock);
extern int  othr_set_lock(othr_lock_t *lock);
extern int  othr_unset_lock(othr_lock_t *lock);
extern int  othr_test_lock(othr_lock_t *lock);

/* Barrier (Optional)
 * If you uncomment the #define below, you must provide the othr_barrier_t 
 * type and the next 4 functions.
 */
//#define AVOID_OMPI_DEFAULT_BARRIER
#if defined(AVOID_OMPI_DEFAULT_BARRIER)
typedef int othr_barrier_t;
extern void othr_barrier_init(othr_barrier_t *barrier, int nthreads);
extern void othr_barrier_wait(othr_barrier_t *barrier, int nthreads);
extern void othr_barrier_destroy(othr_barrier_t *barrier);
extern int  _ort_barrier_me(int ignoreReason);
#endif

/* Tasks (Optional)
 * If you uncomment the #define below, you must provide the ort_task_node_t
 * (and takgroup_t) types, and the following functions.
 */
//#define AVOID_OMPI_DEFAULT_TASKS
#if defined(AVOID_OMPI_DEFAULT_TASKS)
typedef struct taskgroup_s             /* This should actually not be needed */
{
	struct taskgroup_s *next;            /* For use in garbage collector */
	volatile int        is_canceled;     /* Cancel flag for taskgroup */
} taskgroup_t;

typedef struct { <whatever> } ort_task_node_t;
extern void othr_new_task(int final, int flag, void *(*func)(void *arg), void *arg);
extern void othr_taskwait(int how, void *info, int thread_num);
extern void othr_set_currtask(void **eeinfo, ort_task_node_t *tw);
extern ort_task_node_t *othr_get_currtask(void **eeinfo, int thread_num);
extern void *othr_taskenv_alloc(int size, void *task_func);
extern othr_taskenv_free(void *arg);
/* The following functions will NOT get called as long as you compile user
 * programs with --taskopt=size option, but you will need to provide dummy
 * implementations for them to keep the compiler happy. 
 */
extern void *othr_task_immediate_start(int final);
extern void othr_task_immediate_end(void *new_node);
#endif  /* defined(AVOID_OMPI_DEFAULT_TASKS) */

#endif
