/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <stdlib.h>
#include <pthread.h>
#include <vulkan/vulkan.h>
#include "context.h"
#include "vkgpu.h"
#include "assorted.h"
#include <assert.h>

/* context.c -- init/destroy thread-specific Vulkan contexts */

static 
uint32_t _find_compute_queue_family(vk_gpu_t *gpu) 
{
	uint32_t queueFamilyCount = 0, i;
	VkQueueFamilyProperties *queue_families;
	vkGetPhysicalDeviceQueueFamilyProperties(gpu->pdev.physical_device, &queueFamilyCount, NULL);

	queue_families = (VkQueueFamilyProperties *) smalloc(queueFamilyCount * sizeof(VkQueueFamilyProperties));

	vkGetPhysicalDeviceQueueFamilyProperties(gpu->pdev.physical_device, &queueFamilyCount, queue_families);

	for (i = 0; i < queueFamilyCount; i++)
		if (queue_families[i].queueFlags & VK_QUEUE_COMPUTE_BIT)
		{
			free(queue_families);
			return i;
		}

	free(queue_families);
	fprintf(stderr, "[vulkan] vkgpu: error: Failed to find a compute queue family!\n");
	exit(EXIT_FAILURE);
}


static 
void create_command_pool(vk_gpu_t *gpu, vk_gpu_ctx_t *context) 
{
	VkCommandPoolCreateInfo pool_info = {0};

	pool_info.sType = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO;
	pool_info.queueFamilyIndex = _find_compute_queue_family(gpu);

	if (vkCreateCommandPool(context->device, &pool_info, NULL, &(context->command_pool)) != VK_SUCCESS) 
	{
		fprintf(stderr, "[vulkan] vkgpu: error: Failed to create command pool!\n");
		exit(EXIT_FAILURE);
	}
}


static 
void create_command_buffer(vk_gpu_t *gpu, vk_gpu_ctx_t *context) 
{
	VkCommandBufferAllocateInfo alloc_info = {0};

	alloc_info.sType = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO;
	alloc_info.commandPool = context->command_pool;
	alloc_info.level = VK_COMMAND_BUFFER_LEVEL_PRIMARY;
	alloc_info.commandBufferCount = 1;

	if (vkAllocateCommandBuffers(context->device, &alloc_info, &(context->command_buffer)) != VK_SUCCESS) 
	{
		fprintf(stderr, "[vulkan] vkgpu: error: Failed to allocate command buffer!\n");
		exit(EXIT_FAILURE);
	}
}


static 
void create_logical_device(vk_gpu_t *gpu, vk_gpu_ctx_t *context)
{
	VkDeviceQueueCreateInfo  queue_create_info = {0};
	VkPhysicalDeviceFeatures device_features = {0};
	VkDeviceCreateInfo       create_info = {0};
	float queue_priority = 1.0f;
	
	queue_create_info.sType = VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO;
	queue_create_info.queueFamilyIndex = _find_compute_queue_family(gpu);
	queue_create_info.queueCount = 1;
	queue_create_info.pQueuePriorities = &queue_priority;

	create_info.sType = VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO;
	create_info.pQueueCreateInfos = &queue_create_info;
	create_info.queueCreateInfoCount = 1;
	create_info.pEnabledFeatures = &device_features;

	if (vkCreateDevice(gpu->pdev.physical_device, &create_info, NULL, &(context->device)) != VK_SUCCESS)
	{
		fprintf(stderr, "[vulkan] vkgpu: error: Failed to create logical device!\n");
		exit(EXIT_FAILURE);
	}
	
	vkGetDeviceQueue(context->device, queue_create_info.queueFamilyIndex, 0, &(context->compute_queue));
}


/**
 * Tests if a thread-specific Vulkan context is set. If not,
 * it initializes and sets a new one.
 * 
 * @param gpu 
 */
void vkgpu_ctx_test_init(vk_gpu_t *gpu)
{
	vk_gpu_ctx_t *context;
#ifdef VK_ENABLE_CONCURRENCY
	context = (vk_gpu_ctx_t*) pthread_getspecific(gpu->context_key);
#else
	context = gpu->context;
#endif

	if (context == NULL)
	{
		context = (vk_gpu_ctx_t*) scalloc(1, sizeof(vk_gpu_ctx_t));
		create_logical_device(gpu, context);
		create_command_pool(gpu, context);
		create_command_buffer(gpu, context);
#ifdef VK_ENABLE_CONCURRENCY
		pthread_setspecific(gpu->context_key, context);
#else
		gpu->context = context;
#endif
	}
}


/**
 * Tests if a thread-specific Vulkan context is set. If so,
 * it destroys and frees it.
 * 
 * @param gpu 
 */
void vkgpu_ctx_test_destroy(vk_gpu_t *gpu)
{
	vk_gpu_ctx_t *context;
#ifdef VK_ENABLE_CONCURRENCY
	context = (vk_gpu_ctx_t*) pthread_getspecific(gpu->context_key);
#else
	context = gpu->context;
#endif

	if (context != NULL)
	{
		vkDeviceWaitIdle(context->device);
		vkDestroyCommandPool(context->device, context->command_pool, NULL);
		vkDestroyDevice(context->device, NULL);
		free(context);

#ifdef VK_ENABLE_CONCURRENCY
		pthread_setspecific(gpu->context_key, NULL);
#else
		gpu->context = NULL;
#endif
	}
}


/** 
 * Returns the thread-specific Vulkan context. If not set,
 * it exits with an error.
 * 
 * @param gpu 
 * @return the thread-specific Vulkan context
 */
vk_gpu_ctx_t* vkgpu_ctx_get(vk_gpu_t *gpu)
{
	vk_gpu_ctx_t *context;
#ifdef VK_ENABLE_CONCURRENCY
	context = (vk_gpu_ctx_t*) pthread_getspecific(gpu->context_key);
#else
	context = gpu->context;
#endif

	if (context == NULL)
	{
		fprintf(stderr, "[vulkan] vkgpu: context is not initialized; exiting.\n");
		exit(1);
	}

	return context;
}
