/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <stdlib.h>
#include <pthread.h>
#include <cuda.h>
#include "stream.h"
#include "cudautils.h"

/* stream.c -- init/destroy thread-specific CUDA streams */

/**
 * Tests if a thread-specific CUDA stream is set. If not,
 * it initializes and sets a new one.
 */
void cuda_stream_test_init(cuda_dev_t *cuda_gpu)
{
	CUstream *stream = (CUstream*) pthread_getspecific(cuda_gpu->stream_key);

	if (stream == NULL)
	{
		stream = (CUstream*) malloc(sizeof(CUstream));
		cuda_do(cuStreamCreate(stream, CU_STREAM_NON_BLOCKING));
		pthread_setspecific(cuda_gpu->stream_key, stream);
	}
}

/**
 * Tests if a thread-specific CUDA stream is set. If so,
 * it destroys and frees it.
 */
void cuda_stream_test_destroy(cuda_dev_t *cuda_gpu)
{
	CUstream *stream = (CUstream*) pthread_getspecific(cuda_gpu->stream_key);

	if (stream != NULL)
	{
		cuda_do(cuStreamDestroy(*stream));
		free(stream);
	}
}

void cuda_stream_sync(cuda_dev_t *cuda_gpu)
{
	CUstream *stream = (CUstream*) pthread_getspecific(cuda_gpu->stream_key);

	if (stream != NULL)
		cuda_do(cuStreamSynchronize(*stream));
}

CUstream *cuda_stream_get(cuda_dev_t *cuda_gpu)
{
	CUstream *stream = (CUstream*) pthread_getspecific(cuda_gpu->stream_key);

	if (stream == NULL)
	{
		fprintf(stderr, "[cuda] stream is not initialized; exiting.\n");
		exit(1);
	}

	return stream;
}
