/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <cuda.h>
#include <stdio.h>
#include <string.h>

typedef struct {
	int minor;
	int major;
} sm_t;

sm_t smnums[16];
int numdevices = 0;

sm_t get_device_sm_num(int devnum)
{
	CUresult err;
	CUdevice device;
	sm_t smnum;

	err = cuDeviceGet(&device, devnum);

	if (err != CUDA_SUCCESS)
	{
		fprintf(stderr, "[cuda-smversion] could not get CUDA device %d\n", devnum);
		exit(-1);
	}

	/* Get SM version major and minor */
	err = cuDeviceGetAttribute(&(smnum.major), 
	                          CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MAJOR, 
	                          device);

	if (err != CUDA_SUCCESS)
	{
		fprintf(stderr, "[cuda-smversion] could not get CUDA device %d SM major\n", devnum);
		exit(-1);
	}

	err = cuDeviceGetAttribute(&(smnum.minor), 
	                          CU_DEVICE_ATTRIBUTE_COMPUTE_CAPABILITY_MINOR, 
	                          device);

	if (err != CUDA_SUCCESS)
	{
		fprintf(stderr, "[cuda-smversion]could not get CUDA device %d SM minor\n", devnum);
		exit(-1);
	}

	return smnum;
}

void get_all_devices_sm_num(sm_t *s)
{
	int i = 0;
	int device_count = 0;

	sm_t temp;

	cuDeviceGetCount(&device_count);

	if (device_count == 0) {
		fprintf(stderr, "[cuda-smversion] error: no CUDA-capable devices found\n");
		exit(-1);
	}

	for (i=0; i<device_count; i++)
	{
		temp = get_device_sm_num(i);
		
		/* Count only GPUs that have compute capability >= 3.5 */
		if ((temp.major << 4) + temp.minor >= 0x35)
		{
			s[numdevices].minor = temp.minor;
			s[numdevices].major = temp.major;
			numdevices++;
		}
	}
}

void print_gencode_flags(sm_t *s)
{
	int i;

	for (i=0; i<numdevices; i++)
		printf("-gencode arch=compute_%1$d%2$d,code=sm_%1$d%2$d ", s[i].major, s[i].minor);
	printf("\n");
}

void print_arch_nums(sm_t *s)
{
	int i;

	for (i=0; i<numdevices; i++)
	{
		printf("%d%d", s[i].major, s[i].minor);
		if (i!=numdevices-1)
			printf(",");
	}
	printf("\n");
}

int main(int argc, char **argv) 
{
	CUresult err;
	err = cuInit(0);	

	if (err != CUDA_SUCCESS)
	{
		fprintf(stderr, "[cuda-smversion] could not initialize CUDA\n");
		fprintf(stderr, "  If CUDA is properly installed, please also check your user permissions.\n");
		exit(1);
	}

	get_all_devices_sm_num(smnums);

	if ((argc == 2) && (strcmp(argv[1], "--numbers-only") == 0))
		print_arch_nums(smnums);
	else
		print_gencode_flags(smnums);
	
	exit(0);
}
