#!/bin/bash

# OMPi OpenMP Compiler
# == Copyright since 2001 the OMPi Team
# == Dept. of Computer Science & Engineering, University of Ioannina
# 
# This file is part of OMPi.
# 
# OMPi is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# OMPi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with OMPi; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

# Utility that compiles an OMPi kernel as a CUDA fatbinary and links with
# devpart, using the nvcc compiler toolchain.

#
# Argument handling
#
if [ $# -lt 2 ]; then
	echo "[cuda-compile.sh] Syntax: ./cuda-compile.sh <ompi-installation-dir> "\
	"<ompi-cuda-libpath> <kernel-input-filename> "\
	"[--devpart=<libdevpart>] [--output=<kernel-output-filename>] [--compiler-flags \"<flags>\"] "\
	"[--linker-flags \"<flags>\"]";
	exit 1;
fi

OMPI_INSTALL_DIR="$1"
OMPI_CUDA_LIBPATH="$2"
KERNEL_FILE="$3"

if [ ! -f "$OMPI_INSTALL_DIR"/bin/ompicc ]; then
	echo "[cuda-compile.sh] Incorrect OMPi installation directory \"$OMPI_INSTALL_DIR\". Exiting.";
	exit 1;
elif [ ! -d "$OMPI_CUDA_LIBPATH" ]; then
	echo "[cuda-compile.sh] OMPi CUDA device not found in \"$OMPI_CUDA_LIBPATH\". Exiting.";
	exit 1;
fi

if [ ! -f "$KERNEL_FILE" ]; then
	echo "[cuda-compile.sh] Kernel file \"$KERNEL_FILE\" does not exist. Exiting.";
	exit 1;
fi

EXTENSION="fatbin"
KERNEL_DIR=$(dirname "$KERNEL_FILE")
KERNEL_INPUT_FILENAME="${KERNEL_FILE##*/}"
KERNEL_INPUT_FILENAME="${KERNEL_INPUT_FILENAME%.cu}"
CUCFLAGS=""
CULDFLAGS=""

for arg in "${@}"
do
	if [ "${arg}" == "--fatbin" ]; then
		EXTENSION="fatbin"
	elif [ "${arg}" == "--ptx" ]; then
		EXTENSION="ptx"
	elif [ "${arg}" == "--cubin" ]; then
		EXTENSION="cubin"
	fi
done

KERNEL_OUTPUT_FILENAME=${KERNEL_FILE%.cu}.${EXTENSION}

LIBDEVPART=devpart

if [ $# -ge 4 ]; then
	for arg in "${@:3}"
	do
		if [ "${arg:0:9}" == "--output=" ]; then
			KERNEL_OUTPUT_FILENAME=${arg:9}
			OUTPUT_EXTENSION=${KERNEL_OUTPUT_FILENAME##*.}
			if [ "$OUTPUT_EXTENSION" != "${EXTENSION}" ]; then
				echo "[cuda-compile.sh] Output extension is not "${EXTENSION}". Exiting.";
				exit 1;
			fi
		elif [ "${arg:0:10}" == "--devpart=" ]; then
			LIBDEVPART=${arg:10}
		elif [ "${arg:0:16}" == "--compiler-flags" ]; then
			CUCFLAGS=${arg:17}
		elif [ "${arg:0:14}" == "--linker-flags" ]; then
			CULDFLAGS=${arg:15}
		fi
	done
fi


#
# NVIDIA compiler toolchain variables
#
CPUARCH=$(uname -m)
LFLAG=""
if [[ $CPUARCH == *"64"* ]]; then
	MFLAG="-m64"
	LFLAG="64"
else
	MFLAG=""
	LFLAG=""
fi

#
# First check if there is an environment variable
#
if [ -z "$OMPI_CUDA_ARCH_TARGET" ]; then
	#
	# Then check if there is a corresponding smversion file
	#
	OMPI_CUDA_ARCH_TARGET=$(cat "$OMPI_CUDA_LIBPATH"/cuda-smversion.txt 2>/dev/null)
	if [ -z "$OMPI_CUDA_ARCH_TARGET" ]; then
		OMPI_CUDA_ARCH_TARGET=$("$OMPI_CUDA_LIBPATH"/cuda-smversion)
		exitval=$?

		if [ "$exitval" -eq 1 ]; then
			echo "[cuda-compile.sh] ""${KERNEL_INPUT_FILENAME}"",arch=""${arch}"": CUDA is not properly installed; exiting."
		fi
	fi
fi

#
# The actual compilation
#
nvcc -O3 -I$OMPI_INSTALL_DIR/include --compiler-options "${CUCFLAGS}" --linker-options "${CULDFLAGS}" \
	--disable-warnings -Xnvlink="--suppress-stack-size-warning" -c ${MFLAG} ${OMPI_CUDA_ARCH_TARGET} \
	--${EXTENSION} -dlink -L "$OMPI_CUDA_LIBPATH" -l${LIBDEVPART} \
	-o "$KERNEL_OUTPUT_FILENAME" "$KERNEL_DIR/$KERNEL_INPUT_FILENAME.cu"
	
if [ "$EXTENSION" == "ptx" ]; then
	ptxfile=$(tail -n +2 "$KERNEL_OUTPUT_FILENAME")
	ompilinkstr="// ompi-link-with:lib${LIBDEVPART}.a"
	printf "%s\n%s\n" "${ompilinkstr}" "${ptxfile}" > "$KERNEL_OUTPUT_FILENAME"
fi

exit 0
