/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* This is the host-side part of the module; it should be compiled
 * to a shared library. It is dynamically linked to the host runtime at runtime.
 */

#include "rt_common.h"


int hm_unified_medaddr = 0; /* use 1 if internal and usable mediary addresses are the same */
char *modulename = NULL;


/* Pointers to lock functions of the host runtime
 */
void (*init_lock)(void **lock, int type);
void (*lock)(void **lock);
void (*unlock)(void **lock);
int  (*hyield)(void);
int  (*strprintf)(str s, char *fmt, ...);


/**
 * Calculates the number of available devices supported by this module
 *
 * @return number of devices
 */
int hm_get_num_devices(void)
{
	return 1;
}


/**
 * Prints information for this module and its available devices, 
 * to a dedicated string (info). The information is printed using the 
 * registered strprintf function.
 * While the devices this module serves are numbered starting from 0 (local
 * device id), the global device ids are set by ORT; the devid_offset parameter
 * gives the global device id of local device 0.
 *
 * @param devid_offset  the global id of the 1st device served by this module
 * @param info          the string which all the information will be printed to
 */
void hm_print_information(int devid_offset, str info)
{
	strprintf(info, "");
}


/**
 * Registers host runtime functions (currently it registers functions for locks)
 *
 * @param str_printf_in pointer to the function used for printing to a string
 */
void hm_register_str_printf(int  (*str_printf_in)(str s, char *fmt, ...))
{
	strprintf = str_printf_in;
}


/** 
 * Initializes the module
 * 
 * @param modname the name of the module
 * @param global_omp_id_of_first_device the global OpenMP ID of the first device
 * @param init_lock_in pointer to the function used for initializing a lock.
 *                     It's parameters are the address of a "void *" variable
 *                     and one of the "ORT_LOCK_*" defines denoting the type of
 *                     the lock
 * @param lock_in      pointer to the function used for acquiring a lock
 * @param unlock_in    pointer to the function used for releasing a lock
 * @param hyield_in    pointer to the function used for thread yield
 * 
 * @return             the number of available devices on success, 0 on failure
 */
int hm_initialize(char *modname, int global_omp_id_of_first_device,
                  void (*init_lock_in)(void **lock, int type),
                  void (*lock_in)(void **lock),
                  void (*unlock_in)(void **lock),
                  int  (*hyield_in)(void), int *argc, char ***argv)
{
	modulename = strdup(modname ? modname : "noname");

	init_lock = init_lock_in;
	lock      = lock_in;
	unlock    = unlock_in;
	hyield    = hyield_in;

	return hm_get_num_devices();
}


/** 
 * Finalizes the module
 */
void hm_finalize(void)
{
	if (modulename)
	{
		free(modulename);
		modulename = NULL;
	}
}


/**
 * Initializes a device
 *
 * @param dev_num     the (local) id of the device to initialize
 *                    (0 <= dev_num < hm_get_num_devices())
 * @param ort_icv     Pointer to struct with
 *                    initial values for the device ICVs.
 * @param sharedspace (ret) set to true if the device address space 
 *                    is identical to host (default: false)
 * @param argc        Pointer to main function's argc.
 * @param argv        Pointer to main function's argv.
 *
 * @return device_info: arbitrary pointer that will be passed back in
 *         following calls (see below).
 *         Return NULL only if it failed to initialize.
 */
void *hm_dev_init(int dev_num, ort_icvs_t *ort_icv, int *sharedspace)
{
	return NULL;
}


/**
 * Finalizes a device
 *
 * @param device_info the pointer returned by hm_dev_init()
 */
void hm_dev_end(void *device_info)
{
}


/**
 * Offloads and executes a kernel file, returnin 0 if all goes OK.
 * num_teams, num_threads and thread_limit are the values of the homonymous
 * clauses, if the offloading construct was a combined target-teams-*, or 
 * target-*-parallel-* construct; otherwise they have a 0 value.
 * Returns non-zero if there was an error.
 *
 * @param device_info         the device (the pointer returned by hm_dev_init())
 * @param host_func pointer   to offload function on host address space
 * @param dev_data pointer    to a struct containing kernel variables
 * @param decl_data pointer   to a struct containing globally declared variables
 * @param kernel_filename_prefix filename of the kernel (without the suffix)
 * @param num_teams           the clause value from teams construct
 * @param num_threads         the clause value from combined parallel constructs
 * @param thread_limit        the clause value from teams construct
 * @param teamdims            an unsigned long long that contains the
 *                            dimensions of the launched league, encoded as follows:
 *                            x: bits 0-20, y: bits 21-41, z: bits 42-62 
 * @param thrdims             an unsigned long long that contains the
 *                            dimensions of each thread team, encoded as follows:
 *                            x: bits 0-20, y: bits 21-41, z: bits 42-62 
 * @param num_args            an array that contains the number of declare variables, 
 *                            firstprivates and mapped variables
 * @param args                the addresses of all target data and target
 *                            declare variables
 * @return                    0 on success, non-zero otherwise
 *
 * NOTE: `teamdims' and `thrdims' can be decoded using the _ull_decode3 function.
 */
int hm_offload(void *device_info, void *(*host_func)(void *), void *dev_data,
               void *decl_data, char *kernel_filename_prefix, int num_teams,
               int num_threads, int thread_limit, 
               unsigned long long teamdims, unsigned long long thrdims, 
               int *num_args, void **args)
{
	return 0;
}


/**
 * Allocates memory "on the device"
 *
 * @param device_info the device (the pointer returned by hm_dev_init())
 * @param size        the number of bytes to allocate
 * @param map_memory  used in OpenCL, when set to 1 additionaly to the memory
 *                    allocation in shared virtual address space, the memory
 *                    is mapped with read/write permissions so the host cpu
 *                    can utilize it.
 * @param hostaddr    used in MPI to allocate #declare target link variables;
 *                    you can safely ignore this argument.
 * @param map_type    the mapping type that triggered this allocation (to/from/tofrom/alloc)
 * @return            pointer to the allocated space (internal mediary address)
 */
void *hm_dev_alloc(void *device_info, size_t size, int map_memory, void *hostaddr,
                   int map_type)
{
	return NULL;
}


/**
 * Allocates & initializes memory "on the device" for a global variable
 *
 * @param device_info the device (the pointer returned by hm_dev_init())
 * @param global_id   the ID of the global variable
 * @param size        the number of bytes to allocate

 * @return            pointer to the allocated space (internal mediary address)
 */
void *hm_dev_init_alloc_global(void *device_info, void *initfrom, size_t size, int global_id,
                               void *hostaddr)
{
	return NULL;
}


/**
 * Frees data allocated with hm_dev_alloc
 *
 * @param device_info  the device (the pointer returned by hm_dev_init())
 * @param imedaddr     pointer to the memory that will be released
 * @param unmap_memory used in OpenCL, when set to 1 prior to the memory
 *                     deallocation, the memory is unmapped.
 */
void hm_dev_free(void *device_info, void *imedaddr, int unmap_memory)
{
}


/**
 * Frees a global variable allocated with hm_dev_init_alloc_global
 *
 * @param device_info  the device (the pointer returned by hm_dev_init())
 * @param iaddr        pointer to the memory that will be released
 * @param global_id    the ID of the global variable that will be released
 */
void hm_dev_free_global(void *device_info, void *iaddr, int global_id)
{
}


/**
 * Transfers data from the host to a device
 *
 * @param device_info the device (the pointer returned by hm_dev_init())
 * @param hostaddr    the source memory
 * @param hostoffset  offset from hostaddr
 * @param imedaddr    the target memory (internal mediary address)
 * @param devoffset   offset from imedaddr
 * @param size        the size of the memory block
 */
void hm_todev(void *device_info, void *hostaddr, size_t hostoffset,
               void *imedaddr, size_t devoffset, size_t size)
{
}


/**
 * Transfers data from a device to the host
 *
 * @param device_info the device (the pointer returned by hm_dev_init())
 * @param hostaddr    the target memory
 * @param hostoffset  offset from hostaddr
 * @param imedaddr    the source memory (internal mediary address)
 * @param devoffset   offset from imedaddr
 * @param size        the size of the memory block
 */
void hm_fromdev(void *device_info, void *hostaddr, size_t hostoffset,
                 void *imedaddr, size_t devoffset, size_t size)
{
}


/**
 * Given an internal mediary address, it returns a usable mediary address
 *
 * @param device_info the device (the pointer returned by hm_dev_init())
 * @param imedaddr    allocated memory from hm_dev_alloc
 *
 * @return usable mediary address to pass to a kernel
 */
void *hm_imed2umed_addr(void *device_info, void *imedaddr)
{
	return imedaddr;
}


/**
 * Given a usable mediary address, it returns the internal mediary address
 *
 * @param device_info the device (the pointer returned by hm_dev_init())
 * @param umedaddr    allocated memory from hm_dev_alloc
 *
 * @return internal mediary address to be used by ORT
 */
void *hm_umed2imed_addr(void *device_info, void *umedaddr)
{
	return umedaddr;
}


/* new-hostpart-func.sh:funcdef */
