/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* sem.h - Semantic/type analysis stuff */

#ifndef __SEM_H__
#define __SEM_H__

#include "ompi.h"
#include "ast_assorted.h"

typedef enum { /* numeric types are <= 12 (see below - UBOOL should be 63?) */ 
               ARRAY_T = 64, POINTER_T = 128, STRUNI_T = 256, 
               ENUM_T = 512, FUNC_T = 1024, VOID_T = 2048 
             } typeclass_e;

#define type_isnum(t)    ((t) >= 0 && (t) <= 12)
#define type_isarray(t)  ((t) == ARRAY_T)
#define type_isptr(t)    ((t) == POINTER_T)
#define type_isstruni(t) ((t) == STRUNI_T)
#define type_isenum(t)   ((t) == ENUM_T)
#define type_isfunc(t)   ((t) == FUNC_T)
#define type_isvoid(t)   ((t) == VOID_T)
/* Arithmetic types and pointer types are collectively called scalar types. 
   Array and structure types are collectively called aggregate types. */
#define type_isscalar(t)    (type_isnum(t) || type_isptr(t))
#define type_isaggregate(t) (type_isarray(t) || type_isstruni(t))

extern typeclass_e get_typeclass(astspec s, astdecl d);

typedef struct _semtype semtype_t;
struct _semtype {
	typeclass_e class;   /* Encoded type class */
	bool        isconst; /* Whether a numeric type is const-qualified */
	symbol      name;    /* The name of struct/union/enum */
	astspec     spec;    /* The specifier of an aggregate var declaration */
	astdecl     decl;    /* The declarator of an aggregate var declaration */
	semtype_t  *next;    /* When lists are needed (e.g. pointer-to) */
};

/* Semantic types */
extern semtype_t *sem_expr_type(astexpr e);
extern semtype_t *sem_declaration_type(astspec s, astdecl d);
extern void sem_type_to_declaration(semtype_t *type, astspec *s, 
                                    astdecl *d, bool abstract);
extern void sem_type_show(semtype_t *t);
  /* typeof() facilities directly from expressions */
extern bool sem_expr_typeof(astexpr e, astspec *s, astdecl *d);
extern bool sem_expr_typeof_declare(astexpr expr,symbol v,astspec *s,astdecl*d);


/* Value categories */
extern int  sem_expr_islvalue(astexpr tree);   /* > 0 if expr is l-value
                                                  > 1 if non-modifiable */
extern bool sem_expr_isfuncdes(astexpr tree);  /* true if function designator */
extern void sem_expr_showinfo(astexpr tree);

/*
 * Numeric semtypes
 */

/* Do NOT change the values or type encoding breaks */
typedef enum { NORMAL_T=0, SHORT_T=1, LONG_T=2, LONGLONG_T=3 } nt_size;
typedef enum { SIGNED_T=0, UNSIGNED_T=4 } nt_sign;
typedef enum { INT_T=0, CHAR_T=8, DOUBLE_T=9, FLOAT_T=10, UBOOL_T=63 } 
             nt_basetype;
extern nt_size     speclist_size(astspec s);
extern nt_sign     speclist_sign(astspec s);
extern nt_basetype speclist_basetype(astspec s);

/* Numeric type encoding [0-12] */
extern int numtype(nt_basetype b, nt_sign s, nt_size z);
extern int spec_to_numtype(astspec spec);
extern astspec numtype_to_spec(int nt);

/* Macros for the encoded arithmetic type */
#define numtype_isint(t)      ((t) <= 7)
#define numtype_ischar(t)     ((t) == 8 || (t) == 12)
#define numtype_isintegral(t) ((t) <= 8 || (t) == 12)
#define numtype_isfloat(t)    ((t) == 10)
#define numtype_isdouble(t)   ((t) == 9 || (t) == 11)
#define numtype_isreal(t)     ((t) >= 9 && (t) <= 11)
#define numtype_isunsigned(t) (((t) >= 4 && (t) <= 7) || (t) == 12)
#define numtype_issigned(t)   (((t) < 4 || (t) > 7) && (t) != 12)
#define numtype_isshort(t)    ((t) == 1 || (t) == 5) 
#define numtype_islong(t)     ((t) == 2 || (t) == 6 || (t) == 11)
#define numtype_islonglong(t) ((t) == 3 || (t) == 7)

#endif /* __SEM_H__ */
