/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* sem.c - Semantic/type analysis stuff */

/* A type is a linked list of semtype_t nodes.
 * If the type is basic (e.g. numeric or a SUE) then the list has 1 node only;
 * otherwise it may contain many nodes (e.g. arrayof -> pointerto -> ...).
 * This representation simplifies many operations such as pointer 
 * dereferencing (one needs only remove the head pointerto node).
 */

#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <stack.h>
#include "ast_traverse.h"
#include "ast_show.h"
#include "sem.h"


/* Allocate new sem node */
static
semtype_t *semtype(int baseclass, symbol name, astspec spec, astdecl decl)
{
	semtype_t *t = smalloc(sizeof(semtype_t));
	t->class = baseclass;
	t->isconst = false;
	t->name  = name;
	t->spec  = spec;
	t->decl  = decl;
	t->next  = NULL;
	return t;
}


/* Free a whole sem node list */
static
void semtype_listfree(semtype_t *l)
{
	if (!l) return;
	semtype_listfree(l->next);
	free(l);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     NUMERIC TYPES AND RULES                                   *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* Returns the encoded numeric type of a constant */
static
int constant_type(char *c)
{
	int ucount=0, lcount=0;
	
	if (*c == '\'')
		return numtype(CHAR_T, SIGNED_T, NORMAL_T);
		
	if (*c == '.')
	{
		CFLOAT:
		while (*c)  /* Go to the end */
			c++;
		c--;        /* last character */
		if (*c == 'l' || *c == 'L')   /* Long double */
			return numtype(DOUBLE_T, SIGNED_T, LONG_T);
		if (*c == 'f' || *c == 'F')   /* Float */
			return numtype(FLOAT_T, SIGNED_T, NORMAL_T);
		return numtype(DOUBLE_T, SIGNED_T, NORMAL_T);
	}
	
	if (*c == '0')
	{
		c++;
		if (*c == 'x' || *c == 'X')
			goto CINT;
	}
	
	/* Now in all cases numbers follow */
	while (isdigit(*c))
		c++;
	if (*c == '.' || *c == 'e' || *c == 'E')
		goto CFLOAT;
			
	CINT:
	while (*c)  /* Go to the end */
		c++;
	for (--c, ucount = lcount = 0; *c=='u'||*c=='U'||*c=='l'||*c=='L'; c--)
		if (*c == 'u' || *c == 'U')
			ucount++;
		else
			lcount++;
	return numtype(INT_T, ucount ? UNSIGNED_T : SIGNED_T,
	                lcount == 1 ? LONG_T : (lcount == 2 ? LONGLONG_T : NORMAL_T));
}


/* Integer promotion */
static
int int_promote(int t)
{
	if (numtype_ischar(t))
		t -= CHAR_T;                  /* convert to int */
	else
		if (numtype_isshort(t))       /* convert to normal int */
			t += (NORMAL_T - SHORT_T);
	return t;
}


/* Usual arithmetic conversions (in the order given):
 * if one is long double, the other is converted to long double 
 * if one is double, the other is converted to double 
 * if one is float, the other is converted to float 
 * Then, both are integers. They both get integer promotions and then:
 * if of the same type, that is the common type
 * if of the same signedness, the lower rank is promoted to the higher one
 * if the unsigned has equal or higher rank, the other converts to unsigned
 * if the signed can reperesent all unsigned vals, the unsigned becomes signed
 * Else both turn into the unsigned counterpart of the signed operand type
 */
static
int numtype_usual_convert(int a, int b)
{
	if ((numtype_isdouble(a) && numtype_islong(a)) || 
	    (numtype_isdouble(b) && numtype_islong(b)))
		return numtype(DOUBLE_T, SIGNED_T, LONG_T);
	if (numtype_isdouble(a) || numtype_isdouble(b))
		return numtype(DOUBLE_T, SIGNED_T, NORMAL_T);
	if (numtype_isfloat(a) || numtype_isfloat(b))
		return numtype(FLOAT_T, SIGNED_T, NORMAL_T);
	a = int_promote(a);
	b = int_promote(b);
	if ((numtype_isunsigned(a) && numtype_isunsigned(b)) ||
	    (numtype_issigned(a) && numtype_issigned(b)))
	{
		if (numtype_islonglong(a)) return a;
		if (numtype_islonglong(b)) return b;
		if (numtype_islong(a)) return a;
		if (numtype_islong(b)) return b;
		return a;
	}
	/* Make sure a is the signed one */
	if (numtype_isunsigned(a))
	{
		a = a^b;     /* swap */
		b = b^a;
		a = b^a;
	}
	if (numtype_islonglong(b)) return b;
	if (numtype_islong(b)) return numtype_islonglong(a) ? a : b;
	return a;
}


/* Checks whether the speclist includes "long long" (2 long in a row) */
#define speclist_has_longlong(s) _two_longs(s,0)

static
int _two_longs(astspec s, int one_long)
{
	if (s == NULL) return (0);
	if (s->type == SPECLIST)
	{
		if (speclist_getspec(s->body, SPEC, SPEC_long) == NULL)
			return (_two_longs(s->u.next, 0));
		else  /* Found 1 instance of long */
			return (one_long ? 1 : _two_longs(s->u.next, 1));
	}
	return ((s->type != SPEC || s->subtype != SPEC_long) ? 0 : one_long);
}


nt_size speclist_size(astspec s)
{
	if (speclist_has_longlong(s)) return (LONGLONG_T);
	if (speclist_getspec(s, SPEC, SPEC_long) != NULL) return (LONG_T);
	if (speclist_getspec(s, SPEC, SPEC_short) != NULL) return (SHORT_T);
	return (NORMAL_T);
}


nt_sign speclist_sign(astspec s)
{
	if (speclist_getspec(s, SPEC, SPEC_unsigned) != NULL) return (UNSIGNED_T);
	return (SIGNED_T);
}


nt_basetype speclist_basetype(astspec s)
{
	if (speclist_getspec(s, SPEC, SPEC_char) != NULL) return (CHAR_T);
	if (speclist_getspec(s, SPEC, SPEC_float) != NULL) return (FLOAT_T);
	if (speclist_getspec(s, SPEC, SPEC_double) != NULL) return (DOUBLE_T);
	if (speclist_getspec(s, SPEC, SPEC_ubool) != NULL) return (UBOOL_T);
	return (INT_T);
}


/**
 * Numeric/arithmetic type encoding [0-12]:
 *   ints are from 0 to 7 (pure int is 0),
 *   char is 8, double is 9 and float is 10;
 *   short adds 1, long adds 2, longlong adds 3, unsigned adds 4.
 * This is why the enums have the values they have...
 */
int numtype(nt_basetype b, nt_sign s, nt_size z)
{
	return ( b + s + z );
}


/* Get the numeric type out of a specifier */
int spec_to_numtype(astspec spec)
{
	return 
		numtype(speclist_basetype(spec), speclist_sign(spec), speclist_size(spec));
}


/* Get the specifier out of the numeric type */
astspec numtype_to_spec(int nt)
{
	astspec specs = Declspec(
	                  numtype_isint(nt)    ? SPEC_int :    /* basetype */
	                  numtype_ischar(nt)   ? SPEC_char : 
	                  numtype_isfloat(nt)  ? SPEC_float : 
	                  numtype_isdouble(nt) ? SPEC_double : SPEC_ubool
	                );
	if (numtype_isshort(nt))
		specs = Speclist_right(Declspec(SPEC_short), specs);
	if (numtype_islong(nt))
		specs = Speclist_right(Declspec(SPEC_long), specs);
	if (numtype_islonglong(nt))
		specs = Speclist_right(Declspec(SPEC_long),
		                       Speclist_right(Declspec(SPEC_long), specs));
	if (numtype_isunsigned(nt))
		specs = Speclist_right(Declspec(SPEC_unsigned), specs);
	return specs;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     GET THE TYPE OUT OF DECLARATOR                            *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* semlist
 * We are pushing to a stack during traversal and reversing at the end
 */
 
/* Prepend a node to a sem node list (appending is costly) */
static
semtype_t *semtype_listpush(semtype_t *l, typeclass_e class, astdecl d)
{
	semtype_t *n = semtype(class, NULL, NULL, d);
  n->next = l;
	return (n);
}

/* Recursive construction needs to reverse the list ultimately 
 * (so as to avoid appending entries...)
 */
static 
semtype_t *semtype_listreverse(semtype_t *me, semtype_t *previous)
{
	semtype_t *n = me->next;
	
	if (!me) return NULL;
	me->next = previous;
	if (n == NULL) 
		return me;
	return semtype_listreverse(n, me);
}


static void dfunc_c(astdecl t, void *state, int ignore)
{
	semtype_t **l = (semtype_t **) &((travopts_t *) state)->starg;
	if (t->decl)
		ast_decl_traverse(t->decl, (travopts_t *) state);
	// t->u.params can be found from t; we skip getting their types for now
	*l = semtype_listpush(*l, FUNC_T, t);
}


/**
 * Assume a speclist SP whose body is SPEC_star (i.e. a pointer); this checks
 * if there is a CONST qualifier that follow in the speclist (before reaching
 * the next UOP_star).
 * @param s this is SP->u.next
 * @return 1 if this is a const-qualified pointer, or false otherwise
 */
static bool _check_for_const_ptr(astspec s)
{
	if (s->type == SPECLIST)
	{
		astspec next = (s->subtype == SPEC_Rlist) ? s->u.next : s->body;
		
		s = (s->subtype == SPEC_Rlist) ? s->body : s->u.next;
		if (s->type == SPEC && s->subtype == SPEC_star)
			return false;                            /* stop, found next pointer */
		return _check_for_const_ptr(s) ? true : 
		       _check_for_const_ptr(next);
	}
	else
		return (s->type == SPEC && s->subtype == SPEC_const);
}


/* We assume the speclist is indeed a list and not a tree */
static void _add_pointers(astspec s, semtype_t **l)
{
	if (!s) return;
	if (s->type == SPECLIST)
	{
		astspec next = (s->subtype == SPEC_Rlist) ? s->u.next : s->body;
		
		s = (s->subtype == SPEC_Rlist) ? s->body : s->u.next;
		_add_pointers(next, l);
		if (s->type == SPEC && s->subtype == SPEC_star)
		{
			*l = semtype_listpush(*l, POINTER_T, NULL);
			(*l)->isconst = _check_for_const_ptr(next);
		}
	}
	else 
		if (s->type == SPEC && s->subtype == SPEC_star)
			*l = semtype_listpush(*l, POINTER_T, NULL);     /* surely non-const */
}


/* same as declcbf_t absdeclarator_c, dcasttype_c */
static void declarator_c(astdecl t, void *state, int ignore)
{
	semtype_t **l = (semtype_t **) &((travopts_t *) state)->starg;

	if (t->decl)
		ast_decl_traverse(t->decl, (travopts_t *) state);
	if (t->spec)  /* Exists only for a pointer */
		_add_pointers(t->spec, l);
}


/* same as dbit_c, dinit_c (ditching u.expr) */
static void dparen_c(astdecl t, void *state, int ignore)
{
	if (t->decl)
		ast_decl_traverse(t->decl, (travopts_t *) state);
}


static void darray_c(astdecl t, void *state, int ignore)
{
	semtype_t **l = (semtype_t **) &((travopts_t *) state)->starg;
	if (t->decl)
		ast_decl_traverse(t->decl, (travopts_t *) state);
	// t->u.expr can be found from t if needed
	*l = semtype_listpush(*l, ARRAY_T, t);
}


/* It creates a flat sem node list that represents the full type of
 * a declaration. It basically comes from a post-order traversal 
 * of the AST.
 */
static
semtype_t *declarator_ast2flat(astspec sp, astdecl de)
{
	semtype_t *l = NULL;
	
	if (de)
	{
		travopts_t tropts;
		
		travopts_init_noop(&tropts);
		tropts.guided = 1;
		tropts.starg = l;          /* The semlist as it is now */
		
		tropts.declc.dinit_c =
		tropts.declc.dbit_c =
		tropts.declc.dparen_c = dparen_c;
		tropts.declc.darray_c = darray_c;
		tropts.declc.dfunc_c = dfunc_c;
		tropts.declc.dcasttype_c = 
		tropts.declc.declarator_c = 
		tropts.declc.absdeclarator_c = declarator_c;
	
		ast_decl_traverse(de, &tropts);
		l = tropts.starg;
	}
	if (sp)
	{
		l = semtype_listpush(l, get_typeclass(sp, NULL), NULL);
		if (l->class == STRUNI_T || l->class == ENUM_T)
		{
			astspec u = speclist_getspec(sp, SUE, 0);
			l->name = u ? u->name : NULL;
		}
		l->isconst = speclist_getspec(sp, SPEC, SPEC_const);
	}
	return semtype_listreverse(l, NULL);   /* Must reverse the final stack */
}


/**
 * @brief Takes a declaration and forms the semtype list.
 * @param s the specifier
 * @param d the declarator
 * @return the semtype
 */
semtype_t *sem_declaration_type(astspec s, astdecl d)
{
	semtype_t *t;
	
	if (!d && !s)
	{
		warning("[%s] received NULL arguments\n", __func__); 
		return NULL; 
	}
	t = declarator_ast2flat(s, d);
	if (t->class == STRUNI_T || t->class == ENUM_T)
	{
		s = speclist_getspec(s, SUE, 0);
		t->name = s ? s->name : NULL;     /* NULL should not happen */
	}
	return t;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     TYPE OF EXPRESSION                                        *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


static
semtype_t *sem_uop(astexpr e)
{
	semtype_t *t = sem_expr_type(e->left);
	switch (e->opid)
	{
		case UOP_neg:
		case UOP_bnot:
			if (numtype_isintegral(t->class))  /* If it is integral type */
				t->class = int_promote(t->class);
			return t;
		case UOP_star:
		{
			semtype_t *res;
			if (!type_isarray(t->class) && !type_isptr(t->class))
				exit_error(1, "rhs of * op is not array/pointer (%d)\n", t->class);
			res = t->next;  /* Just get rid of the head node */
			free(t);
			return res;
		}
		case UOP_addr:
		{
			semtype_t *res = semtype(POINTER_T, NULL, NULL, NULL);
			/* Just add a pointer to the head of the type list */
			res->next = t;
			return res;
		}
		case UOP_lnot:
		case UOP_sizeof:
			t->class = numtype(INT_T, SIGNED_T, NORMAL_T);
			return t;
		default:
			break;
	}
	return t;
}


static
semtype_t *sem_bop(astexpr e)
{
	semtype_t *lhs = sem_expr_type(e->left), *rhs = sem_expr_type(e->right);
	
	switch (e->opid)
	{
		case BOP_shl:
		case BOP_shr:
			/* type of lhs after integer promotions */
			if (type_isnum(lhs->class) && numtype_isintegral(lhs->class))
				exit_error(1, "[%s()]: non-numeric type in a shift operation\n", 
				              __func__);
			semtype_listfree(rhs);
			lhs->class = int_promote(lhs->class);
			return lhs;
		case BOP_band:
    case BOP_bor:
    case BOP_xor:
    case BOP_mod:
    case BOP_div:
    case BOP_mul:
			lhs->class = numtype_usual_convert(lhs->class, rhs->class);
			free(rhs);
			return lhs;
		case BOP_add:
		case BOP_sub:
			/* If both operands have arithmetic types, then
         first, usual arithmetic conversions are performed.
         If one is pointer the other must be integer
         If BOP_sub and both pointers, then the result is
         ptrdiff_t type (which is a signed integer) */
			if (type_isnum(lhs->class))
			{
				if (type_isptr(rhs->class) || type_isarray(rhs->class))
				{
					if (e->opid != BOP_add || !numtype_isintegral(lhs->class))
						goto ERROR;
					free(lhs);
					return rhs;
				}
				if (!type_isnum(rhs->class))
					goto ERROR;
				lhs->class = numtype_usual_convert(lhs->class, rhs->class);
				free(rhs);
				return lhs;
			}
			if (type_isptr(lhs->class) || type_isarray(lhs->class))
			{
				if (type_isptr(rhs->class) || type_isarray(rhs->class))
				{
					if (e->opid != BOP_sub)
						goto ERROR;
					semtype_listfree(lhs);
					semtype_listfree(rhs); /* result should be ptrdiff_t */
					return semtype(numtype(INT_T, UNSIGNED_T, NORMAL_T), 
					               NULL, NULL, NULL);
				}
				if (!type_isnum(rhs->class) || !numtype_isintegral(rhs->class))
					goto ERROR;
				free(rhs);
				return lhs;
			}
			ERROR:
				exit_error(1, "[%s()]: invalid operand type in +/- operator (%d, %d)\n",
				              __func__, lhs->class, rhs->class);
			return NULL;
		case BOP_cast:
			free(rhs);
			return lhs;
		case BOP_leq:
		case BOP_geq:
		case BOP_eqeq:
		case BOP_neq:
		case BOP_land:
		case BOP_lor:
		case BOP_lt: 
		case BOP_gt:
			lhs->class = numtype(INT_T, SIGNED_T, NORMAL_T);
			free(rhs);
			return lhs;
		default:
			break;
	}
	free(lhs);
	free(rhs);
	return NULL;
}


semtype_t *sem_expr_type(astexpr tree)
{
	switch (tree->type)
	{
		case IDENT:
		{
			stentry e = symtab_get(stab, tree->u.sym, IDNAME);
			if (!e)  /* Handle FUNCNAME */
			{
				e = symtab_get(stab, tree->u.sym, FUNCNAME);
				if (!e)
					return NULL;
				return sem_declaration_type(e->spec, e->decl);
			}
			return sem_declaration_type(e->spec, e->decl);
		}
		case CONSTVAL:
			return semtype(constant_type(tree->u.str), NULL, NULL, NULL);
		case STRING:
		{
			semtype_t *t = semtype(numtype(CHAR_T, SIGNED_T, NORMAL_T),
			                       NULL, NULL, NULL);
			t->isconst = true;  /* String literals are const */
			return semtype_listpush(t, POINTER_T, NULL);
		}
		case FUNCCALL:
		{
			semtype_t *t = sem_expr_type(tree->left), *res;
			if (!type_isfunc(t->class))
			{
				/* cover the case of function pointer with the * */
				if (type_isptr(t->class) && t->next->class == FUNC_T)
				{
					res = t;       /* ditch the pointer */
					t = t->next;
					free(res);
				}
				else
					exit_error(1, "lhs of funccall is not a function (%d)\n", t->class);
			}
			res = t->next;  /* Just get rid of the head (FUNC) node */
			free(t);
			return res;
		}
		case ARRAYIDX:
		{
			semtype_t *t = sem_expr_type(tree->left), *res;
			if (!type_isarray(t->class) && !type_isptr(t->class))
				exit_error(1, "lhs of arrayidx is not array/pointer (%d)\n", t->class);
			res = t->next;  /* Just get rid of the head node */
			free(t);
			return res;
		}
		case DOTFIELD:
		{
			semtype_t *t = sem_expr_type(tree->left);
			astdecl f;
			
			if (!type_isstruni(t->class))
				exit_error(1, "lhs of dotfield is not a struct/union (%d)\n", t->class);
			if (!(f = get_sufield(t->name, tree->u.sym)))
				exit_error(1, "struct/union %s does not have field %s\n", 
				              t->name->name, tree->u.sym->name);
			free(t);
			return sem_declaration_type(f->spec, f->decl);
		}
		case PTRFIELD:
		{
			semtype_t *t = sem_expr_type(tree->left);
			semtype_t *old = t;
			astdecl f;
			
			if (!type_isptr(t->class))
				exit_error(1, "lhs of -> op is not a pointer (%d)\n", t->class);
			t = t->next;  /* Just get rid of the head node */
			free(old);
			
			/* Repeat DOTFIELD code */
			if (!type_isstruni(t->class))
				exit_error(1, "lhs of -> doesn't point to struct/union(%d)\n",t->class);
			if (!(f = get_sufield(t->name, tree->u.sym)))
				exit_error(1, "struct/union %s does not have field %s\n", 
				              t->name->name, tree->u.sym->name);
			free(t);
			return sem_declaration_type(f->spec, f->decl);
		}
		case BRACEDINIT:
			/* TODO?? */
			return semtype(VOID_T, NULL, NULL, NULL);
		case CASTEXPR:
			return sem_declaration_type(tree->u.dtype->spec, tree->u.dtype->decl);
		case CONDEXPR:
		{
			/* Returns the common type of the 2 expressions defined as:
			   1) if the expressions have arithmetic type, the common type is the 
			      type after usual arithmetic conversions
			   2) if the expressions have struct/union type, the common type is 
			      that struct/union type
			   3) if the expressions are both void, the entire conditional operator
			      expression is a void expression
			   4) if one is a pointer and the other is a null pointer constant, the 
			      type is the type of that pointer
			   5) if both are pointers, the result is the pointer to the type that 
			      combines cvr-qualifiers of both pointed-to types (that is, if one 
			      is const int* and the other is volatile int*, the result is 
			      const volatile int*), and if the types were different, the 
			      pointed-to type is the composite type.
			   6) if one is a pointer to void, the result is a pointer to void with 
			      combined cvr-qualifiers
			*/
			semtype_t *lhs = sem_expr_type(tree->left), *rhs = sem_expr_type(tree->right);
			if (type_isnum(lhs->class) && type_isnum(rhs->class))
			{
				lhs->class = numtype_usual_convert(lhs->class, rhs->class);
				free(rhs);
				return lhs;
			}
			if ((type_isstruni(lhs->class) && type_isstruni(rhs->class) 
			                            && lhs->name == rhs->name) ||
			    (type_isvoid(lhs->class) && type_isvoid(rhs->class)))
			{
				free(rhs);
				return lhs;
			}
			if (!type_isptr(lhs->class) && !type_isptr(rhs->class))
				exit_error(1, "incompatible lhs/rhs of conditional operator (%d, %d)\n",
				              lhs->class, rhs->class);
			free(rhs);
			return lhs;    /* TODO: somehow check for rules 4-6 ?? */
		}
		case UOP:
			return sem_uop(tree);
		case BOP:
			return sem_bop(tree);
		case PREOP:
		case POSTOP:
		case ASS:
			return sem_expr_type(tree->left);
		case DESIGNATED:
		case IDXDES:
		case DOTDES:
			return NULL;
		case COMMALIST:
		case SPACELIST:
			return sem_expr_type(tree->right);
		default:
			fprintf(stderr, "[%s]: b u g !!\n", __func__);
	}
	return NULL;
}


/* Arithmetic types and pointer types are collectively called scalar types. 
 * Array and structure types are collectively called aggregate types.
 * Also, because we do not keep all the info, results are subject to 
 * lvalue conversions (i.e. basically we drop any non-CONST qualifiers).
 */
 
/**
 * @brief Given a semtype, this function returns the corresponding type tree 
 * in the form of specifier and (abstract or concrete) declarator, after any 
 * lvalue conversions (i.e. basically non-CONST qualifiers and storage class 
 * specifiers are dropped). The specifier and declarator can be used e.g. for 
 * casts or variable declarations.
 * 
 * @param type the semtype
 * @param s (ret) the specifier
 * @param d (ret) the declararor
 * @param abstract if true, derive the abstract declarator otherwise 
 *                 derive the concrete declarator
 */
void sem_type_to_declaration(semtype_t *type, astspec *s, astdecl *d, 
                             bool abstract)
{
	if (!type)
		return;
	for (; type; type = type->next)
		switch (type->class)
		{
			case ARRAY_T:     /* done */
				/* Avoid parentheses if *d is identifier or array */
				if (*d && (*d)->type != DIDENT && 
				    (!(*d)->decl || (*d)->decl->type != DARRAY))
					*d = ParenDecl(*d);
				*d = abstract ?
				       AbstractDeclarator(NULL, ArrayDecl(*d,NULL,type->decl->u.expr)) :
				       Declarator(NULL, ArrayDecl(*d, NULL, type->decl->u.expr));
				break;
			case POINTER_T:   /* done; except maybe const? */
				/* Avoid parentheses if *d is identifier or pointer; actually for 
				 * pointers we need to check if they have the same qualifiers...
				 * so leave it for the future.
				 */
				if (*d && (*d)->type != DIDENT) // && !decl_ispointer(*d))
					*d = ParenDecl(*d);
				if (type->isconst)
				  *d = abstract ?
				       AbstractDeclarator(
					       Speclist_right(Declspec(SPEC_star), Declspec(SPEC_const)), *d):
				       Declarator(
					       Speclist_right(Declspec(SPEC_star), Declspec(SPEC_const)), *d);
				else
					*d = abstract ?
				         AbstractDeclarator(Declspec(SPEC_star), *d) :
				         Declarator(Declspec(SPEC_star), *d);
				break;
			case STRUNI_T:
				*s = SUdecl(SPEC_struct /* TODO */, type->name, type->decl, NULL);
				break;
			case ENUM_T:
				*s = Enumdecl(type->name, NULL, NULL);
				*d = IdentifierDecl(Symbol("dummy"));
				break;
			case FUNC_T:      /* done (with no parames of course) */
				if (*d && (*d)->type != DIDENT)
					*d = ParenDecl(*d);
				*d = abstract ?
				       AbstractDeclarator(NULL, FuncDecl(*d, type->decl->u.params)) :
				       Declarator(NULL, FuncDecl(*d, type->decl->u.params));
				break;
			case VOID_T:      /* done */
				*s = Declspec(SPEC_void);
				break;
			default:          /* numeric types (done) */
				*s = numtype_to_spec(type->class);
				if (type->isconst)
					*s = Speclist_right(Declspec(SPEC_const), *s);
				break;
		};
}


/**
 * @brief Given an expression, this function returns the type tree of the 
 * expression in the form of specifier and abstract declarator, after any 
 * lvalue conversions. The specifier and abstract declarator can be used to
 * form casts or as operands of the sizeof operator.
 * 
 * @param expr the expression
 * @param s (ret) the specifier
 * @param d (ret) the declarator
 * @return false is something went wrong, true otherwise.
 */
bool sem_expr_typeof(astexpr expr, astspec *s, astdecl *d)
{
	semtype_t *type = sem_expr_type(expr);
	*s = NULL; *d = NULL;
	if (type == NULL) return false;
	sem_type_to_declaration(type, s, d, true);
	semtype_listfree(type);
	return (*s != NULL || *d != NULL);
}


/**
 * @brief Given an expression, this function prepares the declaration of 
 * a variable with type identical to type of the expression. Actually, 
 * the specifier and the declarator parts of the declaration are returned, 
 * after any lvalue conversions. The specifier and abstract declarator 
 * can be used to form variable or function parameter declaration statements.
 * 
 * @param expr the expression
 * @param v the variable which will be declared
 * @param s (ret) the specifier
 * @param d (ret) the declarator
 * @return false is something went wrong, true otherwise.
 */
bool sem_expr_typeof_declare(astexpr expr, symbol v, astspec *s, astdecl *d)
{
	semtype_t *type = sem_expr_type(expr);
	
	*s = NULL; *d = IdentifierDecl(v);
	if (type == NULL) return false;
	sem_type_to_declaration(type, s, d, true);
	semtype_listfree(type);
	return true;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     VALUE CATEGORY OF EXPRESSION                              *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Expressions in C have a *type* and a *value category*; the value
 * categories are *lvalues*, normal *(r)values* or *function designator* values.
 */

 
static bool su_has_const_fields(astdecl d)
{
	if (d == NULL) return false;
	if ((d)->type == DLIST && d->subtype == DECL_fieldlist)
		return su_has_const_fields(d->u.next) ? true : su_has_const_fields(d->decl);
	else 
		if (d->type == DSTRUCTFIELD)
			return speclist_getspec(d->spec, SPEC, SPEC_const);
		else  /* should never happen */
			return false;
}


/* Given an expression tree, this function returns: 
 *   0 if it is not an lvalue
 *   1 if it is a modifiable lvalue, 
 *   2 if it is a non-modifiable lvalue
 *   3 if it is an lvalue but must be checked for modifiability
 * If 1 or 2 is returned, then a check for modifiability is not needed.
 *
 * Non-modifiable lvalue:
 * A modifiable lvalue is any lvalue expression of complete, non-array type
 * which is not const-qualified, and, if it's a struct/union, has no members 
 * that are const-qualified, recursively.
 *
 * An incomplete type is an object type that lacks sufficient information to 
 * determine the size of the objects of that type. An incomplete type may be 
 * completed at some point in the translation unit. These are incomplete types:
 *   - the type void
 *   - array type of unknown size
 *   - structure or union type of unknown content
 */
 int sem_expr_islvalue(astexpr tree)
{
	semtype_t *type;
	stentry e;

	if (!tree) return 0;
	switch (tree->type)
	{
		case IDENT:
			if ((e = symtab_get(stab, tree->u.sym, IDNAME)) == NULL) return 0;
			if (e->isarray) return 2;
			return speclist_getspec(e->spec, SPEC, SPEC_const) ? 2 : 1;
		case STRING:
			return 2;
		case ARRAYIDX:
			return sem_expr_islvalue(tree->left);
		case DOTFIELD:
			if (sem_expr_islvalue(tree->left) == 0)
				return 0;
				
			/* Now check for modifiability: no field should be const qualified */
			type = sem_expr_type(tree->left);
			e = (type->class == STRUNI_T) ? 
			         symtab_get(stab, type->name, SUNAME) : NULL;
			semtype_listfree(type);
			if (!e || !e->spec)     /* should not happen */
				return 0;
			return su_has_const_fields(e->spec->u.decl) ? 2 : 1;
		case PTRFIELD:         /* This is always an lvalue */
			/* Now check for modifiability: no field should be const qualified */
			type = sem_expr_type(tree->left)->next;   /* Get what it points to */
			e = (type->class == STRUNI_T) ? 
			         symtab_get(stab, type->name, SUNAME) : NULL;
			semtype_listfree(type);
			if (!e || !e->spec)     /* should not happen */
				return 0;
			return su_has_const_fields(e->spec->u.decl) ? 2 : 1;
		case UOP:
			if (tree->opid == UOP_paren)
				return sem_expr_islvalue(tree->left);
			if (tree->opid == UOP_star)
			{
				int retval;
				
				type = sem_expr_type(tree->left);
				if (!type || type->class != POINTER_T || !type->next)
					retval = 0;
				else
					if (type->next->class == FUNC_T)  /* It is a function designator */
						retval = 0;
					else       /* Now, it is 100% an lvalue; check for modifiability */
				    retval = (type->next->isconst ||
				              speclist_getspec(type->next->spec, SPEC, SPEC_const)) ?
				             2 : 1;
				semtype_listfree(type);
				return retval;
			}
		default:
			return 0;
	}
}


/* Given an expression tree, this function returns: 
 *   false if it is not a function designator
 *   true  if it is a function designator
 *
 * C defines a function designator as an expression that has function type. 
 * It can be the name of a function or the result of dereferencing a function 
 * pointer.
 *
 * Note (irrelevant): 
 * When used in any context other than the address-of operator, sizeof, and
 * _Alignof, the function designator is always converted to a non-lvalue pointer 
 * to function. Note that the function-call operator is defined for pointers to 
 * functions and not for function designators themselves.
 */
bool sem_expr_isfuncdes(astexpr tree)
{
	if (!tree) return 0;
	switch (tree->type)
	{
		case IDENT:
			return symtab_get(stab, tree->u.sym, FUNCNAME);
		case UOP:
			if (tree->opid == UOP_paren)
				return sem_expr_isfuncdes(tree->left);
			if (tree->opid == UOP_star)
			{
				semtype_t *type = sem_expr_type(tree->left);
				bool isfuncdes = (type->class==POINTER_T && type->next->class==FUNC_T);
				semtype_listfree(type);
				return isfuncdes;
			}
		default:
			return false;
	}
	return false;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     UTILITIES AND DEBUGGING                                   *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


typeclass_e get_typeclass(astspec s, astdecl d)
{
	if (!d && !s) { warning("[%s] NULL arguments\n", __func__); return -1; }
	
	if (d)
	{
		if (decl_ispointer(d)) 
			return POINTER_T;  /* TODO: pointer to what? */
		if (decl_getkind(d) == DARRAY)
			return ARRAY_T;    /* TODO: array of what? */
	}
	if (s)
	{
		if (speclist_getspec(s, SUE, SPEC_struct))
			return STRUNI_T;
		if (speclist_getspec(s, SUE, SPEC_union))
			return STRUNI_T;
		if (speclist_getspec(s, SUE, SPEC_enum))
			return ENUM_T;
		if (speclist_getspec(s, SPEC, SPEC_void))
			return VOID_T;
		/* Must be numeric */
		return spec_to_numtype(s);
	}
	return VOID_T;
}


static
void _show_type(semtype_t *t, int level)
{
	if (level)
		fprintf(stderr, "%*s", 2*level, "");
	switch (t->class)
	{
		case FUNC_T:
 			fprintf(stderr, "function (");
 			if (t->decl && t->decl->u.params)
 			  ast_decl_show_stderr(t->decl->u.params);
 			fprintf(stderr, ") returning\n");
			break;
		case ARRAY_T:
			fprintf(stderr, "array ["); 
			if (t->decl && t->decl->u.expr) ast_expr_show_stderr(t->decl->u.expr); 
			fprintf(stderr, "] of\n");
			break;
		case POINTER_T:
			if (t->isconst) fprintf(stderr, "(const) ");
			fprintf(stderr, "pointer to\n");
			break;
		case STRUNI_T:
		case ENUM_T:
			fprintf(stderr, "struct/union/enum (%s)", t->name ? t->name->name : "");
			fprintf(stderr, "\n");
			break;
		case VOID_T:
		  fprintf(stderr, "void\n");
		  break;
		default: /* numeric */
		{
			if (t->isconst) fprintf(stderr, "(const) ");
			if (numtype_isdouble(t->class))
				fprintf(stderr, "%sdouble\n", numtype_islong(t->class) ? "long " : "");
			if (numtype_isfloat(t->class))
				fprintf(stderr, "float\n");
			if (numtype_isintegral(t->class))
			{
				if (numtype_isunsigned(t->class))
					fprintf(stderr, "unsigned ");
				if (numtype_isshort(t->class))
					fprintf(stderr, "short ");
				if (numtype_islonglong(t->class))
					fprintf(stderr, "long long ");
				else
					if (numtype_islong(t->class))
						fprintf(stderr, "long ");
				if (numtype_ischar(t->class))
					fprintf(stderr, "char\n");
				if (numtype_isint(t->class))
					fprintf(stderr, "int\n");
			}
		}
	}
}


void sem_type_show(semtype_t *l)
{
	int level = 0;
	if (l)
		for (; l; l = l->next)
			_show_type(l, level++);
}


void sem_expr_showinfo(astexpr tree)
{
	int lvalue = sem_expr_islvalue(tree);
	
	fprintf(stderr, "Expression: ");
	ast_expr_show_stderr(tree);
	fprintf(stderr, "\nIs lvalue : %s\nIs function designator : %s\nType: ",
	        lvalue == 0 ? "no" : lvalue == 1 ? "yes, modifiable" : 
	        lvalue == 2 ? "yes, non-modifiable" : "yes", 
	        sem_expr_isfuncdes(tree) ? "yes" : "no");
	sem_type_show(sem_expr_type(tree));
	fprintf(stderr, "\n");
}
