/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/*
 * 2014/01/07
 *   integrated ptester
 * 2009/05/10
 *   fixed '#include <string.h>' bug
 */

/* ompi.c -- the starting point */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <sys/time.h>
#include <assert.h>
#include "ompi.h"
#include "ast.h"
#include "ast_free.h"
#include "ast_gv.h"
#include "ast_csource.h"
#include "ast_xform.h"
#include "ast_show.h"
#include "ast_types.h"
#include "ast_xformrules.h"
#include "x_target.h"
#include "x_task.h"
#include "x_clauserules.h"
#include "cfg.h"
#include "tac.h"
#include "sem.h"
#include "callgraph.h"
#include "str.h"
#include "config.h"
#include "x_reduction.h"
#include "x_requires.h"
#include "codetargs.h"
#ifdef OMPI_REMOTE_OFFLOADING
	#include "roff_config.h"
#endif

static aststmt ast;           /* The AST we use as our original */
symtab         stab;          /* Our symbol table */
bool           testingmode;   /* For internal testing only */
char           *filename;     /* The file we parse */
char           *filename_noext; /* The file we parse, without the extension */
char           advert[1024];  /* A string added to the top of generated files */

#ifdef PORTABLE_BUILD
	char InstallPath[PATHSIZE], LibDir[PATHSIZE];
#endif

/* This is for taking care of main() in the parsed code;
 * OMPi generates its own main() and replaces the original one.
 */
char *MAIN_NEWNAME   = "__original_main";  /* Rename the original main */
bool hasMainfunc     = false; /* true if main() is defined in the file */
bool includes_omph   = false; /* true if user program included <omp.h> */
bool knowMemcpy      = false; /* true if memcpy() prototype exists */
bool knowMemset      = false; /* true if memset() prototype exists */
bool needMemcpy      = false; /* true if generated code includes memcpy()s */
bool needMemset      = false; /* true if generated code includes memset()s */
bool needLimits      = false; /* true if need limits.h constants (min/max) */
bool needFloat       = false; /* true if need float.h constants (min/max) */
bool needKernDimsEnc = false; /* true if need kernel dimension encoding */
bool enablePortability = false; /* true if user app compiled with --portable */
bool forceInitOrt    = false; /* true if we must generate an _ort_init call */
bool nonewmain       = false; /* If true, we won't output a new "main()" */
bool processmode     = false; /* If true, turn on process mode */
bool threadmode      = false; /* Will become true by default */
bool enableOpenMP    = true;  /* If false, ignore OpenMP constructs */
bool enableOmpix     = true;  /* Enable OMPi-extensions */
bool cppLineNo       = true;  /* Output precompiler line directives (# n) */
bool showdbginfo     = false; /* Allow assorted debugging info on stderr */
bool analyzeKernels  = false; /* Force kernel analysis for our C.A.R. system */
bool oldReduction    = false; /* Force newer reduction code */
bool enableAutoscope = false; /* No auto scoping analysis */
int  mainfuncRettype = 0;     /* 0 = int, 1 = void */
taskopt_e taskoptLevel = OPT_FAST;  /* Task code optimizations for speed */
bundling_e bundleKernels = BUNDLE_NONE; /* No kernel bundling unless told so */

str devtargs;      /* String containing all used module names; for codetargs */
str modstr;        /* String containing all used module names quoted; only used 
                      as an option for ort_init(). */
int usedmods;      /* Number of used modules */

int __needs_devices = 0;
int __has_devfuncs = 0;
int __has_onlydecltarg = 0;
int __has_ompfuncs = 0;
int __has_ompxfuncs = 0;


/*
 * Options
 */


#define OPTNAME(opt)   "-" #opt
#define OPTNAME_L(opt) "--" #opt "="
#define OPTION(opt)    OPT_##opt

typedef enum {
	OPTION(unknown) = -1, /* unknown option */
	OPTION(nomain) = 0,   OPTION(procs),       OPTION(threads),
	OPTION(nomp),         OPTION(nox),         OPTION(oldred), 
	OPTION(nolineno),     OPTION(showdbginfo), OPTION(usemod),
	OPTION(taskopt0),     OPTION(taskopt1),    OPTION(taskopt2),
	OPTION(drivecar),     OPTION(autoscope),   OPTION(rtlibred),
	OPTION(portable),     OPTION(initort),     OPTION(bundlebins),
	OPTION(bundlesrcs),
	OPTION(lastoption)    /* dummy */
} option_t;

static char *optnames[] = {
	OPTNAME(nomain),      OPTNAME(procs),       OPTNAME(threads),
	OPTNAME(nomp),        OPTNAME(nox),         OPTNAME(oldred),
	OPTNAME(nolineno),    OPTNAME(showdbginfo), OPTNAME_L(usemod),
	OPTNAME(taskopt0),    OPTNAME(taskopt1),    OPTNAME(taskopt2),
	OPTNAME(drivecar),    OPTNAME(autoscope),   OPTNAME(rtlibred),
	OPTNAME(portable),    OPTNAME(initort),     OPTNAME(bundlebins),
	OPTNAME(bundlesrcs)
};


static void showopts()
{
	int i;

	for (i = 0; i < OPTION(lastoption); i++)
		fprintf(stderr, "\t%s\n", optnames[i]);
}

static option_t optid(char *arg)
{
	int i;

	for (i = 0; i < OPTION(lastoption); i++)
		if (optnames[i][1] == '-')     /* Long option */
		{
			if (strncmp(optnames[i], arg, strlen(optnames[i])) == 0)
				return ((option_t) i);
		}
		else
			if (strcmp(optnames[i], arg) == 0)
				return ((option_t) i);
	return OPTION(unknown);
}


static int check_for_codetarg_cmdarg(char *arg)
{
	char *p;
	for (p=arg+2 /* skip -- */; *p != '-'; p++)
		;
	if (*p == '-')
		return codetarg_id_len(arg+2, p-(arg+2));
	else
		return -1;
}


/* Catch any option that starts with --<modname> 
 */
static int getdevopts(int argc, char **argv)
{
	int err = 0, dt, i;

	for (i = 0; i < argc; i++)
		if ((dt = check_for_codetarg_cmdarg(argv[i])) >= 0)
			if ((err = codetarg_cmdarg(dt, argv[i] + 2))) 
				break;

	return err;
}

static
int getopts(int argc, char **argv)
{
	int i = 0;

#ifdef PORTABLE_BUILD
	/* In portable builds the 1st argument is the library installation path */
	strncpy(LibDir, argv[0], PATHSIZE - 1);
	argc--;
	argv++;

	strncpy(InstallPath, argv[0], PATHSIZE - 1);
	argc--;
	argv++;
	// fprintf(stderr, "LibDir=%s, InstallPath=%s, argc=%d, *argv=%s\n", LibDir, InstallPath, argc, *argv);
#endif
	
	devtargs = Strnew();
	modstr = Strnew();
	
	for (i = 0; i < argc; i++)
		switch ( optid(argv[i]) )
		{
			case OPTION(nomain):      nonewmain = true; break;
			case OPTION(procs):       processmode = true; 
			                          /* Force old reduction style since newer one
			                             passes local pointers to the master; array
			                             reductions are, hence, precluded. */
			                          oldReduction = true;
			                          break;
			case OPTION(threads):     threadmode = true; break;
			case OPTION(nomp):        enableOpenMP = false; break;
			case OPTION(nox):         enableOmpix = false; break;
			case OPTION(taskopt0):    taskoptLevel = OPT_NONE; break;
			case OPTION(taskopt1):    taskoptLevel = OPT_FAST; break;
			case OPTION(taskopt2):    taskoptLevel = OPT_ULTRAFAST; break;
			case OPTION(nolineno):    cppLineNo = false; break;
			case OPTION(showdbginfo): showdbginfo = true; break;
			case OPTION(drivecar):    analyzeKernels = true; break;
			case OPTION(oldred):      oldReduction = true; break;
			case OPTION(rtlibred):    red_codegen_style = REDCODE_RTLIB; break;
			case OPTION(autoscope):   enableAutoscope = true; break;
			case OPTION(bundlebins):  bundleKernels = BUNDLE_BINS; break;
			case OPTION(bundlesrcs):  bundleKernels = BUNDLE_SRCS; break;
			case OPTION(portable):    enablePortability = true; break;
			case OPTION(initort):     forceInitOrt = true; break;
			case OPTION(usemod):
				str_printf(devtargs, "%s ", argv[i]+strlen(OPTNAME_L(usemod)));
				str_printf(modstr, ", \"%s\"", argv[i]+strlen(OPTNAME_L(usemod)) );
				usedmods++;
				break;
			default:
				return (0);
		};
	return (0);
}


static void reveal_inside_info()
{
	fprintf(stderr, ">> package = %s-%s\n", PKG_NAME, PKG_VERSION);
#ifdef PORTABLE_BUILD
	fprintf(stderr, ">> this is a portable build\n");
#endif
#ifdef MODULES_CONFIG
	fprintf(stderr, ">> configured modules:   %s\n", MODULES_CONFIG);
#endif
#ifdef ENABLE_KERNEL_BUNDLING
	fprintf(stderr, ">> kernel file bundling: enabled by default\n");
#else
	fprintf(stderr, ">> kernel file bundling: disabled by default\n");
#endif
	fprintf(stderr, ">> sizes of int, long, ptr:    %d, %d, %d\n", 
	        SIZEOF_INT, SIZEOF_LONG, SIZEOF_CHAR_P);
#ifdef TLS_KEYWORD
	fprintf(stderr, ">> Thread-Local Storage: available\n");
#else
	fprintf(stderr, ">> Thread-Local Storage: unavailable.\n");
#endif
	fprintf(stderr, ">> _ompi __ompi__ <options>:\n");
	showopts();
}


static
int privatemode(int argc, char *argv[])
{
	enum { 
		SHOWINTS = 0, GV_STMT, GV_EXPR, GV_PROG, CODE_STMT, CODE_EXPR, CODE_PROG, 
		SEM_DECL, SEM_EXPR, CFG_SHOW, CFG_SHOW_VERBOSE, CFG_SHOW_VERYVERBOSE, 
		CG_SHOW, ALL_CL_RULES, ALL_DIR_RULES, SINGLE_CL_RULES, 
		SINGLE_DIR_RULES, SEARCH_OMP, VARACC,
		PRVCMD_LAST /* dummy */
	} cmd;
	struct { char *name, *info; } privcmds[] = {
		{ "showints",    "show various internal infos" },
		{ "gv_stmt",     "print statement node parse tree" },
		{ "gv_expr",     "print expression node parse tree" },
		{ "gv_prog",     "print the AST of a whole program" },
		{ "code_stmt",   "print C code for the tree" },
		{ "code_expr",   "print C code for an expression" },
		{ "code_prog",   "print C code for a whole program" },
		{ "sem_decl",    "explain the semantics of a declaration statement" },
		{ "sem_expr",    "give the type of an expression" },
		{ "cfg",         "print the CFG for the tree" },
		{ "cfgv",        "print the CFG for the tree, detailed" },
		{ "cfgvv",       "print the CFG for the tree, more detailed" },
		{ "cg",          "print the call graph of a whole program" },
		{ "allclauses",  "show directives accepting each OpenMP clause" },
		{ "alldirs",     "show clauses for every OpenMP directive" },
		{ "clausedirs",  "show directives accepting the given OpenMP clause clause" },
		{ "dirclauses",  "show clauses accepted by the given OpenMP directive" },
		{ "searchomp",   "show all OpenMP entities that contain the given string" },
		{ "varaccess",   "show variables accessed in an expression"}
	};

	str     argstr = Strnew();
	aststmt ast;
	astexpr expr;
	ompdirt_e directive;
	ompclt_e clause;
	char *s;

	MAIN_NEWNAME = "main";    /* Do NOT change it!! */

	if (argc < 3)
	{
	PRIVATEMODE_FAILURE:
		fprintf(stderr, "Usage: %s __internal__ <command> "
		        "[ file | directive | clause]\n"
		        "where <command> is one of the following:\n", argv[0]);
		for (cmd = 0; cmd < PRVCMD_LAST; cmd++)
			fprintf(stderr, "%11s   %s\n", privcmds[cmd].name, privcmds[cmd].info);
		fprintf(stderr, "[ If gv output is produced, use 'dot -Tpng' "
		                "to make an image out of it. ]\n");
		return (1);
	}
	filename = (argc > 3) ? argv[3] : NULL;
	filename_noext = filename ? strdup(filename) : NULL;
	if (filename_noext)
		if ((s = strrchr(filename_noext, '.')) != NULL) 
			*s = 0;

	if (argc > 3)  /* Get all late args into a single string */
	{
		int i;
		
		for (i = 3; i < argc; i++)
		{
			str_printf(argstr, "%s", argv[i]);
			if (i != argc-1)
				str_putc(argstr, ' ');
		}
	}

	ast_initomp();
	clr_init_rules();

	for (cmd = 0; cmd < PRVCMD_LAST; cmd++)
		if (strcmp(argv[2], privcmds[cmd].name) == 0)
			break;
	/* Check for 1st batch of private options */
	switch(cmd)
	{
		case PRVCMD_LAST:
			goto PRIVATEMODE_FAILURE;
		case SHOWINTS:
			reveal_inside_info();
			return (1);
		case ALL_CL_RULES:
			clr_print_all_clauses();
			return (0);
		case ALL_DIR_RULES:
			clr_print_all_dirs();
			return (0);
		case SINGLE_CL_RULES:
			if (argc < 4)
				goto PRIVATEMODE_FAILURE;
			if ((clause = clr_clause_id(str_string(argstr))) != OCNOCLAUSE)
				clr_print_clause_dirs(clause);
			return (0);
		case SINGLE_DIR_RULES:
			if (argc < 4)
				goto PRIVATEMODE_FAILURE;
			if ((directive = clr_dir_id(str_string(argstr))) != DCNONE)
				clr_print_dir_clauses(directive);
			return (0);
		case SEARCH_OMP:
			if (argc < 4)
				goto PRIVATEMODE_FAILURE;
			ast_search_omp(str_string(argstr));
			return (0);
		default:
			break;
	}

	stab = Symtab();                            /* Create the symbol table */
	symtab_put(stab, Symbol("__builtin_va_list"), TYPENAME);
	symtab_put(stab, Symbol("__extension__"), TYPENAME);
	symtab_put(stab, Symbol("__uint128_t"), TYPENAME);
	symtab_put(stab, Symbol("_Float16"), TYPENAME);
	symtab_put(stab, Symbol("__float128"), TYPENAME);
	symtab_put(stab, Symbol("__func__"), IDNAME);
	symtab_put(stab, Symbol("nullptr"), IDNAME);      /* C23 */

	/* Read the file into a string */
	if (1)
	{
		char s[1024];
		FILE *fp;

		if (filename)
		{
			if ((fp = fopen(filename, "r")) == NULL)
				goto PRIVATEMODE_FAILURE;
			A_str_truncate();
			for (; fgets(s, 1024, fp);)
				str_printf(strA(), "%s", s);
			fclose(fp);
		}
		else
		{
			fprintf(stderr, "Enter %s:\n", 
			                (cmd == GV_EXPR || cmd == CODE_EXPR || cmd == SEM_EXPR ||
			                 cmd == VARACC) ?
			                   "expression" : "statement");
			if (!fgets(s, 1024, stdin))
				exit_error(1, "fgets failed\n");
			str_printf(strA(), "%s", s);
		}
	}

	switch (cmd)
	{
		case GV_STMT:
			ast = parse_blocklist_string(A_str_string());
			if (ast != NULL)
				ast_stmt_gviz_doc(ast, A_str_string());
			break;
		case GV_EXPR:
			expr = parse_expression_string(A_str_string());
			if (expr != NULL)
				ast_expr_gviz_doc(expr, A_str_string());
			break;
		case GV_PROG:
			ast = parse_transunit_string(A_str_string());
			if (ast != NULL)
				ast_stmt_gviz_doc(ast, A_str_string());
			break;
		case CODE_STMT:
			ast = parse_blocklist_string(A_str_string());
			if (ast != NULL)
			{
				ast_stmt_csource(ast);
				printf("\n");
			}
			break;
		case CODE_EXPR:
			expr = parse_expression_string(A_str_string());
			if (expr != NULL)
			{
				ast_expr_csource(expr);
				printf("\n");
			}
			break;
		case CODE_PROG:
			ast = parse_transunit_string(A_str_string());
			if (ast != NULL)
			{
				ast_csource(ast);
				printf("\n");
			}
			break;
		case CFG_SHOW:
		case CFG_SHOW_VERBOSE:
		case CFG_SHOW_VERYVERBOSE:
			ast = parse_blocklist_string(A_str_string());
			if (ast != NULL)
				cfg_test(ast, (cmd==CFG_SHOW) ? 0 : ((cmd==CFG_SHOW_VERBOSE) ? 1 : 2));
			break;
		case CG_SHOW:
			ast = parse_transunit_string(A_str_string());
			if (ast != NULL)
				cg_call_graph_test(ast);
			break;
		case SEM_DECL:
			ast = parse_blocklist_string(A_str_string());
			if (ast != NULL && ast->type == DECLARATION)
			{
				semtype_t *l;
				astspec sp = NULL;
				astdecl de = IdentifierDecl(Symbol("dummy"));
				
				sem_type_show(
					l = sem_declaration_type(ast->u.declaration.spec, 
					                         ast->u.declaration.decl)
				);
				sem_type_to_declaration(l, &sp, &de, false);
				printf("Typeof:\n");
				printf("  Spec: "); if (sp) ast_spec_show(sp);
				printf("\n  Decl: "); if (de) ast_decl_show(de);
				printf("\n");
#ifdef TEST_ABSTRACT_TO_CONCRETE_CONVERTER
				/* int (*)[] */
				sp = Declspec(SPEC_int);
				de = AbstractDeclarator(NULL, ArrayDecl(ParenDecl(AbstractDeclarator(Declspec(SPEC_star), NULL)), NULL, NULL));
				astdecl res, con;
				con = Declarator(Declspec(SPEC_star), IdentifierDecl(Symbol("dummy")));
				res = ast_xt_abstract_to_concrete_declarator(de, con);
				printf("\nabstract declarator  = "); ast_decl_show(de);
				printf("\nconcrete replacement = "); ast_decl_show(con);
				printf("\nresultant declarator = "); ast_decl_show(res);
				printf("\n");
				/* (*x)[5] */
				con = Declarator(NULL, ArrayDecl(ParenDecl(Declarator(Declspec(SPEC_star), IdentifierDecl(Symbol("x")))), NULL,  Constant("5")));
				res = ast_xt_abstract_to_concrete_declarator(de, con);
				printf("\nabstract declarator  = "); ast_decl_show(de);
				printf("\nconcrete replacement = "); ast_decl_show(con);
				printf("\nresultant declarator = "); ast_decl_show(res);
				printf("\n");
#endif
			}
			break;
		case SEM_EXPR:
			expr = parse_expression_string(A_str_string());
			if (expr != NULL)
			{
				astspec sp = NULL;
				astdecl de = NULL;
				
				if (sem_expr_typeof(expr, &sp, &de))
				{
					printf("\tTypeof is: ");
					if (sp) { ast_spec_show(sp); printf(" "); }
					if (de) ast_decl_show(de);
					printf("\n");
					
					printf("\tA declaration: ");
					sp = NULL; de = NULL;
					sem_expr_typeof_declare(expr, Symbol("var"), &sp, &de);
					if (sp) { ast_spec_show(sp); printf(" "); }
					if (de) ast_decl_show(de);
					printf(";\n");
				}
			}
			break;
		case VARACC:
			expr = parse_expression_string(A_str_string());
			if (expr != NULL)
			{
				acclist_t l = { NULL, 0 };
				expr2acclist(expr, &l);
				acclist_show(&l);
				printf("\n");
			}
			break;
		default:
			break;
	}
	return (0);
}


/* Some initializations before creating the AST */
static 
void _initialize_compiler_parts()
{
	stab = Symtab();       /* Create the symbol table */
	
	/* Take care of GCC and other extensions */
	symtab_put(stab, Symbol("__builtin_va_list"), TYPENAME);
	symtab_put(stab, Symbol("__extension__"), TYPENAME);
	symtab_put(stab, Symbol("__uint128_t"), TYPENAME);
	symtab_put(stab, Symbol("_Float16"), TYPENAME);
	symtab_put(stab, Symbol("__float128"), TYPENAME);
	
	/* This is a reserved identifier in C99; it is actually supposed to be
	 * "declared" at the top of each function; we simply insert it
	 * @ global scope so it is visible everywhere.
	 */
	symtab_put(stab, Symbol("__func__"), IDNAME);
	symtab_put(stab, Symbol("nullptr"), IDNAME);      /* C23 */

	ast_initomp();         /* Initialize OpenMP stuff */
	clr_init_rules();      /* Clause rules located in omp/x_* files & ast.c */

	codetargs_init(str_string(devtargs)); /* Prepare multi-target compilation */
}


/**
 * Check for required runtime functionality 
 */
static 
void find_required_functionality()
{
	set(cgfun) calledfuncs;
	setelem(cgfun) e;
	
	/* The following lines are part of a general effort to know whether
	 * the OMPi runtime and device functionality should be included.
	 * The concept is that for simple applications that do not use any
	 * OpenMP functionality, we need not initialize the OMPi runtime at all.
	 * Likewise, if no target-related constructs or runtime calls are used,
	 * we need not initialize the device functionality. 
	 *
	 * See also: add_reqfuncs() at ast_xform.c
	 */
	 
	/* (1) Check if any omp_target_* or generally omp_* functions are used */
	cg_find_defined_funcs(ast);   /* Discover all defined functions first */
	calledfuncs = cg_find_called_funcs(ast);
	
	for (e = calledfuncs->first; e; e = e->next)
	{
		if ((strncmp(e->key->name, "omp_target", strlen("omp_target")) == 0) ||
		    ((strcmp(e->key->name, "omp_get_num_devices") == 0)))
		{
			__has_devfuncs = 1;
			if (__has_ompfuncs && __has_ompxfuncs) break; /* no need to continue */
		}
		else if (strncmp(e->key->name, "omp_", strlen("omp_")) == 0)
		{
			__has_ompfuncs = 1;
			if (__has_devfuncs && __has_ompxfuncs) break; /* no need to continue */
		}
		else if (strncmp(e->key->name, "ompx_", strlen("ompx_")) == 0)
		{
			__has_ompxfuncs = 1;
			if ((strstr(e->key->name, "device") != NULL) || 
			    (strstr(e->key->name, "module") != NULL))
			  __has_devfuncs = 1;
			if (__has_devfuncs && __has_ompfuncs) break; /* no need to continue */
		}
	}
		
	/* (2) Fix the '__has_omp' flag */
	if (!__has_omp)
	{
		if (__has_devfuncs)
			__has_omp = 1;
		else if (__has_decltarg)
			__has_omp = __has_onlydecltarg = 1;
	}

	/* (3) The parser has left the symbol table at global scope; check stuff */
	includes_omph = (symtab_get(stab, Symbol("omp_lock_t"), TYPENAME) != NULL);
	knowMemcpy = (symtab_get(stab, Symbol("memcpy"), FUNCNAME) != NULL);
	knowMemset = (symtab_get(stab, Symbol("memset"), FUNCNAME) != NULL);
	
#if 1
	/* (4) Optionally force libort initialization */
	/* Sometimes we need to intialize libort even if there is no main() in the
	 * user program (e.g. in android). Thus, we force a call to _ort_init(). 
	 */
	if (forceInitOrt)
	{
		if (hasMainfunc)
			warning("[warning]: --initort option ignored when main() is present.\n");
		else
			bld_autoinits_add(  /* _ort_init(NULL, NULL, 0, 0); */
				FuncCallStmt(
					"_ort_init", 
					Comma4(NullExpr(), NullExpr(), ZeroExpr(), ZeroExpr())
				)
			);
	}
#endif
}


/**
 * Get options, set working mode, initialize stuff
 */
static
void _initializations(int argc, char *argv[])
{
	char *s;
	if (argc < 2)
	{
	OMPI_FAILURE:
		fprintf(stderr, "** %s should not be run directly; use %scc instead\n",
		        argv[0], argv[0]+1);
		exit(20);
	}
	
	if (strcmp(argv[1], "__internal__") == 0)
		exit(privatemode(argc, argv));
	if (argc < 3)
		goto OMPI_FAILURE;

	if (strcmp(argv[2], "__ompi__") != 0)
	{
		if (strcmp(argv[2], "__intest__") == 0)
			testingmode = 1;
		else
			goto OMPI_FAILURE;
	}
	
	if (argc > 3 && getopts(argc - 3, argv + 3))
		goto OMPI_FAILURE;
	
	filename = argv[1];
	filename_noext = strdup(filename);
	if ((s = strrchr(filename_noext, '.')) != NULL) *s = 0;

	if (!processmode) threadmode = true;  /* By default */

	_initialize_compiler_parts();
	
	/* Parse device options after codetargs initialization
	 * (as it requires knowing all module names)
	 */
	if (argc > 3 && getdevopts(argc - 3, argv + 3))
		goto OMPI_FAILURE;
}


void exit_error(int exitvalue, char *format, ...)
{
	va_list ap;
	va_start(ap, format);
	vfprintf(stderr, format, ap);
	va_end(ap);
	exit(exitvalue);
}


void warning(char *format, ...)
{
	va_list ap;
	va_start(ap, format);
	vfprintf(stderr, format, ap);
	va_end(ap);
}


int main(int argc, char *argv[])
{
	int     r;
	aststmt p, b = NULL;

	_initializations(argc, argv);
	
	/*
	 * 1. Parse & get the AST
	 */

	ast = parse_file(filename, &r);
	if (r) return (r);
	if (ast == NULL)        /* Cannot open file */
	{
		fprintf(stderr, "Error opening file %s for reading!\n", filename);
		return (30);
	}
	
	find_required_functionality();
	
	if ((!__has_omp || !enableOpenMP) && (!__has_ompix || !enableOmpix)
	    && !testingmode && !processmode)
		return (33);          /* Leave it as is */

	/*
	 * 2. Transform
	 */

	symtab_drain(stab);     /* Drain the symbol table */
	
	bld_prepare_for_xform(&ast);   /* Prepare the AST */
	if (__has_target)              /* Prepare for multiple targets */
		codetargs_kernels_prepare();

	if ((__has_omp  && enableOpenMP) || (__has_ompix && enableOmpix) 
	    || testingmode || processmode)
	{
		symtab_drain(stab);    /* Empty it; new globals will be traversed again */
	
		ast_xform(&ast);       /* The transformation phase */

		p = bld_top_comment(); /* Add top comment and needed stuff */ 
		p = bld_add_needed_std_prototypes(p);
		if ((__has_omp && enableOpenMP) && usedmods && bundleKernels)
		{
			b = Verbatim("");
			p = BlockList(p, b);
		}
		ast = BlockList(p, ast);
	}

	/*
	 * 3. Add final stuff and output
	 */

	/* Append new main() */
	if (hasMainfunc)
	{
		ast = BlockList(ast, bld_new_main());
	}
	
	cg_find_defined_funcs(ast);   /* The AST is finalized; mark all functions */

	if (__has_omp && enableOpenMP && usedmods)
	{
		codetargs_kernels_generate_kernelfiles(); /* IMPORTANT: global scope is still alive! */
		
		/* Replace b with actual bubins tree; we do it after calling
		 * cg_find_defined_funcs() to avoid bugs.
		 */
		if (bundleKernels)
		{
			if ((p = codetargs_kernels_bubins_tree()) != NULL)
				*b = *p; /* replacement */
		}
	}

	ast_show(ast);                /* Generate the output file */

	if (__has_target)             /* Free all kernel-related data */
		codetargs_kernels_free();

	if (testingmode)
	{
		/* Clean up (not needed actually; we do it only when testing) */
		ast_free(ast);
		symtab_drain(stab);
		symbols_allfree();
		ast_xt_free_retired();
	}
	
	return (0);
}
