/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* x_taskloop.c */

#include "ast_xform.h"
#include "ast_copy.h"
#include "ast_free.h"
#include "ast_renlabs.h"
#include "x_task.h"
#include "x_clauses.h"
#include "ompi.h"
#include "outline.h"

#define DEPARRAY "_deps"

static vartype_t xt_implicitDefault(setelem(vars) s, void *arg)
{
	stentry orig = symtab_get(stab, s->key, IDNAME);
	if ((orig->scopelevel == 0 && closest_parallel_scope < 0) ||
	    orig->scopelevel <= closest_parallel_scope ||
	    orig->isthrpriv)
		return DCT_BYREF;
	return DCT_BYVALUE;
}


/* Produces an expression that calls the outlined function */
static astexpr xt_callsite_xtraargs;
astexpr xt_callsite_expr(symbol func, astexpr funcargs)
{
	/* Add the extra parameters */
	funcargs = funcargs ? CommaList(funcargs, xt_callsite_xtraargs) : 
	                      xt_callsite_xtraargs;
	return FunctionCall(
		       IdentName("_ort_new_task"),
		       CommaList(Identifier(func), funcargs)
		     );
}


void xform_taskloop(aststmt *t, taskopt_e opt)
{
	static int taskloopnum = 0;

	aststmt   body_copy, parent, vardecls, varinits;
	aststmt   fast, fastdecls = NULL;                   /* For the fast path */
	astexpr   ifexpr = NULL, finalexpr = NULL, prioexpr = NULL, fastif;
	ompclause c;
	bool      untied, mergeable;
	outcome_t op;
	char      clabel[22];

	static outline_opts_t oo =
	{
		/* structbased             */  true,              
		/* functionName            */  "",                
		/* functionCall  (func)    */  xt_callsite_expr,  
		/* byvalue_type            */  BYVAL_bycopy,      
		/* byref_type              */  BYREF_pointer,     
		/* byref_copyptr (2 funcs) */  NULL, NULL,        
		/* global_byref_in_struct  */  false,             
		/* structName              */  "__taskenv__",     
		/* structVariable          */  "_tenv",           
		/* structInitializer       */  NULL,              
		/* implicitDefault (func)  */  xt_implicitDefault,
		/* implicitDefault (args)  */  NULL,
		/* deviceexpr              */  NULL,              
		/* addComment              */  true,              
		/* thestmt                 */  NULL,
		/* userType                */  NULL,
		/* usePointers             */  true,
		/* makeReplCode            */  true,
		/* makeWrapper             */  false,
		/* wrapperType             */  WRAPPER_none
	};

	/* The name of the label used for canceling. We use line number to avoid
	 * conflicts
	 */
	snprintf(clabel, 22, "CANCEL_task_%d", (*t)->u.omp->l);

	/* We insert the label before creating a copy, to prevent doing it twice and
	 * to make sure that the label gets renamed correctly (in case mergeable is
	 * present)
	 */
	if ((*t)->u.omp->body->type == COMPOUND)
		ast_stmt_in_place_append((*t)->u.omp->body->body,
		                 Labeled(Symbol(clabel), Expression(NULL)));
	else
		(*t)->u.omp->body = Compound(
		                      BlockList(
		                        (*t)->u.omp->body,
		                        Labeled(Symbol(clabel), Expression(NULL))
		                      )
		                    );

	body_copy = ast_stmt_copy((*t)->u.omp->body);         /* Keep a full copy */

	/* (1) Take care of special clauses:
	 *     - Check for untied, mergeable, if and final clauses and keep a copy
	 *     - Prepare a dependence array, if corresponding clauses are present
	 */
	if ((c = xc_ompcon_get_clause((*t)->u.omp, OCIF)) != NULL)
		ifexpr = ast_expr_copy(c->u.expr);
	if ((c = xc_ompcon_get_clause((*t)->u.omp, OCFINAL)) != NULL)
		finalexpr = ast_expr_copy(c->u.expr);
	if ((c = xc_ompcon_get_clause((*t)->u.omp, OCPRIORITY)) != NULL)
		prioexpr = ast_expr_copy(c->u.expr);
	untied = (xc_ompcon_get_clause((*t)->u.omp, OCUNTIED) != NULL);
	mergeable = (xc_ompcon_get_clause((*t)->u.omp, OCMERGEABLE) != NULL);

	/* (2) Outline
	 */
	sprintf(oo.functionName, "_taskloopFunc%d_", taskloopnum++);
	/* (struct structType *) _ort_taskenv_alloc(sizeof(struct), functionName); */
	oo.structInitializer =
	  CastedExpr(
	    Casttypename(
	      SUdecl(SPEC_struct, Symbol(oo.structType), NULL, NULL),
	      AbstractDeclarator(Pointer(), NULL)
	    ),
	    FunctionCall(
	      IdentName("_ort_taskenv_alloc"),
	      CommaList(
	        Sizeoftype(Casttypename(
	                     SUdecl(SPEC_struct, Symbol(oo.structType), NULL, NULL),
	                     NULL
	                   )),
	        IdentName(oo.functionName)
	      )
	    )
	  );
	xt_callsite_xtraargs = 
		Comma8(
			ifexpr ? ast_expr_copy(ifexpr) : ZeroExpr(),       /* if expression */
			finalexpr ? ast_expr_copy(finalexpr) : ZeroExpr(), /* finalxpression */
			numConstant(untied ? 1 : 0),                       /* tied(0)/untied(1) */
			prioexpr ? ast_expr_copy(prioexpr) : ZeroExpr(),   /* priority expr */
			NullExpr(),                                        /* dependece array */
			ZeroExpr(),                                        /* #outdeps */
			ZeroExpr(),                                        /* #indeps */
			ZeroExpr()                                         /* #inoutdeps */
		);
	oo.thestmt = *t;
	
	op = outline_OpenMP(t, oo); //TODO fix placement of comment
	
	/* (3) Produce a "fast" path:
	 *       {
	 *         <possible local declarations / initializations>
	 *         _ort_task_immediate_start(...)
	 *           <original code>
	 *         _ort_task_immediate_end(...)
	 *       }
	 */
	parent = op.replacement->parent;

	fast = Block3(
	         Expression(    /* _itn = _ort_task_immediate_start(<finalexpr>); */
	           Assignment(IdentName("_itn"), ASS_eq,
	                      FunctionCall(
	                        IdentName("_ort_task_immediate_start"),
	                        finalexpr ? finalexpr : ZeroExpr()
	                      ))
	         ),
	         body_copy,
	         Expression(  /* _ort_task_immediate_end(_itn); */
	           FunctionCall(IdentName("_ort_task_immediate_end"),
	                        IdentName("_itn"))
	         )
	       );
	
	/* Insert local declarations in the top of the fast path */
	vardecls = out_inline_firstprivate(op->usedvars[DCT_BYVALUE], &varinits);
	if (vardecls)
	{
		if (varinits)
			vardecls = BlockList(vardecls, varinits);
		fastdecls = vardecls;
	}
	vardecls = out_inline_private(op->usedvars[DCT_PRIVATE]);
	if (vardecls)
		fastdecls = BlockList(vardecls, fastdecls);

	/* Insert declaration: "void *_itn;" */
	if (!mergeable || !hasdeps || ifexpr)
		fast = BlockList(
		         Declaration(
		           Declspec(SPEC_void),
		           Declarator(Declspec(SPEC_star), IdentifierDecl(Symbol("_itn")))
		         ),
		         fast
		       );

	if (fastdecls)
		fast = BlockList(fastdecls, fast);

	/* void *_itn; */
	fast = BlockList(
	         Declaration(
	           Declspec(SPEC_void),
	           Declarator(Declspec(SPEC_star), IdentifierDecl(Symbol("_itn")))
	         ),
	         fast
	       );

	fast = Compound(fast);

	/* (4) Combine slow & fast paths & a possibly even faster third path.
	 */
	if (opt == OPT_ULTRAFAST && mergeable)
	{
		fastif = Call0_expr("_ort_task_throttling");
		if (ifexpr)
			fastif = BinaryOperator(
			           BOP_lor,
			           UnaryOperator(UOP_lnot, Parenthesis(ifexpr)),
			           fastif
			         );

		*t = If(Call0_expr("omp_in_final"),
		        Compound(
		          fastdecls ?
		          BlockList(ast_stmt_copy(fastdecls), ast_stmt_copy(body_copy)) :
		          ast_stmt_copy(body_copy)
		        ),
		        fastif ? If(fastif, fast, op.replacement) : op.replacement
		     );
		/* Rename labels on the first if block */
		ast_stmt_renlabs((*t)->body);
	}
	else
	{
		fastif = Call0_expr("omp_in_final");
		if (ifexpr)
			fastif = BinaryOperator(
			           BOP_lor,
			           UnaryOperator(UOP_lnot, Parenthesis(ifexpr)),
			           fastif
			         );
		fastif = BinaryOperator(
		           BOP_lor,
		           fastif,
		           Call0_expr("_ort_task_throttling")
		         );
			
		if (opt != OPT_NONE)
			*t = If(fastif, fast, op.replacement);
		else
		{
			*t = Compound(op.replacement);
			if (fast) ast_stmt_free(fast);
			if (fastif) ast_expr_free(fastif);
		}
	}

	(*t)->parent = parent;

	//If we used a struct for byref/byvalue variables call _ort_taskenv_free
	if (op.func_struct)
		ast_stmt_in_place_prepend(op.func_return,
		                          FuncCallStmt(
		                            "_ort_taskenv_free",
		                            CommaList(
		                              IdentName(oo.structName),
		                              IdentName(oo.functionName)
		                            )
		                          )
		                         );
}
