/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* x_requires.c -- Handle OpenMP #requires directive */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ompi.h"
#include "sem.h"
#include "set.h"
#include "ast_free.h"
#include "ast_xform.h"
#include "ast_copy.h"
#include "ast_assorted.h"
#include "ast_types.h"
#include "ast_arith.h"
#include "builder.h"
#include "x_arrays.h"
#include "x_clauses.h"
#include "x_reduction.h"
#include "x_requires.h"

/* Standard OpenMP requirements */
_requires_flag(OCREVERSEOFFLOAD) = false;
_requires_flag(OCUNIFIEDADDRESS) = false;
_requires_flag(OCUNIFIEDSHAREDMEMORY) = false;
_requires_flag(OCDYNAMICALLOCATORS) = false;
_requires_flag(OCATOMICDEFAULTMEMORDER) = false;
ompclsubt_e __req_memorder_value;
unsigned int num_requirements = 0;

/* Extensions: implementation requirements */
static struct {
		char *name; /* the name of the requirement (the XXXX part of ext_XXX) */
		bool __req; /* true if required */
	} _ext_required_flags[] = {
	                            { "dummyreq", false },
	                            { NULL, false }         /* mark the end */
	                          };


/* Sets an implementation-specific requirement to true */
static bool set_ext_required(symbol s)
{
	int i;
	
	for (i = 0; _ext_required_flags[i].name != NULL; i++)
		if (strcmp(_ext_required_flags[i].name, s->name) == 0)
		{
			_ext_required_flags[i].__req = true;
			num_requirements++;
			return true;
		};
	return false;
}


/* Returns the implementation-specific requirement flag */
bool _ext_required(char *name)
{
	int i;
	
	for (i = 0; _ext_required_flags[i].name != NULL; i++)
		if (strcmp(_ext_required_flags[i].name, name) == 0)
			return _ext_required_flags[i].__req;

	return false;
}


#define _check_and_print_req(s,cl,subt,q) \
	if (_required(cl)) \
	{ \
		str_printf((s), "%s%s", (q), \
		                ompclauseinfo[cl].name); \
		if ((subt) != OC_DontCare) \
			str_printf(s, "(%s)", clausesubs[(subt)]); \
		str_printf((s), "%s%s", (q), (cnt++ < num_requirements-1) ? ", " : ""); \
	}


/* Dumps requirements in a string like this:
 *   quoted == false: req1, req2, ...
 *   quoted == true:  "req1", "req2", ...
 */
void stringify_reqs(str s, bool quoted)
{
	int i;
	char *q = (quoted) ? "\"" : "";
	int cnt = 0;

	if (num_requirements == 0) 
		return;

	/* Print standard requirements */
	_check_and_print_req(s, OCDYNAMICALLOCATORS,     OC_DontCare, q);
	_check_and_print_req(s, OCUNIFIEDSHAREDMEMORY,   OC_DontCare, q);
	_check_and_print_req(s, OCUNIFIEDADDRESS,        OC_DontCare, q);
	_check_and_print_req(s, OCREVERSEOFFLOAD,        OC_DontCare, q);
	_check_and_print_req(s, OCATOMICDEFAULTMEMORDER, __req_memorder_value, q);

	/* Print ext_* requirements */
	for (i = 0; _ext_required_flags[i].name != NULL; i++)
		if (_ext_required_flags[i].__req)
			str_printf(s, "%sext_%s%s%s", q, _ext_required_flags[i].name, q,
			              _ext_required_flags[i+1].name != NULL ? ", " : "");
}


void xform_requires(aststmt *t)
{
	aststmt parent = (*t)->parent, v;
	ompclause c;
	v = ompdir_commented((*t)->u.omp->directive);
	static int num_atomic_default_memorder = 0;

	if (xc_ompcon_get_clause((*t)->u.omp, OCREVERSEOFFLOAD) ||
	    xc_ompcon_get_clause((*t)->u.omp, OCUNIFIEDADDRESS) ||
	    xc_ompcon_get_clause((*t)->u.omp, OCUNIFIEDSHAREDMEMORY))
		exit_error(1, "(%s, line %d) openmp error:\n\t"
		              "the reverse_offload, unified_address and "
		              "unified_shared_memory\n\trequirements are not supported\n",
		              (*t)->u.omp->directive->file->name, 
		              (*t)->u.omp->directive->l);

	if (xc_ompcon_get_clause((*t)->u.omp, OCDYNAMICALLOCATORS))
	{
		_required(OCDYNAMICALLOCATORS) = true;
		num_requirements++;
	}

	c = xc_ompcon_get_clause((*t)->u.omp, OCATOMICDEFAULTMEMORDER);
	if (c)
	{
		if (num_atomic_default_memorder && __req_memorder_value != c->subtype)
			exit_error(1, "(%s, line %d) openmp error:\n\t"
			              "at most one type of atomic default memory order may "
			              "appear in all\n\t#requires directives\n",
			              (*t)->u.omp->directive->file->name, 
			              (*t)->u.omp->directive->l);
		num_atomic_default_memorder++;
		__req_memorder_value = c->subtype;
		_required(OCATOMICDEFAULTMEMORDER) = true;
		num_requirements++;
	}
	
	/* Check for implementation-defined requirements (ext_*) 
	 */
	if ((c = xc_ompcon_get_every_clause((*t)->u.omp, OCEXT)))
	{
		while (c != NULL)
		{
			if (c->type != OCLIST)
			{
				if (!set_ext_required(c->s))
					exit_error(1, "(%s, line %d) openmp error:\n\t"
					              "unknown requirement 'ext_%s'.\n",
					              (*t)->u.omp->directive->file->name, 
					              (*t)->u.omp->directive->l, c->s->name);
				break;
			}
			if (!set_ext_required(c->u.list.elem->s))
				exit_error(1, "(%s, line %d) openmp error:\n\t"
				              "unknown requirement 'ext_%s'.\n",
				              (*t)->u.omp->directive->file->name, 
				              (*t)->u.omp->directive->l, c->s->name);
			c = c->u.list.next; /* Move to the next item */
		}
		ast_ompclause_free(c); /* Free the list */
	}

	ast_free(*t);
	*t = v; /* We are done */
	(*t)->parent = parent;
}
