/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <stdlib.h>
#include <assert.h>
#include "x_ordered.h"
#include "ast_xform.h"
#include "ast_free.h"
#include "ast_assorted.h"
#include "ast_copy.h"
#include "ast.h"
#include "x_clauses.h"
#include "ompi.h"
#include "x_loops.h"


void xform_ordered_doacross(aststmt *t)
{
	aststmt   parent = (*t)->parent, 
	          v = ompdir_commented((*t)->u.omp->directive);
	ompclause cl;
	ompcon    enclosing;
	int       i, ordnum, ncla;
	symbol    *indices;
	astexpr   args, vec, initer;

	/* Find enclosing #for and get its ordered clause parameter */
	enclosing = ast_get_enclosing_ompcon((*t)->parent, 0);
	if ((enclosing->type != DCFOR && enclosing->type != DCFOR_P) || 
	    !(cl = xc_ompcon_get_clause(enclosing, OCORDEREDNUM)))
		exit_error(1, "(%s, line %d) openmp error:\n\t"
			"stand-alone ordered constructs should be closely nested in\n\t"
			"doacross loops.\n",
			(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);

	/* Get loop nest depth */
	ordnum = cl->subtype; 
	if (ordnum <= 0)
		exit_error(1, "(%s, line %d) openmp error:\n\t"
			"ordered loop nest depth must be > 0.\n",
			(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);

	/* Get the depend clause(s) */
	cl = xc_ompcon_get_clause((*t)->u.omp, OCDEPEND);
	if (!cl)    /* impossible for stand-alone ordered (see parser.y) */
		exit_error(1, "(%s, line %d) openmp error:\n\t"
			"stand-alone ordered constructs should contain depend clause(s).\n",
			(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	if (cl->subtype != OC_source && cl->subtype != OC_sink)   /* impossible;  */
		exit_error(1, "(%s, line %d) openmp error:\n\t"         /* see parser.y */
			"stand-alone ordered constructs should contain depend(source) or\n\t"
			"depend(sink: ) clause(s).\n",
			(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);

	/* Get the names of the loop nest indices (FIXME: we assume they are ints) */
	indices = loopnest_get_indices(enclosing->body, ordnum);
		
	if (cl->subtype == OC_source)
	{
		args = Identifier(indices[0]);
		for (i = 1; i < ordnum; i++)
			args = CommaList(args, Identifier(indices[i]));
		v = BlockList(
					v,
					FuncCallStmt(
						"_ort_doacross_post", 
						Comma2(
							IdentName(DOACCPARAMS),
							CastedExpr(         /* (long[]) { indices } */
								Casttypename(
									Declspec(SPEC_long),
									AbstractDeclarator(
										NULL,
										ArrayDecl(NULL, NULL, NULL)
									)
								),
								BracedInitializer(args)
							)
						)
					)
				);
	}
	else   /* sink */
	{
		initer = NULL;
		ncla = 0;
		cl = xc_ompcon_get_every_clause((*t)->u.omp, OCDEPEND);  /* memory leak */
		do
		{
			ncla++;
			vec = (cl->type == OCLIST) ? cl->u.list.elem->u.expr : cl->u.expr;
			args = NULL;
			for (i = 0; i < ordnum; i++)
			{
				if ((i != ordnum-1 && vec->type != COMMALIST) ||   /* sanity */
						(i == ordnum-1 && vec->type == COMMALIST))
					exit_error(1, "(%s, line %d) openmp error:\n\t"
						"sink indices (%d) are %s than the doacrros loop indices (%d).\n",
						(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l,
						i+1, (i == ordnum-1) ? "more" : "fewer", ordnum);
				if (i != ordnum-1)
				{
					assert(vec->left->left->type == IDENT);
					if (vec->left->left->u.sym != indices[i])
						exit_error(1, "(%s, line %d) openmp error:\n\t"
							"sink index '%s' does not correspond to loop index #%d (%s).\n",
							(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l,
							vec->left->left->u.sym->name, i+1, indices[i]->name); 
					args = args ? CommaList(args, ast_expr_copy(vec->left)) : 
					              ast_expr_copy(vec->left);
					vec = vec->right;
				}
				else
				{
					assert(vec->left->type == IDENT);
					if (vec->left->u.sym != indices[i])
						exit_error(1, "(%s, line %d) openmp error:\n\t"
							"sink index '%s' does not correspond to loop index #%d (%s).\n",
							(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l,
							vec->left->u.sym->name, i+1, indices[i]->name); 
					args = args ? CommaList(args,ast_expr_copy(vec)) : ast_expr_copy(vec);
				}
			}
			initer = initer ? CommaList(initer, args) : args;
			if (cl->type != OCLIST)
				break;
			else
				cl = cl->u.list.next;
		}
		while (cl != NULL);
		
		v = BlockList(
					v,
					FuncCallStmt(
						"_ort_doacross_wait", 
						Comma3(
							IdentName(DOACCPARAMS),
							numConstant(ncla), 
							CastedExpr(         /* (int[]) { deps } */
								Casttypename(
									Declspec(SPEC_long),
									AbstractDeclarator(
										NULL,
										ArrayDecl(NULL, NULL, NULL)
									)
								),
								BracedInitializer(initer)
							)
						)
					)
				);
	}
	free(indices);
	ast_free(*t);
	*t = v;
	(*t)->parent = parent;
}


void xform_ordered(aststmt *t)
{
	aststmt s, parent, v;
	bool    stlist;   /* See comment above */

	/* First transform the body */
	ast_stmt_xform(&((*t)->u.omp->body));

	s = (*t)->u.omp->body;
	parent = (*t)->parent;

	if (OMPCON_IS_STANDALONE((*t)->u.omp))
	{
		xform_ordered_doacross(t);
		return;
	}
	
	/* Good ol' ORDERED */
	stlist = ((*t)->parent->type == STATEMENTLIST ||
						(*t)->parent->type == COMPOUND);

	v = ompdir_commented((*t)->u.omp->directive);
	(*t)->u.omp->body = NULL;     /* Make it NULL so as to free t easily */
	ast_free(*t);                 /* Get rid of the OmpStmt */

	*t = Block5(
				v, 
				Call0_stmt("_ort_ordered_begin"),
				s,
				Call0_stmt("_ort_ordered_end"),
				linepragma(s->l + 1 - (!stlist), s->file)
			);
	if (!stlist)
		*t = Compound(*t);
	(*t)->parent = parent;
}
