/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* x_combine.c -- OpenMP statement combinations and splits. */

//#define DEBUGTHIS

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include "ompi.h"
#include "x_combine.h"
#include "x_clauserules.h"
#include "x_clauses.h"
#include "set.h"
#include "ast_copy.h"
#include "ast_free.h"
#include "ast_assorted.h"
#include "codetargs.h"
#include "ast_xform.h"

#ifdef DEBUGTHIS
	#include "ast_show.h"
	#include "codetargs.h"
	#include "ast_xform.h"
	#define IFDEBUG(a) a;
#else
	#define IFDEBUG(a)
#endif



/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *             RULES FOR COMBINING/SPLITTING                 *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
 
 
/* Combination rules describe pairs of directives that can be combined into
 * a single directive. Split rules describe which two directives should
 * a combined directive be split into. We organize them into 
 * arbitraty combination rulesets.
 *
 * To add a combination rule, simply list the outer, the inner and 
 * the combined directive names, as in 	{ DCTEAMS, DCDISTRIBUTE, DCTEAMSDIST }.
 * At this point we do not care if the resulting construct is a combined
 * or a composite one.
 *
 * To add a split rule, simply list the combined directive name, as well
 * as the produced outer and inner directive names, as in 
 * { DCTARGETTEAMSDIST, DCTARGETTEAMS, DCDISTRIBUTE }.
 */

 
/* 
 * COMBINING 
 */


#define COMBSET_DEFINE(m) combrule_t *m ## _mergeinfo = (combrule_t[])
#define COMBSET(m)        &m ## _mergeinfo
#define COMBSEND          { DCNONE, DCNONE, DCNONE }

typedef struct ompdir_merge_rule
{
	ompdirt_e outer;
	ompdirt_e inner;
	ompdirt_e result;
} combrule_t;


COMBSET_DEFINE(distribute) 
{
	{ DCDISTRIBUTE, DCPARFOR, DCDISTPARFOR },   /* composite */
	COMBSEND
};

COMBSET_DEFINE(teams) 
{
	{ DCTEAMS, DCDISTRIBUTE, DCTEAMSDIST },
	{ DCTEAMS, DCDISTPARFOR, DCTEAMSDISTPARFOR },
	COMBSEND
};

COMBSET_DEFINE(target) 
{
	{ DCTARGET, DCTEAMS, DCTARGETTEAMS },
	{ DCTARGET, DCTEAMSDIST, DCTARGETTEAMSDIST },
	{ DCTARGET, DCTEAMSDISTPARFOR, DCTARGETTEAMSDISTPARFOR },
	{ DCTARGETTEAMS, DCDISTPARFOR, DCTARGETTEAMSDISTPARFOR },
	{ DCTARGETTEAMS, DCDISTRIBUTE, DCTARGETTEAMSDIST },
	{ DCTARGET, DCPARALLEL, DCTARGETPARALLEL },
	{ DCTARGET, DCPARFOR, DCTARGETPARFOR },
	COMBSEND
};

combrule_t **all_combrules[] = 
{ 
	COMBSET(target), COMBSET(teams), COMBSET(distribute),
	NULL
};


/* 
 * SPLITTING 
 */


#define SPLITSET_DEFINE(m) splitrule_t *m ## _splitinfo = (splitrule_t[])
#define SPLITSET(m)        &m ## _splitinfo
#define SPLITSEND          { DCNONE, DCNONE, DCNONE }

typedef struct ompdir_split_rule
{
	ompdirt_e combined;
	ompdirt_e outer;
	ompdirt_e inner;
} splitrule_t;


SPLITSET_DEFINE(target) 
{
	{ DCTARGETTEAMSDIST, DCTARGETTEAMS, DCDISTRIBUTE },
	{ DCTARGETTEAMSDISTPARFOR, DCTARGETTEAMS, DCDISTPARFOR },
	{ DCTARGETTEAMS, DCTARGET, DCTEAMS },
	{ DCTARGETPARFOR, DCTARGET, DCPARFOR },
	{ DCTARGETPARALLEL, DCTARGET, DCPARALLEL },
	SPLITSEND
};

SPLITSET_DEFINE(parallel) 
{
	{ DCPARSECTIONS, DCPARALLEL, DCSECTIONS },
	{ DCPARFOR, DCPARALLEL, DCFOR_P },
	SPLITSEND
};

SPLITSET_DEFINE(teams) 
{
	{ DCTEAMSDIST, DCTEAMS, DCDISTRIBUTE },
	{ DCTEAMSDISTPARFOR, DCTEAMS, DCDISTPARFOR },
	SPLITSEND
};


splitrule_t **all_splitrules[] = 
{ 
	SPLITSET(target), SPLITSET(parallel), SPLITSET(teams),
	NULL
};


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *      COMBINE (MERGE) OPENMP STATEMENTS (CLAUSES)          *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
 
 
/**
 * This adds the directive name modifier required for a given clause, if it 
 * does not already have one. But when the given OpenMP directive is compound,
 * then if the clause applies to more than one constituent directives
 * (according to OpenMP v6.0 rules), we return an integer vector of all
 * the needed modifiers.
 *
 * @param dir the type of directive the clause belongs to
 * @param c   the clause (gets modified if a single modifier is needed)
 * @return    NULL if the clause itself already has a modifier or it iapplies
 *            to a single directive; otherwise, a vector of directive name 
 *            modifiers, one for each constituent directive
 */
intvec_t auto_dirname_modifiers_for_clause(ompdirt_e dir, ompclause c)
{
	intvec_t parts, res;
	int      i, nused = 0;
	
	/* Don't do anything if a modifer exists */
	if (get_dirname_modifier(c) != DCNONE) return NULL;
	
	if (!clr_allowed(dir, c->type))  /* Sanity */
		exit_error(1, "[%s] bug: clause '%s' unsuppored in directive %s.\n",
		           __func__, ompclauseinfo[c->type].name, ompdirnames[dir]);
	
	/* Compound directive: get constituent directives */
	if ((parts = constituent_directives(dir)) == NULL)
	{
		c->modifs = intvec_append(c->modifs, 1, DirnameModifier(dir));
		return NULL;
	}
	
	/* Filter-out special cases */
	/* TODO: handle FIRSTPRIVATE, too */
	if (c->type == OCREDUCTION && intvec_contains(parts, DCPARALLEL) &&
	    (intvec_contains(parts, DCFOR) || intvec_contains(parts, DCSECTIONS)))
	  parts = intvec_remove_all(parts, DCPARALLEL);

	if (ompclauseinfo[c->type].incomp == APPLY_ONCE ||
	    ompclauseinfo[c->type].incomp == APPLY_INNER)
	{
		APPLY_TO_INNER:
		for (i = intvec_len(parts)-1; i >= 0; i--)
			if (clr_allowed(parts[i], c->type))
			{
				c->modifs = intvec_append(c->modifs, 1, DirnameModifier(parts[i]));
				intvec_free(parts);
				return NULL;
			};
	}
	
	if (ompclauseinfo[c->type].incomp == APPLY_OUTER)
	{
		for (i = 0; i < intvec_len(parts); i++)
			if (clr_allowed(parts[i], c->type))
			{
				c->modifs = intvec_append(c->modifs, 1, DirnameModifier(parts[i]));
				intvec_free(parts);
				return NULL;
			};
	}
	
	/* APPLY_ALL */
	/* FIXME: must take special care for FIRSTPRIVATE, too */
	/* First count */
	for (i = intvec_len(parts)-1; i >= 0; i--)
		if (clr_allowed(parts[i], c->type))
			nused++;

	if (nused == 1)
		goto APPLY_TO_INNER;
	
	res = intvec_new(nused);
	for (i = intvec_len(parts)-1; i >= 0; i--)
		if (clr_allowed(parts[i], c->type))
			res[--nused] = DirnameModifier(parts[i]);
	intvec_free(parts);
	return res;
}


#if 0
/* 
 * Decides if an ompclause should be included to the list, during the
 * combination of two constructs (outer + inner).
 */
static void checkNadd_innerdir_clauses(ompclause *list, ompclause c, 
                                       ompdirt_e outerdir, ompdirt_e innerdir)
{
	/* shared() is completely ignored */
	if (c->type == OCSHARED)
	{
		/* #target + #parallel/#parallel for */
		if (outerdir == DCTARGET && (innerdir == DCPARALLEL || innerdir == DCPARFOR))
			return;
		/* #target teams + #distribute parallel for */
		if (outerdir == DCTARGETTEAMS && innerdir == DCDISTPARFOR)
			return;
	}

	*list = OmpClauseList(*list, c); /* Add to the list, normally */
}
#endif


/**
 * Given a clause and two directives that are about to be combined (assuming 
 * the clause belongs to the first one), modify it so as to be appropriate
 * for inclusion in the merged list of clauses; and add it to the list. 
 *
 * @param c      the clause
 * @param dir1   one of the directives, containing c
 * @param dir2   the other directived
 * @param mlc    the merged list of clauses
 * @return true  if all OK, false if there was an unresovled conflict.
 */
static void merge_one_clause_from_dirs(ompclause c, ompdirt_e dir1, 
                                       ompdirt_e dir2, ompclause *mlc)
{
	intvec_t mods;
	int      i;
	
	if (c == NULL || c->type == OCLIST) return;
	
	/* Add the clause to the list as is (it maybe modified later) */
	*mlc = (*mlc) ? OmpClauseList(*mlc, c) : c;
	
	/* If the clause has already a directive name modifer, or if the clause 
	 * is not supported in the other directive, we have nothing else to do.
	 */
	if (get_dirname_modifier(c) || !clr_allowed(dir2, c->type))
		return;
	
	/* In order to make sure the clause won't be applied to dir2 when the 
	 * combined construct is formed, we simply put directive name modifier(s); 
	 * if multiple such modifiers, then multiple copies of the clause are 
	 * necessary.
	 */
	if ((mods = auto_dirname_modifiers_for_clause(dir1, c)) == NULL)
		return;
	
	/* Create multiple copies of the cluse, appropriately modified */
	c->modifs = intvec_append(c->modifs, 1, mods[0]);
	for (i = intvec_len(mods)-1; i > 0; i--)
	{
		c = ast_ompclause_copy(c);                       /* copy clause */
		c->modifs[ intvec_len(c->modifs)-1 ] = mods[i];  /* change modifier */
		*mlc = OmpClauseList(*mlc, c);                   /* append to list */
	}
}


/**
 * Merge the clause lists of two directives into one (assuming the two 
 * directives will become a combined construct). The resulting clause list
 * consists of totally new nodes, i.e. the original clauses are not freed.
 *
 * @param outerdir The outer directive type
 * @param innerdir The inner directive type
 * @param all      The resulting merged clause list
 * @param fcl      The first (outer) clause list
 * @param scl      The second (inner) clause list
 * @return true    if all OK, false if there was an unresovled conflict.
 */
static void merge_clauses_from_dirs(ompdirt_e outerdir, ompdirt_e innerdir,
                                   ompclause fcl, ompclause scl, ompclause *all)
{
	*all = NULL;
	
	/* Add the outer clauses */
	while (fcl)
		if (fcl->type == OCLIST)
		{
			merge_one_clause_from_dirs(ast_ompclause_copy(fcl->u.list.elem), 
			                           outerdir, innerdir, all);
			fcl = fcl->u.list.next; /* Proceed */
		}
		else
		{
			merge_one_clause_from_dirs(ast_ompclause_copy(fcl),outerdir,innerdir,all);
			break;
		};
		
	/* Add the inner clauses */
	while (scl)
		if (scl->type == OCLIST)
		{
			merge_one_clause_from_dirs(ast_ompclause_copy(scl->u.list.elem), 
			                           innerdir, outerdir, all);
			scl = scl->u.list.next; /* Proceed */
		}
		else
		{
			merge_one_clause_from_dirs(ast_ompclause_copy(scl),innerdir,outerdir,all);
			break;
		};
}


/**
 * Get corresponding combined version of two given OMP statements
 *
 * @param  t     The outer OMP statement
 * @param  with  The inner OMP statement
 * @return The directive type of the combined construct, otherwise DCNONE.
 */
static
ompdirt_e ccc_combtype(aststmt t, aststmt with) 
{
	ompdirt_e ttype, withtype, outer;
	int i, j;

	if ((t->type != OMPSTMT) || (with->type != OMPSTMT))
		return DCNONE;

	ttype = OmpStmtDir(t)->type;
	withtype = OmpStmtDir(with)->type;
	for (i = 0; all_combrules[i] != NULL; i++)
		for (j = 0; (outer = (*all_combrules[i])[j].outer) != DCNONE; j++)
			if ((outer == ttype) && ((*all_combrules[i])[j].inner == withtype))
				return (*all_combrules[i])[j].result;
		
	return DCNONE;
}


/**
 * Check if one OMP statement can be combined with another, according to
 * the combination rules
 *
 * @param  t The OMP statement
 * @return true if given OMP statement can be combined with another one.
 */
bool ccc_iscombinable(aststmt t) 
{
	ompdirt_e mtype, outer, inner;
	aststmt innerstmt;
	int i, j;

	if (t->type != OMPSTMT)
		return false;

	for (innerstmt = t->u.omp->body; innerstmt->type == COMPOUND; )
		innerstmt = innerstmt->body;  /* Should contain only 1 OpenMP statement */

	if (ccc_combtype(t, innerstmt) == DCNONE)
		return false;

	mtype = OmpStmtDir(t)->type;
	for (i = 0; all_combrules[i] != NULL; i++)
	{
		for (j = 0; (outer = (*all_combrules[i])[j].outer) != DCNONE; j++)
			if (outer == mtype)
				return true;
		for (j = 0; (inner = (*all_combrules[i])[j].inner) != DCNONE; j++)
			if (inner == mtype)
				return true;
	}

	return false;
}


/**
 * Produce the combined version of two directives. The result 
 * replaces the first directive.
 *
 * @param t1      The parent (outer) directive that encloses t2
 * @param t2      The child (inner) directive
 * @param newtype The resulting directive type
 * @return true if the combination was successful, false otherwise.
 */
static
void combine_ompstmts(aststmt *t1, aststmt t2, ompdirt_e newtype)
{
	ompclause newcls = NULL;
	ompdirt_e outerdir = (*t1)->u.omp->type, 
	          innerdir = t2->u.omp->type;

	IFDEBUG((fprintf(stderr, "\nCombining (%s):\n\t",codetarg_name(xformingFor))))
	IFDEBUG((ast_ompdir_show_stderr(OmpStmtDir(*t1))))
	IFDEBUG((fprintf(stderr, "\t  ")))
	IFDEBUG((ast_ompdir_show_stderr(OmpStmtDir(t2))))
	
	/* The result is stored in the first stmt */
	(*t1)->u.omp->type = newtype;  
	OmpStmtDir(*t1)->type = newtype;

	/* Merge the clauses of the two directives */
	merge_clauses_from_dirs(outerdir, innerdir, OmpStmtDir(*t1)->clauses, 
	                        OmpStmtDir(t2)->clauses, &newcls);
	ast_ompclause_parent(OmpStmtDir(*t1), newcls); /* Fix the clauses' parent */
	ast_ompclause_free(OmpStmtDir(*t1)->clauses);
	OmpStmtDir(*t1)->clauses = newcls;

	/* Some bookkeeping... */
	(*t1)->u.omp->body->file = OmpStmtDir(*t1)->file;
	(*t1)->u.omp->body->l    = OmpStmtDir(*t1)->l;
	(*t1)->u.omp->body->c    = OmpStmtDir(*t1)->c;

	/* Store the actual body in the result & free t2 */
	(*t1)->u.omp->body = t2->u.omp->body;
	(*t1)->u.omp->body->parent = *t1;              /* Fix the parent */
	t2->u.omp->body = NULL;
	ast_stmt_free(t2);

	IFDEBUG((fprintf(stderr, "gives:\n\t")))
	IFDEBUG((ast_ompdir_show_stderr(OmpStmtDir(*t1))))
}


/**
 * Try to combine a directive with its enclosed one, according to the
 * combination rules. The combination will take place only if the 
 * enclosed (inner) directive is the only child of the node, either 
 * contained in a compound or not.
 *
 * @param  t     the given directive that may enclose another directive
 * @return true  if a combined statement was indeed possible
 */
bool ccc_try_combining(aststmt *t)
{
	ompdirt_e resulttype;
	aststmt   inner, comp;

	if (!(*t) || !ccc_iscombinable(*t))
		return false;

	for (inner = (*t)->u.omp->body; inner->type == COMPOUND; inner = inner->body)
		;   /* Should contain only 1 OpenMP statement */

	if ((resulttype = ccc_combtype((*t), inner)) == DCNONE)/* Should not happen */
		return false;
		
	/* Redo to free intermediate compound nodes */
	for (inner = (*t)->u.omp->body; inner->type == COMPOUND; )
	{
		inner = (comp = inner)->body;
		free(comp);
	}

	combine_ompstmts(t, inner, resulttype);
	return true;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                           *
 *        SPLIT (BREAK) OPENMP STATEMENTS (CLAUSES)          *
 *                                                           *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
 
 
 #define xformingForGPU (xformingFor == CODETARGID(opencl) || \
           xformingFor == CODETARGID(cuda) || xformingFor == CODETARGID(vulkan))
 
static set(vars) break_parallel_defnone_set;
static void 
_break_combined_parallel_clauses(ompclause all, ompclause *parc, 
                                ompclause *wshc, int def)
{
	if (all == NULL) { *parc = *wshc = NULL; return; }
	if (all->type == OCLIST)
	{
		_break_combined_parallel_clauses(all->u.list.next, parc, wshc, def);
		all = all->u.list.elem;
		assert(all != NULL);
	}
	all = ast_ompclause_copy(all);
	
	/* Check for modifiers */
	if (get_dirname_modifier(all) != DCNONE)
	{
		if (get_dirname_modifier(all) != DCPARALLEL)
			*wshc = (*wshc) ? OmpClauseList(*wshc, all) : all;
		else
			*parc = (*parc) ? OmpClauseList(*parc, all) : all;
		return;
	}
	
	switch (all->type)
	{
		/* Hack for target teams distribute */
		case OCTHREADLIMIT:
		case OCNUMTEAMS:
		case OCMAP:
		/* End hack */
		case OCIF:
		case OCSHARED:
		case OCCOPYIN:     /* OpenMP 5.2 gives the reduction to the WS */
		case OCREDUCTION:  /* We however prefer it @ the parallel */
		case OCNUMTHREADS: 
		case OCPRIVATE:
		case OCDEFAULT:
		case OCAUTO:
			*parc = (*parc) ? OmpClauseList(*parc, all) : all;
			break;
		case OCFIRSTPRIVATE:
			if (xformingForGPU)  /* special case: parallel takes it */
			{
				*parc = (*parc) ? OmpClauseList(*parc, all) : all;
				break;
			}
		case OCDISTSCHEDULE: /* should copy to both constructs */
		case OCLASTPRIVATE:  /* should copy to both constructs */
		case OCCOLLAPSE:
			if (def)           /* Remember them */
				ast_varlist2set(all->u.varlist, break_parallel_defnone_set);
		default:
			*wshc = (*wshc) ? OmpClauseList(*wshc, all) : all;
			break;
	}
}


//static set(vars) break_defnone_set;
static void 
_do_break_clauses_to_dirs(ompclause all, ompdirt_e firstdir,ompdirt_e seconddir,
                      ompclause *fcl, ompclause *scl, int def)
{
	int supported1st, supported2nd, nused = 0;
	
	if (all == NULL) { *fcl = *scl = NULL; return; }
	if (all->type == OCLIST)
	{
		_do_break_clauses_to_dirs(all->u.list.next,firstdir,seconddir,fcl,scl,def);
		all = all->u.list.elem;
		assert(all != NULL);
	}
	all = ast_ompclause_copy(all);

	/* First check if a directive name modifier is present and settle this */
	if ((supported1st = get_dirname_modifier(all)) != DCNONE)
	{
		if (is_or_combines(firstdir, supported1st)) 
			*fcl = (*fcl) ? OmpClauseList(*fcl, all) : all;
		else
			*scl = (*scl) ? OmpClauseList(*scl, all) : all;
		return;
	}
	
	/* Use clause rules to determine which clause goes where */
	supported1st = clr_allowed(firstdir, all->type);
	supported2nd = clr_allowed(seconddir, all->type);
	
	/* Now use clause properties to determin how to split */
	switch (ompclauseinfo[all->type].incomp)
	{
		case APPLY_ONCE:
			/* collapse */
		case APPLY_INNER:
			/* private */
			GIVE_TO_INNER:
			if (supported2nd)
				*scl = (*scl) ? OmpClauseList(*scl, all) : all;
			else
				*fcl = (*fcl) ? OmpClauseList(*fcl, all) : all;
			break;
		case APPLY_OUTER:
			/* nowait */
			GIVE_TO_OUTER:
			if (supported1st)
				*fcl = (*fcl) ? OmpClauseList(*fcl, all) : all;
			else
				*scl = (*scl) ? OmpClauseList(*scl, all) : all;
			break;
		case APPLY_ALL:
		default:
			/* all other clauses */
			switch (all->type)
			{
				case OCFIRSTPRIVATE:
					if (is_or_combines(firstdir, DCDISTRIBUTE) ||
					    (is_or_combines(firstdir, DCTEAMS) && 
					    !is_or_combines(seconddir, DCDISTRIBUTE)) ||
					    is_or_combines(firstdir, DCFOR) || 
					    is_or_combines(firstdir, DCSECTIONS) ||
					    /* is_or_combines(firstdir, DCTASKLOOP) || -- ignored for now*/
					    (is_or_combines(firstdir, DCPARALLEL) && 
					     !is_or_combines(seconddir, DCFOR) && 
					     !is_or_combines(seconddir, DCSECTIONS)) ||
					    is_or_combines(firstdir, DCTARGET)) /* FIXME: &&	
					                        a var does NOT appear on lastprivate or map */
					{
						*fcl = (*fcl) ? OmpClauseList(*fcl, all) : all;
						nused++;
					}
					if (is_or_combines(seconddir, DCDISTRIBUTE) ||
					    (is_or_combines(seconddir, DCTEAMS) && 
					    !is_or_combines(firstdir, DCDISTRIBUTE)) ||
					    is_or_combines(seconddir, DCFOR) || 
					    is_or_combines(seconddir, DCSECTIONS) ||
					    /* is_or_combines(seconddir, DCTASKLOOP) || -- ignored for now*/
					    (is_or_combines(seconddir, DCPARALLEL) && 
					     !is_or_combines(firstdir, DCFOR) && 
					     !is_or_combines(firstdir, DCSECTIONS)) ||
					    is_or_combines(seconddir, DCTARGET)) /* FIXME: &&	
					                        a var does NOT appear on lastprivate or map */
					{
						if (nused > 0) /* if we have already used it */
							all = ast_ompclause_copy(all); /* get another copy */
						*scl = (*scl) ? OmpClauseList(*scl, all) : all;
					}
					break;
				case OCREDUCTION:
					/* If we have teams / parallel and OpenCL then give reduction 
					 * only to teams 
					 */
					if (is_or_combines(firstdir, DCTEAMS) && 
					    is_or_combines(seconddir, DCPARALLEL) && xformingForGPU)
						goto GIVE_TO_OUTER;
						
					/* If we have teams/loop or parallel/for then apply to inner only
					 * Since we do not have loop yet, we only have one case to check...
					 */
					if (is_or_combines(firstdir, DCPARALLEL) && 
					    (is_or_combines(seconddir, DCFOR) || 
					     is_or_combines(seconddir, DCSECTIONS)))
					  goto GIVE_TO_INNER;
					/* else fall through (i.e. apply to both) */
				default:            /* all others */
					if (supported2nd)
					{
						*scl = (*scl) ? OmpClauseList(*scl, all) : all;
						if (supported1st)  /* get another copy */
							all = ast_ompclause_copy(all);
					}
					if (supported1st)   
						*fcl = (*fcl) ? OmpClauseList(*fcl, all) : all;
			}
			break;
	}
	
#if 0
		default:        /* FIXME: 
		                   The rest of the cases; I left the code as it was,
		                   Ilia it must be reviewed.
		                 */
			if ((all->type == OCNUMTHREADS) && (firstdir == DCTARGETTEAMS) && 
			    (seconddir == DCDISTPARFOR))
				*fcl = (*fcl) ? OmpClauseList(*fcl, all) : all;
			else 
				if ((all->type == OCNUMTHREADS) && (seconddir == DCTARGETTEAMS) && 
				    (firstdir == DCDISTPARFOR))
					*scl = (*scl) ? OmpClauseList(*scl, all) : all;
				else 
					if ((all->type == OCFIRSTPRIVATE) || (all->type == OCLASTPRIVATE) || 
					    (all->type == OCREDUCTION && firstdir == DCTEAMS && 
					     seconddir == DCDISTPARFOR))
					{
						if ((supported1st) && (supported2nd))
						{
							*fcl = (*fcl) ? OmpClauseList(*fcl, all) : all;
							*scl = (*scl) ? OmpClauseList(*scl, ast_ompclause_copy(all)) : 
							                ast_ompclause_copy(all);
						}
					}
					else 
						if (supported1st) 
						{
							if (all->type == OCCOLLAPSE)
								if (def)            /* Remember them */
									ast_varlist2set(all->u.varlist, break_defnone_set);
							*fcl = (*fcl) ? OmpClauseList(*fcl, all) : all;
						}
						else 
							if (supported2nd)
							{
								if (all->type == OCCOLLAPSE)
									if (def)            /* Remember them */
										ast_varlist2set(all->u.varlist, break_defnone_set);
								*scl = (*scl) ? OmpClauseList(*scl, all) : all;
							}
							else
								*fcl = (*fcl) ? OmpClauseList(*fcl, all) : all;
			break;
	}
#endif
}


/**
 * Divides a set of clauses among two directives.
 *
 * It takes a set of clauses and assigns them to two given directives; it is 
 * basically assumed that the two directives come from a combined construct,
 * and thus we follow the rules of OpenMP 5.2/6.0 to decide which clause goes 
 * to which of the two constituent constructs. If a clause is not supported 
 * by any of the two directives, it is given to the first of the two.
 *
 * @param all the list of clauses
 * @param firstdir the first directive (should be the outermost)
 * @param seconddir the second directive (should be the innermost)
 * @param fcl (ret) the list of clauses assigned to the first directive
 * @param scl (ret) the list of clauses assigned to the second directive
 */
static void break_clauses_to_dirs(ompclause all, ompdirt_e firstdir, 
                            ompdirt_e seconddir, ompclause *fcl, ompclause *scl)
{
	ompclause ocl = xc_clauselist_get_clause(all, OCDEFAULT, DCIGNORE, 0);
	int       hasdef = false;

	/* We must prepare for the case there is a #parallel with default(none)  */
	if (firstdir == DCPARALLEL) 
	{
		/* FIXME: should also check if the clausa has a modifier */
		hasdef = (ocl != NULL && ocl->subtype == OC_defnone);
		if (hasdef)
			set_init(vars, &break_parallel_defnone_set);
		
		/* TODO: merge with _do_break_clauses_to_dirs */
		_break_combined_parallel_clauses(all, fcl, scl, hasdef);
		
		/* Create a SHARED clause to give to the #parallel construct */
		if (hasdef && !set_isempty(break_parallel_defnone_set))
		{
			ocl = VarlistClause(OCSHARED, ast_set2varlist(break_parallel_defnone_set));
			*fcl = (*fcl) ? OmpClauseList(*fcl, ocl) : ocl;
		}
	}
	else 
		_do_break_clauses_to_dirs(all, firstdir, seconddir, fcl, scl, hasdef);
}


/**
 * Produce the split version of a combined directive. The result 
 * replaces the directive with one construct that encloses another.
 * No checks are made for the validity of the split.
 *
 * @param c     The combined directive
 * @param type1 The resulting type of the outer directive
 * @param type2 The resulting type of the inner directive
 */
void ccc_split_ompstmts(aststmt *t, ompdirt_e type1, ompdirt_e type2)
{
	ompclause ocls = NULL;
	ompclause icls = NULL;

	IFDEBUG((fprintf(stderr, "\nSplitting (%s):\n\t",codetarg_name(xformingFor))))
	IFDEBUG((ast_ompdir_show_stderr(OmpStmtDir(*t))))

	if (OmpStmtDir(*t)->clauses)
		break_clauses_to_dirs(OmpStmtDir(*t)->clauses,type1,type2, &ocls, &icls);

	ast_ompclause_free(OmpStmtDir(*t)->clauses); /* Not needed */
	(*t)->u.omp->type = type1;                   /* Change the node */
	OmpStmtDir(*t)->type = type1;
	OmpStmtDir(*t)->clauses = ocls;
	ast_ompclause_parent(OmpStmtDir(*t), ocls);  /* Fix the clauses' parent */
	(*t)->u.omp->body = OmpStmt(                 /* New body */
	                      OmpConstruct(
	                        type2,
	                        OmpDirective(type2, icls),
	                        (*t)->u.omp->body
	                      )
	                    );
	(*t)->u.omp->body->parent = *t;              /* Re-set the parent */                   
	(*t)->u.omp->body->file =                    /* Fix infos */
		(*t)->u.omp->body->u.omp->file =
		(*t)->u.omp->body->u.omp->directive->file =
			OmpStmtDir(*t)->file;
	(*t)->u.omp->body->l =
		(*t)->u.omp->body->u.omp->l =
		(*t)->u.omp->body->u.omp->directive->l =
			OmpStmtDir(*t)->l;
	(*t)->u.omp->body->c =
		(*t)->u.omp->body->u.omp->c =
		(*t)->u.omp->body->u.omp->directive->c =
			OmpStmtDir(*t)->c;

	IFDEBUG((fprintf(stderr, "gives:\n\t")))
	IFDEBUG((ast_ompdir_show_stderr(OmpStmtDir(*t))))
	IFDEBUG((fprintf(stderr, "\t  ")))
	IFDEBUG((ast_ompdir_show_stderr(OmpStmtDir((*t)->u.omp->body))))
}


/**
 * Check if an OMP statement can be split to two statements, one nested 
 * within the other, according to the split rules.
 *
 * @param  t The OMP statement
 * @return true if given OMP statement can be split.
 */
static bool _is_splittable(aststmt t, ompdirt_e *outer, ompdirt_e *inner)
{
	ompdirt_e mtype, combined;
	int i, j;

	if (t->type != OMPSTMT)
		return false;

	mtype = OmpStmtDir(t)->type;
	for (i = 0; all_splitrules[i] != NULL; i++)
		for (j = 0; (combined = (*all_splitrules[i])[j].combined) != DCNONE; j++)
			if (combined == mtype)
			{
				if (outer) *outer = (*all_splitrules[i])[j].outer;
				if (inner) *inner = (*all_splitrules[i])[j].inner;
				return true;
			};
	return false;
}


/**
 * Try to split a directive into two directives, according to the
 * split rules. 
 *
 * @param  t      the given combined directive
 * @return true   if a combined statement was indeed possible
 */
bool ccc_try_splitting(aststmt *t)
{
	ompdirt_e resulttype1, resulttype2;

	if (!_is_splittable((*t), &resulttype1, &resulttype2))
		return false;

	ccc_split_ompstmts(t, resulttype1, resulttype2);
	return true;
}
