/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* x_clauses.c -- everything related to openmp clauses & their generated code */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "ast_free.h"
#include "ast_vars.h"
#include "ast_xform.h"
#include "ast_copy.h"
#include "ast_assorted.h"
#include "ast_csource.h"
#include "ast_print.h"
#include "ast_show.h"
#include "ast.h"
#include "x_clauses.h"
#include "x_clauserules.h"
#include "ast_arith.h"
#include "x_arrays.h"
#include "x_reduction.h"
#include "x_target.h"
#include "x_parallel.h"
#include "ompi.h"
#include "codetargs.h"
#include "builder.h"


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     EXTENDED LIST ITEMS STUFF                                 *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/**
 * Returns an expression with the base element of an xlitem
 * @param t  The xlitem
 * @return   The expression
 */
astexpr xc_xlitem_baseelement(ompxli t)
{
	astexpr e = Identifier(t->id);
	
	assert(t != NULL);
	if (t->xlitype == OXLI_ARRSEC)
		return ( arr_section_baseelement(t, NULL) );
	if (symtab_get(stab, t->id, IDNAME)->isarray)
		return ( Parenthesis(Deref(e)) );
	else     /* even in the case of a pointer.. */
		return ( Parenthesis(e) );
}


/**
 * The expression that gives the total number of elements in an xlitem
 * @param  arrsec  the xlitem
 * @return an expression with the total size of the array section (#elememts)
 */
astexpr xc_xlitem_length(ompxli arrsec)
{
	assert(arrsec != NULL);
	if (arrsec->xlitype == OXLI_ARRSEC)
		return ( arr_section_length(arrsec, ALLDIMS) );
	else     /* pointers?? */
	{
		stentry e = symtab_get(stab, arrsec->id, IDNAME);
		if (e->isarray)
			return ( arr_num_elems(e->decl, 0) );
		else
			return ( OneExpr() );
	}
}


/**
 * Gives expressions for the basic quantities of an xlitem
 * @param xl the xlitem
 * @param itemaddr if non-NULL, it gets the starting address of the entire array
 * @param nbytes   if non-NULL, it gets the total size (in bytes)
 * @param addrlb   if non-NULL, it gets the address of the section's 1st element
 */
void xc_xlitem_copy_info(ompxli xl,
                         astexpr *itemaddr, astexpr *nbytes, astexpr *addrlb)
{
	if (itemaddr) *itemaddr = UOAddress(Identifier(xl->id));
	if (xl->xlitype == OXLI_IDENT)
	{
		if (nbytes) *nbytes = Sizeof(Identifier(xl->id));
		if (addrlb) *addrlb = ast_expr_copy(*itemaddr);
	}
	else    /* Array section */
	{
		if (nbytes) *nbytes = arr_section_size(xl);
		if (addrlb) *addrlb = arr_section_baseaddress(xl);
	}
}


/* Check a list of extended-list-item OpenMP clauses and try to find if a 
 * variable is included in them; returns the xlitem.
 */
ompxli xc_xlitem_find_in_clause(ompclt_e type, ompclause t, symbol var)
{
	ompxli xl;

	if (!t) return (NULL);
	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			if ( (xl = xc_xlitem_find_in_clause(type, t->u.list.next, var)) != NULL )
				return (xl);
		t = t->u.list.elem;
		assert(t != NULL);
	}
	if (t->type == type)
		for (xl = t->u.xlist; xl; xl = xl->next)
			if (xl->id == var)
				return (xl);
	return (NULL);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     FIRSTPRIVATE RELATED STUFF                                *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* This produces a statement declaring a firstprivate variable (plus
 * another one which is need). The declaration contains an
 * initializer, only when the variable is scalar; otherwise it
 * contains a plain declaration and initialization should be done
 * through xc_ompdir_array_initializers()
 */
aststmt xc_firstprivate_declaration(symbol var)
{
	char    flvar[256];
	stentry e = symtab_get(stab, var, IDNAME);

	snprintf(flvar, 255, "_fip_%s", var->name);    /* a temp var _fip_<name> */
	return ( flr_privatize(var, Symbol(flvar), e->isarray,
	              e->isarray ? 
	                NULL :                         /* var = fip_var; */
	                IdentName(flvar)) );
}


/* Take a varlist and generate correct initialization statements
 * for lastprivate vars that are non-scalar.
 * It works for all OpenMP constructs except *parallel* which produces
 * its own declarations.
 */
static
aststmt array_initializations_from_varlist(astdecl d,
                                           ompclt_e type, ompdir ompd)
{
	aststmt list = NULL, st = NULL;

	if (d->type == DLIST && d->subtype == DECL_idlist)
	{
		list = array_initializations_from_varlist(d->u.next, type, ompd);
		d = d->decl;
	}
	assert(d->type == DIDENT);
	if (type == OCFIRSTPRIVATE)
	{
		char flvar[256];
		if ((ompd->type == DCFOR || ompd->type == DCSECTIONS || /* special case */
		     ompd->type == DCFOR_P) &&
		    symtab_get(stab, d->u.id, IDNAME)->isarray &&  /* first&last private */
		    xc_isvar_in_dataclause(d->u.id, ompd, OCLASTPRIVATE))       /* array */
			snprintf(flvar, 255, "_lap_%s", d->u.id->name);
		else
			snprintf(flvar, 255, "_fip_%s", d->u.id->name);
		if (symtab_get(stab, d->u.id, IDNAME)->isarray)
			st = xc_memcopy(Identifier(d->u.id),   /* *flvar */
			                Deref(IdentName(flvar)),
			                Sizeof(Identifier(d->u.id)));
	}
	if (st)
		list = ((list != NULL) ? BlockList(list, st) : st);
	return (list);
}


static
aststmt array_initializations_from_clauses(ompclause t, ompdir d)
{
	aststmt list = NULL, st = NULL;

	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			list = array_initializations_from_clauses(t->u.list.next, d);
		t = t->u.list.elem;
		assert(t != NULL);
	}

	if (t->type == OCFIRSTPRIVATE)
		if ((st = array_initializations_from_varlist(t->u.varlist, t->type, d)))
			list = ((list != NULL) ? BlockList(list, st) : st);
	return (list);
}


/* Memory copying statements for fip vars (firstprivate) */
aststmt xc_ompdir_fiparray_initializers(ompdir t)
{
	return (t->clauses ? array_initializations_from_clauses(t->clauses, t)
	        : NULL);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     LASTPRIVATE RELATED STUFF                                 *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* This produces a statement declaring a lastprivate variable (plus
 * another one which is needed), exactly as in firstprivate; the only
 * difference is that there is no initializer.
 */
aststmt xc_lastprivate_declaration(symbol var)
{
	char flvar[256];
	
	snprintf(flvar, 255, "_lap_%s", var->name);    /* a temp var _lap_<name> */
	return ( flr_privatize(var, Symbol(flvar), 1, NULL) );
}


/* This produces a statement declaring a lastprivate variable (plus
 * another one which is need), exactly as in firstprivate; the only
 * difference is that there is no initializer.
 */
aststmt xc_firstlastprivate_declaration(symbol var)
{
	char    flvar[256];

	/* Declare and intialize a temp var _flp_<name> */
	/* We call it _lap_<name> because xc_lastprivate_assignments()
	 * has no way of telling whether the var is also firstprivate
	 */ 
	snprintf(flvar, 255, "_lap_%s", var->name);
	return ( flr_privatize(var, Symbol(flvar), 1,
	              symtab_get(stab, var, IDNAME)->isarray ? 
	                NULL :        /* var = *flp_var; */
	                Deref(IdentName(flvar))) );
}


/* Take a lastprivate varlist and generate correct assignement statements */
static
aststmt lastprivate_assignments_from_varlist(astdecl d)
{
	char    flvar[256];
	aststmt list = NULL, st = NULL;

	if (d->type == DLIST && d->subtype == DECL_idlist)
	{
		list = lastprivate_assignments_from_varlist(d->u.next);
		d = d->decl;
	}
	assert(d->type == DIDENT);
	snprintf(flvar, 255, "_lap_%s", d->u.id->name);
	if (symtab_get(stab, d->u.id, IDNAME)->isarray)
		//st = xc_array_assigner(d->u.id,   /* *flvar */
		//                       Deref(IdentName(flvar)));
		st = xc_memcopy(IdentName(flvar), Identifier(d->u.id),
		                Sizeof(Identifier(d->u.id)));
	else  /* Scalar */
		st = AssignStmt( Deref(IdentName(flvar)), Identifier(d->u.id) );
	list = ((list != NULL) ? BlockList(list, st) : st);
	return (list);
}


static
aststmt lastprivate_assignments_from_clauses(ompclause t)
{
	aststmt list = NULL, st = NULL;

	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			list = lastprivate_assignments_from_clauses(t->u.list.next);
		t = t->u.list.elem;
		assert(t != NULL);
	}
	if (t->type == OCLASTPRIVATE)
		if ((st = lastprivate_assignments_from_varlist(t->u.varlist)))
			list = ((list != NULL) ? BlockList(list, st) : st);
	return (list);
}


/* Assignement / memory copying statements for lap vars (lastprivate) */
aststmt xc_ompdir_lastprivate_assignments(ompdir t)
{
	return (t->clauses ? lastprivate_assignments_from_clauses(t->clauses)
	                   : NULL);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     DECLARATIONS FROM THE DATA CLAUSE VARS                    *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* Take a varlist and generate a correct declaration for each of the
 * vars. It takes care of private/firstprivate/lastprivate/etc types
 * of varlists.
 *
 * It works for all OpenMP constructs except *parallel* which produces
 * its own declarations.
 */
static
aststmt declarations_from_varlist(astdecl d, ompclt_e type, int operator)
{
	aststmt list = NULL, st = NULL;

	if (d->type == DLIST && d->subtype == DECL_idlist)
	{
		list = declarations_from_varlist(d->u.next, type, operator);
		d = d->decl;
	}
	assert(d->type == DIDENT);
	switch (type)
	{
		case OCFIRSTPRIVATE:
			st = xc_firstprivate_declaration(d->u.id);
			break;
		case OCLASTPRIVATE:
			st = xc_lastprivate_declaration(d->u.id);
			break;
		case OCPRIVATE:
			st = xform_clone_declaration_nosc(d->u.id, NULL, false, NULL);
			break;
		case OCCOPYIN:        /* This is handled entirely in x_parallel.c */
		case OCSHARED:        /* These do not produce any declarations */
		case OCCOPYPRIVATE:   /* This is handled entirely in x_single.c */
			break;
		default:
			break;
	}
	if (st) list = ((list != NULL) ? BlockList(list, st) : st);
	return (list);
}


static
aststmt declarations_from_xlist(ompxli xl, ompclt_e type, int operator)
{
	aststmt list = NULL, st;

	for (; xl; xl = xl->next)
		if (type == OCREDUCTION)
		{
			st = red_generate_declaration(xl->id, operator, xl);
			list = ((list != NULL) ? BlockList(list, st) : st);
		};
	return (list);
}


/* It is assumed that a validity check for clauses and variables
 * appearing in them has already taken place.
 * ompdir is only needed for error messages & validity checks
 */
static
aststmt declarations_from_clauses(ompdirt_e dirtype, ompclause t)
{
	aststmt list = NULL, st = NULL;

	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			list = declarations_from_clauses(dirtype, t->u.list.next);
		t = t->u.list.elem;
		assert(t != NULL);
	}
	switch (t->type)
	{
		case OCSHARED:          /* Only data clauses matter */
		case OCCOPYIN:
		case OCPRIVATE:
		case OCFIRSTPRIVATE:
		case OCCOPYPRIVATE:
		case OCLASTPRIVATE:
			if ((st = declarations_from_varlist(t->u.varlist, t->type, t->subtype)))
				list = ((list != NULL) ? BlockList(list, st) : st);
			break;
		case OCREDUCTION:
			if ((st = declarations_from_xlist(t->u.xlist, t->type, t->subtype)))
				list = ((list != NULL) ? BlockList(list, st) : st);
			break;
		default:
			break;
	}
	return (list);
}


/* This is actually called only from x_single.c (!!) */
aststmt xc_ompdir_declarations(ompdir t)
{
	return (t->clauses ? declarations_from_clauses(t->type, t->clauses) : NULL);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     DATA CLAUSE VARIABLES                                     *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* The following 4 functions are used to gather all variables appearing
 * in a directive's data clauses. They also detect duplicates.
 * They use a local symbol table to do it.
 *
 * For each var in the table, the ival field stores the clause type
 * and the "value" field encodes the operator, in case of reduction vars.
 */
static symtab    dc_vars = NULL;    /* Store the list here */
static ompclause dc_vars_clause;    /* Only needed for error messages */


static
void checkNstore_varlist_vars(astdecl d, int clausetype, int opid)
{
	stentry e;

	if (d->type == DLIST && d->subtype == DECL_idlist)
	{
		checkNstore_varlist_vars(d->u.next, clausetype, opid);
		d = d->decl;
	}
	if (d->type != DIDENT)
		exit_error(1, "[checkNstore_varlist_vars]: !!BUG!! not a DIDENT ?!\n");
	if ((e = symtab_get(dc_vars, d->u.id, IDNAME)) != NULL)
	{
		if ((clausetype == OCFIRSTPRIVATE && e->ival == OCLASTPRIVATE) ||
		    (clausetype == OCLASTPRIVATE && e->ival == OCFIRSTPRIVATE))
		{
			e->ival = OCFIRSTLASTPRIVATE;       /* Change it -- special case! */
			return;                             /* Don't put it in again */
		}
		else
			exit_error(1, "(%s, line %d) openmp error:\n\t"
			           "variable `%s' appears more than once in "
			           "the directive's clause(s)\n",
			           dc_vars_clause->file->name, dc_vars_clause->l,
			           d->u.id->name);
	}
	switch (clausetype)
	{
		case OCCOPYIN:         /* Copyin is only for threadprivate vars */
			if (!symtab_get(stab, d->u.id, IDNAME)->isthrpriv)
				exit_error(1, "(%s, line %d) openmp error:\n\t"
				           "copyin clause variable '%s' is not threadprivate.\n",
				           dc_vars_clause->file->name, dc_vars_clause->l, d->u.id->name);
			break;
		case OCCOPYPRIVATE:   /* Nothing here */
			break;
		default:              /* All others */
			if (symtab_get(stab, d->u.id, IDNAME)->isthrpriv)
				exit_error(1, "(%s, line %d) openmp error:\n\t"
				           "threadprivate variable '%s' cannot appear in a %s clause.\n",
				           dc_vars_clause->file->name, dc_vars_clause->l,
				           d->u.id->name, ompclauseinfo[clausetype].name);
			break;
	}
	/* Put the opid in the "value" field */
	e = symtab_put(dc_vars, d->u.id, IDNAME);
	e->vval = opid;
	e->ival = clausetype;
	e->pval = NULL;        /* No xlitem */
}


static
void checkNstore_xlist_vars(ompxli xl, int clausetype, int opid)
{
	stentry e;

	for (; xl; xl = xl->next)
	{
		if ((e = symtab_get(dc_vars, xl->id, IDNAME)) != NULL)
		{
			if (clausetype == OCMAP)
			{
				if (e->ival == OCREDUCTION)
				{
					if (opid != OC_tofrom)
						exit_error(1, "(%s, line %d) openmp error:\n\t"
								"variable `%s' cannot be declared as `map(%s)', "
								"as it appears in a `reduction' clause.\n",
								dc_vars_clause->file->name, dc_vars_clause->l,
								xl->id->name, clausesubs[opid]);
					return;
				}
			}
			else if (clausetype == OCREDUCTION)
			{
				if (e->ival == OCMAP)
				{
					if (e->vval != OC_tofrom)
						exit_error(1, "(%s, line %d) openmp error:\n\t"
								"variable `%s' cannot be declared as `map(%s)', "
								"as it appears in a `reduction' clause.\n",
								dc_vars_clause->file->name, dc_vars_clause->l,
								xl->id->name, clausesubs[e->vval]);
					return;
				}
			}

			exit_error(1, "(%s, line %d) openmp error:\n\t"
								"variable `%s' appears more than once in "
								"the directive's clause(s)\n",
								dc_vars_clause->file->name, dc_vars_clause->l,
								xl->id->name);
		}
			
		if (symtab_get(stab, xl->id, IDNAME)->isthrpriv)
			exit_error(1, "(%s, line %d) openmp error:\n\t"
								"threadprivate variable '%s' cannot appear in a %s clause.\n",
								dc_vars_clause->file->name, dc_vars_clause->l,
								xl->id->name, ompclauseinfo[clausetype].name);
		/* Put the opid in the "value" field */
		e = symtab_put(dc_vars, xl->id, IDNAME);
		e->vval = opid;
		e->ival = clausetype;
		e->pval = xl;          /* Remeber the xlitem */
	}
}


static
void checkNstore_dcclause_vars(ompclause t)
{
	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			checkNstore_dcclause_vars(t->u.list.next);
		assert((t = t->u.list.elem) != NULL);
	}
	dc_vars_clause = t;
	switch (t->type)
	{
		case OCPRIVATE:
		case OCFIRSTPRIVATE:
		case OCLASTPRIVATE:
		case OCCOPYPRIVATE:
		case OCCOPYIN:
		case OCSHARED:
		case OCAUTO:
			if (t->u.varlist)   /* t->subtype is the opid in case of reduction */
				checkNstore_varlist_vars(t->u.varlist, t->type, t->subtype);
			break;
		case OCREDUCTION:
		case OCMAP:
		case OCTASKREDUCTION:
			if (t->u.xlist)
				checkNstore_xlist_vars(t->u.xlist, t->type, t->subtype);
			break;
		default:
			break;
	}
}


/* Checks for duplicates AND keeps the list of vars; returns the table */
/* TODO: replace symtab with sets */
symtab xc_validate_store_dataclause_vars(ompdir d)
{
	if (dc_vars == NULL)
		dc_vars = Symtab();
	else
		symtab_drain(dc_vars);
	if (d->clauses)
		checkNstore_dcclause_vars(d->clauses);
	return (dc_vars);
}


/* Only checks for duplicates */
void xc_validate_only_dataclause_vars(ompdir d)
{
	xc_validate_store_dataclause_vars(d);
	symtab_drain(dc_vars);                   /* Not needed any more */
}


/*
 * The following 3 functions are used to search whether a particular
 * variable appears in any (or a specific) data clause.
 */


static
int findvar_varlist(symbol var, astdecl d)
{
	if (d->type == DLIST)
	{
		if (findvar_varlist(var, d->u.next))
			return (1);
		d = d->decl;
	}
	if (d->type != DIDENT)
		exit_error(1, "[findvar_varlist]: !!BUG!! not a DIDENT ?!\n");
	return (d->u.id == var);
}


static
int findvar_xlist(symbol var, ompxli xl)
{
	for (; xl; xl = xl->next)
		if (xl->id == var)
			return (1);
	return (0);
}


static
int findvar_dataclause(symbol var, ompclause t, ompclt_e c)
{
	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			if (findvar_dataclause(var, t->u.list.next, c))
				return (1);
		assert((t = t->u.list.elem) != NULL);
	}
	if (t->type == c && t->u.varlist)
		return ( (c != OCREDUCTION) ? 
		             findvar_varlist(var, t->u.varlist) :
		             findvar_xlist(var, t->u.xlist) );
	return (0);
}


static
ompclt_e findvar_any_dataclause(symbol var, ompclause t)
{
	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			if (findvar_any_dataclause(var, t->u.list.next))
				return (OCPRIVATE);  /* anything would do */
		assert((t = t->u.list.elem) != NULL);
	}
	switch (t->type)
	{
		case OCPRIVATE:
		case OCFIRSTPRIVATE:
		case OCLASTPRIVATE:
		case OCCOPYPRIVATE:
		case OCCOPYIN:
		case OCSHARED:
			if (t->u.varlist && findvar_varlist(var, t->u.varlist))
				return (t->type);
			break;
		case OCREDUCTION:
			if (t->u.varlist && findvar_xlist(var, t->u.xlist))
				return (t->type);
			break;
		default:
			break;
	}
	return (OCNOCLAUSE);
}


/* This one searches to see whether var appears in a type c
 * data clauses of d; returns 1 if so, 0 if not.
 */
int xc_isvar_in_dataclause(symbol var, ompdir d, ompclt_e c)
{
	return (findvar_dataclause(var, d->clauses, c) != OCNOCLAUSE);
}


/* Find whether var appears in any of the data clauses of d.
 * If found, it returns the clause type, otherwise 0.
 */
ompclt_e xc_dataclause_of_var(symbol var, ompdir d)
{
	return (findvar_any_dataclause(var, d->clauses));
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     DECLARATIONS FROM A SET OF VARS                           *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* We have 2 ways of declaring the variables in the data clauses.
 *   (1) by scanning all the data clauses and declaring from each clause
 *   (2) by collecting all data clause variables in a set, along with
 *       the dataclause type of each one and declaring them from there.
 * Both have some advantages. The second one does not need recursion,
 * plus it makes it trivial to discover vars that are both firstprivate
 * and lastprivate.
 */


/* Produces a list of declarations for all variables in the dc_vars table
 */
aststmt xc_stored_vars_declarations(bool *has_last, bool *has_both,
                                    bool *has_red)
{
	stentry e;
	aststmt st = NULL, list = NULL;

	*has_last = *has_both = *has_red = false;
	for (e = dc_vars->top; e; e = e->stacknext)
	{
		switch (e->ival)
		{
			case OCFIRSTPRIVATE:
				st = xc_firstprivate_declaration(e->key);
				break;
			case OCREDUCTION:
				*has_red = true;
				st = red_generate_declaration(e->key, e->vval, e->pval);
				break;
			case OCLASTPRIVATE:
				*has_last = true;
				st = xc_lastprivate_declaration(e->key);
				break;
			case OCFIRSTLASTPRIVATE:
				*has_both = true;
				*has_last = true;
				st = xc_firstlastprivate_declaration(e->key);
				break;
			case OCPRIVATE:
				st = xform_clone_declaration_nosc(e->key, NULL, false, NULL);
				break;
			case OCCOPYIN:        /* Taken care in x_parallel.c */
			case OCSHARED:        /* These do not produce any declarations */
			case OCCOPYPRIVATE:   /* This is handled entirely in x_single.c */
				break;
		}

		if (st)
			list = ((list == NULL) ? st : BlockList(list, st));
	}
	return (list);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     UTILITY FUNCTIONS                                         *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/**
 * Creates a local variable with the same name (with an optional initializer), 
 * plus a copy of/pointer to the original one.  I.e. the following:
 *   <spec> varbak = var, var [ = initer ]; or
 *   <spec> *varbak = &var, var [ = initer ]; 
 *
 * @param var       The original variable whose declaration we want to reproduce
 * @param varbak    The copy of/pointer to the original variable
 * @param isptr     A flag for making varbak a pointer or not
 * @param initer    An optional initializer for the new local variable
 * @return          A statement with the declaration
 */
aststmt flr_privatize(symbol var, symbol varbak, int isptr, astexpr initer)
{
	astdecl bakdecl, id, localvar;
	stentry e = symtab_get(stab, var, IDNAME);
	
	bakdecl = ast_decl_copy(e->decl);      /* Make the varbak */
	id = IdentifierDecl(varbak);
	*(decl_getidentifier(bakdecl)) = *id;
	free(id);
	if (isptr)
		bakdecl = InitDecl( decl_topointer(bakdecl),UOAddress(Identifier(var)) );
	else
		bakdecl = InitDecl( bakdecl, Identifier(var) );

	localvar = xform_clone_declonly(e);    /* Make the local var declarator */ 
	if (initer)
		localvar = InitDecl(localvar, initer);
	
	/* Originally it was: <spec> *varbak = &var, var = <initializer>; 
	 * However, we split it in two declarations because something goes wrong 
	 * when later analyzing this declaration list
	 */
	//return Declaration(ast_spec_copy_nosc(e->spec), DeclList(bakdecl,localvar));
	return BlockList(Declaration(ast_spec_copy_nosc(e->spec), bakdecl), 
	                 Declaration(ast_spec_copy_nosc(e->spec), localvar));
}


static bool clauses_contain(ompclause t, ompclt_e c, ompdirt_e dnmod)
{
	if (t == NULL) return false;
	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			if (clauses_contain(t->u.list.next, c, dnmod))
				return true;
		assert((t = t->u.list.elem) != NULL);
	}
	return (t->type == c && get_dirname_modifier(t) == dnmod);
}


/**
 * Checks rules to ensure validity and OpenMP compliance of the given clause(s).
 * It won't check the variables inside the clauses.
 *
 * @param dirtype the directive type
 * @param cl      the clause (list)
 */
void xc_validate_clauses(ompdirt_e dirtype, ompclause cl)
{
	ompclause t;
	int n;
	ompdirt_e dnmod;

	/* Count the clauses */
	for (n = 0, t = cl; t; t = (t->type == OCLIST) ? t->u.list.next : NULL)
	{
		assert(((t->type == OCLIST) ? t->u.list.elem : t) != NULL);
		n++;
	}
	
	for ( ; cl; cl = (cl->type == OCLIST) ? cl->u.list.next : NULL)
	{
		t = (cl->type == OCLIST) ? cl->u.list.elem : cl;

		if (t->type <= OCNOCLAUSE || t->type >= OCLASTCLAUSE)
			exit_error(1, "(%s, line %d) openmp error:\n\t"
			         	 "unknown clause type (%d) in `%s' directive\n",
			         	 t->file->name, t->l, t->type, ompdirnames[dirtype]);
		
		/* check if clause is supported */
		if (!clr_allowed(dirtype, t->type))
			exit_error(1, "(%s, line %d) openmp error:\n\t"
			           "%s clauses aren't allowed in `%s' directives\n",
			           t->file->name, t->l, ompclauseinfo[t->type].name, 
			           ompdirnames[dirtype]);

		/* check for the ultimate property */
		/* TODO: check also for directive-name modifier... */
		if ((ompclauseinfo[t->type].props & CMSP_ULTIMATE) && cl->type == OCLIST)
			exit_error(1, "(%s, line %d) openmp error:\n\t"
			         "a %s clause must be the last clause on a directive.\n",
			         t->file->name, t->l, ompclauseinfo[t->type].name);
			
		/* check for exclusiveness */
		/* TODO: check also for directive-name modifier... */
		if ((ompclauseinfo[t->type].props & CMSP_EXCLUSIVE) && n > 1)
			exit_error(1, "(%s, line %d) openmp error:\n\t"
			         "%s clauses must be exclusive (i.e. no other clauses allowed)\n",
			         t->file->name, t->l, ompclauseinfo[t->type].name);
			
		/* check for directive-name modifier */
		dnmod = get_dirname_modifier(t);
		if (dnmod != DCNONE && !is_or_combines(dirtype, dnmod))
			exit_error(1, "(%s, line %d) openmp error:\n\t"
			           "the %s clause directive name modifier (%s) does not\n\t"
			           "refer to the directive the clause belongs to (%s)\n",
			           t->file->name, t->l, ompclauseinfo[t->type].name, 
			           ompdirnames[dnmod], ompdirnames[dirtype]);

		/* check for uniqueness */
		if ((ompclauseinfo[t->type].props & CMSP_UNIQUE) && (cl->type == OCLIST) &&
		    clauses_contain(cl->u.list.next, t->type, get_dirname_modifier(t)))
		{
			exit_error(1, "(%s, line %d) openmp error:\n\t"
			           "clause %s can only appear once in a directive\n",
			           t->file->name,t->l, ompclauseinfo[t->type].name);
		}
		
		/* special case: autoscoping */
		if (t->type == OCDEFAULT)
		{
			/* This one is needed for Aggelo's auto scoping */
			if (t->subtype == OC_auto)
			{
				if (!enableAutoscope)
					exit_error(1, "(%s, line %d) error:\n\t"
										"autoscoping has not been enabled (use --autoscope)\n",
										t->file->name, t->l);
				if (dirtype != DCPARALLEL && dirtype != DCPARFOR && 
						dirtype != DCPARSECTIONS)
					exit_error(1, "(%s, line %d) openmp error:\n\t"
										"default(auto) clause isn't allowed in `%s' directives\n",
										t->file->name, t->l, ompdirnames[dirtype]);
			}
			else
				if (xc_clauselist_get_clause(cl, OCAUTO, dirtype, false))
					exit_error(1, "(%s, line %d) openmp error:\n\t"
										"auto and default clauses collide\n",
										t->file->name, t->l);
		}
		if (t->type == OCAUTO)
		{
			/* This one is needed for Aggelo's auto scoping */
			if (!enableAutoscope)
				exit_error(1, "(%s, line %d) error:\n\t"
									"autoscoping has not been enabled (use --autoscope)\n",
									t->file->name, t->l);
			if (xc_clauselist_get_clause(cl, OCDEFAULT, dirtype, false))
				exit_error(1, "(%s, line %d) openmp error:\n\t"
										"auto and default clauses collide\n",
										t->file->name, t->l);
		}
	}
}


/**
 * Get clause(s) of a given type from a clauselist.
 *
 * @param t    The OpenMP clauses we are searching in
 * @param type The type of clause we want to search for
 * @param dir  If the found clause has a modifier, then dir is the directive 
 *             it should refer to, i.e. the clause directive name modifier must
 *             be equal to dir or (if compound) equal to a directive included
 *             in dir; dir = DCNONE *requires* that no directive name modifier 
 *             exists; use dir = DCIGNORE to ignore the modifier. 
 * @param all  false if loking for any, true if looking for all clauses
 * @return     The clause(s) of the given type, or NULL if none exists;
 *             if all clauses are needed, the returned clause(s) are copies 
 *             (i.e. freeable).
 */
ompclause xc_clauselist_get_clause(ompclause t, ompclt_e type, ompdirt_e dir,
                                   bool all)
{
	ompclause c = NULL;
	ompdirt_e mod;

	if (t == NULL) return (NULL);
	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			if ((c = xc_clauselist_get_clause(t->u.list.next, type, dir, all)) != NULL
			    && !all)
				return (c);        /* Found one, stop here */
		assert((t = t->u.list.elem) != NULL);
	}
	mod = get_dirname_modifier(c);
	if (t->type == type && 
	    (mod == DCNONE || dir == DCIGNORE || is_or_combines(dir, mod)))
	{
		if (all)
			t = ast_ompclause_copy(t);     /* get a copy */
		c = c ? OmpClauseList(c, t) : t;
	}
	return (c);
}


/**
 * Find a clause of a given type in a construct.
 * It only returns the leftmost clause found.
 *
 * @param t    The OpenMP construct we are searching in
 * @param type The type of clause we want to search for
 * @return     The leftmost clause of the given type (or NULL)
 */
ompclause xc_ompcon_get_clause(ompcon t, ompclt_e type)
{
	return xc_clauselist_get_clause(t->directive->clauses, type, t->type, false);
}


/**
 * Search for a clause of a given type in perfectly nested constructs.
 * If the clause is not found it generates an error
 *
 * @param t    The outermost OpenMP construct we are searching in
 * @param type The type of clause we want to search for
 * @return     The clause of the given type (or NULL)
 */
ompclause xc_ompcon_search_unique_clause(ompcon t, ompclt_e type)
{
	ompclause c = NULL;
	ompcon *tptr = &t;

	if (t == NULL) return NULL;

	while ((c = xc_ompcon_get_clause(*tptr, type)) == NULL)
	{
		if ((*tptr == NULL) || ((*tptr)->body == NULL)) return NULL;
		if ((*tptr)->body->type == COMPOUND) 
		{
			if ((*tptr)->body->body && (*tptr)->body->body->type == OMPSTMT)
			{
				tptr = &((*tptr)->body->body->u.omp);
				continue;
			}
		}
		if ((*tptr)->body->type != OMPSTMT) return NULL;
		tptr = &((*tptr)->body->u.omp);
	}

	return c;
}

/**
 * Search for all clauses of a given type in perfectly nested constructs
 * (one per construct).
 * If no clauses are found it generates an error
 *
 * @param t    The outermost OpenMP construct we are searching in
 * @param type The type of clauses we want to search for
 * @return     A list of clauses of the given type (or NULL)
 */
static ompclause xc_ompcon_search_all_clauses(ompcon t, ompclt_e type)
{
	ompclause c = NULL, c2 = NULL;
	ompcon *tptr = &t;

	if (t == NULL) return NULL;

	while ((*tptr) != NULL)
	{
		if (((*tptr)->body == NULL)) return xc_ompcon_get_clause(*tptr, type);
		if ((*tptr)->body->type == COMPOUND) 
		{
			if ((*tptr)->body->body && (*tptr)->body->body->type == OMPSTMT)
				tptr = &((*tptr)->body->body->u.omp);
		}

		if ((c2 = xc_ompcon_get_clause(*tptr, type)) != NULL)
			c = (c) ? OmpClauseList(c, c2) : c2;

		if ((*tptr)->body->type != OMPSTMT) return c;
		tptr = &((*tptr)->body->u.omp);
	}

	return c;
}


/* Calculates 3 dimensions given an expression that contains
 * a product.
 */
astexpr expr_product_to_dimensions(astexpr expr)
{
	astexpr alldims, dimX = NULL, dimY = NULL, dimZ = NULL;
	set(factor_st) factors = set_new(factor_st);
	setelem(factor_st) e;

	/* (1) Find and store all factors f1, f2, ..., fN from `expr' */
	xar_create_factors_set(factors, expr);

	/* (2) dimX = f1, dimY = f2 */
	dimX = factors->first->key;
	dimY = factors->first->next ? factors->first->next->key : numConstant(0);

	/* (3) dimZ = f3 * f4 * ... * fN */
	if (set_size(factors) >= 3)
		for (e = factors->first->next->next; e; e = e->next)
			dimZ = dimZ ? BinaryOperator(BOP_mul, dimZ, e->key) : e->key;
	else
		dimZ = numConstant(0);

	alldims = Comma3(dimX, dimY, dimZ);
		
	return alldims;
}

/* 
 * This function works on a single thread_limit() clause. We first find out
 * which construct the clause belongs to. According to the cases we generate
 * different initialization declarations:
 * 
 * CASE 1: #target teams thread_limit(expr)
 * 
 *   int __ompi_target_thread_limit = expr;
 *   int __ompi_teams_thread_limit = __ompi_target_thread_limit;
 * 
 * 
 * CASE 2: #target thread_limit(expr) { ... }
 * 
 *   int __ompi_target_thread_limit = expr;
 * 
 * SUBCASE 2.1: #target thread_limit(expr1) { #teams }
 * 
 *   int __ompi_teams_thread_limit = __ompi_target_thread_limit;
 * 
 * (as specified by OpenMP V6.0, 12.2, p. 394)
 * 
 * 
 * CASE 3: #target { #teams thread_limit(expr) }
 * 
 *   int __ompi_teams_thread_limit = expr;
 * 
 * 
 * ALL OTHER CASES:
 * We keep the expression as is and do not generate an initdecl.
 */
static astexpr _handle_single_thread_limit_clause(ompclause *c)
{
	ompdirt_e parent_type = (*c)->parent->type;
	astexpr c_copy = ast_expr_copy((*c)->u.expr),
			tc = IdentName(clexpr_var_name(CLV_THREAD_LIMIT));
	aststmt thrl_decl, teams_thrl_decl;

	if (!xar_expr_has_constant_value(c_copy))
	{
		/* CASE 1 */
		if (parent_type == DCTARGETTEAMS)
		{
			/* int __ompi_target_thread_limit = expr; */
			thrl_decl = clexpr_var_new_initdecl(CLV_THREAD_LIMIT, &c_copy, true);
			clexpr_var_set(CLV_THREAD_LIMIT, parent_type, c_copy, thrl_decl);
			(*c)->u.expr = IdentName(clexpr_var_name(CLV_TEAMS_THREAD_LIMIT));

			/* int __ompi_teams_thread_limit = __ompi_target_thread_limit; */
			teams_thrl_decl = clexpr_var_new_initdecl(CLV_TEAMS_THREAD_LIMIT, &tc, false);

			/* __ompi_teams_thread_limit */
			c_copy = IdentName(clexpr_var_name(CLV_TEAMS_THREAD_LIMIT));
			clexpr_var_set(CLV_TEAMS_THREAD_LIMIT, parent_type, c_copy, teams_thrl_decl);
		}
		/* CASE 2 */
		else if (is_or_combines(parent_type, DCTARGET))
		{
			/* int __ompi_target_thread_limit = expr; */
			thrl_decl = clexpr_var_new_initdecl(CLV_THREAD_LIMIT, &c_copy, true);
			clexpr_var_set(CLV_THREAD_LIMIT, parent_type, c_copy, thrl_decl);
	
			if (search_nested_construct((*c)->parent->parent, DCTEAMS))
			{
				/* int __ompi_teams_thread_limit = __ompi_target_thread_limit; */
				teams_thrl_decl = clexpr_var_new_initdecl(CLV_TEAMS_THREAD_LIMIT, &tc, false);
				c_copy = IdentName(clexpr_var_name(CLV_TEAMS_THREAD_LIMIT));
				clexpr_var_set(CLV_TEAMS_THREAD_LIMIT, parent_type, c_copy, teams_thrl_decl);
			}
		}
		/* CASE 3 */
		else if (is_or_combines(parent_type, DCTEAMS))
		{
			teams_thrl_decl = clexpr_var_new_initdecl(CLV_TEAMS_THREAD_LIMIT, &(*c)->u.expr, true);
			c_copy = IdentName(clexpr_var_name(CLV_TEAMS_THREAD_LIMIT));
			clexpr_var_set(CLV_TEAMS_THREAD_LIMIT, parent_type, c_copy, teams_thrl_decl);
		}
	}

	return c_copy;
}

/*
 * This function works on a list of 2 thread_limit() clauses; one deriving from
 * a #target construct and the other from a #teams-related construct. The list 
 * was constructed by xc_ompcon_search_all_clauses(), ensuring that #teams is nested 
 * in the #target construct. 
 * 
 * We first find out which clause belongs to which construct and then we handle the
 * following two cases: 
 * 
 * CASE 1: thread_limit(expr1) === thread_limit(expr2)
 * 
 *   int __ompi_target_thread_limit = expr;
 *   int __ompi_teams_thread_limit = __ompi_target_thread_limit;
 * 
 * CASE 2: thread_limit(expr1) !== thread_limit(expr2)
 * 
 *   int __ompi_target_thread_limit = expr1;
 *   int __ompi_teams_thread_limit = expr2;
 */
static astexpr _handle_list_thread_limit_clause(ompclause *list)
{
	clexprtype_e types[2];
	ompclause clauses[2] = { (*list)->u.list.elem, (*list)->u.list.next };
	aststmt decls[2];
	astexpr thread_limit_ident = IdentName(clexpr_var_name(CLV_THREAD_LIMIT));
	bool first_is_thread_limit = is_or_combines(clauses[0]->parent->type, DCTARGET);
	int teams_idx = first_is_thread_limit ? 1 : 0,
	    thread_idx = 1 - teams_idx, i;

	/* Set types */
	types[teams_idx] = CLV_TEAMS_THREAD_LIMIT;
	types[thread_idx] = CLV_THREAD_LIMIT;

	/* CASE 1 */
	if (!xar_expr_has_constant_value(clauses[thread_idx]->u.expr) && 
	    same_expr(clauses[0]->u.expr, clauses[1]->u.expr))
	{
		/* int __ompi_target_thread_limit = expr; 
		 * expr gets replaced by thread_limit(__ompi_target_thread_limit)
		 */
		decls[thread_idx] = clexpr_var_new_initdecl(types[thread_idx], &clauses[thread_idx]->u.expr, true);

		/* int __ompi_teams_thread_limit = __ompi_target_thread_limit;
		 * we ditch the #teams thread_limit expr by replacing it with 0
		 */
		decls[teams_idx] = clexpr_var_new_initdecl(types[teams_idx], &thread_limit_ident, true);
		clauses[teams_idx]->u.expr = numConstant(0);

		clexpr_var_set(types[thread_idx], clauses[thread_idx]->parent->type,
		               NULL, decls[thread_idx]);
		clexpr_var_set(types[teams_idx], clauses[teams_idx]->parent->type,
		               NULL, decls[teams_idx]);

		/* We return __ompi_teams_thread_limit */
		return IdentName(clexpr_var_name(CLV_TEAMS_THREAD_LIMIT));
	}

	/* CASE 2 */
	for (i = 0; i < 2; ++i)
	{
		decls[i] = clexpr_var_new_initdecl(types[i], &clauses[i]->u.expr, true);
		clexpr_var_set(types[i], clauses[i]->parent->type, clauses[i]->u.expr, decls[i]);
	}

	/* We return the #teams thread_limit() expression which might have changed to
	 * __ompi_teams_thread_limit, if it was non-constant.
	 */
	return ast_expr_copy(clauses[teams_idx]->u.expr);
}


astexpr xc_ompcon_search_offload_params(ompcon t)
{
	astexpr numteamsmin = NULL, numteamsmax = NULL,
	        thrlim = NULL, numthreads = NULL;
	aststmt ntmindecl = NULL, ntmaxdecl = NULL, nthrdecl = NULL, teamdims_decl = NULL,
	        threaddims_decl = NULL;
	ompclause c;

	if (t == NULL) return Comma4(ZeroExpr(), ZeroExpr(), ZeroExpr(), ZeroExpr());

	/* num_teams([min]:max) */
	if ((c = xc_ompcon_search_unique_clause(t, OCNUMTEAMS)) != NULL) 
	{
		if (c->u.x2.expr1)
		{
			ntmindecl = clexpr_var_new_initdecl(CLV_NUM_TEAMS_MIN, &(c->u.x2.expr1), true);
			clexpr_var_set(CLV_NUM_TEAMS_MIN, c->parent->type, c->u.x2.expr1, ntmindecl);

			/* expr will be either the same or __ompi_num_teams_min */
			numteamsmin = ast_expr_copy(c->u.x2.expr1);
		}

		ntmaxdecl = clexpr_var_new_initdecl(CLV_NUM_TEAMS_MAX, &(c->u.x2.expr2), true);
		teamdims_decl = Declaration(
		                  Usertype(Symbol("unsigned long long")), 
		                  InitDecl(
		                    Declarator(
		                      NULL, IdentifierDecl(Symbol(clexpr_var_name(CLV_TEAMDIMS_LIST)))
		                    ), 
		                    FunctionCall(
		                      IdentName("ENCODE3_ULL"),
		                      clexpr_var_expr(CLV_TEAMDIMS_LIST) != NULL
		                        ? ast_expr_copy(clexpr_var_expr(CLV_TEAMDIMS_LIST))
		                        : Comma3(ast_expr_copy(c->u.x2.expr2), ZeroExpr(), ZeroExpr())
		                    ) 
		                  )
		                );
		clexpr_var_set(CLV_TEAMDIMS_LIST, c->parent->type, 
		               expr_product_to_dimensions(c->u.x2.expr2), teamdims_decl);
		clexpr_var_set(CLV_NUM_TEAMS_MAX, c->parent->type, c->u.x2.expr2, ntmaxdecl);

		/* expr will be either the same or __ompi_num_teams_max */
		numteamsmax = ast_expr_copy(c->u.x2.expr2);
	}

	/* thread_limit(lim) */
	if ((c = xc_ompcon_search_all_clauses(t, OCTHREADLIMIT)) != NULL)
	{
		if (c->type == OCLIST)
			thrlim = _handle_list_thread_limit_clause(&c);
		else
			thrlim = _handle_single_thread_limit_clause(&c);
	}

	/* num_threads(nthr) */
	if ((c = xc_ompcon_search_unique_clause(t, OCNUMTHREADS)) != NULL)
	{
		nthrdecl = clexpr_var_new_initdecl(CLV_NUM_THREADS, &(c->u.expr), true);
		threaddims_decl = Declaration(
		                  Usertype(Symbol("unsigned long long")), 
		                  InitDecl(
		                    Declarator(
		                      NULL, IdentifierDecl(Symbol(clexpr_var_name(CLV_THREADDIMS_LIST)))
		                    ), 
		                    FunctionCall(
		                      IdentName("ENCODE3_ULL"),
		                      clexpr_var_expr(CLV_THREADDIMS_LIST) != NULL
		                        ? ast_expr_copy(clexpr_var_expr(CLV_THREADDIMS_LIST))
		                        : Comma3(ast_expr_copy(c->u.expr), ZeroExpr(), ZeroExpr())
		                    ) 
		                  )
		                );
		clexpr_var_set(CLV_THREADDIMS_LIST, c->parent->type, 
		               expr_product_to_dimensions(c->u.expr), threaddims_decl);
		clexpr_var_set(CLV_NUM_THREADS, c->parent->type, c->u.expr, nthrdecl);

		numthreads = ast_expr_copy(c->u.expr);
	}
	/* Otherwise find the # of nested parallel regions: if at least one nested parallel
	 * region exists (combined) use -1 to decide at runtime.
	 */
	else
		if (search_nested_construct(t, DCPARALLEL) >= 1) 
			numthreads = numConstant(DEVICETHREADS_RUNTIME);

	return Comma5(
		numteamsmin ? numteamsmin : ZeroExpr(),
		numteamsmax ? numteamsmax : ZeroExpr(),
		numthreads, // if NULL, it will be handled by xform_target() funcs
		thrlim ? thrlim : ZeroExpr(),
		Comma2(
			IdentName(clexpr_var_name(CLV_TEAMDIMS_LIST)), 
			IdentName(clexpr_var_name(CLV_THREADDIMS_LIST))
		)
	);
}


/**
 * Get all clauses of a given type in a construct.
 * The returned list is a copy (so it can be freed).
 *
 * @param t    The OpenMP construct we are searching in
 * @param type The type of clause we want to search for
 * @return     A list of all clauses of the given type (or NULL)
 */
ompclause xc_ompcon_get_every_clause(ompcon t, ompclt_e type)
{
	return xc_clauselist_get_clause(t->directive->clauses, type, t->type, true);
}


/* Take the clauses of a V4.5 target construct and split them in two:
 * the ones that refer to the actual (V4.0) target and the ones that
 * refer to the implied target task. The nowait one gets nowhere.
 */
void xc_split_target_clauses(ompclause all, ompclause *targc, ompclause *taskc)
{
	if (all == NULL) { *targc = *taskc = NULL; return; }
	if (all->type == OCLIST)
	{
		xc_split_target_clauses(all->u.list.next, targc, taskc);
		all = all->u.list.elem;
		assert(all != NULL);
	}
	all = ast_ompclause_copy(all);
	switch (all->type)
	{
		case OCDEPEND:
			*taskc = (*taskc) ? OmpClauseList(*taskc, all) : all;
			break;
		default:
			*targc = (*targc) ? OmpClauseList(*targc, all) : all;
			break;
	}
}


/* Next 4 functions return a set containing all the variables in clauses of
 * a given type.
 */
static
void varlist_get_vars(astdecl d, set(vars) s, 
                      ompclt_e type, ompclsubt_e subt, intvec_t mods)
{
	setelem(vars) e;

	if (d->type == DLIST && d->subtype == DECL_idlist)
	{
		varlist_get_vars(d->u.next, s, type, subt, mods);
		d = d->decl;
	}
	assert(d->type == DIDENT);
	e = set_put_unique(s, d->u.id);
	e->value.clause = type;
	e->value.clsubt = subt;
	e->value.clmods = intvec_copy(mods);
}


static
void xlist_get_vars(ompclause t, set(vars) s)
{
	setelem(vars) e;
	ompxli        xl;

	for (xl = t->u.xlist; xl; xl = xl->next)
	{
		e = set_put_unique(s, xl->id);      // TODO: array sections 
		e->value.ptr    = xl;
		e->value.clause = t->type;
		e->value.clsubt = t->subtype;
		e->value.clmods = intvec_copy(t->modifs);
	}
}


static
void clauselist_get_vars(ompclause t, 
         ompclt_e type, ompclsubt_e subt, set(vars) s)
{
	if (t == NULL) return;

	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			clauselist_get_vars(t->u.list.next, type, subt, s);
		t = t->u.list.elem;
		assert(t != NULL);
	}
	if (t->type == type && (subt == OC_DontCare || t->subtype == subt))
	{
		if (type == OCDEPEND || type == OCMAP || type == OCREDUCTION  || 
		    type == OCTASKREDUCTION)
			xlist_get_vars(t, s);
		else
			varlist_get_vars(t->u.varlist, s, type, t->subtype, t->modifs);
	}
}


/**
 * Find varlist clauses of a given type/subtype and return a set with the 
 * variables that appear in those clauses.
 *
 * @param t    The OpenMP construct we are searching in
 * @param type The type of clause we want to search for
 * @param subt The clause subtype (pass OC_DontCare to ignore)
 * @param s    An initiliazed set (must not be NULL) that after execution will
 * contain all the variables that appear in a "type" clause
 */
void xc_ompcon_get_vars(ompcon t, ompclt_e type, ompclsubt_e subt, set(vars) s)
{
	clauselist_get_vars(t->directive->clauses, type, subt, s);
}


/* Next 3 functions return a set containing all the extended list items
 * in clauses of a given type.
 */
static
void xlist_get_items(ompxli xl, set(xlitems) s, 
                     ompclt_e type, ompclsubt_e subt, intvec_t mods)
{
	setelem(xlitems) e;
	
	for (; xl; xl = xl->next)
	{
		e = set_put(s, xl->id);
		e->value.clause = type;
		e->value.clsubt = subt;
		e->value.clmods = intvec_copy(mods);
		e->value.xl     = xl;
	}
}


static
void clauselist_get_xlitems(ompclause t, 
         ompclt_e type, ompclsubt_e subt, set(xlitems) s)
{
	if (t == NULL) return;

	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			clauselist_get_xlitems(t->u.list.next, type, subt, s);
		t = t->u.list.elem;
		assert(t != NULL);
	}
	if (t->type == type && (subt == OC_DontCare || t->subtype == subt))
		xlist_get_items(t->u.xlist, s, type, t->subtype, t->modifs);
}

/**
 * Find clauses of a given type/subtype and return a set with the extended
 * list items that appear in those clauses. This only refers to
 * REDUCTION/MAP/DEPEND/TO/FROM clauses (the only with extended list items).
 *
 * @param t    The OpenMP construct we are searching in
 * @param type The type of clause we want to search for
 * @param subt The clause subtype (pass OC_DontCare to ignore)
 * @param s    An initiliazed set (must not be NULL) that after execution will
 * contain all the extended list items that appear in a "type" clause
 */
void xc_ompcon_get_xlitems(ompcon t, 
         ompclt_e type, ompclsubt_e subt, set(xlitems) s)
{
	clauselist_get_xlitems(t->directive->clauses, type, subt, s);
}


/**
 * Produces a memory copy statement
 *
 *    memcpy( (void *) to, (void *) from, size );
 *
 * @param to   The identifier we are copyin to
 * @param from The identifier we are copyin from
 * @param size The size we will copy (usually either SizeOf(to) or SizeOf(from)
 *             depending on which one is not a pointer
 * @return     The new statement
 */
aststmt xc_memcopy(astexpr to, astexpr from, astexpr size)
{
	needMemcpy = true;
	return (
	         FuncCallStmt(
	           "memcpy",
	           Comma3(CastVoidStar(to), CastVoidStar(from), size)
	         )
	       );
}


/**
 * Produces memory fill code (either a memset() call if filling with 0s 
 * or an explicit loop that sets each element to the given value).
 *
 * @param addr     The start of the area we are filling
 * @param nelems   Number of elements
 * @param nbytes   Size in bytes
 * @param what     The filler value
 * @return         The new statement
 */
aststmt xc_memfill(astexpr addr, astexpr nelems, astexpr nbytes, astexpr what)
{
	symbol loopidx = Symbol("_tmp_i_");
	
	if (what->type == CONSTVAL && strcmp(what->u.str, "0") == 0)
	{
		needMemset = true;                     /* Fill with zeroes */
		return FuncCallStmt("memset", Comma3(CastVoidStar(addr), what, nbytes));
	}
	
	/* { int i; for (i = (<nelems>)-1; i >= 0; i--)  (<addr>)[i] = <what>; } */
	return
		Compound(  
			BlockList(
				Declaration(
					Declspec(SPEC_int), Declarator(NULL, IdentifierDecl(loopidx))), 
				For(
					AssignStmt(
						Identifier(loopidx),
						BinaryOperator( BOP_sub, Parenthesis(nelems), OneExpr() )
					), 
					BinaryOperator(BOP_geq, Identifier(loopidx), ZeroExpr()), 
					PostOperator(Identifier(loopidx), UOP_dec),    
					Expression(
						Assignment(
							ArrayIndex(UnaryOperator(UOP_paren, addr), Identifier(loopidx)), 
							ASS_eq, 
							what
						)
					)
				)
			)
		);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     CLAUSE EXPRESSION STORE                                   *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Expressions from clauses, when non-constant, should be evaluated
 * and stored in a new variable. We hold specific structs that hold
 * the clause, the construct it belongs to and the initialization
 * declaration for the variable.
 * 
 * Right now, The whole functionality relies on a static array, which 
 * is not safe at all. Consider it as a temporary solution, it will be
 * fixed in the future. Also, clexpr_var_new_initdecl() is independent
 * of clause_var_set -- the main reason for that was to make it usable
 * by x_teams.c and not mess up variables set by the outermost construct.
 * 
 * The flow is the following:
 * 
 * HOST:
 *   Before transforming #target, the host searches for the offload 
 *   params and creates initdecls for the clause expression vars 
 *   calling clexpr_var_new_initdecl() / clexpr_var_set(). During
 *   #target transformation, outline will catch those vars and put 
 *   them in the DCT_UNSPECIFIED set, to declare them as firstprivate.
 *   After outlining, it just outputs the initdecl for each var, if
 *   it exists (=if the expr was non-constant).
 * 
 * GENERIC CODE TARGET:
 *   Before transforming #target, it just creates initdecls for the
 *   clause expression vars, replacing non-constant expressions with
 *   the variable name.
 * 
 * CODE TARGETS (except for generic):
 *   Before transforming #target, they create initdecls for the clause
 *   expression vars. Within the transformation and during outlining they 
 *   place the declared variables in the DCT_UNSPECIFIED set.
 * 
 */
static struct { 
	astexpr   expr;
	ompdirt_e dirtype;
	aststmt   initdecl;
} clexpr_store[CLV_SIZE];

char *clxvarnames[CLV_SIZE] = { 
	"__ompi_num_teams_min", "__ompi_num_teams_max", "__ompi_target_thread_limit",
	"__ompi_teams_thread_limit", "__ompi_num_threads", "__ompi_team_dimensions", 
	"__ompi_thread_dimensions"
};


/* Given a non-constant clause expression, it creates an init declaration
 * using a variable with the specified name. 
 */
aststmt clexpr_var_new_initdecl(clexprtype_e type, astexpr *clexpr, bool change_expr)
{
	symbol sym = Symbol(clxvarnames[type]);
	aststmt initdecl = NULL;

	if (!(*clexpr)) return NULL;

	if (((*clexpr)->type) != IDENT || ((*clexpr)->u.sym != sym))
	{
		if (!xar_expr_has_constant_value(*clexpr))
		{
			initdecl = Declaration(Declspec(SPEC_int),
							InitDecl(
							Declarator(NULL, IdentifierDecl(sym)),
							*clexpr
							)
						);
			if (change_expr) 
				*clexpr = Identifier(sym);
		}
	}

	return initdecl;
}

/* Returns true if there is an initdecl for a specific variable type 
 */
bool clexpr_var_has_initdecl(clexprtype_e type)
{
	assert((type > CLV_FIRST) && (type < CLV_SIZE));
	return ((clexpr_store[type].initdecl != NULL));
}

/* Prepends the initdecl of a specific variable to a statement,
 * if it exists.
 */
void clexpr_var_prepend_initdecl(clexprtype_e type, aststmt where)
{
	if (!clexpr_var_has_initdecl(type)) return;
	ast_stmt_in_place_prepend(where, clexpr_store[type].initdecl);
}

/* Getters 
 */
aststmt clexpr_var_initdecl(clexprtype_e type)
{
	if (!clexpr_var_has_initdecl(type)) return NULL;
	return clexpr_store[type].initdecl;
}

astexpr clexpr_var_expr(clexprtype_e type)
{
	return clexpr_store[type].expr;
}

char *clexpr_var_name(clexprtype_e type)
{
	return clxvarnames[type];
}

ompdirt_e clexpr_var_parent_dirtype(clexprtype_e type)
{
	return clexpr_store[type].dirtype;
}


/* Set a clexpr variable 
 */
void clexpr_var_set(clexprtype_e type, ompdirt_e dirtype, astexpr expr, aststmt initdecl)
{
	assert(type > CLV_FIRST && type < CLV_SIZE);

	clexpr_store[type].expr = expr;
	clexpr_store[type].dirtype = dirtype;
	if (initdecl)
	{
		ast_declare_decllist_vars(initdecl->u.declaration.spec, initdecl->u.declaration.decl);
		clexpr_store[type].initdecl = initdecl;
	}
}

/* Reset everything 
 */
void clexpr_vars_reset(void)
{
	clexprtype_e c;
	for (c = CLV_FIRST + 1; c < CLV_SIZE; c++)
	{
		clexpr_store[c].expr = NULL;
		clexpr_store[c].initdecl = NULL;
		clexpr_store[c].dirtype = DCNONE;
	}
}
