/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "x_atomic.h"
#include "ast_xform.h"
#include "ast_free.h"
#include "ast_copy.h"
#include "ast.h"
#include "ompi.h"
#include "builder.h"
#include "sem.h"
#include "ast_assorted.h"
#include "ast_arith.h"
#include "x_target.h"
#include "x_clauses.h"
#include "x_requires.h"


/* The following are used when the CAPTURE clause is present. Given an 
 * atomically-updated var:
 * - captureway specifies when to capture the var value: before or after it 
 *              is updated; if 0, no capturing is done.
 * - capturevar is where the captured value is stored.
 * - capturestmt is the statement which assigns the value to capturevar.
 *
 * The above are used in all atomic operations. When CAS is involved we
 * further have the following:
 * - casstatus is used to store the return value (true/false) of a CAS.
 * - casfalsecapture is true if we must capture the value of the var only in 
 *                   the case of a CAS failure (i.e. the CAS returns false).
 */
static enum { NOCAPTURE = 0, PRECAPTURE, POSTCAPTURE } captureway;
static astexpr capturevar, casstatus;
static aststmt capturestmt;     /* for capturing through distinct statement */
static bool    casfalsecapture; /* for capturing only when CAS fails */

/* Central error handling */
typedef enum { 
		ERR_NCUPD,  ERR_NCUVAR, ERR_RWASS,  ERR_UPDOP,  ERR_MOD,    ERR_BOP, 
		ERR_MATCH,  ERR_BADOP,  ERR_UPDASS, ERR_COMP,	  ERR_NCCUS,  ERR_WEAKEQ, 
		ERR_ONECC,  ERR_BOTHCC, ERR_NCCAPT, ERR_CCELSE, ERR_REQCMP, ERR_CUVAR,
		ERR_CCCVAR,
	} atomicerr_e;

static char *atomicerr[] = {
		"non-conforming update statement.\n",
		"non-conforming update statement (check variable usage).\n",
		"expecting a plain assignment.\n",
		"expecting an update operation (increment/decrement/etc).\n",
		"the \% operator is not allowed in an atomic update operation.\n",
		"expecting a binary operator on the right side of the assignment.\n",
		"none of the two operands on the right side of the assignment\n"
			"\tmatches the left side of the assignment\n",
		"illegal operator in an atomic update operation.\n",
		"expecting an assignement in the update statement.\n",
		"expecting a conditional update statement with a comparison (==, <, >).\n",
		"non-conforming conditional update statement.\n",
		"the WEAK clause must be used with a conditional update statement\n"
			"\twhich tests for equality.\n",
		"this syntax requires a COMPARE or CAPTURE clause to be present.\n",
		"the given conditional expression statement requires both CAPTURE\n"
			"\tand UPDATE caluses\n",
		"non-conforming atomic capture body.\n",
		"ELSE part require when both CAPTURE and UPDATE clauses are present.\n",
		"this syntax requires the COMPARE clause to be present.\n",
		"the captured variable does not match the updated variable\n",
		"the condition must match the right-hand side capturing variable\n",
	};

static atomicerr_e _errid;

static void _atomic_error(aststmt t, atomicerr_e errid)
{
	exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t%s",
	           t->u.omp->directive->file->name, t->u.omp->directive->l+1,
	           atomicerr[errid]);
}


/* Fall back transformation */
void _lock_based_atomic(aststmt *t)
{
	aststmt s;
	
	s = (*t)->u.omp->body;
	(*t)->u.omp->body = NULL;     /* Make it NULL so as to free t easily */
	ast_free(*t);                 /* Get rid of the OmpStmt */

	if (!symbol_exists("_atomic_lock"))    /* Check for the lock */
		bld_globalvar_add(
			Declaration(
			  Declspec(SPEC_void), /* Dont use omp_lock_t */
			  InitDecl(
			    Declarator(Pointer(),IdentifierDecl(Symbol("_atomic_lock"))),
			    NullExpr()
			  )
			)
		)->isindevenv = due2INJNOMAP;    /* make sure it is never mapped */

	if (captureway)
	{
		if (capturestmt)
			s = captureway==PRECAPTURE? Block2(capturestmt,s) : Block2(s,capturestmt);
		else
			s->u.expr = Assignment(capturevar, ASS_eq, s->u.expr); /* + capture var */
	}
	s->file = (*t)->file;
	*t = Block4(
			Call0_stmt("_ort_atomic_begin"),
			s, 
			Call0_stmt("_ort_atomic_end"),
			linepragma(s->l + 1, s->file)
		);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     CONVERSIONS AND UTILITIES                                 *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


static ompclt_e memop_subtype2type(ompclsubt_e memsubop)
{
	switch (memsubop)
	{
		case OC_seq_cst: return OCSEQCST;
		case OC_acq_rel: return OCACQREL;
		case OC_relaxed: return OCRELAXED;
		default:         break;
	}
	fprintf(stderr, "[%s]: spurious memory order (%d).\n", __func__, memsubop);
	return OCRELAXED;
}


static symbol memop2memorder(ompclt_e memop)
{
	switch (memop)
	{
		case OCSEQCST:  return Symbol("__ATOMIC_SEQ_CST");
		case OCACQREL:  return Symbol("__ATOMIC_ACQ_REL");
		case OCACQUIRE: return Symbol("__ATOMIC_ACQUIRE");
		case OCRELEASE: return Symbol("__ATOMIC_RELEASE");
		case OCRELAXED: return Symbol("__ATOMIC_RELAXED");
		default: 
			return NULL;
	}
}


static symbol memop2memorder_fail(ompclsubt_e memsubop)
{
	switch (memsubop)
	{
		case OC_seq_cst: return Symbol("__ATOMIC_SEQ_CST");
		case OC_acquire: return Symbol("__ATOMIC_ACQUIRE");
		case OC_relaxed: return Symbol("__ATOMIC_RELAXED");
		default:         break;
	}
	return NULL;
}


/* Given the operator, it returns the corresponding atomic function to call
 */
static char *assop2atomicfunc(int opid, enum exprtype optype)
{
	if ((optype == POSTOP || optype == PREOP) && 
	    opid != UOP_inc && opid != UOP_dec)
	  return NULL;
	if (optype == POSTOP)
	{
		/* Take care if CAPTURE is needed */
		if (captureway == POSTCAPTURE)
			return opid==UOP_inc? "__atomic_add_fetch" : "__atomic_sub_fetch";
		else
			return opid==UOP_inc? "__atomic_fetch_add" : "__atomic_fetch_sub";
	}
	if (optype == PREOP)   /* pre-capture only through statement */
	{
		if (captureway == PRECAPTURE && capturestmt)
			return opid==UOP_inc? "__atomic_fetch_add" : "__atomic_fetch_sub";
		else
			return opid==UOP_inc? "__atomic_add_fetch" : "__atomic_sub_fetch";
	}
	if (optype == ASS)     /* there is no pre-capture here, too */
	{
		if (captureway == PRECAPTURE && capturestmt)
			switch (opid)
			{
				case ASS_add: return "__atomic_fetch_add";
				case ASS_sub: return "__atomic_fetch_sub";
				case ASS_and: return "__atomic_fetch_and";
				case ASS_xor: return "__atomic_fetch_xor";
				case ASS_or:  return "__atomic_fetch_or";
				default:
					return NULL;
			}
		else
			switch (opid)
			{
				case ASS_add: return "__atomic_add_fetch";
				case ASS_sub: return "__atomic_sub_fetch";
				case ASS_and: return "__atomic_and_fetch";
				case ASS_xor: return "__atomic_xor_fetch";
				case ASS_or:  return "__atomic_or_fetch";
				default:
					return NULL;
			}
	}
	warning("[%s]: spurious operator (%d) in ATOMIC\n", __func__, opid);
	return NULL;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     ATOMIC READ & WRITE                                       *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


 void _xform_atomic_read(aststmt *t, ompclt_e memop)
{
	astexpr e = (*t)->u.omp->body->u.expr;
	aststmt parent = (*t)->parent;
	
	if (e == NULL || e->type != ASS || e->opid != ASS_eq)
		_atomic_error(*t, ERR_RWASS);
	if (memop == OCACQREL)
		memop = OCACQUIRE;
	/* Use the generic version to cover all types */
	*t = Expression(FunctionCall(
	                  IdentName("__atomic_load"),
	                  Comma3(UOAddress(Parenthesis(e->right)), 
	                         UOAddress(Parenthesis(e->left)),
	                         Identifier(memop2memorder(memop))))
	               );
	(*t)->parent = parent;
}


void _xform_atomic_write(aststmt *t, ompclt_e memop)
{
	astexpr e = (*t)->u.omp->body->u.expr;
	aststmt parent = (*t)->parent;
	
	if (e == NULL || e->type != ASS || e->opid != ASS_eq)
		_atomic_error(*t, ERR_RWASS);
	if (memop == OCACQREL)
		memop = OCRELEASE;
	
	if (sem_expr_islvalue(e->right))
	{
		free(*t);
		*t = FuncCallStmt(    /* Use the generic version to cover all types */
	         "__atomic_store",
	         Comma3(UOAddress(Parenthesis(e->left)), 
	                UOAddress(Parenthesis(e->right)),
	                Identifier(memop2memorder(memop)))
	       );
	}
	else
	{
		semtype_t *tp = sem_expr_type(e->left);
		if (!numtype_isintegral(tp->class))
		{
			free(tp);
			_lock_based_atomic(t); /* fallback code; maybe we use CAS loops someday */
			return;
		}
		free(tp);
		free(*t);
		*t = FuncCallStmt(
			       "__atomic_store_n",
			       Comma3(UOAddress(Parenthesis(e->left)), 
			              e->right,
			              Identifier(memop2memorder(memop)))
			     );
	}
	
	if (captureway == PRECAPTURE)
		*t = Block2(AssignStmt(capturevar, ast_expr_copy(e->left)), *t);
	else if (captureway == POSTCAPTURE)
		*t = Block2(*t, AssignStmt(capturevar, ast_expr_copy(e->left)));
	(*t)->parent = parent;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *    ATOMIC UPDATE (PLAIN)                                      *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/**
 * Produces a compara-and-swap expression out of the given expressions.
 */
static 
astexpr _cas_expr(astexpr var, astexpr expected, astexpr desired,
                  bool weak, int succmemord, int failmemord)
{
	astexpr e =
		FunctionCall(
			IdentName("__atomic_compare_exchange_n"),
			Comma6(
				UOAddress(var->type == IDENT ? var : Parenthesis(var)),
				UOAddress(expected->type == IDENT ? expected : Parenthesis(expected)),
				desired,
				numConstant(weak ? 1 : 0),
				Identifier(memop2memorder(succmemord)),
				Identifier(memop2memorder_fail(failmemord))
			)
		);
	return e;
}


/* Assuming rhs is an l-value, the following statement is returned:
 *    {
 *      <type> _rhs_tmp = rhs, _upd_val = expr;
 *      while (!__atomic_compare_exchange_n(&(rhs), &_rhs_tmp, 
 *                                _rhs_tmp <op> _upd_val, memop, failop));
 *        ;
 *		  <capture the value of rhs or _rhs_tmp if needed>
 *    }
 * If expr is a constant or an identifier, then _upd_val is skipped.
 * In addition, it uses capturever if capture is active.
 */
static aststmt _cas_loop(astexpr rhs, astexpr expr, int bopid, 
                         bool weak, ompclt_e memop, ompclsubt_e failop)
{
	int capturebak = captureway;
	aststmt loop, tmpdecl, uvdecl = NULL;
	astexpr upd = expr->type == CONSTVAL || expr->type == IDENT ? 
	                expr : IdentName("_upd_val");
	
	if (rhs->type == IDENT)
	{
		tmpdecl = xform_clone_declaration_nosc_notq(rhs->u.sym, rhs, false, 
		                                            Symbol("_rhs_tmp"));
		if (expr->type != CONSTVAL && expr->type != IDENT)
			uvdecl = xform_clone_declaration_nosc_notq(rhs->u.sym, expr, false,
			                                           Symbol("_upd_val"));
	}
	else /* go through typeof() */
	{
		astspec sp;
		astdecl de;
		if (!sem_expr_typeof_declare(rhs, Symbol("_rhs_tmp"), &sp, &de))
			return NULL;
		tmpdecl = Declaration(sp, InitDecl(Declarator(NULL, de), rhs));
		if (expr->type != CONSTVAL && expr->type != IDENT)
		{
			if (!sem_expr_typeof_declare(rhs, Symbol("_upd_val"), &sp, &de))
				return NULL;
			uvdecl = Declaration(sp, InitDecl(Declarator(NULL, de), expr));
		}
	}
	
	if (captureway) captureway = NOCAPTURE;  /* Don't let _cas_expr see it */
	
	loop = While(
		       UnaryOperator(
		         UOP_lnot,
		         _cas_expr(ast_expr_copy(rhs), IdentName("_rhs_tmp"), 
		                   BinaryOperator(bopid, IdentName("_rhs_tmp"), upd),
		                   weak, memop, failop)
		       ),
		       Expression(NULL)
		     );
	
	if (capturebak) captureway = capturebak;      /* Restore it */ 
	
	loop = uvdecl ? Block3(tmpdecl, uvdecl, loop) : Block2(tmpdecl, loop);
	if (captureway)
		loop = Block2(loop, AssignStmt(capturevar, captureway == PRECAPTURE ?
		                       IdentName("_rhs_tmp") : ast_expr_copy(rhs)));
	return Compound(loop);
}


/* Assuming rhs is an l-value, the following statement is returned:
 *    {
 *      <type> _rhs_tmp = rhs, _upd_val = expr;
 *      while (_rhs_tmp <cmpop> _upd_val && 
 *             !__atomic_compare_exchange_n(&(rhs), &_rhs_tmp, 
 *                                _upd_val, memop, failop));
 *        ;
 *		  <capture the value of rhs or _rhs_tmp if needed>
 *    }
 * If expr is a constant or an identifier, then _upd_val is skipped.
 * In addition, it uses capturever if capture is active.
 */
static aststmt _cas_loop_cmp(astexpr rhs, astexpr expr, int bopid, 
                             bool weak, ompclt_e memop, ompclsubt_e failop)
{
	int capturebak = captureway;
	aststmt loop, tmpdecl, uvdecl = NULL; 
	astexpr upd = expr->type == CONSTVAL || expr->type==IDENT ? 
	                ast_expr_copy(expr) : IdentName("_upd_val");
	
	if (rhs->type == IDENT)
	{
		tmpdecl = xform_clone_declaration_nosc_notq(rhs->u.sym, rhs, false,
		                                            Symbol("_rhs_tmp"));
		if (expr->type != CONSTVAL && expr->type != IDENT)
			uvdecl = xform_clone_declaration_nosc_notq(rhs->u.sym, expr, false, 
			                                           Symbol("_upd_val"));
	}
	else /* go through typeof() */
	{
		astspec sp;
		astdecl de;
		if (!sem_expr_typeof_declare(rhs, Symbol("_rhs_tmp"), &sp, &de))
			return NULL;
		tmpdecl = Declaration(sp, InitDecl(Declarator(NULL, de), rhs));
		if (expr->type != CONSTVAL && expr->type != IDENT)
		{
			if (!sem_expr_typeof_declare(rhs, Symbol("_upd_val"), &sp, &de))
				return NULL;
			uvdecl = Declaration(sp, InitDecl(Declarator(NULL, de), expr));
		}
	}

	if (captureway) captureway = NOCAPTURE;  /* Don't let _cas_expr see it */
	
	loop = While(
		       BinaryOperator(
		         BOP_land,
		         BinaryOperator(bopid, IdentName("_rhs_tmp"), upd),
		         UnaryOperator(
		           UOP_lnot,
		           _cas_expr(ast_expr_copy(rhs), IdentName("_rhs_tmp"), 
		                     ast_expr_copy(upd), weak, memop, failop)
						 )
		       ),
		       Expression(NULL)
		     );

	if (capturebak) captureway = capturebak;      /* Restore it */ 
	
	loop = uvdecl ? Block3(tmpdecl, uvdecl, loop) : Block2(tmpdecl, loop);
	if (captureway)
		loop = Block2(loop, AssignStmt(capturevar, captureway == PRECAPTURE ?
		                       IdentName("_rhs_tmp") : ast_expr_copy(rhs)));
	return Compound(loop);
}


static 
void _atomic_update_statement(aststmt *t, int opid, enum exprtype optype, 
                   astexpr var, astexpr val, ompclt_e memop, ompclsubt_e failop)
{
	if (optype == ASS && 
	    (opid == ASS_div || opid == ASS_mul || opid == ASS_shl || opid ==ASS_shr))
	{
		aststmt new = _cas_loop(var, val, assop2bop(opid), false,memop,failop);
		if (new == NULL)
			_lock_based_atomic(t);
		else
		{
			free(*t);
			*t = new;
		}
	}
	else
	{
		free(*t);  /* maybe a little mem leak here? */
		/* A semantic check would be useful here... */
		*t =
			captureway ? 
				AssignStmt(      /* If capture, assign the returned value */
					capturevar,
					FunctionCall(
						IdentName(assop2atomicfunc(opid, optype)),
						Comma3(UOAddress(Parenthesis(var)), val,
						       Identifier(memop2memorder(memop)))
					)
				) :              /* If no capture, just call the atomic function */
				FuncCallStmt(
					assop2atomicfunc(opid, optype),
					Comma3(UOAddress(Parenthesis(var)), val,
					       Identifier(memop2memorder(memop)))
				);
	}
}


void _xform_atomic_update_plain(aststmt *t, ompclt_e memop, ompclsubt_e failop)
{
	astexpr e = (*t)->u.omp->body->u.expr;
	semtype_t *tp;
	
	if (e == NULL || (e->type != ASS && e->type != POSTOP && e->type != PREOP))
		_atomic_error(*t, ERR_UPDOP);
	
	/* Use fallback code for non-integral vars; someday we may do CAS loops */
	tp = sem_expr_type(e->left);
	if (!numtype_isintegral(tp->class))
	{
		free(tp);
		_lock_based_atomic(t);
		return;
	}
	free(tp);

	switch (e->type)
	{
	  case POSTOP:
	  case PREOP:
			_atomic_update_statement(t, e->opid, e->type, e->left, numConstant(1), 
			                         memop, failop);
			break;
		case ASS:
			if (e->opid != ASS_eq)
			{
				if (e->opid == ASS_mod)
					_atomic_error(*t, ERR_MOD);
				_atomic_update_statement(t, e->opid, ASS, e->left, e->right, 
				                         memop, failop);
			}
			else
			{
				int opid, useleft;
				
				if (!e->left || !e->right || e->right->type != BOP)
					_atomic_error(*t, ERR_BOP);
				if ((opid = e->right->opid) == BOP_mod)
					_atomic_error(*t, ERR_MOD);
				useleft = same_expr(e->left, e->right->right) ? 1 :
				          same_expr(e->left, e->right->left)  ? 0 : -1;
				if (useleft == -1)
					_atomic_error(*t, ERR_MATCH);
				if ((opid = bop2assop(opid)) == ASS_eq) /* this is error */
					_atomic_error(*t, ERR_BADOP);
				
				if (useleft && opid == ASS_sub)  /* strange fruit */
					_lock_based_atomic(t);
				else
				{
					_atomic_update_statement(t, opid, ASS, e->left, 
				             useleft ? e->right->left : e->right->right, memop, failop);
					if (opid!=ASS_div && opid!=ASS_mul && opid!=ASS_shl && opid !=ASS_shr)
					{
						ast_expr_free(useleft ? e->right->right : e->right->left);
						free(e->right);
					}
				}
			}
		default:
			break;
	}
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *    ATOMIC UPDATE (COMPARE/CAPTURE)                            *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


typedef enum {
		FORM_ERR = 0, 
		FORM_CUPEXP,   /* (+ CONDITIONTYPE) */ /* x = expr ordop x ? expr : x; 
		                                          x = x ordop expr ? expr : x;
		                                          x = x == e ? d : x;  */
		FORM_CUPIFS,   /* (+ CONDITIONTYPE) */ /* if (expr ordop x) { x = expr; }
		                                          if (x ordop expr) { x = expr; }
		                                          if (x == e) { x = d; } */
		FORM_CAPEXPR,  /* v = expr-stmt (+ expression statement type 
		                                 + conditional statement type) */
		FORM_CAPBLK1,  /* { v = x; expr-stmt } (+ expression statement type) */
		FORM_CAPBLK2,  /* { expr-stmt v = x; } (+ expression statement type) */
		FORM_COMCAP1,  /* { v = x; cond-update-stmt } */
		FORM_COMCAP2,  /* { cond-updte-stmt v = x; } */
		FORM_COMCAP3,  /* if () {...} else {...} */
		FORM_COMCAP4,  /* { r = x == e; if (r) <no else> } */
		FORM_COMCAP5,  /* { r = x == e; if (r) then else } */
	} comcapform_e;
#define FORMTYPE(x) ((x) & 31)

/* expression statement types (used with CAPTURE) */
#define EXPRESSIONWR 32   /* write-expr-stmt */
#define EXPRESSIONUP 64   /* update-expr-stmt */
#define EXPRESSIONCU 128  /* cond-update-expr-stmt */
#define EXPRESSIONTYPE(x) (((x) & 255) & ~31)

/* conditional statement types (for cond-expr-stmt and cond-update-stmt) */
#define CONDITIONEX 256  /* conditional as in (expr >|< x) */
#define CONDITIONXE 512  /* conditional as in (x >|< expr) */
#define CONDITIONEQ 1024 /* conditional as in (x == expr)  */
#define CONDITIONTYPE(x) ((x) & ~255)

#define FORMERROR(err) (_errid = err, FORM_ERR)


/**
 * Checks if an expression is a cond-expr; when checking fot the type of 
 * atomic statement, this should be checked last since it is the most general.
 */
static bool _is_write_expr(astexpr e)
{
	if (!e || e->type != ASS || e->opid != ASS_eq) return false;
	return true;
}


/**
 * Checks if an expression is an update-expr
 */
static bool _is_update_expr(astexpr e)
{
	if (e == NULL || (e->type != ASS && e->type != POSTOP && e->type != PREOP))
		return false;
	if (e->type == POSTOP || e->type == PREOP)
		return true;
	if (e->type != ASS)
		return false;
		
	if (e->opid != ASS_eq) return e->opid == ASS_mod ? false : true;
	if (!e->left || !e->right || e->right->type != BOP) return false;
	if (e->right->opid == BOP_mod) return false;
	if (!same_expr(e->left,e->right->right) && !same_expr(e->left,e->right->left)) 
	  return false;
	return (bop2assop(e->right->opid) != ASS_eq);
}


/* 
 * Checks if an expression is a cond-expr 
 */
static bool _is_cond_expr(astexpr e)
{
	if (!e || e->type != ASS || e->opid != ASS_eq) return false;
	if (!e->right || e->right->type != CONDEXPR) return false;
	return (same_expr(e->left, e->right->right));
}


/* Given an expression, it returns what type of cond-update-expr it is. 
 */
int _cond_expr_type(astexpr e)
{
	astexpr cond;
	
	if (e->type != ASS || e->opid != ASS_eq) return FORMERROR(ERR_UPDASS);
	e = e->right;
	if (e->type != CONDEXPR) return FORMERROR(ERR_COMP);
	
	/* Check the condition */
	cond = e->u.cond;
	if (cond->type != BOP) return FORMERROR(ERR_COMP);
	if (cond->opid == BOP_eqeq)
		return same_expr(cond->left, e->right) ? CONDITIONEQ : FORMERROR(ERR_NCCUS);
	if (cond->opid != BOP_lt && cond->opid != BOP_gt) 
		return FORMERROR(ERR_NCCUS);
	if (same_expr(cond->right, e->right))
		return CONDITIONEX;
	if (same_expr(cond->left, e->right))
		return CONDITIONXE;
	return FORMERROR(ERR_NCCUS);

}


/* Given an if-statement, it returns what type of cond-update-stmt it is. 
 */
int _cond_update_stmt_type(aststmt t)
{
	aststmt s;
	astexpr cond;
	
	/* Just to make sure */
	if (t->type != SELECTION || t->subtype != SIF) return FORMERROR(ERR_NCCUS);
	if (t->u.selection.elsebody) return FORMERROR(ERR_NCCUS);
	
	/* Check the then part */
	s = t->body;
	if (s->type != COMPOUND || !s->body || s->body->type != EXPRESSION)
		return FORMERROR(ERR_NCCUS);
	s = s->body;
	if (s->u.expr->type != ASS || s->u.expr->opid != ASS_eq)
		return FORMERROR(ERR_NCCUS);
	
	/* Check the condition */
	cond = t->u.selection.cond;
	if (cond->type != BOP) return FORMERROR(ERR_NCCUS);
	if (cond->opid == BOP_eqeq)
		return same_expr(cond->left, s->u.expr->left) ? CONDITIONEQ : 
		                                                FORMERROR(ERR_NCCUS);
	if (cond->opid != BOP_lt && cond->opid != BOP_gt) 
		return FORMERROR(ERR_COMP);
		
	/* Check that vars match */
	if (same_expr(s->u.expr->left, cond->right))
		return CONDITIONEX;
	if (same_expr(s->u.expr->left, cond->left))
		return CONDITIONXE;
	return FORMERROR(ERR_NCUVAR);
}



/* Expression-statement form of atomic update; the given statement is assumed
 * to be an expression one.
 */
int _update_expression_form_type(aststmt t, bool compare, bool capture)
{
	astexpr e = t->u.expr;
	int     condtype;
	
	if (e->type != ASS || e->opid != ASS_eq)  
		return FORMERROR(ERR_NCCAPT);
	e = t->u.expr->right;  /* the conditional or the expr-stmt */
	if (capture)           /* must be the expr-stmt */
	{
		if (_is_update_expr(e)) 
			return compare ? FORMERROR(ERR_NCCAPT) : FORM_CAPEXPR + EXPRESSIONUP;
		if (_is_cond_expr(e)) /* this actually *requires* the compare clause */
		{
			if (!compare) return FORMERROR(ERR_BOTHCC);
			if ((condtype = _cond_expr_type(e)) != FORM_ERR)
				condtype += FORM_CAPEXPR + EXPRESSIONCU;
			return condtype;
		}
		if (_is_write_expr(e) && !compare)
			return FORM_CAPEXPR + EXPRESSIONWR;
		return FORMERROR(ERR_NCCAPT);
	}
	if (compare)              /* must be the conditional */
	{
		if (!same_expr(t->u.expr->left, e->right)) 
			return FORMERROR(ERR_NCCUS);
		if ((condtype = _cond_expr_type(t->u.expr)) != FORM_ERR)
			condtype += FORM_CUPEXP;
		return condtype;
	}
	/* what if !compare && !capture ? */
	return FORMERROR(ERR_NCCUS); // dummy 
}


/* If-statement form of atomic update; the given statement is assumed to be
 * an if statement.
 */
int _update_if_form_type(aststmt t, bool compare, bool capture)
{
	aststmt s, u;
	int     condtype;
	
	if (!compare) return FORMERROR(ERR_REQCMP);
	if (!capture)    /* compare form: if () { ... } */
	{
		if ((condtype = _cond_update_stmt_type(t)) != FORM_ERR)
			condtype += FORM_CUPIFS;
		return condtype;
	}
	else    /* compare & capture 3rd form: if () then else */
	{
		s = t->u.selection.elsebody;
		if (!s) return FORMERROR(ERR_CCELSE);
		if (s->type != COMPOUND) return FORMERROR(ERR_NCCAPT);
		s = s->body;
		if (s->type != EXPRESSION || s->u.expr->type != ASS 
		                          || s->u.expr->opid != ASS_eq)
			return FORMERROR(ERR_NCCUS);;
		if (!same_expr(s->u.expr->right, t->u.selection.cond->left))
			return FORMERROR(ERR_NCUVAR);  /* Check if vars match failed */
			
		/* disable the else part temporarily to check the if-then part */
		u = t->u.selection.elsebody; t->u.selection.elsebody = NULL;
		condtype = _cond_update_stmt_type(t);
		t->u.selection.elsebody = u;           /* restore the else */
		return condtype == CONDITIONEQ ? FORM_COMCAP3 : FORMERROR(ERR_NCCAPT);
	}
}


/* Compound-statement form of atomic update; the given statement is assumed
 * to be a compound one.
 */
int _update_compound_form_type(aststmt t, bool compare, bool capture)
{
	aststmt s, u;
	astexpr cond;
	comcapform_e form;
	int condtype;
	
	if (!capture && !compare) return FORMERROR(ERR_ONECC);
	t = t->body;
	if (t->type != STATEMENTLIST)
		return compare ? FORMERROR(ERR_NCCUS) : FORMERROR(ERR_NCCAPT);
	
		/* 1st or 2nd form { v = x; expr-stmt } or { expr-stmt v = x; } */
	if (t->body->type == EXPRESSION && t->u.next->type == EXPRESSION)
	{
		if (!capture)  /* capture required in such forms */
			return FORMERROR(ERR_NCCUS);

		/* Given {v=x; expr-stmt} (or equivallently {expr-stmt v=x;}), is "v=x;" 
		 * the capture statment or the update statement? Thus we need to check 
		 * expr-stmt; if it is anything but a simple assignment, then we assume 
		 * that the "v=x;" is the capture statement. If it is a simple assignment 
		 * (i.e. a write-expr-stmt), then we need to check whether its left-hand 
		 * side is "x" or its right-hand side is "v".
		 */
		if (t->u.next->type == EXPRESSION && _is_write_expr(t->u.next->u.expr))
		{
			if (t->body->type == EXPRESSION && _is_write_expr(t->body->u.expr))
			{
				if (same_expr(t->u.next->u.expr->right, t->body->u.expr->left))
					form = FORM_CAPBLK1;   /* PRECAPTURE */ 
				else
					if (same_expr(t->u.next->u.expr->left, t->body->u.expr->right))
						form = FORM_CAPBLK2; /* POSTCAPTURE */ 
					else
						return FORMERROR(ERR_NCCAPT);
			}
			else 
				form = FORM_CAPBLK1;     /* PRECAPTURE */ 
		}
		else
		{
			if (t->body->type == EXPRESSION && _is_write_expr(t->body->u.expr))
				form = FORM_CAPBLK2;     /* POSTCAPTURE */ 
			else
				return FORMERROR(ERR_NCCAPT);
		}
		s = form == FORM_CAPBLK1 ? t->u.next : t->body;
		u = form == FORM_CAPBLK1 ? t->body : t->u.next;    /* update stmt */
		
		if (!u || !s || !same_expr(s->u.expr->right, u->u.expr->left)) 
			return FORMERROR(ERR_NCCAPT);

		if (u->type == EXPRESSION && _is_update_expr(u->u.expr))
			return form + EXPRESSIONUP;
		if (u->type == EXPRESSION && _is_cond_expr(u->u.expr)) /* compare needed */
			return compare ? form + EXPRESSIONCU : FORMERROR(ERR_BOTHCC);
		if (u->type == EXPRESSION && _is_write_expr(u->u.expr))
			return form + EXPRESSIONWR;
		return FORMERROR(ERR_NCCAPT);
	}
	
	/* compare & capture 1st, 2nd, 4th, 5th forms */
	if (t->u.next->type == SELECTION && t->u.next->subtype == SIF)
	{
		/* 2nd form: { if () ; v = x; } (postcapture) */
		s = t->body;      /* the "v = x;" part */
		t = t->u.next; /* the if-statement */
		form = FORM_COMCAP2;
		
		COMCAP12:
		if (s->type != EXPRESSION || s->u.expr->type != ASS 
		                          || s->u.expr->opid != ASS_eq)
			return FORMERROR(ERR_NCCAPT);
		if ((condtype = _cond_update_stmt_type(t)) == FORM_ERR)
			return FORM_ERR;
		/* Finally check that "x" is the same */
		return same_expr(s->u.expr->right, t->body->body->u.expr->left) ?
		         form + condtype : FORMERROR(ERR_CUVAR);
	}
	
	if (t->u.next->type != EXPRESSION)   /* any of the other 3 forms */
		return FORMERROR(ERR_NCCAPT);
	s = t->u.next;   /* the assignment part */
	t = t->body;     /* the if-statement */
	if (s->u.expr->type != ASS || s->u.expr->opid != ASS_eq)
		return FORMERROR(ERR_NCCAPT);
	
	/* 1st form: { x = v; if () ; } (precapture) */
	if (sem_expr_islvalue(s->u.expr->right))
	{
		form = FORM_COMCAP1;
		goto COMCAP12;
	}
	
	/* 4 & 5th form { r = x == e; if (r) ... } */
	if (s->u.expr->right->type != BOP || s->u.expr->right->opid != BOP_eqeq)
		return FORMERROR(ERR_NCCAPT);
		
	if (t->type != SELECTION || t->subtype != SIF || 
	    !same_expr(s->u.expr->left, t->u.selection.cond))
		return FORMERROR(ERR_CCCVAR);        /* check that "r" matches */
	/* temporarily replace the condition with "x==e" */
	cond = t->u.selection.cond;
	t->u.selection.cond = s->u.expr->right;
	
	if (!t->u.selection.elsebody)
	{
		/* 4th form { r = x == e; if (r) <no else> } */
		condtype = _cond_update_stmt_type(t);
		t->u.selection.cond = cond;  /* restore condition */
		return condtype == CONDITIONEQ ? FORM_COMCAP4 : FORMERROR(ERR_NCCAPT);
	}

	/* 5th form { r = x == e; if (r) then else } */
	s = t->u.selection.elsebody;     /* check the "v = x;" part */
	if (s->type != COMPOUND) 
	{
		t->u.selection.cond = cond;  /* restore condition */
		return FORMERROR(ERR_NCCAPT);
	}
	s = s->body;
	if (s->type != EXPRESSION || s->u.expr->type != ASS 
	                          || s->u.expr->opid != ASS_eq)
	{
		t->u.selection.cond = cond;  /* restore condition */
		return FORMERROR(ERR_NCCAPT);
	}
	if (!same_expr(s->u.expr->right, t->u.selection.cond->left))
	{
		t->u.selection.cond = cond;  /* restore condition */
		return FORMERROR(ERR_NCCAPT);   /* Check if vars match */
	}
	
	/* disable the else part temporarily to check the if-then part */
	u = t->u.selection.elsebody; t->u.selection.elsebody = NULL;
	condtype = _cond_update_stmt_type(t);
	t->u.selection.elsebody = u;     /* restore the else */
	t->u.selection.cond = cond;      /* restore the condition */
	return condtype == CONDITIONEQ ? FORM_COMCAP5 : FORMERROR(ERR_NCCAPT);
}


int _update_type(aststmt t, bool compare, bool capture)
{
	t = t->u.omp->body;
	switch (t->type)
	{
		case EXPRESSION: return _update_expression_form_type(t, compare, capture);
		case SELECTION:  return _update_if_form_type(t, compare, capture);
		case COMPOUND:   return _update_compound_form_type(t, compare, capture);
		default:         break;
	}
	return capture ? FORMERROR(ERR_NCCAPT) : FORMERROR(ERR_NCUPD); 
}


static void _xform_atomic_update_compare(aststmt *t, bool weak, ompclt_e memop, 
                                         ompclsubt_e failop)
{
	astexpr e, cond, condvar, condexpr, resvar, resexpr;
	int cmpopid;
	aststmt body = (*t)->u.omp->body;

	if (body->type == EXPRESSION) 
	{
		e = body->u.expr;
		e = e->right;
		cond     = e->u.cond;
		resvar   = e->right;
		resexpr  = e->left;
	}
	else    /* an IF SELECTION statement */
	{
		e = body->body->body->u.expr;
		cond = body->u.selection.cond;
		resvar = e->left;
		resexpr = e->right;
	}
	
	/* Unified code for cond-expr-stmt and cond-update-stmt */
	cmpopid  = cond->opid;
	if (cmpopid == BOP_eqeq)  /* plain CAS */
	{
		condvar  = ast_expr_copy(cond->left);    /* to deeply free *t */
		condexpr = ast_expr_copy(cond->right);
		resexpr  = ast_expr_copy(resexpr);
		
		/* If expected is an lvalue, we just do CAS, otherwise, we should assign 
		 * the value to a temporary variable and use that variable in the CAS.
		 */
		if (sem_expr_islvalue(condexpr))
		{
			ast_stmt_free(*t);
			e = _cas_expr(condvar, condexpr, resexpr, weak, memop, failop);
			if (casstatus)
				e = Assignment(casstatus, ASS_eq, e);
			if (casfalsecapture)
				*t = If(UnaryOperator(UOP_lnot, Parenthesis(e)), capturestmt, NULL);
			else 
			{
				if (captureway == PRECAPTURE)
					e = Comma2(Assignment(capturevar, ASS_eq, ast_expr_copy(condvar)), e);
				if (captureway == POSTCAPTURE)
					e = Comma2(e, Assignment(capturevar, ASS_eq, ast_expr_copy(condvar)));
				*t = Expression(e);
			}
		}
		else                   /* temp variable + plain CAS */
		{
			astspec sp;
			astdecl de;
			
			if (sem_expr_typeof_declare(condexpr, Symbol("_cond_tmp"), &sp, &de))
			{
				ast_stmt_free(*t);
				e = _cas_expr(condvar,IdentName("_cond_tmp"),resexpr,weak,memop,failop);
				if (casstatus)
					e = Assignment(casstatus, ASS_eq, e);
				if (casfalsecapture)
					*t = If(UnaryOperator(UOP_lnot, Parenthesis(e)), capturestmt, NULL);
				else
				{
					if (captureway == PRECAPTURE)
						e = Comma2(Assignment(capturevar,ASS_eq,ast_expr_copy(condvar)), e);
					if (captureway == POSTCAPTURE)
						e = Comma2(e, Assignment(capturevar,ASS_eq,ast_expr_copy(condvar)));
					*t = Expression(e);
				}
				*t = Compound(
				       Block2(
				         Declaration(sp, InitDecl(Declarator(NULL, de), condexpr)),
				         *t
				       )
				     );
			}
			else               /* should not happen... */
			{
				ast_expr_free(condvar); ast_expr_free(condexpr); ast_expr_free(resexpr);
				_lock_based_atomic(t);
			}
		}
	}
	else                              /* atomic min/max using a CAS loop */
	{
		if (weak)
			_atomic_error(*t, ERR_WEAKEQ);

		if (same_expr(resvar, cond->left))
		{
			condvar  = cond->left;
			condexpr = cond->right;
		}
		else /* Swap condexpr and condvar and invert the operator */
		{
			condvar  = cond->right;
			condexpr = cond->left;
			cmpopid = cmpopid == BOP_lt ? BOP_gt : BOP_lt;
		}
		resvar = ast_expr_copy(resvar);
		resexpr = ast_expr_copy(resexpr);
		if (captureway) capturevar = ast_expr_copy(capturevar);
		ast_stmt_free(*t);
		*t = _cas_loop_cmp(resvar, resexpr, cmpopid, weak, memop, failop);
	}
}


static void _xform_atomic_update(aststmt *t, bool compare, bool capture, 
                              bool weak, ompclt_e memop, ompclsubt_e failop)
{
	astexpr e;
	int form;
	aststmt body = (*t)->u.omp->body, updstmt;
	semtype_t *tp;
	
	if (!compare && !capture)
	{
		_xform_atomic_update_plain(t, memop, failop);
		return;
	}
	
	form = _update_type(*t, compare, capture);
	switch (FORMTYPE(form))
	{
		/* 3 forms of x = a condop b ? expr : x; */
		case FORM_CUPEXP:
			e = body->u.expr->right->right;
			goto  CUP_EXP_IFS;
			
		/* 3 forms of if (a condop b) { x = expr; } */
		case FORM_CUPIFS:
			e = body->body->body->u.expr->left;
			
			CUP_EXP_IFS:
				/* Use fallback code for non-integral vars for now */
				tp = sem_expr_type(e);
				if (numtype_isintegral(tp->class))
					_xform_atomic_update_compare(t, weak, memop, failop);
				else
					_lock_based_atomic(t);
				free(tp);
				break;
	
		/* v = expr-stmt */
		case FORM_CAPEXPR:
			e = body->u.expr->right;
			capturestmt = NULL;              /* capture through expression */
			captureway = POSTCAPTURE;
			capturevar = body->u.expr->left;
			free(body->u.expr); /* Modify the expression to only contain the update */
			body->u.expr = e;

			tp = sem_expr_type(e);
			if (!numtype_isintegral(tp->class))
				_lock_based_atomic(t); /* Fallback code for non-integral vars for now */
			else
				if (EXPRESSIONTYPE(form) == EXPRESSIONWR)
					_xform_atomic_write(t, memop);
				else if (EXPRESSIONTYPE(form) == EXPRESSIONCU)
					_xform_atomic_update_compare(t, weak, memop, failop);
				else
				{
					if (e->type == POSTOP)
						captureway = PRECAPTURE;
					_xform_atomic_update_plain(t, memop, failop);
				}
			free(tp);
			break;
			
		/* { v = x; expr-stmt } (+ expression statement type) */
		case FORM_CAPBLK1:
			body = body->body;
			captureway = PRECAPTURE;
			capturestmt = body->u.next;
			capturevar = capturestmt->u.expr->left;
			updstmt = body->body;
			goto CAPBLK_1_2;
			
		/* { expr-stmt v = x; } (+ expression statement type) */
		case FORM_CAPBLK2:
			body = body->body;
			captureway = POSTCAPTURE;
			capturestmt = body->body;
			capturevar = capturestmt->u.expr->left;
			updstmt = body->u.next;
			
			CAPBLK_1_2:
				free(body);                   /* the statementlist */
				free((*t)->u.omp->body);      /* the compound */
				(*t)->u.omp->body = updstmt;  /* keep the update statement only */
				
				tp = sem_expr_type(updstmt->u.expr);
				if (!numtype_isintegral(tp->class))
					_lock_based_atomic(t); /* Fallback code for non-integrals for now */
				else
					if (EXPRESSIONTYPE(form) == EXPRESSIONWR)
						_xform_atomic_write(t, memop);
					else
						if (EXPRESSIONTYPE(form) == EXPRESSIONUP)
							_xform_atomic_update_plain(t, memop, failop);
						else
							_xform_atomic_update_compare(t, weak, memop, failop);
				free(tp);
				break;
			
		/* { v = x; cond-update-stmt } */
		case FORM_COMCAP1:  
			body = body->body;
			captureway = PRECAPTURE;
			capturestmt = body->u.next;
			capturevar = capturestmt->u.expr->left;
			free((*t)->u.omp->body->body);   /* the statment list */
			free((*t)->u.omp->body);         /* the compound */
			(*t)->u.omp->body = body->body;  /* just the if statement */
			
			tp = sem_expr_type(body->body->u.expr->left);
			if (!numtype_isintegral(tp->class))
				_lock_based_atomic(t); /* Fallback code for non-integrals for now */
			else
				_xform_atomic_update_compare(t, weak, memop, failop);
			free(tp);
			break;
			
		/* { cond-updte-stmt v = x; } */
		case FORM_COMCAP2:  
			body = body->body;
			captureway = POSTCAPTURE;
			capturestmt = body->body;
			capturevar = capturestmt->u.expr->left;
			free((*t)->u.omp->body->body);     /* the statment list */
			free((*t)->u.omp->body);           /* the compound */
			(*t)->u.omp->body = body->u.next;  /* just the if statement */
			
			tp = sem_expr_type(body->u.next->u.expr->left);
			if (!numtype_isintegral(tp->class))
				_lock_based_atomic(t); /* Fallback code for non-integrals for now */
			else
				_xform_atomic_update_compare(t, weak, memop, failop);
			free(tp);
			break;
			
		/* if () {...} else {...} */
		case FORM_COMCAP3: 
			capturestmt = body->u.selection.elsebody->body;
			casfalsecapture = true;
			captureway = POSTCAPTURE;
			capturevar = capturestmt->u.expr->left;
			free(body->u.selection.elsebody);   /* the compound */
			body->u.selection.elsebody = NULL;  /* no else branch */
			
			tp = sem_expr_type(body->body->body->u.expr->left);
			if (!numtype_isintegral(tp->class))
				_lock_based_atomic(t); /* Fallback code for non-integrals for now */
			else
				_xform_atomic_update_compare(t, weak, memop, failop);
			free(tp);
			break;

		/* { r = x == e; if (r) <no else> } */
		case FORM_COMCAP4:  
			body = body->body;
			capturestmt = body->u.next;
			casstatus = capturestmt->u.expr->left;  /* remember the CAS status var */
			body = body->body;
			
			tp = sem_expr_type(body->body->body->u.expr->left);
			if (!numtype_isintegral(tp->class))
				_lock_based_atomic(t); /* Fallback code for non-integrals for now */
			else 
			{
				ast_expr_free(body->u.selection.cond);  /* replace cond with "x==e" */
				body->u.selection.cond = capturestmt->u.expr->right;
				free(capturestmt);              /* the assignment */
				free((*t)->u.omp->body->body);  /* the statment list */
				free((*t)->u.omp->body);        /* the compound */
				(*t)->u.omp->body = body;       /* just the if statement */
				capturestmt = NULL;
				_xform_atomic_update_compare(t, weak, memop, failop);
			}
			free(tp);
			break;
	
		/* { r = x == e; if (r) then else } */
		case FORM_COMCAP5:  
			body = body->body;
			capturestmt = body->u.next;
			casstatus = capturestmt->u.expr->left;  /* remember the CAS status var */
			body = body->body;
			
			tp = sem_expr_type(body->body->body->u.expr->left);
			if (!numtype_isintegral(tp->class))
				_lock_based_atomic(t); /* Fallback code for non-integrals for now */
			else 
			{
				ast_expr_free(body->u.selection.cond);  /* replace cond with "x==e" */
				body->u.selection.cond = capturestmt->u.expr->right;
				free(capturestmt);              /* the assignment */
				free((*t)->u.omp->body->body);  /* the statment list */
				free((*t)->u.omp->body);        /* the compound */
				(*t)->u.omp->body = body;       /* just the if statement */
				
				capturestmt = body->u.selection.elsebody->body;
				casfalsecapture = true;
				captureway = POSTCAPTURE;
				capturevar = capturestmt->u.expr->left;
				free(body->u.selection.elsebody);   /* the compound */
				body->u.selection.elsebody = NULL;  /* no else branch */
				_xform_atomic_update_compare(t, weak, memop, failop);
			}
			free(tp);
			break;
		
		default:
			_atomic_error(*t, ERR_NCUPD);
			break;
	}
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *    MAIN TRANSFORMATION FUNCTION                               *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


 void _dbg_update_type(aststmt t, bool compare, bool capture)
{
	int form;
	
	if (!compare && !capture)
	{
		fprintf(stderr, "Plain atomic update\n");
		return;
	}
		
	form = _update_type(t, compare, capture);
	if (form == FORM_ERR)
		_atomic_error(t, _errid);
	fprintf(stderr, "Form: %d", FORMTYPE(form));
	if (EXPRESSIONTYPE(form))
		fprintf(stderr, " (expr-stmt is '%s')", 
			EXPRESSIONTYPE(form) == EXPRESSIONWR ? "write" :
			EXPRESSIONTYPE(form) == EXPRESSIONUP ? "update" : "cond-update");	
	if (CONDITIONTYPE(form))
		fprintf(stderr, " (condition type is '%s')", 
			CONDITIONTYPE(form) == CONDITIONEX ? "expr >|< x" :
			CONDITIONTYPE(form) == CONDITIONXE ? "x >|< expr" : "x == expr");
	fprintf(stderr, "\n");
}


/**
 * Transforms an OpenMP v5.2 #atomic construct
 * @param t pointer to the construct
 */
void xform_atomic(aststmt *t)
{
	int         atomicclause = 0, memclause = 0;
	bool        capture = false, compare = false, fail = false, weak = false;
	ompclt_e    atomicop, memop;
	ompclsubt_e failop;
	aststmt     v, parent = (*t)->parent;
	
	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */
	captureway = NOCAPTURE;
	capturestmt = NULL;
	capturevar = casstatus = NULL;
	casfalsecapture = false;
	
	/* ====> 1. First transform the body */
	ast_stmt_xform(&((*t)->u.omp->body));
	
	/* ====> 2. Get the clauses and check conformance */
	  /* atomic clauses */
	if (xc_ompcon_get_clause((*t)->u.omp, OCREAD))
	{
		atomicclause++;
		atomicop = OCREAD;
	}
	if (xc_ompcon_get_clause((*t)->u.omp, OCWRITE))
	{
		atomicclause++;
		atomicop = OCWRITE;
	}
	if (xc_ompcon_get_clause((*t)->u.omp, OCUPDATE))
	{
		atomicclause++;
		atomicop = OCUPDATE;
	}
	if (atomicclause == 0)
		atomicop = OCUPDATE;
	if (atomicclause > 1)
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
			"at most one of the READ/WRITE/UPDATE clauses is allowed.\n",
			(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);

	  /* memory order clauses */
	if (xc_ompcon_get_clause((*t)->u.omp, OCSEQCST))
	{
		memclause++;
		memop = OCSEQCST;
	}
	if (xc_ompcon_get_clause((*t)->u.omp, OCACQREL))
	{
		memclause++;
		memop = OCACQREL;
	}
	if (xc_ompcon_get_clause((*t)->u.omp, OCACQUIRE))
	{
		memclause++;
		memop = OCACQUIRE;
	}
	if (xc_ompcon_get_clause((*t)->u.omp, OCRELEASE))
	{
		memclause++;
		memop = OCRELEASE;
	}
	if (xc_ompcon_get_clause((*t)->u.omp, OCRELAXED))
	{
		memclause++;
		memop = OCRELAXED;
	}
	/* OpenMP 5.2, p. 312:
	 * If a memory-order clause is present, or implicitly provided by a 
	 * requires directive, it specifies the effective memory ordering. 
	 * Otherwise the effect is as if the relaxed memory ordering clause
	 * is specified.
	 */
	if (memclause == 0)
		memop = _required(OCATOMICDEFAULTMEMORDER) ? 
		                  memop_subtype2type(__req_memorder_value) : OCRELAXED;
	if (memclause > 1)
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
			"at most one of the SEQ_CST/ACQ_REL/ACQUIRE/RELEASE/RELAXED\n"
			"\tclauses is allowed.\n",
			(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	if (memop == OCRELEASE && atomicop == OCREAD)
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
			"RELEASE clause not allowed when the READ clause is used.\n",
			(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	if (memop == OCACQUIRE && atomicop == OCWRITE)
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
			"ACQUIRE clause not allowed when the WRITE clause is used.\n",
			(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	
	  /* extended atomic clauses */
	capture = xc_ompcon_get_clause((*t)->u.omp, OCCAPTURE) != NULL;
	compare = xc_ompcon_get_clause((*t)->u.omp, OCCOMPARE) != NULL;
	if ((capture || compare) && atomicop != OCUPDATE)
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
				"CAPTURE/COMPARE clauses not allowd when READ/WRITE clauses exist.\n",
				(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	
	  /* the hint() clause is ignored */
	fail = xc_ompcon_get_clause((*t)->u.omp, OCFAIL) != NULL;
	weak = xc_ompcon_get_clause((*t)->u.omp, OCWEAK) != NULL;
	if (( fail || weak) && atomicop != OCUPDATE && !capture)
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
				"FAIL/WEAK clauses not allowed when READ/WRITE clauses exist.\n",
				(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	if (fail)
		failop = xc_ompcon_get_clause((*t)->u.omp, OCFAIL)->subtype;
	else 
		failop = (memop == OCSEQCST)  ? OC_seq_cst :
		         (memop == OCACQUIRE) ? OC_acquire : OC_relaxed;
	
	/* ===> 3. Check body of construct based on the clauses used */
	if ((atomicop == OCREAD || atomicop == OCWRITE) && 
	    (*t)->u.omp->body->type != EXPRESSION)
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
				"the body of ATOMIC READ/WRITE construct must be an expression.\n",
				(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	if (atomicop == OCUPDATE && !capture && !compare && 
	    (*t)->u.omp->body->type != EXPRESSION)
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
				"the body of such an ATOMIC construct must be an expression.\n",
				(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	if (compare && !capture && (*t)->u.omp->body->type != EXPRESSION && 
	    ((*t)->u.omp->body->type != SELECTION || (*t)->u.omp->body->subtype!=SIF))
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
				"the body of ATOMIC construct with a COMPARE clause must be an "
				"\texpression or an if statement.\n",
				(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	if (!compare && capture && 
	    (*t)->u.omp->body->type != EXPRESSION && 
	    (*t)->u.omp->body->type != COMPOUND)
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
				"the body of ATOMIC construct with a CAPTURE clause must be an "
				"\texpression or a compound statement.\n",
				(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	if (compare && capture && 
	    (*t)->u.omp->body->type != EXPRESSION && 
	    (*t)->u.omp->body->type != COMPOUND &&
	    ((*t)->u.omp->body->type != SELECTION || (*t)->u.omp->body->subtype!=SIF))
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
				"the body of ATOMIC construct with COMPARE and CAPTURE clauses must be "
				"\tan expression, a compound or an if statement.\n",
				(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	if (fail && (!compare || (*t)->u.omp->body->type != SELECTION || 
	                         (*t)->u.omp->body->subtype != SIF))
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
				"the FAIL clause must be used with a conditional update statement.\n",
				(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	/* For WEAK we need to additionally check that the comparison operator 
	 * tests for equality 
	 */
	if (weak && !compare && !capture)
		exit_error(1, "(%s, line %d) openmp error on ATOMIC construct:\n\t"
				"the WEAK clause requires the COMPARE or the CAPTURE clause.\n",
				(*t)->u.omp->directive->file->name, (*t)->u.omp->directive->l);
	
	/* ===> 4. Now transform */
	if (!capture) 
		captureway = NOCAPTURE;
	if (atomicclause && atomicop == OCREAD)
		_xform_atomic_read(t, memop);
	if (atomicclause && atomicop == OCWRITE)
		_xform_atomic_write(t, memop);
	if (atomicop == OCUPDATE)
	{
		//_dbg_update_type(*t, compare, capture);
		_xform_atomic_update(t, compare, capture, weak, memop, failop);
	}
	*t = Block2(v, *t); /* Add comment */
	if (parent->type != STATEMENTLIST && parent->type != COMPOUND &&
	    (*t)->body->type != COMPOUND)
		*t = Compound(*t);
	(*t)->parent = parent;
}
