/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* x_single.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "ast_free.h"
#include "ast_xform.h"
#include "ast_copy.h"
#include "x_single.h"
#include "x_clauses.h"
#include "symtab.h"
#include "ompi.h"
#include "vulkan.h"
#include "ast_assorted.h"

/* Trivial implementation */
void xform_task_vulkan(aststmt *t)
{
	if (*t)
		(*t) = (*t)->u.omp->body;
}

/* possible clauses:
 *   private, firstprivate, copyprivate, nowait
 */
void xform_single_vulkan(aststmt *t)
{
	xform_ompcon_body((*t)->u.omp);

	aststmt   s = (*t)->u.omp->body, parent = (*t)->parent,
	          decls, inits = NULL;
	int       stlist;   /* See comment above */
	ompclause nw = xc_ompcon_get_clause((*t)->u.omp, OCNOWAIT);
	bool      needbarrier = (nw == NULL &&
	                         xform_implicit_barrier_is_needed((*t)->u.omp));

	/*
	 * Checks
	 */
	stlist = ((*t)->parent->type == STATEMENTLIST ||
	          (*t)->parent->type == COMPOUND);

	/*
	 * Preparations
	 */

	/* declarations from private/firstprivate vars */
	decls = xc_ompdir_declarations((*t)->u.omp->directive);
	/* initialization statments for firstprivate non-scalar vars */
	if (decls)
		inits = xc_ompdir_fiparray_initializers((*t)->u.omp->directive);
	/* checks for ditching the implicit barrier; if we have cp we MUST block */
	if (nw == NULL && !xform_implicit_barrier_is_needed((*t)->u.omp))
		nw = (ompclause) 123;       /* Non-null anyway */

	/*
	 * Do the job
	 */

	(*t)->u.omp->body = NULL;     /* Make it NULL so as to free it easily */
	ast_free(*t);                 /* Get rid of the OmpStmt */

	if (decls)
	{
		/* Check for a special case: we have decls & inits.
		 * In this case, the inits should be placed after the decls and *before*
		 * the body. But what if the body's first statements are declarations?
		 * Consequently, unless the body itself is a compound statement, we
		 * must enclose the body in another compound.
		 */
		if (inits)
			s = BlockList(BlockList(decls, inits),
			              (s->type != COMPOUND) ? Compound(s) : s);
		else
			s = BlockList(decls, s);
	}
	if (decls)
		s = Compound(s);

	*t = BlockList(
	        If(
	          BinaryOperator(BOP_neq, FunctionCall(IdentName("_ort_mysingle"),
	                       Constant(strdup(nw ? "1" : "0"))), numConstant(0)),
	          s,
	          NULL
	        ),
	        Call0_stmt("_ort_leaving_single")
	     );

	if (needbarrier)
		*t = BlockList(*t, BarrierCall(OMPBAR_IMPLICIT));
	else
		if (!nw)   /* We ditched the barrier; but should at least flush */
			*t = BlockList(*t, Call0_stmt("_ort_fence")); 

	if (!stlist)
		*t = Compound(*t);
	(*t)->parent = parent;
}


void xform_error_vulkan(aststmt *t)
{
	aststmt parent = (*t)->parent, v;
	ompclause c;
	ompclsubt_e time = OC_compilation, action = OC_fatal;  /* the defaults */
	
	if ((c = xc_ompcon_get_clause((*t)->u.omp, OCAT)))
		time = c->subtype;
	if ((c = xc_ompcon_get_clause((*t)->u.omp, OCSEVERITY)))
		action = c->subtype;
	c = xc_ompcon_get_clause((*t)->u.omp, OCMESSAGE);
	
	if (time == OC_compilation)
	{ 
		if (action == OC_fatal)
			exit_error(1, "[compilation aborted]: %.*s\n", 
			              c ? strlen(c->s->name)-2 : 20,  /* avoid the quotes */
			              c ? c->s->name+1 : "error directive met.");
		else
			warning("[warning]: %.*s\n", 
			        c ? strlen(c->s->name)-2 : 20,  /* avoid the quotes */
			        c ? c->s->name+1 : "error directive met.");
	}
	else
	{
		if (action == OC_fatal && xformingTarget) /* TODO: support in devices?? */
		{
			if (xformingForHOST)   /* Avoid re-printing it for all code targets */
				warning("(%s, line %d) warning:\n\tFatal #error is only supported in "
				        "the host currently; ignoring.\n",
				        (*t)->u.omp->directive->file->name, 
				        (*t)->u.omp->directive->l);
		}
		else
			if (action != OC_fatal)
				v = FuncCallStmt("debugPrintfEXT",
				      c ? Comma2(
				            String(strdup("\"[vulkan module warning]: \%s\\n\"")), 
				            String(strdup(c->s->name))
				          )
				        : String(strdup("\"[vulkan module warning]: an #error directive"
				                        " was executed\\n\""))
				    );
	}
	ast_free(*t);
	*t = v; /* We are done */
	(*t)->parent = parent;
}


void xform_critical_vulkan(aststmt *t)
{
	aststmt s, parent, v, decl, inittree;
	stentry e;
	char    shlock[128]; //lock[128], 
	bool    stlist;
	void (*sharedadjust)(aststmt) = codetarg_get_adjuster(CODETARGID(vulkan), ADJ_SHARED_STRUCT);

	/* First transform the body */
	ast_stmt_xform(&((*t)->u.omp->body));
	
	s = (*t)->u.omp->body;
	parent = (*t)->parent;
	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */
	stlist = ((*t)->parent->type == STATEMENTLIST ||
	          (*t)->parent->type == COMPOUND);

	/* A lock named after the region name */
	if ((*t)->u.omp->directive->u.region)
		snprintf(shlock,127,"_vk_ompi_crity_%s", (*t)->u.omp->directive->u.region->name);
	else
		strcpy(shlock, "_vk_ompi_crity");

	/* Add declaration to globals (includes check for duplicates) */
	decl = Declaration(
	         Usertype(Symbol("uint")),
	         Declarator(NULL, IdentifierDecl(Symbol(shlock)))
	       );

	if (sharedadjust)
		sharedadjust(decl); /* Add __shared__ qualifier */
	
	inittree = Block2(If(
	             BinaryOperator(BOP_eqeq, 
	               FunctionCall(IdentName("omp_get_thread_num"), NULL), 
	               numConstant(0)
	             ),
	             Expression(
	               Assignment(
	                 IdentName(shlock), ASS_eq, numConstant(0)
	               )
	             ),
	             NULL
	           ), FuncCallStmt("barrier", NULL));

	codetargs_kernel_add_global(decl, inittree, CODETARGID(vulkan));
	ast_stmt_free(decl);         /* They keep a copy */

	/* Remove __shared__ qualifier from every declaration (except for the global) */
	e = symtab_get(stab, Symbol(shlock), IDNAME);
	e->spec = Declspec(SPEC_int);

	(*t)->u.omp->body = NULL;    /* Make it NULL so as to free it easily */
	ast_free(*t);                /* Get rid of the OmpStmt */

	/* Produced code:
	 *   _vulkan_set_lock(&<intlock>);
	 *   <body> // critical section
	 *   _vulkan_unset_lock(&<intlock>);
	 */		 
	*t = Block3(
	       v,
	       Block3(
	         Expression(
	           FunctionCall(IdentName("_vulkan_set_lock"), 
	           IdentName(shlock))
	         ),
	         s,
	         Expression(
	           FunctionCall(IdentName("_vulkan_unset_lock"), 
	           IdentName(shlock))
	         )
	       ),
	       linepragma(s->l + 1 - (!stlist), s->file)
	     );

	if (!stlist)
		*t = Compound(*t);
	(*t)->parent = parent;
}
