/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* x_target_ocl.c -- transform OpenCL target constructs */

#include <string.h>
#include "ast_free.h"
#include "ast_xform.h"
#include "x_target.h"
#include "x_target_ocl.h"
#include "x_clauses.h"
#include "x_teams_ocl.h"
#include "outline.h"
#include "x_combine.h"
#include "codetargs.h"
#include "sem.h"

// #define DEVENV_DBG
#ifdef DEVENV_DBG
#include "ast_show.h"
#endif


void xform_targetparallel_opencl(aststmt *t)
{
	XFORM_CURR_DIRECTIVE->iscombpar = 1;

	/* This is called just to set the clause vars */
	xc_ompcon_search_offload_params((*(t))->u.omp);
	
	xform_ompcon_body((*t)->u.omp);
	_do_target_opencl(t);
}


void xform_targparfor_opencl(aststmt *t)
{
	ccc_split_ompstmts(t, DCTARGETPARALLEL, DCFOR);
	xform_targetparallel_opencl(t);
}


void xform_target_opencl(aststmt *t)
{
	int savecpl = cur_parallel_line;
	int savectgl = cur_taskgroup_line;
	cur_parallel_line = cur_taskgroup_line = 0;	

	/* This is called just to set the clause vars */
	xc_ompcon_search_offload_params((*(t))->u.omp);

	if (!search_nested_construct((*t)->u.omp, DCPARALLEL)
	    && !search_nested_construct((*t)->u.omp, DCDISTPARFOR))
	{
		#if PARALLEL_SCHEME == SCHEME_MASTERWORKER
		// masterworker_stmt(&((*t)->u.omp->body));
		#else
		// ifmaster_stmt(&((*t)->u.omp->body), false);
		#endif
	}
	else
		XFORM_CURR_DIRECTIVE->iscombpar = 1;
		
	xform_ompcon_body((*(t))->u.omp);
	_do_target_opencl(t);

	cur_parallel_line = savecpl;
	cur_taskgroup_line = savectgl;
}


/**
 * Wrapper function that simply handles defaultmap clauses and calls 
 * xtarget_implicitDefault for everything else.
 * @return The decided mapping attribute (i.e. the corresponding set to join).
 */
vartype_t xtarget_implicitDefault_ocl(setelem(vars) s, void *arg)
{
	struct { bool hasdefaultmap; } *impdefargs = arg;
	if (impdefargs->hasdefaultmap)
		return DCT_MAPTOFROM;   /* all treated as map(tofrom:) */
	return xtarget_implicitDefault(s, arg);
}


#if 0
/* Add stuff to the wrapper */
static void _wrapper_extras(aststmt wr)
{
	astdecl wrapperparams = wr->u.declaration.decl->decl->u.params;
	aststmt funcbody = wr->body->body, kfunccall; 
	
	/* Pass the device id in front of the normal arguments */
	wrapperparams = ParamList(
	                  ParamDecl(
	                    Declspec(SPEC_int),
	                    Declarator(NULL, IdentifierDecl(Symbol("_dev_id")))
	                  ),
	                  wrapperparams
	                );
#if OCL_PASS_LOCALMEM_AS_KERNEL_ARG 
	/* Pass local mem as an extra param */
	wrapperparams = ParamList(
	                  ParamDecl(
	                    Speclist_right(Usertype(Symbol("__local")), 
	                                   Declspec(SPEC_long)),
	                    Declarator(
	                      Declspec(SPEC_star), 
	                      IdentifierDecl(Symbol("_localmem"))
	                    )
	                  ), 
	                  wrapperparams
	                );
#else
	/* Declare local memory within the wrapper */
	ast_stmt_in_place_place_prepend(
	           funcbody,
	           Declaration(
	             Speclist_right(Usertype(Symbol("__local")), Declspec(SPEC_long)), 
	             Declarator(
	               NULL,
	               ArrayDecl(IdentifierDecl(Symbol("_localmem")), NULL, 
	                         numConstant(1024))
	             )
	           )
	         );
#endif

	wr->u.declaration.decl->decl->u.params = wrapperparams;
	
	/* Prepend two calls to set the device id and the local memory area */
	kfunccall = Block3(
	              FuncCallStmt("_ort_set_device_id", IdentName("_dev_id")),
	              FuncCallStmt("_ort_set_local_mem", IdentName("_localmem")),
	              kfunccall
	            );

}
#endif


void _do_target_opencl(aststmt *t)
{
	outcome_t oc;
	ompcon    ompc = (*t)->u.omp;
	kernel_t *kernel = codetargs_get_kernel_from_copy(ompc, xformingFor);
	bool hasdefaultmap =
		(xc_ompcon_get_clause((*t)->u.omp, OCDEFAULTMAP) != NULL);
	struct { bool hasdefaultmap; } impdefargs = { hasdefaultmap };

	/* 1) Outline
	 */
	static outline_opts_t op =
	{
		/* structbased             */  true,                   
		/* functionName            */  "test",                 
		/* functionCall  (func)    */  NULL,  
		/* byvalue_type            */  BYVAL_bycopy,           
		/* byref_type              */  BYREF_pointer,          
		/* byref_copyptr (2 funcs) */  NULL, NULL,             
		/* global_byref_in_struct  */  true,                   
		/* structName              */  "__dev_struct",         
		/* structVariable          */  DEVENV_STRUCT_NAME,     
		/* structInitializer       */  NULL,                   
		/* implicitDefault (func)  */  xtarget_implicitDefault_ocl,
		/* implicitDefault (args)  */  NULL,
		/* deviceexpr              */  NULL,                   
		/* addComment              */  true,                   
		/* thestmt                 */  NULL,
		/* userType                */  NULL,
		/* usePointers             */  true,
		/* makeReplCode            */  true,
		/* makeWrapper             */  true,
		/* wrapperType             */  WRAPPER_default
	};
	sprintf(op.functionName, "_kernelFunc%d_opencl", kernel->kid);
	op.structInitializer = NullExpr();
	op.deviceexpr = numConstant(DFLTDEV_ALIAS);  /* dummy, just != NULL */
	op.implicitDefault_args = &impdefargs;
	op.thestmt = *t;

	oc = outline_OpenMP(t, op);

	/* 2) Wrapper
	 */
	if (oc.func_struct)
		gpuize_struct(oc.func_struct, set_size(oc.usedvars[DCT_BYVALUE]));
	//_wrapper_extras(oc.wrapper);
	
	ast_free(op.thestmt);          /* Get rid of the OmpStmt */

	/* 3) Store the generated code
	 */
	ast_parentize(kernel->kfuncstmt[xformingFor]);
	analyze_pointerize_decltarg_varsfuncs(kernel->kfuncstmt[xformingFor]);
	kernel->kfuncname[xformingFor] = strdup(op.functionName);

	// The wrapper statement goes last
	kernel->kfuncstmt[xformingFor] = BlockList(kernel->kfuncstmt[xformingFor], 
	                                           oc.wrapper);
}
