/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* x_assorted_ocl.c -- transform assorted OpenCL constructs */

#include "x_assorted_ocl.h"
#include "ast_xform.h"
#include "ast_free.h"
#include "ast_copy.h"
#include "ast.h"
#include "ompi.h"
#include "builder.h"
#include "ast_assorted.h"
#include "ast_arith.h"
#include "x_target.h"
#include "x_clauses.h"
#include "codetargs.h"
#include "opencl.h"
#include <string.h>


static bool _do_atomic = false;


#if 0
	   { 
	     int _stop = 0;
	     while (!_stop) {
	       if (atomic_xchg(&<intlock>, 1) == 0) {
	         <critical section>
	         _stop = 1;
	         atomic_xchg(&<intlock>, 0);
	       };
	     }
  	 }
#endif



/* Unfortunately it won't work because a kernel argument (the lock which 
 * is mapped by other code targets).
 */
void xform_critical_opencl(aststmt *t)
{
	aststmt body, parent, v, decl;
	char    lock[128];
	bool    stlist;
	int     bodyline;

	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */

	if ((*t)->u.omp->body == NULL)
	{
		ast_free(*t);        /* Get rid of the OmpStmt */
		*t = v;              /* Jut replace by a comment */
		return;
	}
		
	/* First transform the body */
	bodyline = (*t)->u.omp->body->l;
	ast_stmt_xform(&((*t)->u.omp->body));
	
	body = (*t)->u.omp->body;
	parent = (*t)->parent;
	stlist = ((*t)->parent->type == STATEMENTLIST ||
	          (*t)->parent->type == COMPOUND);

	snprintf(lock, 127, "%s",
	         (_do_atomic) ? "_ocl_atomicy" :  /* just use another lock for atomic */
	                      ((*t)->u.omp->directive->u.region ? 
	                         (*t)->u.omp->directive->u.region->name : "_ocl_crity"));

	/* Add declaration to globals (includes check for duplicates) */
	decl = Declaration(
	         Declspec(SPEC_int),
	         InitDecl(
	           Declarator(NULL, IdentifierDecl(Symbol(lock))),
	           ZeroExpr()
	         )
	       );
	codetargs_kernel_add_global(decl, NULL, CODETARGID(opencl));
	ast_stmt_free(decl);         /* They keep a copy */

	(*t)->u.omp->body = NULL;    /* Make it NULL so as to free it easily */
	ast_free(*t);                /* Get rid of the OmpStmt */
	
	/* Produced code:
	 *    for (volatile int _stop = 0; !stop; )
	 *      if (atomic_xchg(&<intlock>, 1) == 0) {
	 *        <body> // critical section
	 *        _stop = 1;
	 *        mem_fence(GLOBAL);
	 *        atomic_xchg(&<intlock>, 0);
	 *      };
	*/
	body = 
		For(
		  Declaration(
				Speclist_right(Declspec(SPEC_volatile), Declspec(SPEC_int)),
		    InitDecl(Declarator(NULL, IdentifierDecl(Symbol("_stop"))), ZeroExpr())
		  ),
		  UnaryOperator(UOP_lnot, IdentName("_stop")),
		  NULL,
		  If(
		    BinaryOperator(BOP_eqeq,Constant("atomic_xchg(&%s,1)",lock),ZeroExpr()),
		    Compound(
		      Block5(
		        linepragma(bodyline, body->file),
		        body,        /* body goes here */
		        Expression(Assignment(IdentName("_stop"), ASS_eq, numConstant(1))),
						verbit("mem_fence(CLK_GLOBAL_MEM_FENCE);"),
		        verbit("atomic_xchg(&%s, 0);", lock)
	        )
		    ),
		    NULL
		  )
		);

	*t = Block3(v, body, linepragma(bodyline + 1 - (!stlist), body->file));
	if (!stlist)
		*t = Compound(*t);
	(*t)->parent = parent;
}


/* For now implement atomic using lcoks (i.e. through critical) */
void xform_atomic_opencl(aststmt *t)
{
	_do_atomic = true;
	xform_critical_opencl(t);
	_do_atomic = false;
}


/**
 * This is a special version od flr_privatize for OpenCL.
 * Creates a local variable with the same name (with an optional initializer), 
 * plus a copy of/pointer to the original one. I.e. the following:
 *   __global <spec> varbak = var;
 *   <spec>
 * or
 *   <spec> *varbak = &var, var [ = initer ]; 
 *
 * @param var       The original variable whose declaration we want to reproduce
 * @param varbak    The copy of/pointer to the original variable
 * @param isptr     A flag for making varbak a pointer or not
 * @param initer    An optional initializer for the new local variable
 * @return          A statement with the declaration
 */
aststmt flr_privatize_opencl(symbol var, symbol vbak, int isptr, astexpr initer)
{
	astdecl bakdecl, id, localvar;
	stentry e = symtab_get(stab, var, IDNAME);
	
	bakdecl = ast_decl_copy(e->decl);      /* Make the varbak */
	id = IdentifierDecl(vbak);
	*(decl_getidentifier(bakdecl)) = *id;
	free(id);
	if (isptr)
		bakdecl = InitDecl( decl_topointer(bakdecl),UOAddress(Identifier(var)) );
	else
		bakdecl = InitDecl( bakdecl, Identifier(var) );

	localvar = xform_clone_declonly(e);    /* Make the local var declarator */ 
	if (initer)
		localvar = InitDecl(localvar, initer);
	
	/* __global <spec> *varbak = &var; <spec> var = <initializer>; */
	return 
		BlockList(
			Declaration(
				Speclist_right(
					Usertype(Symbol("__global")), 
					ast_spec_copy_nosc(e->spec)
				), 
				bakdecl
			),
			Declaration(ast_spec_copy_nosc(e->spec), localvar)
		);
}


/* Trivial implementation */
void xform_task_opencl(aststmt *t)
{
	if (*t)
		(*t) = (*t)->u.omp->body;
}


/* Output directly the error string (no runtime calls) */
void xform_error_opencl(aststmt *t)
{
	aststmt parent = (*t)->parent, v;
	ompclause c;
	ompclsubt_e time = OC_compilation, action = OC_fatal;  /* the defaults */

	v = ompdir_commented((*t)->u.omp->directive);

	if ((c = xc_ompcon_get_clause((*t)->u.omp, OCAT)))
		time = c->subtype;
	if ((c = xc_ompcon_get_clause((*t)->u.omp, OCSEVERITY)))
		action = c->subtype;
	c = xc_ompcon_get_clause((*t)->u.omp, OCMESSAGE);
	
	if (time == OC_compilation)
	{ 
		if (action == OC_fatal)
			exit_error(1, "[compilation aborted]: %.*s\n", 
			              c ? strlen(c->s->name)-2 : 20,  /* avoid the quotes */
			              c ? c->s->name+1 : "error directive met.");
		else
			warning("[warning]: %.*s\n", 
			        c ? strlen(c->s->name)-2 : 20,  /* avoid the quotes */
			        c ? c->s->name+1 : "error directive met.");
	}
	else
		if (action != OC_fatal)
			v = BlockList(
			      v, 
			      FuncCallStmt("printf",
			        c ? Comma2(
			              String(strdup("\"[opencl module warning]: \%s\\n\"")), 
			              String(strdup(c->s->name))
			            )
			          : String(strdup("\"[opencl module warning]: an #error directive"
			                          " was executed\\n\""))
			      )
			    );

	ast_free(*t);
	*t = v; /* We are done */
	(*t)->parent = parent;
}
