/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* generic.c -- the generic code target */

#include <string.h>
#include "ast.h"
#include "ast_free.h"
#include "ast_assorted.h"
#include "ast_copy.h"
#include "ast_show.h"
#include "codetargs.h"
#include "ast_xformrules.h"
#include "ast_xform.h"
#include "x_target.h"
#include "x_combine.h"
#include "x_decltarg.h"
#include "x_clauses.h"
#include "outline.h"
#include "generic.h"


/*
 * TRANSFORMATIONS
 */


void xform_critical_generic(aststmt *t)
{
	aststmt s, parent, v, decl;
	char    lock[128];
	bool    stlist;   /* See comment above */
	
	/* First transform the body */
	ast_stmt_xform(&((*t)->u.omp->body));

	s = (*t)->u.omp->body;
	parent = (*t)->parent;
	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */
	stlist = ((*t)->parent->type == STATEMENTLIST ||
	          (*t)->parent->type == COMPOUND);

	/* A lock named after the region name */
	if ((*t)->u.omp->directive->u.region)
		snprintf(lock,127,"_gen_crity_%s", (*t)->u.omp->directive->u.region->name);
	else
		strcpy(lock, "_gen_crity");

	/* Add declaration to globals (includes check for duplicates) */
	decl = Declaration(
	         Declspec(SPEC_void),
	         InitDecl(
	           Declarator(Pointer(), IdentifierDecl(Symbol(lock))),
	           NullExpr()
	         )
	       );
	codetargs_kernel_add_global(decl, NULL, CODETARGID(generic));
	ast_stmt_free(decl);         /* They keep a copy */

	(*t)->u.omp->body = NULL;    /* Make it NULL so as to free it easily */
	ast_free(*t);                /* Get rid of the OmpStmt */

	*t = 
		Block5(
			v,
			FuncCallStmt("_ort_critical_begin", UOAddress(IdentName(lock))),
			s,
			FuncCallStmt("_ort_critical_end", UOAddress(IdentName(lock))),
			linepragma(s->l + 1 - (!stlist), s->file)
		);

	if (!stlist)
		*t = Compound(*t);
	(*t)->parent = parent;
}



static void _omp_target_generic(aststmt *t)
{
	outcome_t oc;
	kernel_t *kernel = &__kernels[xformingKernelID];
	bool hasdefaultmap =
		(xc_ompcon_get_clause((*t)->u.omp, OCDEFAULTMAP) != NULL);
	struct { bool hasdefaultmap; } impdefargs = { hasdefaultmap };

	/* 1) Outline
	 */
	static outline_opts_t op =
	{
		/* structbased             */  true,                   
		/* functionName            */  "test",                 
		/* functionCall  (func)    */  NULL,  
		/* byvalue_type            */  BYVAL_bycopy,           
		/* byref_type              */  BYREF_pointer,          
		/* byref_copyptr (2 funcs) */  NULL, NULL,             
		/* global_byref_in_struct  */  true,                   
		/* structName              */  "__dev_struct",         
		/* structVariable          */  DEVENV_STRUCT_NAME,     
		/* structInitializer       */  NULL,                   
		/* implicitDefault (func)  */  xtarget_implicitDefault,
		/* implicitDefault (args)  */  NULL,
		/* deviceexpr              */  NULL,                   
		/* addComment              */  true,                   
		/* thestmt                 */  NULL,
		/* userType                */  NULL,
		/* usePointers             */  true,
		/* makeReplCode            */  true,
		/* makeWrapper             */  true,
		/* wrapperType             */  WRAPPER_defstructs
	};
	
	sprintf(op.functionName, "_kernelFunc%d_", kernel->kid);
	op.structInitializer = NullExpr();
	op.deviceexpr = numConstant(DFLTDEV_ALIAS);  /* dummy, just != NULL */
	op.implicitDefault_args = &impdefargs;
	op.thestmt = *t;

	oc = outline_OpenMP(t, op);

	/* 2) Wrapper
	 */
	if (oc.func_struct)
		gpuize_struct(oc.func_struct, set_size(oc.usedvars[DCT_BYVALUE]));

	ast_free(op.thestmt);          /* Get rid of the OmpStmt */

	/* 3) Store the generated code
	 */
	ast_parentize(kernel->kfuncstmt[xformingFor]);
	analyze_pointerize_decltarg_varsfuncs(kernel->kfuncstmt[xformingFor]);

	kernel->kfuncstmt[xformingFor] = BlockList(kernel->kfuncstmt[xformingFor], oc.wrapper);
	kernel->kfuncname[xformingFor] = strdup(op.functionName);
}


void xform_target_generic(aststmt *t)
{
	int savecpl = cur_parallel_line;
	int savectgl = cur_taskgroup_line;
	cur_parallel_line = cur_taskgroup_line = 0;	

	/* This is called just to set the clause vars */
	xc_ompcon_search_offload_params((*(t))->u.omp);

	xform_ompcon_body((*(t))->u.omp);
	_omp_target_generic(t);

	cur_parallel_line = savecpl;
	cur_taskgroup_line = savectgl;
}


void xform_targetparallel_generic(aststmt *t)
{
	ccc_try_splitting(t);
	xform_target_generic(t);
}


void xform_targetteams_generic(aststmt *t)
{
	int savecpl = cur_parallel_line;
	int savectgl = cur_taskgroup_line;
	cur_parallel_line = cur_taskgroup_line = 0;
	
	/* This is called just to set the clause vars */
	xc_ompcon_search_offload_params((*(t))->u.omp);

	/* (1) Split to target + teams, if possible */
	if (ccc_try_splitting(t))
		xform_ompcon_body((*t)->u.omp);

	xform_ompcon_body((*t)->u.omp); /* TODO: one more xform???? */
	_omp_target_generic(t);

	cur_parallel_line = savecpl;
	cur_taskgroup_line = savectgl;
}


void xform_targetteamsdistparfor_generic(aststmt *t)
{
	//ccc_try_splitting(t);
	xform_targetteams_generic(t);
}


static xfr_t _generic_xfr[] = {
	{ DCTARGET,           xform_target_generic,           XFR_ACTION_NONE },
	{ DCTARGETPARALLEL,   xform_targetparallel_generic,   XFR_ACTION_SPLIT },
	{ DCTARGETTEAMS,      xform_targetteams_generic,      XFR_ACTION_COMBINE },
	{ DCTARGETTEAMSDIST,  xform_targetteams_generic,      XFR_ACTION_SPLIT },
	{ DCTARGETTEAMSDISTPARFOR, xform_targetteamsdistparfor_generic, 
	                                                      XFR_ACTION_NONE },
	{ DCCRITICAL,         xform_critical_generic,         XFR_ACTION_NONE },
	XFR_LASTRULE
};


/*
 * CODE TARGET
 */


aststmt _generic_filterfunc(symbol fsym)
{
	if (fsym == Symbol("memset"))
		return verbit("void *memset(void *, int, unsigned long int); ");
	/* All other cases: just return a copy of the prototype */
	return xform_clone_funcdecl(fsym);
}

int CODETARGID(generic) = -1;

/* If we are to support reductions on long/float/double we need to define 
   the following (e.g. in a header):
   LONG_MIN
   LONG_MAX
   ULONG_MAX
   FLT_MAX
   LDBL_MAX
   DBL_MAX
static char *_generic_kernel_header = NULL;
*/

static char *_generic_kbinsuffixes[] = { "-proc2.out", NULL };

void __codetarg_generic_init() 
{
	/* Use the default (host) rules for the generic code target */
	codetarg_set_xformrules(CODETARGID(generic), _generic_xfr);
	codetarg_set_kernelfiles_suffix(CODETARGID(generic), ".c");
	codetarg_set_kernelbins_suffixes(CODETARGID(generic), _generic_kbinsuffixes);
	codetarg_set_filterfunc(CODETARGID(generic), _generic_filterfunc);
}
