/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* codetargs.h -- code targets (for OpenMP devices) */

#ifndef __CODETARGS_H__
#define __CODETARGS_H__

#include "ast.h"
#include "callgraph.h"
#include "str.h"
#include "ast_xformrules.h"
#include "x_reduction.h"
#include "x_cars.h"

/*------
  Just add your code targets in (1) and (2) below -- nothing else is needed
  ------*/

#define ADD_CODETARG(m) \
	extern int __codetarg_##m##_id; \
	extern void __codetarg_##m##_init();

	/* (1)
	 * Here add all targetted devices; you need to have global variable 
	 * initialized to a negative value: 
	 *      int __codetarg_<name>_id = -1;
	 *    and a function:
	 *      void __codetarg_<name>_init(); 
	 *    defined somewhere (anywhere). <name> is the symbolic name to use
	 *    for the targetted device, e.g. cuda, opencl.
	 *    
	 * IMPORTANT: because checks related to code targets may exist all over the 
	 *            compiler code, make sure **ALL** of them are added here; 
	 *            simply do not ACTIVATE below the ones you don't want
	 */
ADD_CODETARG(host)          /* The host CPU */
ADD_CODETARG(generic)       /* The generic code target */
ADD_CODETARG(cuda)
ADD_CODETARG(opencl)
ADD_CODETARG(vulkan)
/* new-codetarg.sh:addct */
	
	/* (2)
	 * Add the targetted devices here too, so they can be activated; you can 
	 * safely omit any of the added one in (1).
	 */
#define CHECK_AND_ACTIVATE_CODETARGS \
	CHECK_AND_ACTIVATE_CODETARG(cuda); \
	CHECK_AND_ACTIVATE_CODETARG(opencl); \
	CHECK_AND_ACTIVATE_CODETARG(vulkan); \
/* new-codetarg.sh:checkandactivatect */ ;

/*------*/

/* CODETARGID(host) is 0; reserved for the fallback host CPU "device" */
#define CODETARGID(m)     __codetarg_##m##_id        /* ID from symbolic name */

extern int __codetarg_num;   /* The number of code targets (>= 1) */

#define SYMEND "SYMBOLS_END" /* SUE filter-related */

typedef enum adjfunct_ {
	ADJ_FUNCTION = 0,  ADJ_VARDECL_STMT, ADJ_DECLARATION, ADJ_GLOBALS,      
	ADJ_SHARED_STRUCT, ADJ_KERNEL_FUNC,  ADJ_TOPCOMMENT,
	ADJ_SIZE // should always be last
} adjfunct_e;

typedef struct {
	int         kid;           /* The ID of the kernel */
	str         uid;           /* A unique ID of the kernel */
	ompcon      tnode;         /* The original (host) #target AST node */
	aststmt     *tnodecopies;  /* Per-codetarg copy of the construct */
	set(cgfun)  kcalledfuncs;  /* The list of funcs called by the kernel */
 
	char        **kfuncname;   /* Per-codetarg kernel function name */
	aststmt     *kfuncstmt;    /* Per-codetarg kernel function statement */
	aststmt     *kfxfuncs;     /* Extra funcs gen'd (outlined) by kfuncs */
	aststmt     *kxglobs;      /* Per-codetarg extra global vars (unmapped) */
	aststmt     *kxglobinits;  /* Per-codetarg extra global var initializations */

	str         *kfilename;    /* Per-codetarg kernel filename */
	targstats_t *ts;           /* CARS */

	aststmt     rep_struct;    /* Parts of the kernel func filled at kfile time */
	astexpr     decl_struct;   /* Only used by the host's replacement code */
} kernel_t;

extern kernel_t *__kernels;
extern int __kernels_num;

extern void codetargs_init(char *modulenames);
extern char *codetarg_name(int id);   /* Name string from ID */
extern int  codetarg_id(char *codetargname);   /* ID from name string */
extern int  codetarg_id_len(char *codetargname, int len);
extern void codetarg_set_cmdarg_handler(int id, int (*func)(char*));
extern int  codetarg_cmdarg(int id, char *arg);
extern void codetarg_set_xformrules(int id, xfr_t *r);
extern xfr_t *codetarg_get_xformrule(int id, ompdirt_e directive);
extern void codetarg_set_reduction_style(int id, redcodestyle_e rcs);
extern redcodestyle_e codetarg_get_reduction_style(int id);
extern void codetarg_set_kernelfiles_header(int id, char *s);
extern void codetarg_set_kernelfiles_suffix(int id, char *s);
extern char *codetarg_get_kernelfiles_suffix(int id);
extern void codetarg_set_kernelbins_suffixes(int id, char **s);
extern char **codetarg_get_kernelbins_suffixes(int id);

extern void codetarg_set_adjuster(int id, adjfunct_e type, void (*func)());
extern void *codetarg_get_adjuster(int id, adjfunct_e type);

extern void codetarg_set_filterfunc(int id, aststmt (*func)(symbol));

extern void codetargs_kernel_add_outfunc(ompcon t, int id, aststmt f);
extern void codetargs_kernel_add_global(aststmt g, aststmt init, int ctid);
extern aststmt codetargs_kernel_get_global(int kid, int ctid, symbol g);
extern void codetargs_kernel_set_kernelfunc(ompcon t, int id, aststmt f);
extern void codetargs_kernel_add(ompcon t);
extern void codetargs_kernels_prepare(void);
extern void codetargs_kernels_free(void);
extern int codetargs_kernels_getid(ompcon t);
extern aststmt codetargs_kernel_getstmt(ompcon t, int id);
extern void codetargs_kernel_updatestmt(ompcon t, int id, aststmt s);
extern void codetargs_kernel_get_kfuncname(int kdid);
 
extern bool codetargs_function_is_offloaded(symbol fn);
extern void codetargs_userfuncdef_add(aststmt t);
extern aststmt codetargs_get_userfuncdef(int id, symbol fsym);
extern void codetargs_userfunc_add_outfunc(int id, symbol fsym, aststmt fd,
                                           symbol curfunc);

extern kernel_t *codetargs_get_kernel(ompcon t);
extern kernel_t *codetargs_get_kernel_from_copy(ompcon t, int id);
extern int codetargs_kernel_getid(ompcon t);

extern void codetargs_kernels_generate_kernelfiles(void);
extern void codetargs_kernels_add_bubins_autoinits(void);
extern aststmt codetargs_kernels_bubins_tree(void);
#endif
