/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* cfg.c -- control flow graph module */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include "ompi.h"
#include "ast_print.h"
#include "ast_arith.h"
#include "cfg.h"
#include "set.h"
#include "x_clauses.h"

/* Helper nodes are the terminal ones plus the joiners */
#define CFG_HELPER_NODE(n) ((n)->type==CFG_TERMINAL || (n)->type==CFG_JOINER)

#define indegree(n)  ((n)->npred)
#define outdegree(n) ((n)->nsucc)
#define succ0(n)     ((n)->succs[0])
#define succ1(n)     ((n)->succs[1])

/* Construction from the AST */
typedef struct ast2cfg_opt_s ast2cfg_opt_t;
extern cfgnode cfg_from_ast(aststmt tree, ast2cfg_opt_t opts);
extern cfgnode _current_omp_construct;

/* Traversal */
/* walk direction: fwd (through successors), rev (through predecessors), 
 *                 undir (through both successors and predecessors)
 * visit order: preorder, postorder, midorder (only used in walk_undir; visits
 *                 the node after successors and before predecessors)
 */
typedef enum { walk_fwd, walk_rev, walk_undir } walkdir_e;
typedef enum { vis_preord = 0, vis_postord, vis_midord } visord_e;
typedef void cfg_nodefunc_t(cfgnode);
typedef int  cfg_searchfunc_t(cfgnode);

extern void cfg_traverse(cfgnode n, cfg_nodefunc_t ndf, visord_e ord,
                         int (*filter)(cfgnode), walkdir_e wd);
extern cfgnode cfg_search(cfgnode n, cfg_searchfunc_t ndf, visord_e ord,
                         int (*filter)(cfgnode), walkdir_e wd);

/* Utilities */
extern int  cfg_code_escapes(aststmt tree);
extern void cfg_show(cfgnode n);
extern int  cfg_is_cutnode(cfgnode root, cfgnode n);

/* Sets */
SET_TYPE_IMPLEMENT(cfg)
SET_TYPE_IMPLEMENT(lbl)


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * CFG NODES AND EDGES                                               *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* This is only used by cfg_from_ast(); too lazy to make it more modular */
static set(cfg) cfgallnodes;

#define CfgPlain(t)       Cfgnode(t, CFG_PLAIN)
#define CfgTerminal()     Cfgnode(NULL, CFG_TERMINAL)
#define CfgJoiner()       Cfgnode(NULL, CFG_JOINER)
#define CfgCBlockStart(t) Cfgnode(t, CFG_CBLOCKS)
#define CfgCBlockEnd(t)   Cfgnode(t, CFG_CBLOCKE)


static
cfgnode Cfgnode(aststmt t, cfgtype_e type)
{
	cfgnode n = smalloc(sizeof(struct cfgnode_));
	n->astnodetype = STMTNODE;    /* Default */
	n->stmt = t;
	n->type = type;
	n->nsucc = 0;
	n->npred = 0;
	n->succs = NULL;
	n->preds = NULL;
	n->cfgptr = NULL;
	n->ompparent = _current_omp_construct;
	n->stacknext = NULL;
	set_put(cfgallnodes, n);
	return (n);
}


static
cfgnode CfgnodeExpr(cfgnode n, astexpr e)
{
	n->astnodetype = EXPRNODE;
	n->expr = e;
	return (n);
}


static
void cfgnode_free(cfgnode n)
{
	if (outdegree(n) > 0)
		free(n->succs);
	free(n);
}


/* Checks if to is a successor of n */
static
int cfg_is_successor(cfgnode n, cfgnode to)
{
	int i;

	if (n && to)
		for (i = 0; i < outdegree(n); i++)
			if (n->succs[i] == to)
				return (1);
	return (0);
}


/* This adds a new back edge (from-->to) */
static
void cfg_add_backedge(cfgnode from, cfgnode to)
{
	if (!from)
		return;
	if ((indegree(from) & 3) == 0)    /* multiple of 4 */
	{
		cfgnode *a = (cfgnode *)
		             ((indegree(from) == 0) ?
		              malloc(4 * sizeof(cfgnode)) :
		              realloc(from->preds, (indegree(from) + 4) * sizeof(cfgnode)));
		if (a == NULL)
			exit_error(1, "cfg_add_backedge() cannot allocate memory.\n");
		from->preds = a;
	}
	from->preds[ indegree(from)++ ] = to;
}


/* This deletes the back edge from n to to if it exists */
static
void cfg_del_backedge(cfgnode n, cfgnode to)
{
	int i;

	if (n && to)
		for (i = 0; i < indegree(n); i++)
			if (n->preds[i] == to)
			{
				for (; i < indegree(n) - 1; i++)
					n->preds[i] = n->preds[i + 1];
				(indegree(n))--;
				break;
			};
}


/* This replaces the (back) edge from n to to by one from n to newto */
static
void cfg_rep_backedge(cfgnode n, cfgnode to, cfgnode newto)
{
	int i;

	if (n && to)
		for (i = 0; i < indegree(n); i++)
			if (n->preds[i] == to)
			{
				n->preds[i] = newto;
				break;
			};
}

 
#define cfg_add_successor(from,to) cfg_add_edge(from,to,true)
#define cfg_add_successor_noback(from,to) cfg_add_edge(from,to,false)
 
/* This adds a new edge from --> to (and maybe the back edge) */
static
void cfg_add_edge(cfgnode from, cfgnode to, int backedgetoo)
{
	if ((outdegree(from) & 3) == 0)    /* multiple of 4 */
	{
		cfgnode *a = (cfgnode *)
		             ((outdegree(from) == 0) ?
		              malloc(4 * sizeof(cfgnode)) :
		              realloc(from->succs, (outdegree(from)+4) * sizeof(cfgnode)));
		if (a == NULL)
			exit_error(1, "cfg_add_successor() cannot allocate memory.\n");
		from->succs = a;
	}
	from->succs[ outdegree(from)++ ] = to;
	if (to && backedgetoo)
		cfg_add_backedge(to, from);
}


#define cfg_del_successor(from,to) cfg_del_edge(from,to,true)
#define cfg_del_successor_noback(from,to) cfg_del_edge(from,to,false)
 
/* Not tested exhaustively */
/* This deletes the edge n --> to if it exists (and maybe the back edge) */
static
void cfg_del_edge(cfgnode n, cfgnode to, int backedgetoo)
{
	int i;

	if (!n) return;
	for (i = 0; i < outdegree(n); i++)
		if (n->succs[i] == to)
		{
			for (; i < outdegree(n) - 1; i++)   /* shift back the rest */
				n->succs[i] = n->succs[i + 1];
			(outdegree(n))--;
			if (to && backedgetoo)
				cfg_del_backedge(to, n);
			break;
		};
}


/* Not tested exhaustively */
/* Contract a node, i.e. remove it PLUS add its successors to its predecessors*/
static
void cfg_contract_node(cfgnode n)
{
	int i, j;

	for (i = 0; i < indegree(n); i++)
		if (n->preds[i] && n->preds[i] != n)   /* Avoid self-loops */
		{
			cfg_del_successor_noback(n->preds[i], n);
			for (j = 0; j < outdegree(n); j++)
				if (n->succs[j] && n->succs[j] != n)  /* Avoid self-loops */
				{
					cfg_del_backedge(n->succs[j], n);
					cfg_add_successor(n->preds[i], n->succs[j]);
				};
		};
	cfgnode_free(n);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * CFG VISITING                                                      *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/*
 * Visit the CFG:
 *   a) Use cfg_traverse() if you want to apply a function to every node
 *   b) Use cfg_search() if you want to visit till some condition is met
 * In both cases, if the CFG changes along the node visits, we cannot 
 * guarantee anything. However, modifications should work more or less
 * OK if you visit postorder.
 */


static set(cfg)  visited;
static int       (*nodefilter)(cfgnode s);

#define cfg_traverse_pred_edges(n)\
	for (i = indegree(n) - 1; i >= 0; i--)\
		cfg_traverse_node((n)->preds[i], ndf, ord, wdir);
#define cfg_search_pred_edges(n, res)\
	for (i = indegree(n) - 1; i >= 0; i--)\
		if ((res = cfg_search_node((n)->preds[i], ndf, ord, wdir)) != NULL)\
			return res;


/**
 * @brief This creates a free-able copy of the successor array of the node
 * 
 * While traversing, there may be situations where neighbors of the current
 * node change (e.g. in contraction). Consequently, we cannot blindly loop 
 * over neighbors of the node; we have to backup them and loop through the
 * backup.
 */
static
cfgnode *succsarray_dup(cfgnode n)
{
	cfgnode *b;

	if (!n || n->nsucc == 0) return NULL;
	if ((b = malloc(n->nsucc * sizeof(cfgnode))) == NULL) return NULL;
	return memcpy(b, n->succs, outdegree(n)*sizeof(cfgnode));
}


static
void cfg_traverse_node(cfgnode n,cfg_nodefunc_t ndf,visord_e ord,walkdir_e wdir)
{
	int i;

	if (n == NULL)
		return;
	if (set_get(visited, n) != NULL)
		return;
	if (nodefilter)
		if (nodefilter(n))   /* If true, filter this node out */
			return;
			
	/* Record that we traversed this node (nothing to do with the visit order) */
	set_put(visited, n);
	
	if (ord == vis_preord && ndf)    /* Visit order dictates when to apply ndf */
		(*ndf)(n);
	
	if (wdir == walk_fwd || wdir == walk_undir)
	{
		if (outdegree(n) <= 2)
		{
			/* Remember it because it may change */
			cfgnode second = (outdegree(n) > 1) ? succ1(n) : NULL;
			
			if (outdegree(n) > 0)
				cfg_traverse_node(succ0(n), ndf, ord, wdir);
			if (second)
				cfg_traverse_node(second, ndf, ord, wdir);
		}
		else
		{
			cfgnode *succs = succsarray_dup(n);  /* Remember them */
			
			for (i = outdegree(n) - 1; i >= 0; i--)
				cfg_traverse_node(succs[i], ndf, ord, wdir);
			free(succs);
		}
	}
	
	if (ord == vis_midord && ndf)
		(*ndf)(n);
	
	if (wdir == walk_rev || wdir == walk_undir) /* TODO: backup here too? */
		cfg_traverse_pred_edges(n);
		
	if (ord == vis_postord && ndf)
		(*ndf)(n);
}


#define cfg_search_and_return(a,b,c,d,res) \
          if ((res = cfg_search_node(a,b,c,d)) != NULL) \
            return res;

/* TODO: Completely untested */
static cfgnode 
cfg_search_node(cfgnode n, cfg_searchfunc_t ndf, visord_e ord, walkdir_e wdir)
{
	int i;
	cfgnode res;

	if (n == NULL)
		return NULL;
	if (set_get(visited, n) != NULL)
		return NULL;
	if (nodefilter && nodefilter(n))   /* If true, filter this node out */
		return NULL;
	
	/* Record that we traversed this node (nothing to do with the visit order) */
	set_put(visited, n);
	
	if (ord == vis_preord && ndf)    /* Visit order dictates when to apply ndf */
		if ((*ndf)(n))
			return n;
	
	if (wdir == walk_fwd || wdir == walk_undir)
	{
		if (outdegree(n) <= 2)
		{
			/* Remember it because it may change */
			cfgnode second = (outdegree(n) > 1) ? succ1(n) : NULL;
			
			if (outdegree(n) > 0)
				cfg_search_and_return(succ0(n), ndf, ord, wdir, res);
			if (second)
				cfg_search_and_return(second, ndf, ord, wdir, res);
		}
		else
		{
			cfgnode *succs = succsarray_dup(n);  /* Remember them */
			
			for (i = outdegree(n) - 1; i >= 0; i--)
				cfg_search_and_return(succs[i], ndf, ord, wdir, res);
			free(succs);    /* FIXME: memory leak if alrady found */
		}
	}
	
	if (ord == vis_midord && ndf)    /* Visit order dictates when to apply ndf */
		if ((*ndf)(n))
			return n;
	
	if (wdir == walk_rev || wdir == walk_undir) /* TODO: backup here too? */
		cfg_search_pred_edges(n, res);
		
	if (ord == vis_postord && ndf)    /* Visit order dictates when to apply ndf */
		if ((*ndf)(n))
			return n;
			
	return NULL;
}


#define cfg_traverse_fwd(n,ndf,ord) cfg_traverse(n,ndf,ord,NULL,walk_fwd)
#define cfg_traverse_rev(n,ndf,ord) cfg_traverse(n,ndf,ord,NULL,walk_rev)
#define cfg_traverse_undir(n,ndf,ord) cfg_traverse(n,ndf,ord,NULL,walk_undir)
#define cfg_traverse_filter(n,ndf,ord,f) cfg_traverse(n,ndf,ord,f,walk_fwd)
#define cfg_traverse_undir_filter(n,ndf,ord,f) \
          cfg_traverse(n,ndf,ord,f,walk_undir)


/**
 * @brief General CFG traversal with a void function called @ every node
 * 
 * @param n       starting node
 * @param ndf     the function to call @ every node
 * @param ord     the order of traversal
 * @param filter  function to filter-out nodes
 * @param wdir    the direction of traversal (forward/backward/both)
 * @return void
 */
void cfg_traverse(cfgnode n, cfg_nodefunc_t ndf, visord_e ord,
                  int (*filter)(cfgnode), walkdir_e wdir)
{
	set_init(cfg, &visited);
	nodefilter = filter;
	if (ord == vis_midord && wdir != walk_undir)
		ord = vis_postord;
	cfg_traverse_node(n, ndf, ord, wdir);
}


/**
 * @brief CFG search traversal with a true/false function called @ every node
 * 
 * @param n       starting node
 * @param ndf     the function to call @ every node; if it returns true, the
 *                search stops
 * @param ord     the order of traversal
 * @param filter  function to filter-out nodes
 * @param wdir    the direction of traversal (forward/backward/both)
 * @return the node where ndf() returned true (NULL otherwise)
 */
/* TODO: Untested */
cfgnode cfg_search(cfgnode n, cfg_searchfunc_t ndf, visord_e ord,
                   int (*filter)(cfgnode), walkdir_e wdir)
{
	set_init(cfg, &visited);
	nodefilter = filter;
	if (ord == vis_midord && wdir != walk_undir)
		ord = vis_postord;
	return cfg_search_node(n, ndf, ord, wdir);
}


/*
 * USEFUL TRAVERSALS
 */


/* Basic blocks (bblocks) have a single entry (E) and a single exit (X) 
 * statement, and all nodes are passers (except maybe X).
 * To represent a bblock we use a CFG_BBLOCK node who gets the predecessors
 * of E and the successors of X. Then the cfgptr field is used to point to
 * E.
 */
static cfgnode bbstart;   /* Remember the bblock starting node */

void visit_bblock(cfgnode n)
{
	if (bbstart) /* Check if i am the last node in the bblock */
	{
		if (outdegree(n) > 1 || indegree(succ0(n)) > 1 || 
		    succ0(n)->type == CFG_CBLOCKS || succ0(n)->type == CFG_CBLOCKE ||
		    succ0(n)->type == CFG_TERMINAL || succ0(n)->astnodetype == BARNODE)
		{ 
			int j;                  /* Stop the block here */
			
			free(bbstart->succs);   /* These were the successors of the 1st node */
			bbstart->succs = n->succs;
			bbstart->nsucc = n->nsucc;
			for (j = 0; j < outdegree(n); j++)
				cfg_rep_backedge(n->succs[j], n, bbstart);
			n->cfgptr = bbstart;   /* This is the last node of the block */
			bbstart = NULL;
			/* Leave the successors of n as they are, for traversal purposes */
		}
	}
	else
	{   /* Check if i can start a block */
		if (n->type == CFG_PLAIN && outdegree(n) == 1 && n->astnodetype!=BARNODE &&
		    indegree(succ0(n)) == 1 && succ0(n)->type == CFG_PLAIN)
		{ 
			cfgnode s = CfgPlain(NULL);      /* Start a block here */
			s->stmt = n->stmt;
			s->astnodetype = n->astnodetype;
			s->expr = n->expr;
			s->ompparent = n->ompparent;
			cfg_add_successor_noback(s, succ0(n));
			cfg_rep_backedge(succ0(n), n, s);
			n->cfgptr = s;                   /* The original becomes a bblock */
			n->type = CFG_BBLOCK;
			bbstart = n;
		}
	}
}


/**
 * Form basic blocks within a given plain CFG
 */
void cfg_form_basic_blocks(cfgnode root)
{
	if (!CFG_HELPER_NODE(root))     /* Avoid dealing with corner case */
		return;
	/* This needs to be pre-order since we may change successor back edges */
	cfg_traverse_fwd(root, visit_bblock, vis_preord);
}


static void visit_contract_joiner(cfgnode n)
{
	if (n && n->type == CFG_JOINER)
		cfg_contract_node(n);
}


/* Remove any utility nodes (joiners) */
void cfg_remove_joiners(cfgnode root)
{
	if (root->type == CFG_JOINER)   /* We won't handle this case */
		return;
	cfg_traverse_fwd(root, visit_contract_joiner, vis_postord);  /* POSTorder */
}


static void visit_contract_jumpnode(cfgnode n)
{
	if (n && !CFG_HELPER_NODE(n) && n->astnodetype == STMTNODE && 
		  n->stmt->type == JUMP)
		cfg_contract_node(n);
}


/* Contract any jump statement nodes */
void cfg_remove_jumps(cfgnode root)
{
	if (!CFG_HELPER_NODE(root))     /* Avoid dealing with corner case */
		return;
	cfg_traverse_fwd(root, visit_contract_jumpnode, vis_postord);
}


static void visit_remove_paredges(cfgnode n)
{
	if (n && outdegree(n) > 1)
	{
		int i, j;
		for (i = outdegree(n)-1; i > 0; i--)
		{
			for (j = 0; j < i; j++)
				if (n->succs[i] == n->succs[j])    /* it is a multiple edge */
				{
					cfg_del_backedge(n->succs[i], n);
					/* Swap with the (now) last successor and simply reduce the degree */
					if (i != outdegree(n)-1)
						n->succs[i] = n->succs[outdegree(n)-1];
					outdegree(n)--;
					break;
				};
		}
	}
}


/* Remove multiple edges among the same nodes */
void cfg_remove_parallel_edges(cfgnode root)
{
	if (!CFG_HELPER_NODE(root))     /* Avoid dealing with corner case */
		return;
	cfg_traverse_fwd(root, visit_remove_paredges, vis_postord);
}


/* At the end of an OpenMP CBlock, add an explicit barrier node if
 * it is implied.
 */
static void visit_add_barriers(cfgnode n)
{
	aststmt a;
	ompdirt_e contype;
	
	if (n->type != CFG_CBLOCKE) return;

	a = n->cfgptr->stmt;       /* Start of CBlock's ast statement */
	contype = a->u.omp->type;
	if (n->type == CFG_CBLOCKE /* end of CBlock */ && 
	    (contype == DCFOR || contype == DCSECTIONS || contype == DCSINGLE) &&
			!xc_ompcon_get_clause(a->u.omp, OCNOWAIT))
	{
		cfgnode b = CfgPlain(NULL);   /* Make an explicit node */
		int j;
		
		b->astnodetype = BARNODE;
		b->ompparent = n->ompparent;
		b->succs = n->succs;
		b->nsucc = n->nsucc;
		for (j = 0; j < outdegree(n); j++)
			cfg_rep_backedge(n->succs[j], n, b);
		n->succs = NULL;
		n->nsucc = 0;
		cfg_add_successor(n, b);
	}
}


/* Mark barrier nodes and make implicit barriers explicit */
void cfg_make_barriers_explicit(cfgnode root)
{
	if (!CFG_HELPER_NODE(root))     /* Avoid dealing with corner case */
		return;
	cfg_traverse_fwd(root, visit_add_barriers, vis_preord);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * CFG CONSTRUCTION FROM THE AST                                     *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


struct ast2cfg_opt_s {
		bool expandomp;    /* false to avoid entering within an openmp node */
		bool keepjoiners;  /* true if you want to see the temporary joiners */
		bool keepjmpstmt;  /* true if you want to keep explicit jump nodes */
		bool basicblocks;  /* false if no basic blocks wanted */
		bool keepparedge;  /* true if you want to keep parallel edges */
	};
static ast2cfg_opt_t ast2cfg_opts;

static set(cfg) cfgescapes;   /* Holds nodes that jump to nowhere known */
static set(lbl) cfglabels,    /* Holds labels */
                cfggotos;     /* Holds unresolved gotos */
cfgnode cfg_from_ast_stmt(aststmt tree, cfgnode next);


/* Simple stacks to hold nodes that should be successors of jumps.
 * For example, if N is the end node of a loop (usually an auxilliary Joiner), 
 * N is pushed to the stBREAK stack; if a break statement is found in the
 * loop body, the corresponding CFG node will have N as a successor.
 */
#define STACKSIZE 100   /* FIXME */
typedef enum { stBREAK = 0, stCONTINUE, stRETURN, stSWITCH, stSECTIONS, stLAST } 
	cfgstack_t;
cfgnode stops[stLAST];
static void push(cfgstack_t st, cfgnode n)
{
	n->stacknext = stops[st];
	stops[st] = n;
}
static cfgnode pop(cfgstack_t st)
{
	cfgnode t;

	t = stops[st];
	if (t != NULL)
		stops[st] = t->stacknext;
	return (t);
}
static cfgnode peep(cfgstack_t st)
{
	return (stops[st]);
}


/* The following tries to be smart */
#if DO_CFG_OPTIMIZATIONS

#define _is_always_true(e)  _is_always_what((e),1)
#define _is_always_false(e) _is_always_what((e),0)
static
int _is_always_what(astexpr expr, int true)
{
	int n = 0, er = 0;
	if (expr == NULL)
		return (true);
	if (xar_expr_is_constant(expr))
	{
		n = xar_calc_int_expr(expr, &er);
		if (!er)
			return ((true && n != 0) || (!true && n == 0));
	}
	return (0);
}

#else

#define _is_always_true(e)  0
#define _is_always_false(e) 0

#endif

#define _is_empty_stmt(s) (!(s) || ((s)->type == EXPRESSION && !((s)->u.expr)))


static
cfgnode cfg_from_ast_stmt_jump(aststmt tree, cfgnode next)
{
	cfgnode s, t;

	switch (tree->subtype)
	{
		case SBREAK:
			s = CfgPlain(tree);
			if ((t = peep(stBREAK)) == NULL)
				set_put(cfgescapes, s);
			cfg_add_successor(s, t);
			return (s);
		case SCONTINUE:
			s = CfgPlain(tree);
			if ((t = peep(stCONTINUE)) == NULL)
				set_put(cfgescapes, s);
			cfg_add_successor(s, t);
			return (s);
		case SRETURN:
			s = CfgPlain(tree);
			if ((t = peep(stRETURN)) == NULL)
				set_put(cfgescapes, s);
			cfg_add_successor(s, t);
			return (s);
		case SGOTO:
		{
			setelem(lbl) e = set_get(cfglabels, tree->u.label);

			s = CfgPlain(tree);
			if (e == NULL)
				set_put(cfggotos, tree->u.label)->value = s;
			else
				cfg_add_successor(s, e->value);
			return (s);
		}
		default:
			fprintf(stderr, "[cfg_from_ast_stmt_jump]: b u g !!\n");
			return (NULL);
	}
}


static
cfgnode cfg_from_ast_stmt_iteration(aststmt tree, cfgnode next)
{
	cfgnode s = NULL, t = NULL, d = NULL, l = NULL;
	int     splitd = false;

	switch (tree->subtype)
	{
		case SFOR:
			/* This is the most complicated case; there are many different
			 * configurations depending on which parts are present or not 
			 */
			
			if (tree->u.iteration.init)      /* 1st node (s) is the init part */
			{
				s = CfgPlain(tree->u.iteration.init);
				if (tree->u.iteration.init->type == DECLARATION)
					s = CfgPlain(tree->u.iteration.init);
				else  /* normal case: an expression */
				{
					s = CfgnodeExpr(s,  tree->u.iteration.init->u.expr);
					if (_is_always_false(tree->u.iteration.cond))  /* jump to next */
					{
						cfg_add_successor(s, next);
						return s;
					}
				}
			}
			else
			{
				if (_is_always_false(tree->u.iteration.cond))  /* jump to next */
					return next;
				s = NULL;    /* The first node will be the condition part */
			}
			
			if (tree->u.iteration.cond)      /* d goes for the condition part */
			{
				d = CfgnodeExpr(CfgPlain(tree), tree->u.iteration.cond);
				if (!_is_always_true(tree->u.iteration.cond))
					splitd = true;
			}
			else
				d = CfgJoiner();               /* dummy */
				
			if (s)                           /* Place after init */
				cfg_add_successor(s, d);
			else
				s = d;
							
			/* We escape either from the condition or the body (through break) */
			if (!_is_empty_stmt(tree->body) || 
			     (tree->u.iteration.cond && !_is_always_true(tree->u.iteration.cond)))
			{
				t = CfgJoiner();
				cfg_add_successor(t, next);
				push(stBREAK, t);
			}
			else
				t = NULL;                      /* There is no escape node */

			if (!_is_empty_stmt(tree->body))
			{
				if (tree->u.iteration.incr)
				{
					l = CfgnodeExpr(CfgPlain(tree), tree->u.iteration.incr);
					cfg_add_successor(l, d);
				}
				else
					l = d;
				push(stCONTINUE, l);
				cfg_add_successor(d, cfg_from_ast_stmt(tree->body, l));
			}
			else
			{
				if (tree->u.iteration.incr)
				{
					l = CfgnodeExpr(CfgPlain(tree), tree->u.iteration.incr);
					cfg_add_successor(l, d);
					cfg_add_successor(d, l);
					push(stCONTINUE, l);
				}
				else
				{
					cfg_add_successor(d, d);
					push(stCONTINUE, d);
				}
			}
			
			if (splitd)    /* False case is the 2nd successor */
				cfg_add_successor(d, t);
			
			pop(stCONTINUE);
			if (t)
				pop(stBREAK);
			return (s);
		case SWHILE:
			if (_is_always_false(tree->u.iteration.cond))
			{
				s = CfgPlain(tree);    /* Don't even enter the body ... */
				cfg_add_successor(s, next);
				return (s);
			}
			s = CfgnodeExpr(CfgPlain(tree), tree->u.iteration.cond);
			t = CfgJoiner();
			cfg_add_successor(t, next);
			push(stCONTINUE, s);
			push(stBREAK, t);
			cfg_add_successor(s, cfg_from_ast_stmt(tree->body, s));
			pop(stBREAK);
			pop(stCONTINUE);
			if (_is_always_true(tree->u.iteration.cond))
				cfg_add_successor(s, NULL);
			else
				cfg_add_successor(s, t);
			return (s);
		case SDO:
			d = CfgPlain(tree);
			s = CfgnodeExpr(CfgPlain(tree), tree->u.iteration.cond);
			cfg_add_successor(s, d);
			cfg_add_successor(s, t = CfgJoiner());
			cfg_add_successor(t, next);
			push(stCONTINUE, s);
			push(stBREAK, t);
			cfg_add_successor(d, cfg_from_ast_stmt(tree->body, t));
			pop(stBREAK);
			pop(stCONTINUE);
			return (s);
		default:
			fprintf(stderr, "[cfg_from_ast_stmt_iteration]: b u g !!\n");
			return (NULL);
	}
}


static
cfgnode cfg_from_ast_stmt_selection(aststmt tree, cfgnode next)
{
	cfgnode s = NULL, t;

	switch (tree->subtype)
	{
		case SSWITCH:
			if (tree->body == NULL)
			{
				s = CfgPlain(tree);
				cfg_add_successor(s, next);
				return (s);
			}
			s = CfgnodeExpr(CfgPlain(tree), tree->u.selection.cond);
			/* The first successor is the next node; this is for the case there is
			 * no default label. If there is one, this successor gets changed.
			 */
			cfg_add_successor(s, next);
			push(stSWITCH, s);
			push(stBREAK, next);
			/* Old code: cfg_add_successor(s, cfg_from_ast_stmt(tree->body, t));
			 * This is needed in case there is a statement *before* the first CASE.
			 * However, if there is no such statement, unfortunately we end up with
			 * a double edge to the first CASE. This is why we check here if the
			 * node is already a successor, before adding the edge.
			 * FIXME: We are wrong here. If there are statements *before* the first
			 *        case, we must skip them (actually we must keep only the 
			 *        declarations without their initializers); we can't ignore them
			 *        completely though becuase there may contain a label that later
			 *        code jumps into... 
			 *        For the moment, we hope we don't encounter such a case.
			 */
			t = cfg_from_ast_stmt(tree->body, next);
			if (!cfg_is_successor(s, t))
				cfg_add_successor(s, t);
			pop(stBREAK);
			pop(stSWITCH);
			break;
		case SIF:
			if (_is_always_false(tree->u.selection.cond))
			{
				s = CfgPlain(tree);    /* Don't even enter the then part ... */
				if (tree->u.selection.elsebody)
					cfg_add_successor(s, 
					  cfg_from_ast_stmt(tree->u.selection.elsebody, next));
				return (s);
			}
			if (_is_always_true(tree->u.selection.cond))
			{
				s = CfgPlain(tree);    /* Don't even enter the else part ... */
				cfg_add_successor(s, cfg_from_ast_stmt(tree->body, next));
				return (s);
			}
			/* Normal case */
			s = CfgnodeExpr(CfgPlain(tree), tree->u.selection.cond);
			t = CfgJoiner();
			cfg_add_successor(t, next);
			cfg_add_successor(s, cfg_from_ast_stmt(tree->body, t));
			if (tree->u.selection.elsebody)
				cfg_add_successor(s, cfg_from_ast_stmt(tree->u.selection.elsebody, t));
			else
				cfg_add_successor(s, t);
			break;
		default:
			fprintf(stderr, "[ast_stmt_selection_cfg]: b u g !!\n");
	}
	return (s);
}


static
cfgnode cfg_from_ast_stmt_labeled(aststmt tree, cfgnode next)
{
	cfgnode s = NULL, t;

	switch (tree->subtype)
	{
		case SLABEL:
			s = cfg_from_ast_stmt(tree->body, next);
			set_put(cfglabels, tree->u.label)->value = s;
			break;
		case SCASE:
		case SDEFAULT:
			if (tree->body)
				s = cfg_from_ast_stmt(tree->body, next);
			else
				cfg_add_successor(s = CfgJoiner(), next);
			if ((t = peep(stSWITCH)) != NULL)
			{
				if (tree->subtype == SDEFAULT)
				{
					/* Changed 1st successor of the SWITCH node */
					cfg_del_backedge(succ0(t), t);
					succ0(t) = s;
					cfg_add_backedge(s, t);
				}
				else
					cfg_add_successor(t, s);
			}
			break;
		default:
			fprintf(stderr, "[ast_stmt_labeled_cfg]: b u g !!\n");
	}
	return (s);
}


static
void cfg_from_ast_stmt_ompsections_body(aststmt tree, cfgnode next)
{
	cfgnode s;

	if (tree->type == STATEMENTLIST)
	{
		cfg_from_ast_stmt_ompsections_body(tree->body, next);
		cfg_from_ast_stmt_ompsections_body(tree->u.next, next);
		return;
	}
	assert(tree->type == OMPSTMT && tree->u.omp->type == DCSECTION);
	s = peep(stSECTIONS);
	assert(s != NULL);
	cfg_add_successor(s, cfg_from_ast_stmt(tree, next));
}


cfgnode _current_omp_construct;  /* Keep current OpenMP construct */

cfgnode cfg_from_ast_stmt(aststmt tree, cfgnode next)
{
	cfgnode s, t;

	switch (tree->type)
	{
		case JUMP:
			return (cfg_from_ast_stmt_jump(tree, next));
		case ITERATION:
			return (cfg_from_ast_stmt_iteration(tree, next));
		case SELECTION:
			return (cfg_from_ast_stmt_selection(tree, next));
		case LABELED:
			return (cfg_from_ast_stmt_labeled(tree, next));
		case COMPOUND:
			if (tree->body == NULL)
				cfg_add_successor(s = CfgJoiner(), next);
			else
			{
				t = CfgJoiner();
				cfg_add_successor(t, next);
				s = cfg_from_ast_stmt(tree->body, t);
			}
			return (s);
		case STATEMENTLIST:
			t = cfg_from_ast_stmt(tree->body, next);
			if (tree->u.next == NULL)
				return (t);
			return (cfg_from_ast_stmt(tree->u.next, t));
		case EXPRESSION:
			s = CfgnodeExpr(CfgPlain(tree), tree->u.expr);
			cfg_add_successor(s, next);
			return (s);
		case OMPSTMT:
			if (ast2cfg_opts.expandomp)
			{
				if (!tree->u.omp->body)
				{
					s = CfgPlain(tree);
					cfg_add_successor(s, next);
					if (tree->u.omp->type == DCBARRIER)
						s->astnodetype = BARNODE;
				}
				else
				{
					cfgnode _tmp_omp = _current_omp_construct;
					
					s = CfgCBlockStart(tree);
					s->cfgptr = t = CfgCBlockEnd(tree);
					t->cfgptr = s;
					cfg_add_successor(t, next);
					_current_omp_construct = s;
					if (tree->u.omp->type==DCSECTIONS || tree->u.omp->type==DCPARSECTIONS)
					{
						push(stSECTIONS, s);
						cfg_from_ast_stmt_ompsections_body(tree->u.omp->body->body, t);
						pop(stSECTIONS);
					}
					else
						cfg_add_successor(s, cfg_from_ast_stmt(tree->u.omp->body, t));
						
					_current_omp_construct = _tmp_omp;
				}
				return (s);
			}
			/* else: */
		case DECLARATION:   /* Here we need to extract initializations... */
		case FUNCDEF:
		case ASMSTMT:
		case OX_STMT:
			s = CfgPlain(tree);
			cfg_add_successor(s, next);
			return (s);
		case VERBATIM:
			s = CfgJoiner();          /* skip node */
			cfg_add_successor(s, next);
			return (s);
		default:
			fprintf(stderr, "[ast_stmt_cfg]: b u g !!\n");
			return (NULL);
	}
}


cfgnode cfg_from_ast(aststmt tree, ast2cfg_opt_t opts)
{
	cfgnode      n, t;
	setelem(cfg) ec, lc;
	setelem(lbl) el, ll;
	int          i, unchanged;

	ast2cfg_opts = opts;

	/* Initialize the auxiliary sets */
	set_init(lbl, &cfglabels);
	set_init(lbl, &cfggotos);
	set_init(cfg, &cfgallnodes);
	set_init(cfg, &cfgescapes);

	/* Build the CFG */
	n = CfgTerminal();  /* start */
	t = CfgTerminal();  /* end */
	cfg_add_successor(n, cfg_from_ast_stmt(tree, t));
	 
	/* To make implicit OMP barriers explicit you can cancall: */
	//cfg_make_barriers_explicit(n);

	/* Resolve dangling gotos; unresolved ones join the escape set. */
	for (el = cfggotos->first; el; el = el->next)
		if ((ll = set_get(cfglabels, el->key)) != NULL)
			cfg_add_successor(el->value, ll->value);
		else
			set_put(cfgescapes, el->value);
	set_drain(cfggotos);
	set_drain(cfglabels);

	/* Now find reachable nodes from the root.
	 * All nodes not in the visit set get removed from the graph
	 */
	cfg_traverse_fwd(n, NULL, vis_preord);
	do
	{
		for (unchanged = 1, ec = cfgallnodes->first; ec; ec = ec->next)
		{
			t = ec->key;
			if (!set_get(visited, ec->key))   /* unreachable */
			{
				unchanged = 0;
				if ((lc = set_remove(cfgescapes, ec->key)) != NULL)
					free(lc);                     /* if in escape set, delete */
				if (outdegree(t) > 0)
				{
					for (i = 0; i < outdegree(t); i++)
						if (t->succs[i])
							cfg_del_backedge(t->succs[i], t);
				}
				cfgnode_free(t);
				free(set_remove(cfgallnodes, ec->key));
				break;
			}
		}
	}
	while (!unchanged);
	
	set_drain(cfgallnodes);
	set_drain(visited);
	
	/* CFG now is bloated */
	if (!opts.keepjoiners)
		cfg_remove_joiners(n);
		
	if (!opts.keepjmpstmt)
		cfg_remove_jumps(n);
	if (!opts.keepparedge)
		cfg_remove_parallel_edges(n);
	if (opts.basicblocks)
		cfg_form_basic_blocks(n);

	return (n);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * VISUALIZATION & UTILITIES                                         *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


static char *strnodelabel(cfgnode n)
{
	A_str_truncate();
	if (n->astnodetype == EXPRNODE)
		ast_expr_print(strA(), n->expr);
	else 
		if (n->astnodetype == STMTNODE)
		{
			ast_stmt_print(strA(), n->stmt);
			str_seek(strA(), A_str_tell() - 1); /* delete the \n */
		}
		else
			str_printf(strA(), "BARRIER");
	return (A_str_string());
}


static int show_back_edges = 0;
static int show_bblocks_inside = 1;
static int show_omp_tree = 1;
static int show_node_addr = 1;

static void _show_add_escedge(cfgnode from)
{
	if (!show_bblocks_inside)
	{
		TYPICAL:
			printf("  \"%p\" -> \"OUTSIDE\";\n", from);
			return;
	}
	
	if (from->type == CFG_BBLOCK)
		printf("  \"%p\" -> \"OUTSIDE\" [ltail=cluster_%p];\n", 
		       from->cfgptr, from);
	else
		goto TYPICAL;
}


static void _show_add_edge(cfgnode from, cfgnode to)
{
	if (to == NULL)
	{
		_show_add_escedge(from);
		return;
	}
	
	/* CBlock nodes */
	if ((from->type == CFG_CBLOCKS && to->type == CFG_CBLOCKE) ||
	    (from->type == CFG_CBLOCKE && to->type == CFG_CBLOCKS))
	{
		printf("  \"%p\" -> \"%p\" [style=dotted,color=blue];\n", from, to);
		return;
	}
	
	if (!show_bblocks_inside)
	{
		TYPICAL:
			printf("  \"%p\" -> \"%p\";\n", from, to);
			return;
	}
	
	if (to->type == CFG_BBLOCK)  /* Make it point to the 1st bblock node */
	{
		if (from->type == CFG_BBLOCK)
			printf("  \"%p\" -> \"%p\" [ltail=cluster_%p];\n", 
			       from->cfgptr, to->cfgptr, from);
		else
			printf("  \"%p\" -> \"%p\";\n", from, to->cfgptr);
	}
	else
	{
		if (from->type == CFG_BBLOCK)
			printf("  \"%p\" -> \"%p\" [ltail=cluster_%p];\n", 
			       from->cfgptr, to, from);
		else
			goto TYPICAL;
	}
}


static void _show_add_backedge(cfgnode from, cfgnode to)
{
	if (!show_bblocks_inside)
	{
		TYPICAL:
			printf("  \"%p\" -> \"%p\" [style=dashed,color=lightgray];\n", from, to);
			return;
	}
	
	if (to->type == CFG_BBLOCK)  /* Make it point to the cluster */
	{
		if (from->type == CFG_BBLOCK)
			printf("  \"%p\" -> \"%p\" [style=dashed,color=lightgray,"
			       "ltail=cluster_%p,lhead=cluster_%p];\n", 
			       from->cfgptr, to->cfgptr, from, to);
		else
			printf("  \"%p\" -> \"%p\" [style=dashed,color=lightgray,"
			       "lhead=cluster_%p];\n", from, to->cfgptr, to);
	}
	else
	{
		if (from->type == CFG_BBLOCK)
			printf("  \"%p\" -> \"%p\" [style=dashed,color=lightgray,"
			       "ltail=cluster_%p];\n", from->cfgptr, to, from);
		else
			goto TYPICAL;
	}
}


#define NODEADDR (show_node_addr ? ",xlabel=\"\\N\"" : "")

static void _show_add_node(cfgnode n)
{
	switch (n->type)
	{
		case CFG_TERMINAL:
			printf("  \"%p\" [shape=doublecircle,style=filled,"
	           "fillcolor=gray,label=\"%s\"];\n",
	           n, indegree(n) == 0 ? "start" : "end");
			break;
		case CFG_JOINER:
			printf("  \"%p\" [shape=circle,style=filled,"
			       "fillcolor=lightgray,label=\"J\"];\n", n);
			break;
		case CFG_BBLOCK:
			if (!show_bblocks_inside)   /* otherwise handled specially */
				printf("  \"%p\" [style=filled,fillcolor=lightyellow,color=orange,"
				       "label=\"Bblock\"%s];\n", n, NODEADDR);
			break;
		case CFG_CBLOCKS:
				printf("\"%p\" [style=filled,fillcolor=lightblue,color=blue,"
				       "label=\"%.10s\"%s];\n", n, strnodelabel(n), NODEADDR);
				break;
		case CFG_CBLOCKE:
			printf("  \"%p\" [style=filled,fillcolor=lightblue,color=blue,"
			       "label=\"CBlockEnd\"%s];\n", n, NODEADDR);
			break;
		case CFG_PLAIN:
			if (n->astnodetype == BARNODE)
				printf("\"%p\" [fontcolor=blue,label=\"BARRIER\"%s];\n", n, NODEADDR);
			else
				if (n->stmt == NULL)
					printf("  \"%p\" [shape=circle,style=filled,"
					       "fillcolor=lightgray,label=\"P\"%s];\n", n, NODEADDR);
				else
					printf("\"%p\" [label=\"%.10s\"%s];\n", n, strnodelabel(n), NODEADDR);
			break;
	}
}


/* Special treatment for bblocks so as to draw their internals if needed.
 * All internal nodes are drawn in line; outgoing edges will emanate from 
 * the last node.
 */
static void _visit_show_bblock(cfgnode n)
{
	cfgnode t;
	int i;
	
	/* Show inner nodes in the block */
	printf("  subgraph cluster_%p {\n    color=orange bgcolor=lightyellow\n", n);
	for (t = n->cfgptr; t->cfgptr != n; t = succ0(t))
	{
		_show_add_node(t);
		_show_add_edge(t, succ0(t));
	}
	/* Close the cluster here or else the successors will be pulled in */
	_show_add_node(t);
	printf("}\n");
	
	/* Use the successors of the bblock node */
	for (i = outdegree(n) - 1; i >= 0; i--)
		_show_add_edge(t, n->succs[i]);

	if (show_omp_tree && n->ompparent) 
		printf("  \"%p\" -> \"%p\" [color=yellow];\n", n->cfgptr, n->ompparent);
}


static void visit_show_node(cfgnode n)
{
	int i;

	if (n == NULL)
		return;
		
	if (show_bblocks_inside && n->type == CFG_BBLOCK)
		_visit_show_bblock(n);       /* Special case */
	else
	{
		/* Output the node and its out edges */
		_show_add_node(n);
		for (i = outdegree(n) - 1; i >= 0; i--)
			_show_add_edge(n, n->succs[i]);
		/* For cblocks add start/end edges */
		if (n->type == CFG_CBLOCKS || n->type == CFG_CBLOCKE)
			_show_add_edge(n, n->cfgptr);
	if (show_omp_tree && n->ompparent) 
		printf("  \"%p\" -> \"%p\" [color=yellow];\n", n, n->ompparent);
	
	}
	
	if (show_back_edges && n->preds && indegree(n) > 0)
		for (i = 0; i < indegree(n); i++)
			_show_add_backedge(n, n->preds[i]);
}


/* Output for graphviz (dot).
 * For example, if the output from here goes to file cfg.gv, use:
 *   dot -Tpng cfg.gv -o cfg.png
 */
void cfg_show(cfgnode n)
{
	printf("digraph G {\n");
	printf("  compound=true;\n");     /* Needed for clusters (bblocks) */
	printf("  node [shape=box];\n");
	cfg_traverse_fwd(n, visit_show_node, vis_preord);
	printf("}\n");
}


static cfgnode _cutnodeX;
static int filter_cut_node(cfgnode n)
{
	return (n == _cutnodeX);
}

/* Maybe we should also contract joiner nodes */
int cfg_is_cutnode(cfgnode root, cfgnode n)
{
	int vis;

	cfg_traverse_fwd(root, NULL, vis_preord);
	vis = set_size(visited);

	_cutnodeX = n;
	cfg_traverse_undir_filter(root, NULL, vis_preord, filter_cut_node);
	return (set_size(visited) < vis - 1);
}


/* Checks whether there are instructions that escape the given block of code
 * (gotos to unknown labels, orphaned break/continue)
 */
int cfg_code_escapes(aststmt tree)
{
	ast2cfg_opt_t opts = { true, false, false, false };
	cfg_from_ast(tree, opts);
	return (!set_isempty(cfgescapes));
}


set(cfg) cfg_create(aststmt tree, int mode)
{
	cfgnode       n;
	set(cfg)      all = set_new(cfg); // TODO this is a memory leak
	ast2cfg_opt_t opts;

	opts.expandomp = true; /* set to false to avoid entering omp nodes */
	opts.keepjoiners = (mode >= 2);
	opts.keepjmpstmt = (mode >= 1);
	opts.basicblocks = 0; //(mode < 1);
	opts.keepparedge = (mode >= 1);
	n = cfg_from_ast(tree, opts);
	if (n == NULL)
		return NULL;

	show_back_edges = (mode != 0);
	show_omp_tree = opts.expandomp;

	cfg_traverse_fwd(n, NULL, vis_preord);
	set_copy(all, visited);

	return all;
}


void cfg_test(aststmt tree, int mode)
{
	cfgnode      n;
	set(cfg)     all;
	setelem(cfg) e;

	all = cfg_create(tree, mode);
	n = all->first->key;
	fprintf(stderr, "ESCAPING: the code %s escape.\n", !set_isempty(cfgescapes) ?
	        "DOES" : "does NOT");
	show_bblocks_inside = 1;
	cfg_show(n);

	/* Find all cutnodes */
	fprintf(stderr, "\n\nCutnodes:\n");
	for (e = all->first; e; e = e->next)
		if (e->key->type != CFG_TERMINAL && cfg_is_cutnode(n, e->key))
			fprintf(stderr, "  >> cutnode found: [%.20s]\n", strnodelabel(e->key));
	
	{
		extern void cfg_omp_barrier_phases(cfgnode root);
		cfg_omp_barrier_phases(n);
	}
}


bool cfg_is_stmt_cutnode(aststmt t, set(cfg) all)
{
	setelem(cfg) e;
	cfgnode root = all->first->key;

	for (e = all->first; e; e = e->next)
		if (e->key->stmt)
			if (cfg_is_cutnode(root, e->key))
				return true;

	return false;
}


/*
 * Barrier phases (WIP)
 */
 
 
typedef struct _cfgptrl { cfgnode node; struct _cfgptrl *next; } cfgptrlist;
SET_TYPE_DEFINE(cfgptrs, cfgnode, cfgptrlist*, 1031)
SET_TYPE_IMPLEMENT(cfgptrs)
static set(cfgptrs) Sbarphases, Ebarphases; 

#define _is_cblock(n) ((n)->type == CFG_CBLOCKS || (n)->type == CFG_CBLOCKE)
#define _is_omppar(n) ((n)->astnodetype == STMTNODE && \
                       (n)->stmt->type == OMPSTMT && \
                       ( (n)->stmt->u.omp->type == DCPARALLEL || \
                         (n)->stmt->u.omp->type == DCPARSECTIONS ))
#define _is_barnode(n) ((n)->astnodetype == BARNODE || \
                        (_is_cblock(n) && _is_omppar(n)))
                        
static cfgnode _barX;

int phase_starting_barrier(cfgnode n)
{
	setelem(cfgptrs) e = set_get(Sbarphases, n);
	cfgptrlist *ptr;
	
	if (n == _barX) return 0;  /* starting node */
	if (CFG_HELPER_NODE(n) || _is_barnode(n)) return 1;  /* stop at barriers */
	if (!e)
		(e = set_put(Sbarphases, n))->value = NULL;
	ptr = (cfgptrlist *) malloc(sizeof(cfgptrlist));
	ptr->node = _barX;
	ptr->next = e->value;
	e->value = ptr;
	return 0;
}

int phase_ending_barrier(cfgnode n)
{
	setelem(cfgptrs) e = set_get(Ebarphases, n);
	cfgptrlist *ptr;
	
	if (n == _barX) return 0;  /* starting node */
	if (CFG_HELPER_NODE(n) || _is_barnode(n)) return 1;  /* stop at barriers */
	if (!e)
		(e = set_put(Ebarphases, n))->value = NULL;
	ptr = (cfgptrlist *) malloc(sizeof(cfgptrlist));
	ptr->node = _barX;
	ptr->next = e->value;
	e->value = ptr;
	return 0;
}


static set(cfg) _barnodes;

static void visit_barriers(cfgnode n)
{
	if (_is_barnode(n))
		set_put(_barnodes, n);
}


/* Computed barrier phases */
void cfg_omp_barrier_phases(cfgnode root)
{
	setelem(cfg) e;
	
	set_init(cfg, &_barnodes);
	cfg_traverse_fwd(root, visit_barriers, vis_preord);

	set_init(cfgptrs, &Sbarphases);
	set_init(cfgptrs, &Ebarphases);
	
	for (e = _barnodes->first; e; e = e->next)
	{
		fprintf(stderr, "found barnode %p\n", e->key);
		_barX = e->key;
		cfg_traverse(e->key, NULL, vis_preord, phase_starting_barrier, walk_fwd);
		cfg_traverse(e->key, NULL, vis_preord, phase_ending_barrier, walk_rev);
	}
		
	for (setelem(cfgptrs) e = Sbarphases->first; e; e = e->next)
	{
		cfgptrlist *l = e->value;
		fprintf(stderr, "node %p has bar starters:\n\t", e->key);
		for (; l; l = l->next)
			fprintf(stderr, "%p, ", l->node);
		fprintf(stderr, "\n");
	}
	for (setelem(cfgptrs) e = Ebarphases->first; e; e = e->next)
	{
		cfgptrlist *l = e->value;
		fprintf(stderr, "node %p has bar enders:\n\t", e->key);
		for (; l; l = l->next)
			fprintf(stderr, "%p, ", l->node);
		fprintf(stderr, "\n");
	}
}
