/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* builder.c -- bits and pieces needed for building the final code */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <time.h>
#include <sys/time.h>
#include "ompi.h"
#include "ast_xform.h"
#include "ast_print.h"
#include "ast_xformrules.h"
#include "x_clauses.h"
#include "x_target.h"
#include "x_decltarg.h"
#include "x_target.h"
#include "x_requires.h"
#include "ast_show.h"
#include "builder.h"
#include "codetargs.h"
#ifdef OMPI_REMOTE_OFFLOADING
	#include "roff_config.h"
#endif


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *  TOP AND BOTTOM STATEMENT LISTS GOING INTO GENERATED CODE     *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


static aststmt build_head = NULL, /* Statements injected @ top and .. */
               build_tail = NULL; /* .. injected @ bottom of generated code */

               
static void aststmt_add(aststmt *to, aststmt s)
{
	if (*to == NULL)
	{
		*to = s;
		(*to)->parent = NULL;
	}
	else
	{
		aststmt parent = (*to)->parent;
		*to = BlockList(*to, s);
		(*to)->parent = parent;  /* Re-set the parent */
	}
}


void bld_head_add(aststmt s)
{
	aststmt_add(&build_head, s);
}


void bld_tail_add(aststmt s)
{
	aststmt_add(&build_tail, s);
}


/** 
 * This adds a global variable (it is called only for the HOST code target):
 * it adds the symbol to the symbol table, and the declaration statement to
 * the header section of the generated file.
 * @param s the declaration statement
 * @return the symbol table entry
 */
stentry bld_globalvar_add(aststmt s)
{
	astdecl decl;
	stentry e;

	/* Add in the statement list */
	bld_head_add(s);

	/* Add in the symbol table */
	assert(s->type == DECLARATION);    /* Declare it, too */
	decl = s->u.declaration.decl;
	e = symtab_insert_global(stab, decl_getidentifier_symbol(decl), IDNAME);
	if (decl->type == DINIT)
		decl = (e->idecl = decl)->decl;
	e->decl       = decl;
	e->spec       = s->u.declaration.spec;
	e->isarray    = (decl_getkind(decl) == DARRAY);
	e->isthrpriv  = false;
	e->pval       = NULL;
	e->scopelevel = 0;
	return (e);
}


static aststmt _find_global_declaration(aststmt tree, symbol s)
{
	aststmt p;
	
	if (!tree) return NULL;
	if (tree->type == STATEMENTLIST)
	{
		if ((p = _find_global_declaration(tree->u.next, s)) != NULL)
			return (p);
		else
			return (_find_global_declaration(tree->body, s));
	}
	return (tree->type == DECLARATION &&
	        tree->u.declaration.decl != NULL &&
	        decl_getidentifier(tree->u.declaration.decl)->u.id == s) ?
		tree : NULL;
}


/* Inserts build_head at the right place
 */
static void place_build_head(aststmt tree)
{
	aststmt p, parent, newp;

	if (testingmode)
	{
		/* Just place it at the very beginning;
		 * it is guaranteed that the tree begins with a statementlist.
		 */
		for (p = tree; p->type == STATEMENTLIST; p = p->u.next)
			;               /* Go down to the leftmost leaf */
		p = p->parent;    /* Up to parent */
		p->u.next = BlockList(p->u.next, build_head);
		p->u.next->parent = p;           /* Parentize correctly */
		return;
	}

	/* We put all globals right after __ompi_defs__'s declaration
	 * (see bottom of ort.defs).
	 */
	if ((p = _find_global_declaration(tree, Symbol("__ompi_defs__"))) == NULL)
		exit_error(1,"[%s]: __ompi_defs__ not found in symbol table!?\n", __func__);
	
	parent = p->parent;
	newp = smalloc(sizeof(struct aststmt_));    /* Move p's contents to newp */
	*newp = *p; 
	if (cppLineNo)
		tree = Block4(
		         newp,
		         verbit("# 1 \"%s-newglobals\"", filename),
		         build_head,
		         verbit("# 1 \"%s\"", filename)
		       );
	else
		tree = BlockList(newp, build_head);
	*p = *tree;                                 /* Replace p's contents */
	free(tree);
	ast_stmt_parent(parent, p);
}


void bld_headtail_place(aststmt *tree)
{
	aststmt parent;
	
	if (tree && *tree)
	{
		parent = (*tree)->parent;
		if (build_head)
			place_build_head(*tree);
		if (build_tail != NULL)        /* Cannot ast_stmt_xform(&newtail) */
			*tree = BlockList(*tree, build_tail);  /* .. see x_shglob.c why */
		(*tree)->parent = parent;
	}
	if (build_head != NULL)
		build_head = NULL;
	if (build_tail != NULL)
		build_tail = NULL;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *  BITS AND PIECES APPENDED/PREPENDED TO THE AST                *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


#include "ort.defs"

/**
 * Add runtime definitions and some extra nodes in order to prepare the 
 * tree for transformations.
 */
void bld_prepare_for_xform(aststmt *tree)
{
	aststmt p;
	
	if (hasMainfunc && (enableOpenMP || testingmode || processmode))
	{
		/* Need to declare the ort init/finalize functions */
		p = parse_and_declare_blocklist_string(rtlib_onoff);
		assert(p != NULL);
		if (cppLineNo)
			p = BlockList(verbit("# 1 \"ort.onoff.defs\""), p);
		*tree = BlockList(p, *tree);
	}

	if ((__has_omp  && enableOpenMP) ||
	    (__has_ompix && enableOmpix) || testingmode || processmode)
	{
		if (__has_omp && enableOpenMP)
		{
			aststmt st = NULL;
	
			/* If <omp.h> was not included, then we must define a few things */
			if (includes_omph)
				p = NULL;
			else
			{
				p = parse_and_declare_blocklist_string(
				      "#pragma omp declare target\n"     /* Necessary since 4.0 */
				      "typedef void *omp_nest_lock_t;"   /* The only stuff we needed */
				      "typedef void *omp_lock_t; "       /* from <omp.h> */
				      "typedef enum omp_sched_t { omp_sched_static = 1,"
				      "omp_sched_dynamic = 2,omp_sched_guided = 3,omp_sched_auto = 4"
				      " } omp_sched_t;"
				      "\n#pragma omp end declare target\n"
				      "int omp_in_parallel(void); "
				      "int omp_get_thread_num(void); "
				      "int omp_get_num_threads(void); "
				      "int omp_in_final(void); "
				      "int omp_is_initial_device(void); "
				      "void *_ort_memalloc(int size); "
				      "void _ort_memfree(void *ptr); "
				      "void _ort_required(void); "
				      "void _ort_modules_required(void); "
				      "void _ort_set_requirements(unsigned int num_requirements, ...); "
				    );
				assert(p != NULL);
				if (cppLineNo) 
					p = BlockList(verbit("# 1 \"omp.mindefs\""), p);
			}

			/* Notice here that any types in omp.h will be defined *after* this */
			st = parse_and_declare_blocklist_string(rtlib_defs);
			assert(st != NULL);
			if (cppLineNo)
				st = BlockList(verbit("# 1 \"ort.defs\""), st);
			if (p)
				st = BlockList(p, st);
			if (__has_affinitysched)
				st = BlockList(st, parse_and_declare_blocklist_string(
				                     "int ort_affine_iteration(int *);"));
			*tree = BlockList(st, *tree);   /* Prepend ORT defs */
		}
		
		if (cppLineNo) 
		  *tree = BlockList(verbit("# 1 \"%s\"", filename), *tree);
		*tree = BlockList(*tree, verbit("\n"));    /* Dummy node @ bottom */
		(*tree)->file = Symbol(filename);
	}
	
	ast_parentize(*tree);    /* Parentize the whole tree */
}


aststmt bld_top_comment()
{
	time_t now;
	
	time(&now);  /* Advertise us */
	snprintf(advert, 1023, "/* File generated from [%s] by %s %s, %s"
		                     "$OMPi__nk:%d\n*/",
	                       filename, PKG_NAME, PKG_VERSION, ctime(&now),
	                       __kernels_num);
	return verbit(advert); 
}


aststmt bld_add_needed_std_prototypes(aststmt p)
{
	if (needLimits || bundleKernels)
		p = BlockList(p, verbit("#include <limits.h>"));
	if (needKernDimsEnc)
		p = BlockList(p, verbit("#define P2_21M1 (unsigned long long int) 2097151\n"
		                        "#define ENCODE3_ULL(X,Y,Z) \\\n"
			                      "	       ((((Z) & P2_21M1) << 42) |\\\n"
			                      "         (((Y) & P2_21M1) << 21) |\\\n"
			                      "          ((X) & P2_21M1))\n"));
	if (needFloat)
		p = BlockList(p, verbit("#include <float.h>"));
	if (needMemcpy && !knowMemcpy)
		p = BlockList(
		      p,
		      verbit("#if _WIN64 || __amd64__ || __X86_64__ || __aarch64__\n"
		             "  extern void *memcpy(void*,const void*,unsigned long int);\n"
		             "#else\n"
		             "  extern void *memcpy(void*,const void*,unsigned int);\n"
		             "#endif"
		            )
		    );
	if (needMemset && !knowMemset)
		p = BlockList(
		      p,
		      verbit("#if _WIN64 || __amd64__ || __X86_64__ || __aarch64__\n"
		             "  extern void *memset(void*,int,unsigned long int);\n"
		             "#else\n"
		             "  extern void *memset(void*,int,unsigned int);\n"
		             "#endif"
		            )
		    );
	ast_parentize(p);
	return p;
}


aststmt bld_new_main()
{
	if (usedmods == 0)
		modstr = Str(", \"dummy\"");

	A_str_truncate();
	str_printf(strA(),
	           "/* OMPi-generated main() */\n"
	           "int %s(int argc, char **argv)\n{\n",
	           nonewmain ? "__ompi_main" : "main");

	if (mainfuncRettype == 0)
	{
		if (enableOpenMP || testingmode || processmode)
			str_printf(strA(),
			           "  int _xval = 0;\n\n"
			           "  _ort_init(&argc, &argv, 0, %d%s);\n"
			           "  _xval = (int) %s(argc, argv);\n"
			           "  _ort_finalize(_xval);\n"
			           "  return (_xval);\n",
			           usedmods, str_string(modstr), MAIN_NEWNAME);
		else
			str_printf(strA(),
			           "  int _xval = 0;\n\n"
			           "  _xval = (int) %s(argc, argv);\n"
			           "  return (_xval);\n", MAIN_NEWNAME);
	}
	else
	{
		if (enableOpenMP || testingmode || processmode)
			str_printf(strA(),
			           "  _ort_init(&argc, &argv, 0, %d%s);\n"
			           "  %s(argc, argv);\n"
			           "  _ort_finalize(0);\n"
			           "  return (0);\n",
			           usedmods, str_string(modstr), MAIN_NEWNAME);
		else
			str_printf(strA(),
			           "  %s(argc, argv);\n"
			           "  return (0);\n", MAIN_NEWNAME);
	}
	str_printf(strA(), "}\n");

	return Verbatim(strdup(A_str_string()));
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     THE LIST OF THREAD FUNCTIONS                              *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* All produced thread functions are inserted in this LIFO list.
 * They are transformed seperately and each one gets inserted in the AST,
 * just before the function that created it. The LIFO way guarantees
 * correct placement of nested threads.
 * The whole code here is based on the assumption that the FUNCDEF nodes
 * in the original AST won't change (which does hold since the
 * transformation code---see above---only transforms the body of the FUNCDEF,
 * not the FUNCDEF code itself). Otherwise the ->fromfunc pointers might
 * point to invalid nodes.
 */

static funclist outfuncs = NULL;


/* New funcs are inserted @ front */
void bld_outfuncs_add(symbol name, aststmt fd, aststmt curfunc)
{
	funclist e   = (funclist) smalloc(sizeof(struct funclist_));
	e->fname     = name;
	e->funcdef   = fd;
	e->fromfunc  = curfunc;
	e->next      = outfuncs;
	outfuncs     = e;
}


/* Takes the list with the produced outlined functions (from xform_parallel(),
 * xfrom_task() etc.), and transforms them. Notice that no new outlined
 * functions can be added here since before a construct is transformed, all
 * nested constructs have already been transformed (so there will be no openmp
 * constructs in any of the functions here).
 */
static void outfuncs_xform(funclist l)
{
	for (; l != NULL; l = l->next)
		ast_stmt_xform(&(l->funcdef));
}


void bld_outfuncs_xform() 
{ 
	outfuncs_xform(outfuncs); 
}

static void outfuncs_place(funclist l)
{
	aststmt neu, bl, parent;
	funclist nl;

	if (l == NULL) return;
	
	/* Replace l->fromfunc by a small BlockList; notice that anything pointing
	 * to the same func will now "see" this new blocklist. This is why far below
	 * we change the symbol table to point to the correct node.
	 *   ----> (fromfunc) now becomes
	 *
	 *   ---> (BL) -----------> (BL2) --> (outfunc)
	 *         \                   \
	 *          -->(outfunc decl)   -->(fromfunc) (neu node)
	 */
	parent = l->fromfunc->parent;     /* save parent */
	bl = l->fromfunc;                 /* save node */
	
	neu = (aststmt) smalloc(sizeof(struct aststmt_));
	*neu = *(l->fromfunc);                          /* new node for fromfunc */
  neu->body->parent = neu;                        /* Fix parents... */
  if (neu->u.declaration.dlist) neu->u.declaration.dlist->parent = neu;  

	*bl = *Block2(                                 /* BL */
	         Declaration(                          /* outfunc decl */
	           Speclist_right(StClassSpec(SPEC_static), Declspec(SPEC_void)),
	           Declarator(
	             Pointer(),
	             FuncDecl(
	               IdentifierDecl(l->fname) ,
	               ParamDecl(
	                 Declspec(SPEC_void),
	                 AbstractDeclarator(
	                   Pointer(),
	                   NULL
	                 )
	               )
	             )
	           )
	         ),
	         Block2(neu, l->funcdef)   /* BL2 */
	       );
	bl->body->parent   = bl;            /* BL2's parent */
	bl->u.next->parent = bl;            /* Declarations's parent */
	bl->parent         = parent;        /* Parentize (BL) */
	bl->file           = NULL;
	bl->u.next->file   = NULL;
	bl->body->file     = NULL;
	l->funcdef->file   = NULL;
	
	/* Change funcdef link in the symbol table and the rest of the list */
	symtab_get(stab, decl_getidentifier_symbol(neu->u.declaration.decl), 
	           FUNCNAME)->funcdef = neu;
	for (nl=l; nl; nl = nl->next)
		if (nl->fromfunc == bl)
			nl->fromfunc = neu;
	
	if (l->next != NULL)
		outfuncs_place(l->next);
	free(l);    /* No longer needed */
}


void bld_outfuncs_place()
{
	outfuncs_place(outfuncs);
	outfuncs = NULL;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * CONSTRUCTORS                                                      *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* We are going to have the following two functions:
 * 
 * void _<filename>_init_<time>(void) {
 *   ...   // All auto initializations which can be done *after* main() 
 *         // starts should be added here. This will be called by
 *         // _ort_init().
 * }
 * 
 * static void _<filename>_ctor_<time>(void) {
 *   <register the above function with ORT>
 *   ...   // All other initializations that must be done BEFORE main()
 *         // starts should go in here. This function is called automatically.
 *         // In the thread model, its sole purpose is to register the above. 
 * }
 */

static aststmt ortinits, autoinits;


void bld_ortinits_add(aststmt st)
{
	ortinits = ortinits ? BlockList(ortinits, st) : st;
}


void bld_autoinits_add(aststmt st)
{
	autoinits = autoinits ? BlockList(autoinits, st) : st;
}


void bld_ctors_build()
{
	aststmt st;
	struct timeval ts;
	char    funcname[32];
	
	if (!ortinits && !autoinits) return;

	gettimeofday(&ts, NULL); /* unique names for the constructors */
	
	if (ortinits)
	{
		sprintf(funcname,"_ompi_init_%X%X_",(unsigned)ts.tv_sec,(unsigned)ts.tv_usec);
		ortinits = 
			FuncDef(
				Declspec(SPEC_void),
				Declarator(
					NULL,
					FuncDecl(
						IdentifierDecl(Symbol(funcname)),
						ParamDecl(Declspec(SPEC_void), NULL)
					)
				),
				NULL, 
				Compound(ortinits)
			);
		
		/* Add registration to the above function */
		st = FuncCallStmt("_ort_initreqs_add", Identifier(Symbol(funcname)));
		autoinits = autoinits ? BlockList(st, autoinits) : st;
	}
	
	sprintf(funcname,"_ompi_ctor_%X%X_",(unsigned)ts.tv_sec,(unsigned)ts.tv_usec);
	autoinits = 
		FuncDef(
			Declspec(SPEC_void),
			Declarator(
				NULL,
				FuncDecl(
					IdentifierDecl(Symbol(funcname)),
					ParamDecl(Declspec(SPEC_void), NULL)
				)
			),
			NULL, 
			Compound(autoinits)
		);
	
	autoinits = 
		BlockList(
			verbit("#ifdef __SUNPRO_C\n"
			       "  #pragma init(%s)\n"
			       "#else \n"  /* gcc assumed */
			       "  static void __attribute__ ((constructor)) %s(void);\n"
			       "#endif\n", funcname, funcname),
			autoinits
		);
	if (ortinits)
		autoinits = BlockList(ortinits, autoinits);
		
	bld_tail_add(autoinits);   /* Add to tail */
}

#ifdef OMPI_REMOTE_OFFLOADING

void bld_register_roff_snapshot(void)
{
	char *snapshot_name = "_ompi_remote_devices_";
	char *hex = roff_config_encode(snapshot_name);

	bld_autoinits_add(
		parse_blocklist_string(
			"%s\n"
			"_ort_set_ompi_remote_devices(%s, %s_size);", 
			hex, snapshot_name, snapshot_name
		)
	);
}

#endif

void bld_register_moduledir(void)
{
	bld_autoinits_add(
		parse_blocklist_string(
			"_ort_set_moduledir(\"%s/devices\",%s);", 
			LibDir, enablePortability ? "1" : "0"
		)
	);
}

void bld_add_requirements(void)
{
	str reqstr;
	
	if (num_requirements == 0) 
		return; 
		
	reqstr = Strnew();
	stringify_reqs(reqstr, true); /* quoted */
	bld_autoinits_add(
		parse_blocklist_string(
			"_ort_set_requirements(%d, %s);", 
			num_requirements, str_string(reqstr)
		)
	);
	str_free(reqstr);
}
