/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ast_xformrules.c -- transformation rules */

/* This unit registers transformation functions for each OpenMP construct.
 * In addition, it defines the handling policy (split/combine) for combined,
 * composite and exlusively nested(*) constructs.
 * 
 * (*) We call two constructs A and B *exclusively nested* if B is closely 
 * nested within A and there is no other statement nested in A, e.g.
 * #pragma omp parallel            // A
 * {
 *   #pragma omp for               // B
 *   for (i=0;...) { ... }
 * } 
 * 
 * See below how to add customized rules for modules/devices.
 */ 

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "str.h"
#include "ompi.h"
#include "builder.h"
#include "autoscope.h"
#include "set.h"
#include "ast_xform.h"
#include "ast_xformrules.h"
#include "assorted.h"
#include "ast_copy.h"
#include "x_clauses.h"
#include "x_clauserules.h"
#include "x_combine.h"
#include "x_parallel.h"
#include "x_task.h"
#include "x_single.h"
#include "x_sections.h"
#include "x_loops.h"
#include "x_for.h"
#include "x_thrpriv.h"
#include "x_shglob.h"
#include "x_target.h"
#include "x_decltarg.h"
#include "x_cars.h"
#include "x_teams.h"
#include "x_arrays.h"
#include "x_distribute.h"
#include "x_simd.h"
#include "x_ordered.h"
#include "x_atomic.h"
#include "x_cancel.h"
#include "x_requires.h"
#include "x_utilcons.h"

/* Device-specific headers */
#include "x_target_cu.h"
#include "x_parallel_cu.h"
#include "x_teams_cu.h"
#include "x_task_cu.h"
#include "x_parallel_ocl.h"
#include "x_sections_ocl.h"
#include "x_target_ocl.h"
#include "x_teams_ocl.h"


/* 
 * TODO: update this
 *
 * The DEFAULT rules apply to the host and, in absence of overriding rules,
 * to the all modules/devices. To define new, module-specific transformation 
 * rules for any construct, you need to do it in the code target main file.
 
 * Important notes:
 * 
 *   Directives that have been assigned with a policy different
 *   from NONE, will be handled by the transformation function of the
 *   resulted directive.
 *   
 * Examples:
 * (a) #pragma omp target parallel for 
 * 
 *     with XFR_ACTION_SPLIT gets transformed to:
 * 
 *     #pragma omp target 
 *       #pragma omp parallel for
 *   
 *     and handled as a `#pragma omp target'; as such the transformantion 
 *     function of the DCTARGET rule will be called (xform_target), not the
 *     transformation function of the original construct (xform_targparfor).
 * 
 * (b) #pragma omp target teams distribute
 *       #pragma omp parallel for
 * 
 *     with BOTH policy gets transformed to:
 * 
 *     (1) #pragma omp target teams distribute parallel for
 *  
 *     then to:
 *  
 *     (2) #pragma omp target teams
 *           #pragma omp distribute parallel for
 *    
 *     and gets handled as a `#pragma omp target teams'
 *     (i.e. the corresponding DCTARGETTEAMS xform function will be called)
 */

xfr_t *default_host_xform_rules = (xfr_t[])  {
	{ DCTARGET,                    xform_target,                XFR_ACTION_NONE },
	{ DCTARGETUPDATE,              xform_targetupdate,          XFR_ACTION_NONE },
	{ DCTARGETDATA,                xform_targetdata,            XFR_ACTION_NONE },
	{ DCDECLARETARGET,             xform_declaretarget,         XFR_ACTION_NONE },
	{ DCTARGETENTERDATA,           xform_targetenterdata,       XFR_ACTION_NONE },
	{ DCTARGETEXITDATA,            xform_targetexitdata,        XFR_ACTION_NONE },
	{ DCTARGETPARALLEL,            xform_targetparallel,        XFR_ACTION_SPLIT },
	{ DCTARGETPARFOR,              xform_targparfor,            XFR_ACTION_SPLIT },
	{ DCTARGETTEAMS,               xform_targetteams,           XFR_ACTION_SPLIT },
	{ DCTARGETTEAMSDIST,           xform_targetteams,           XFR_ACTION_SPLIT },
	{ DCTARGETTEAMSDISTSIMD,       xform_targetteamsdistsimd,   XFR_ACTION_NONE },
	{ DCTARGETTEAMSDISTPARFOR,     xform_targetteamsdistparfor, XFR_ACTION_SPLIT },
	{ DCTARGETTEAMSDISTPARFORSIMD, xform_targetteamsdistparforsimd, XFR_ACTION_NONE },
	{ DCTEAMS,                     xform_teams,                 XFR_ACTION_NONE },
	{ DCTEAMSDIST,                 xform_teams,                 XFR_ACTION_SPLIT },
	{ DCTEAMSDISTSIMD,             xform_teamsdistsimd,         XFR_ACTION_NONE },
	{ DCTEAMSDISTPARFOR,           xform_teamsdistparfor,       XFR_ACTION_SPLIT },
	{ DCTEAMSDISTPARFORSIMD,       xform_teamsdistparforsimd,   XFR_ACTION_NONE },
	{ DCPARALLEL,                  xform_parallel,              XFR_ACTION_NONE },
	{ DCPARSECTIONS,               xform_parallel,              XFR_ACTION_SPLIT },
	{ DCPARFOR,                    xform_parallel,              XFR_ACTION_SPLIT },
	{ DCPARFORSIMD,                xform_parforsimd,            XFR_ACTION_NONE },
	{ DCDISTRIBUTE,                xform_distribute,            XFR_ACTION_NONE },
	{ DCDISTSIMD,                  xform_distsimd,              XFR_ACTION_NONE },
	{ DCDISTPARFOR,                xform_distparfor,            XFR_ACTION_NONE },
	{ DCDISTPARFORSIMD,            xform_distparforsimd,        XFR_ACTION_NONE },
	{ DCFORSIMD,                   xform_forsimd,               XFR_ACTION_NONE },
	{ DCSIMD,                      xform_simd,                  XFR_ACTION_NONE },
	{ DCCANCEL,                    xform_cancel,                XFR_ACTION_NONE },
	{ DCCANCELLATIONPOINT,         xform_cancellationpoint,     XFR_ACTION_NONE },
	{ DCTHREADPRIVATE,             xform_threadprivate,         XFR_ACTION_NONE },  
	{ DCATOMIC,                    xform_atomic,                XFR_ACTION_NONE },
	{ DCBARRIER,                   xform_barrier,               XFR_ACTION_NONE },
	{ DCTASKWAIT,                  xform_taskwait,              XFR_ACTION_NONE },
	{ DCTASKYIELD,                 xform_taskyield,             XFR_ACTION_NONE },
	{ DCFLUSH,                     xform_flush,                 XFR_ACTION_NONE },
	{ DCCRITICAL,                  xform_critical,              XFR_ACTION_NONE },
	{ DCMASTER,                    xform_master,                XFR_ACTION_NONE },
	{ DCMASKED,                    xform_masked,                XFR_ACTION_NONE },
	{ DCORDERED,                   xform_ordered,               XFR_ACTION_NONE },
	{ DCSINGLE,                    xform_single,                XFR_ACTION_NONE },
	{ DCSECTION,                   xform_section,               XFR_ACTION_NONE },
	{ DCSECTIONS,                  xform_sections,              XFR_ACTION_NONE },
	{ DCFOR_P,                     xform_for,                   XFR_ACTION_NONE },
	{ DCFOR,                       xform_for,                   XFR_ACTION_NONE },
	{ DCTASK,                      xform_task,                  XFR_ACTION_NONE },
	{ DCTASKGROUP,                 xform_taskgroup,             XFR_ACTION_NONE },
	{ DCREQUIRES,                  xform_requires,              XFR_ACTION_NONE },
	{ DCERROR,                     xform_error,                 XFR_ACTION_NONE },
	/* new-directive.sh:xfrules */
	XFR_LASTRULE 
};


/**
 * Gets the host (default) transformation rule for a given directive
 */
xfr_t *get_default_xformrule(ompdirt_e directive)
{
	xfr_t *r;

	for (r = default_host_xform_rules; r->dirtype != DCNONE; r++)
		if (r->dirtype == directive)
			return (r);
	return NULL;
}
