/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ast_xform.h  */

#ifndef __AST_XFORM_H__
#define __AST_XFORM_H__

#include "ast.h"
#include "set.h"
#include "stddefs.h"
#include "symtab.h"
#include "stack.h"

#define XFORM_CURR_DIRECTIVE stack_peep(ompdirectives)->data
#define SPLCOMB(x) SPLCOMB_##x

typedef enum splcomb_
{
	SPLCOMB(NONE) = 0, SPLCOMB(SPLIT), SPLCOMB(COMBINE),
	SPLCOMB(BOTH)
} splcomb_e;

typedef struct ompdiritem_
{
	ompdirt_e dirtype;
	ompdirt_e prevdirtype;
	char *modulename;
  	int nparallel;
  	int iscombpar;
  	int ismasterworker;
} *ompdiritem_t;

STACK_TYPE_DEFINE(ompdirs, ompdiritem_t);

extern stack(ompdirs) ompdirectives;

SET_TYPE_DEFINE(fbstmts, char *, aststmt, DEFAULT_HASHTABLESIZE);
extern set(fbstmts) funcbodycp;

extern void ast_xform(aststmt *tree);
extern void ast_expr_xform(astexpr *t);
extern void ast_stmt_xform(aststmt *t);
extern void ast_decl_xform(astdecl *t);
extern void ast_spec_xform(astspec *t);
extern void ast_ompcon_xform(ompcon *t);

/**
 * Reproduces only the Declarator part of the original declaration; the
 * special thing about this is that it takes care of arrays with missing
 * size for the 1st dimension.
 * 
 * @param e  The symtab entry for the var whose declaratοr we want to reproduce
 * @return   A declaration node (astdecl)
 */
extern astdecl xform_clone_declonly(stentry e);

/* Produces an identical declaration which may optionally have a new var name,
 * an initializer and/or convert the variable into a pointer. 
 * Any storage class specifiers (and const) are discarded; the second function
 * also drops type qualifiers.
 */
extern aststmt xform_clone_declaration_nosc(symbol s, astexpr initer,
                                                 bool mkpointer, symbol snew);
extern aststmt xform_clone_declaration_nosc_notq(symbol s, astexpr initer,
                                                 bool mkpointer, symbol snew);

/* Reproduces the original declaration of a function */
extern aststmt xform_clone_funcdecl(symbol funcname);

/* Call a function by name, with or without args */
#define Call_expr(s,p) FunctionCall(Identifier(Symbol(s)),p)
#define Call_stmt(s,p) Expression(Call_expr(s,p))
#define Call0_expr(s)  Call_expr(s,NULL)
#define Call0_stmt(s)  Call_stmt(s,NULL)

/* Reasons for a barrier; only needed to support OMPT */
typedef enum { OMPBAR_EXPLICIT=0, OMPBAR_IMPLICIT, OMPIBAR_OMPIADDED } ompbar_e;
extern aststmt BarrierCall(ompbar_e reason);  /* Generate a barrier sync */

/* Holds the scope level of the most recently encountered
 * parallel construct
 */
extern int closest_parallel_scope;

/* Used for determining the label for the cancel parallel GOTOs
 */
extern int cur_parallel_line;

/* Used for determining the label for the cancel taskgroup GOTOs
 */
extern int cur_taskgroup_line;

/* Used in ast_declare_varlist_vars for determining if a variable appears in a
 * target data clause in order to set isindevenv to true.
 */
extern int target_data_scope;

/* Used in outline.c for setting isindevenv of functions */
extern int inDeclTarget;

extern bool beforeTarget;

/* The code target we transform for and the current user function */
extern ompcon *xformingTarget; /* The AST #target we currently transform */
extern int xformingKernelID;   /* The id of the kernel we currently transform */
extern symbol xformingFunc;    /* The user function we currently transform */
extern int xformingFor;        /* The code target we transform for */
#define xformingForHOST (xformingFor == 0)

extern astexpr optnthr;

/* Keeps track of combined directives 
 */
extern bool dist_combined;

/* This one checks whether an openmp construct should include an
 * implicit barrier.
 */
extern int xform_implicit_barrier_is_needed(ompcon t);
extern void xform_ompcon_body(ompcon t);

extern void xform_unsupported(aststmt *t);

extern void xform_taskyield(aststmt *t);
extern void xform_masked(aststmt *t);
extern void xform_master(aststmt *t);
extern void xform_critical(aststmt *t);

extern void xform_barrier(aststmt *t);
extern void xform_flush(aststmt *t);
extern void xform_taskwait(aststmt *t);

extern int  empty_bodied_omp(aststmt *t);
extern stackelem(ompdirs) find_target_parent();
extern int  search_nested_construct(ompcon t, ompdirt_e dircontype);

#endif
