/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ast_xform.c */

/*
 * 2010/10/20:
 *   fixed transformation omission of return statements.
 * 2009/12/10:
 *   thread/task functions now placed *after* the parent function so that
 *   recursion works without any warnings.
 * 2009/05/04:
 *   fixed unbelievable omission in master,ordered,atomic & critical constructs
 * 2007/09/04:
 *   brand new thread funcs handling
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <sys/time.h>
#include "ompi.h"
#include "builder.h"
#include "autoscope.h"
#include "codetargs.h"
#include "ast_copy.h"
#include "ast_free.h"
#include "ast_print.h"
#include "ast_vars.h"
#include "ast_xform.h"
#include "ast_types.h"
#include "ast_arith.h"
#include "ast_assorted.h"
#include "ast_xformrules.h"
#include "ast_show.h"
#include "x_clauses.h"
#include "x_clauserules.h"
#include "x_combine.h"
#include "x_parallel.h"
#include "x_task.h"
#include "x_task_cu.h"
#include "x_single.h"
#include "x_sections.h"
#include "x_loops.h"
#include "x_for.h"
#include "x_thrpriv.h"
#include "x_shglob.h"
#include "x_target.h"
#include "x_decltarg.h"
#include "x_cars.h"
#include "x_teams.h"
#include "x_arrays.h"
#include "x_distribute.h"
#include "x_simd.h"
#include "x_ordered.h"
#include "x_atomic.h"
#include "x_cancel.h"
#include "ox_xform.h"
#include "assorted.h"
#include "codetargs.h"

/*
 * 2009/12/09:
 *   added forgotten type transforms in selection & iteration stmt expressions
 * 2009/05/03:
 *   fixed bug (no-xfrom) in barrier
 */


/* The scope level of the most recently encountered parallel construct
 */
int closest_parallel_scope = -1;

/* Used in ast_declare_varlist_vars for determining if a variable appears in a
 * target data clause in order to set isindevenv to true.
 */
int target_data_scope = -1;

/* Used in outline.c for setting isindevenv of functions and here to prevent
 * target pragmas inside declare target
 * -- made it an int to cover multiply #declared items in v45 (VVD)
 */
int inDeclTarget = 0;

/* Used for determining the label for the cancel parallel GOTOs
 */
int cur_parallel_line = 0;

/* Used for determining the label for the cancel taskgroup GOTOs
 */
int cur_taskgroup_line = 0;

/* Keeping track of kernels and code targets
 */
ompcon *xformingTarget = NULL; /* The AST #target we currently transform */
int xformingKernelID = -1;     /* The id of the kernel we currently transform */
int xformingFor = 0;           /* 0 means host */
symbol xformingFunc = NULL;    /* The user func we currently transform */ 
// astexpr optnthr = NULL;

/* True if #distribute directive is combined with a #parallel for 
 */
bool dist_combined = false;

extern bool ingpufuncdef;

/* Handy check macro to avoid repeating the code in every target-related case */
#define CHECKTARGET() do {\
	if (xformingTarget)\
		exit_error(1, "(%s, line %d) openmp error:\n\t"\
		           "%s within a target leads to undefined behavior\n",\
		           (*t)->u.omp->directive->file->name,\
		           (*t)->u.omp->directive->l, \
				   ompdirnames[(*t)->u.omp->type]);\
\
	if (inDeclTarget)\
		exit_error(1, "(%s, line %d) openmp error:\n\t"\
		         "%s within a declare target leads to undefined behavior\n",\
		           (*t)->u.omp->directive->file->name,\
		           (*t)->u.omp->directive->l, \
				   ompdirnames[(*t)->u.omp->type]);\
	} while(0)							


static
bool is_target_related(ompdirt_e type)
{
	return (is_or_combines(type, DCTARGET) || type == DCTARGETENTERDATA || 
	  type == DCTARGETUPDATE || type == DCTARGETEXITDATA || (type==DCTARGETDATA));
}

/* Degenerate case: check whether an empty-bodied omp construct is given.
 * If so, replace it by a comment.
 */
int empty_bodied_omp(aststmt *t)
{
	if ((*t)->u.omp->body == NULL)
		return 0;
	if ((*t)->u.omp->body->type == COMPOUND && (*t)->u.omp->body->body == NULL)
	{
		char *conname = ompdirnames[(*t)->u.omp->type];
		int line = (*t)->u.omp->directive->l; /* the directive line is correct! */

		ast_stmt_free(*t);
		*t = verbit("/* (l%d) empty %s construct removed */", line, conname); 
		return (1);
	}
	return (0);
}

/* This function decides if we should require the OMPi runtime
 * and/or the OMPi device functionality and adds the appropriate
 * calls to the constructor.
 *
 * See ompi.c for more details on this effort. 
 */
static
void add_ort_required_funcs()
{
	int __has_only_decltargprotos = __has_onlydecltarg && !declvars_exist();
	__needs_devices = __has_target || __has_devfuncs || declvars_exist();
	
	/* Require the OMPi runtime */
	if ((__has_omp && !__has_only_decltargprotos) || (__has_ompfuncs) || (__has_ompxfuncs))
		bld_autoinits_add(FuncCallStmt("_ort_required", NULL));
	
	/* Require the OMPi device functionality */
	if (__needs_devices)
		bld_autoinits_add(FuncCallStmt("_ort_modules_required", NULL));
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     AST TRAVERSAL OF NON-OPENMP NODES                         *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* The only thing we do here is correctly start/end scopes and
 * declare all identifiers. Whenever we insert a symbol into the
 * symbol table, we record its specifiers/declarator so as to
 * find it later if needed.
 */
void ast_stmt_iteration_xform(aststmt *t)
{
	switch ((*t)->subtype)
	{
		case SFOR:
		{
			int newscope = false;
			
			if ((*t)->u.iteration.init != NULL)
			{
				if ((*t)->u.iteration.init->type == DECLARATION)
				{
					newscope = true;
					scope_start(stab);  /* New scope for the declared variable(s) */
				}
				ast_stmt_xform(&((*t)->u.iteration.init));
			}
			if ((*t)->u.iteration.cond != NULL)
				ast_expr_xform(&((*t)->u.iteration.cond));
			if ((*t)->u.iteration.incr != NULL)
				ast_expr_xform(&((*t)->u.iteration.incr));
			ast_stmt_xform(&((*t)->body));
			if (newscope)
				scope_end(stab);
			break;
		}
		case SWHILE:
			if ((*t)->u.iteration.cond != NULL)
				ast_expr_xform(&((*t)->u.iteration.cond));
			ast_stmt_xform(&((*t)->body));
			break;
		case SDO:
			if ((*t)->u.iteration.cond != NULL)
				ast_expr_xform(&((*t)->u.iteration.cond));
			ast_stmt_xform(&((*t)->body));
			break;
	}
}

void ast_stmt_selection_xform(aststmt *t)
{
	switch ((*t)->subtype)
	{
		case SSWITCH:
			ast_expr_xform(&((*t)->u.selection.cond));
			ast_stmt_xform(&((*t)->body));
			break;
		case SIF:
			ast_expr_xform(&((*t)->u.selection.cond));
			ast_stmt_xform(&((*t)->body));
			if ((*t)->u.selection.elsebody)
				ast_stmt_xform(&((*t)->u.selection.elsebody));
			break;
	}
}

void ast_stmt_labeled_xform(aststmt *t)
{
	ast_stmt_xform(&((*t)->body));
}

SET_TYPE_IMPLEMENT(fbstmts);
set(fbstmts) funcbodycp;
bool ingpufuncdef = false;


/* Only statement types that matter are considered. */
void ast_stmt_xform(aststmt *t)
{
	if (t == NULL || *t == NULL) return;

	switch ((*t)->type)
	{
		case EXPRESSION:
			ast_expr_xform(&((*t)->u.expr));
			break;
		case ITERATION:
			ast_stmt_iteration_xform(t);
			break;
		case JUMP:   /* 2010-10-20 (PEH) */
			if ((*t)->subtype == SRETURN && (*t)->u.expr != NULL)
				ast_expr_xform(&((*t)->u.expr));
			break;
		case SELECTION:
			ast_stmt_selection_xform(t);
			break;
		case LABELED:
			ast_stmt_labeled_xform(t);
			break;
		case COMPOUND:
			if ((*t)->body)
			{
				scope_start(stab);
				ast_stmt_xform(&((*t)->body));
				scope_end(stab);
			}
			break;
		case STATEMENTLIST:
			ast_stmt_xform(&((*t)->u.next));
			ast_stmt_xform(&((*t)->body));
			break;
		case DECLARATION:
			ast_xt_declaration_xform(t);   /* transform & declare */
			break;
		case FUNCDEF:
		{
			symbol  fsym = decl_getidentifier_symbol((*t)->u.declaration.decl);
			stentry f    = symtab_get(stab, fsym, FUNCNAME);

			/* Before doing anything else, we need to check/transform the return
			 * type of the function in case it is a new struct/union/enum.
			 */
			ast_xt_suedecl_xform(t);
			if ((*t)->type == STATEMENTLIST) /* was broken after all */
			{
				ast_xt_declaration_xform(&(*t)->u.next);  /* declare the struct */
				t = &((*t)->body);
			}

			if (!xformingForHOST)  /* Offloaded version of the function */
			{
				scope_start(stab);         /* New scope here */

				/* See corresponding comments below (for the host) */
				if ((*t)->u.declaration.dlist) /* Old style */
				{
					ast_xt_dlist_array2pointer((*t)->u.declaration.dlist);
					ast_stmt_xform(&((*t)->u.declaration.dlist));
				}
				else                           /* Normal; has paramtypelist */
				{
					ast_xt_barebones_substitute(&((*t)->u.declaration.spec),
					                            &((*t)->u.declaration.decl));
					ast_xt_decl_array2pointer((*t)->u.declaration.decl->decl->u.params);
					ast_declare_function_params((*t)->u.declaration.decl);
				}

				ast_stmt_xform(&((*t)->body));

				scope_end(stab);           /* Scope done! */
				break;                     /* done */
			}

			/* Normal path (transforming for the host) 
			 */

			assert(f==NULL || f->funcdef==NULL); /* should not be def'ed twice */
			
			/* First declare the function & store its definition stmt */
			if (f == NULL)
			{
				f = symtab_put(stab,fsym, FUNCNAME);
				f->spec = (*t)->u.declaration.spec;
				f->decl = (*t)->u.declaration.decl;
			}
			f->funcdef = *t;
			
			if (decltarg_id_isknown(fsym)) /* Note that we have a #declare func */
			{
				inDeclTarget++;
				decltarg_bind_id(f);  /* Mark it, too */
			}

			/* Create copies for all device targets */
			if (codetargs_function_is_offloaded(fsym))
				codetargs_userfuncdef_add(*t);

			scope_start(stab);         /* New scope here */

			if ((*t)->u.declaration.dlist) /* Old style */
			{
				/* Take care of array params *before* declaring them */
				ast_xt_dlist_array2pointer((*t)->u.declaration.dlist);/* array params */
				ast_stmt_xform(&((*t)->u.declaration.dlist));  /* declared themselves */
			}
			else                           /* Normal; has paramtypelist */
			{
				ast_xt_barebones_substitute(&((*t)->u.declaration.spec),
										&((*t)->u.declaration.decl));
				/* Take care of array params *before* declaring them */
				ast_xt_decl_array2pointer((*t)->u.declaration.decl->decl->u.params);
				ast_declare_function_params((*t)->u.declaration.decl);/* decl manualy */
			}

			ast_stmt_xform(&((*t)->body));
			tp_fix_funcbody_gtpvars((*t)->body);    /* take care of gtp vars */

			scope_end(stab);           /* Scope done! */
			
			/* Transform all copies */
			if (codetargs_function_is_offloaded(fsym))
			{
				int i, _xfor;
				symbol _xfun;

				/* Backup; only needed to restore after the last offloaded function */
				_xfor = xformingFor;
				_xfun = xformingFunc;

				for (i = 1; i < __codetarg_num; i++)
				{
					aststmt duf = codetargs_get_userfuncdef(i, fsym);
					ast_stmt_parent((*t)->parent, duf);
					xformingFor = i;
					xformingFunc = fsym;
					ast_stmt_xform(&duf);  /* duf won't change */
				}

				xformingFor = _xfor;   /* Restore */
				xformingFunc = _xfun;
			}

			if (processmode)
				analyze_pointerize_sgl(*t);        /* just replace them with pointers */

			if (decltarg_id_isknown(fsym))  /* #declare function no more */
				inDeclTarget--;
			break;
		}
		case ASMSTMT:
			break;
		case OMPSTMT:
		{
			void ast_omp_xform(aststmt *t);
			if (enableOpenMP || testingmode) 
				ast_omp_xform(t);
			break;
		}
		case OX_STMT:
			if (enableOmpix || testingmode) 
				ast_ompix_xform(t);
			break;
		default:
			break;
	}
}


/* Expressions are only examined for type casts and sizeof(type) */
void ast_expr_xform(astexpr *t)
{
	if (t == NULL || *t == NULL) return;
	switch ((*t)->type)
	{
		case CONDEXPR:
			ast_expr_xform(&((*t)->u.cond));
		/* Then continue below */
		case IDENT:
		case FUNCCALL:
		case BOP:
		case ASS:
		case DESIGNATED:
		case COMMALIST:
		case SPACELIST:
		case ARRAYIDX:
			if ((*t)->right)
				ast_expr_xform(&((*t)->right));
		/* Then continue below */
		case DOTFIELD:
		case PTRFIELD:
		case BRACEDINIT:
		case PREOP:
		case POSTOP:
		case IDXDES:
			ast_expr_xform(&((*t)->left));
			break;
		case CASTEXPR:
			ast_xt_barebones_substitute(&((*t)->u.dtype->spec), 
			                            &((*t)->u.dtype->decl));
			ast_expr_xform(&((*t)->left));
			break;
		case UOP:
			if ((*t)->opid == UOP_sizeoftype || (*t)->opid == UOP_alignoftype || 
			    (*t)->opid == UOP_typetrick)
				ast_xt_barebones_substitute(&((*t)->u.dtype->spec), 
				                            &((*t)->u.dtype->decl));
			else
				ast_expr_xform(&((*t)->left));
			break;
		default:
			break;
	}
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     OPENMP NODES TO BE TRANSFORMED                            *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


void ast_ompcon_xform(ompcon *t)
{
	if ((*t)->body)
		ast_stmt_xform(&((*t)->body));
}

/**
 * If the original array was declared with its 1st dimension size missing,
 * then any copy will be problematic because it will not have an initializer
 * from which to discover the size. This function fixes it by forcing a
 * size based on the cardinality of the original array initializer.
 * 
 * @param copy  A clone of the original array declaration
 * @param orig  The original array declaration (with initializer)
 * @return      The clone (copy), with modified 1st dimension
 */
static
astdecl fix_array_decl_copy(astdecl copy, astdecl orig)
{
	/* arr_dimension_size(copy, 0, NULL) is assumed to be NULL */
	int ic = decl_initializer_cardinality(orig);
	
	if (ic == 0)
		exit_error(2, "(%s, line %d): cannot determine missing dimension "
					  "size of '%s'\n", orig->file->name, orig->l, 
					  decl_getidentifier_symbol(orig)->name);
	arr_dimension_size(copy, 0, numConstant(ic));
	return (copy);
}


/**
 * Reproduces only the Declarator part of the original declaration; the
 * special thing about this is that it takes care of arrays with missing
 * size for the 1st dimension.
 * 
 * @param e  The symtab entry for the var whose declaratοr we want to reproduce
 * @return   A declaration node (astdecl)
 */
astdecl xform_clone_declonly(stentry e)
{
	astdecl decl = ast_decl_copy(e->decl);
	
	if (e->isarray && arr_dimension_size(decl, 0, NULL) == NULL)
	{
		if (e->idecl)
			fix_array_decl_copy(decl, e->idecl);
		else
			exit_error(2, "(%s, line %d): '%s' cannot be cloned; "
						  "unknown dimension size\n", 
						  e->decl->file->name, e->decl->l, e->key->name);
	}
	return (decl);
}


/**
 * Reproduces the original declaration (optionally with a new variable name)
 * which may optionally have an initializer and/or convert the variable into 
 * a pointer.
 *
 * @todo "const" is not reproduced giving a warning
 *
 * @param s       The variable whose declaration we want to reproduce
 * @param initer  An optional initializer
 * @param mkptr   If true the variable will be declared as a pointer
 * @param snew    An optional new name for the variable
 * @param skip    If 1, skip storage class specifiers; if 2, also skip
 *                type qualifiers; otherwise skip nothing.
 * @return        A statement with the declaration
 */
static aststmt 
_clone_declaration(symbol s, astexpr initer, bool mkptr, symbol snew, int skip)
{
	stentry e    = symtab_get(stab, s, IDNAME);
	astdecl decl = xform_clone_declonly(e);

	if (snew)     /* Change the name */
	{
		astdecl id = IdentifierDecl(snew);
		*(decl_getidentifier(decl)) = *id;
		free(id);
	}
	if (mkptr)    /* Turn to pointer */
		decl = decl_topointer(decl);
	return Declaration(
		       skip == 1 ? ast_spec_copy_nosc(e->spec) :
		                   skip == 2 ? ast_spec_copy_nosc_notq(e->spec) : 
		                               ast_spec_copy(e->spec),
		       initer ? InitDecl(decl, initer) : decl
		     );
}


/* TODO: "const" is not reproduced giving a warning */
aststmt 
xform_clone_declaration_nosc(symbol s, astexpr initer, bool mkptr, symbol snew)
{
	return _clone_declaration(s, initer, mkptr, snew, 1);
}


aststmt 
xform_clone_declaration_nosc_notq(symbol s,astexpr initr,bool mkptr,symbol snew)
{
	return _clone_declaration(s, initr, mkptr, snew, 2);
}


/**
 * Reproduces the original declaration of a function
 *
 * @param funcname  The function whose declaration we want to reproduce
 * @return          A statement with the declaration
 */
aststmt xform_clone_funcdecl(symbol funcname)
{
	stentry func = symtab_get(stab, funcname, FUNCNAME);
	if (!func)
		return NULL;
	//assert(func != NULL || fprintf(stderr, "%s\n", funcname->name));
	return (Declaration(ast_spec_copy(func->spec), ast_decl_copy(func->decl)));
}


/* This one checks whether an openmp construct should include an
 * implicit barrier. This is needed only if the nowait is NOT present.
 * It tries to avoid the barrier by checking if it is redundant:
 *    if the construct is the last statement in the body of another
 *    construct which ends there and does have a barrier, then we don't
 *    need to output one more here.
 */
int xform_implicit_barrier_is_needed(ompcon t)
{
	aststmt p = t->parent;    /* the openmp statement */

	/* We must go up the tree to find the closest BlockList ancestor who has
	 * us as the rightmost leaf (i.e. last statement).
	 */
	for (; p->parent->type == STATEMENTLIST && p->parent->body == p; p=p->parent)
		;
	/* Now, climb up the compounds */
	for (; p->parent->type == COMPOUND; p = p->parent)
		;
	/* We may be in luck only if p is an openmp statement */
	if ((p = p->parent)->type == OMPSTMT)
	{
		/* Now this statement must be a convenient & blocking one; loops won't do */
		t = p->u.omp;
		if (t->type == DCPARSECTIONS || t->type == DCSECTIONS ||
			t->type == DCPARALLEL || t->type == DCSINGLE)
			if (xc_ompcon_get_clause(t, OCNOWAIT) == NULL)
				return (0);  /* Not needed after all! */
	}
	return (1);        /* Needed */
}


/* This is the basic function used when calling xform_masked or 
 * the (deprecated) xform_master.
 */
static 
void filter_execution(aststmt *t, astexpr filterexpr)
{
	aststmt s = (*t)->u.omp->body, parent = (*t)->parent, v;
	bool    parisif;

	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */

	/* Check if our parent is an IF statement */
	parisif = (*t)->parent->type == SELECTION && (*t)->parent->subtype == SIF;

	(*t)->u.omp->body = NULL;     /* Make it NULL so as to free t easily */
	ast_free(*t);                 /* Get rid of the OmpStmt */

	/* Here we play a bit risky: this might be a single statement, and we
	 * convert it to a block list; it seems that it won't create problems;
	 * should we better turn it into a compound?
	 */
	/* Explanation of the following code:
	 * We output an "if { ... }" thing. If our parent is an IF statement,
	 * we should enclose the whole thing in curly braces { } (i.e. create a
	 * compound) so as not to cause syntactic problems with a possibly
	 * following "else" statement.
	 */
	*t = Block3(
		   v,
		   If( /* omp_get_thread_num() == filterexpr */
			 BinaryOperator(
			   BOP_eqeq,
			   Call0_expr("omp_get_thread_num"),
			   Parenthesis(filterexpr ? filterexpr : ZeroExpr())
			 ),
			 s,
			 NULL
		   ),
		   linepragma(s->l + 1 - parisif, s->file)
		 );
	if (parisif)
		*t = Compound(*t);
	(*t)->parent = parent;
}


/* OpenMP V5.1 2.8 page 104 */
void xform_masked(aststmt *t)
{
	ompclause c;
	astexpr filterexpr = NULL;

	/* First transform the body */
	ast_stmt_xform(&((*t)->u.omp->body));

	if ((c = xc_ompcon_get_clause((*t)->u.omp, OCFILTER)) != NULL)
		filterexpr = ast_expr_copy(c->u.expr);

	filter_execution(t, filterexpr);
}


/* Deprecated as of V5.1
 */
void xform_master(aststmt *t)
{
	/* First transform the body */
	ast_stmt_xform(&((*t)->u.omp->body));

	filter_execution(t, NULL);
}


/* In many transforms we use an "stlist" flag. This is to
 * check if our parent is a statement list etc so as to
 * avoid enclosing the produced code into a compound statement.
 * If one does not care for the looks, the produced code
 * should *always* be enlosed into a compound statement.
 */

void xform_critical(aststmt *t)
{
	aststmt s, parent, v;
	char    lock[128];
	bool    stlist;   /* See comment above */
	
	/* First transform the body */
	ast_stmt_xform(&((*t)->u.omp->body));

	s = (*t)->u.omp->body;
	parent = (*t)->parent;
	v = ompdir_commented((*t)->u.omp->directive); /* Put directive in comments */
	stlist = (parent->type == STATEMENTLIST || parent->type == COMPOUND);

	/* A lock named after the region name */
	if ((*t)->u.omp->directive->u.region)
		snprintf(lock,127,"_ompi_crity_%s", (*t)->u.omp->directive->u.region->name);
	else
		strcpy(lock, "_ompi_crity");

	(*t)->u.omp->body = NULL;    /* Make it NULL so as to free it easily */
	ast_free(*t);                /* Get rid of the OmpStmt */

	if (!symbol_exists(lock))    /* Check for the lock */
		bld_globalvar_add(
			Declaration(
			  Declspec(SPEC_void), /* Dont use omp_lock_t */
			  InitDecl(
			    Declarator(Pointer(),IdentifierDecl(Symbol(lock))),
			    NullExpr()
			  )
			)
		)->isindevenv = due2INJNOMAP;    /* make sure it is never mapped */

	*t = Block5(
		v,
		FuncCallStmt("_ort_critical_begin", UOAddress(IdentName(lock))),
		s,
		FuncCallStmt("_ort_critical_end", UOAddress(IdentName(lock))),
		linepragma(s->l + 1 - (!stlist), s->file)
		);

	if (!stlist)
		*t = Compound(*t);
	(*t)->parent = parent;
}


/* Ignore the variable list and replace by a call to __atomic_thread_fence() */
void xform_flush(aststmt *t)
{
	aststmt parent = (*t)->parent, v = ompdir_commented((*t)->u.omp->directive);

#ifdef HAVE_NEW_ATOMICS
	char *memorder = "__ATOMIC_SEQ_CST";     /* by default */ 
	if ((*t)->u.omp->directive->clauses)
		switch ((*t)->u.omp->directive->clauses->type)
		{
			case OCACQUIRE: memorder = "__ATOMIC_ACQUIRE"; break;
			case OCRELEASE: memorder = "__ATOMIC_RELEASE"; break;
			case OCACQREL:  memorder = "__ATOMIC_ACQ_REL"; break;
			default: break;
		}
	ast_free(*t);                                  /* Cut the tree here */
	*t = BlockList(v, FuncCallStmt("__atomic_thread_fence",
	                               Identifier(Symbol(memorder))));
#else
	/* flush occurs ONLY as a blocklist */
	ast_free(*t);                                  /* Cut the tree here */
	*t = BlockList(v, Call0_stmt("ort_fence"));
#endif
	(*t)->parent = parent;
}


/** 
 * Produce a barrier synchronization call with ioncorprated cancel logic.
 * @param reason The reason of adding the barrier (only OMPT needs this)
 * @return The statement
 */
aststmt BarrierCall(ompbar_e reason)
{
	char label[23], mabel[23];
	
	if ((!cur_parallel_line && !cur_taskgroup_line) || xformingFor == CODETARGID(vulkan))
		return Call_stmt("_ort_barrier_me", numConstant(reason));
	if (cur_parallel_line && !cur_taskgroup_line)
	{
		snprintf(label, 23, "CANCEL_parallel_%d", cur_parallel_line);
		/* if(_ort_barrier_me()) goto <label>; */
		return If(BinaryOperator(BOP_eqeq, 
		                         Call_expr("_ort_barrier_me", numConstant(reason)), 
		                         OneExpr()), 
		          Goto(Symbol(label)), 
		          NULL);
	}
	if (!cur_parallel_line && cur_taskgroup_line)
	{
		snprintf(label, 23, "CANCEL_taskgroup_%d", cur_taskgroup_line);
		/* if(_ort_barrier_me() goto <label>; */
		return If(BinaryOperator(BOP_eqeq, 
		                         Call_expr("_ort_barrier_me", numConstant(reason)), 
		                        numConstant(2)), 
		          Goto(Symbol(label)), 
		          NULL);
	}
	/* 4th case: both parallel and taskgroup constructs active */
	snprintf(label, 23, "CANCEL_parallel_%d", cur_parallel_line);
	snprintf(mabel, 23, "CANCEL_taskgroup_%d", cur_taskgroup_line);
	return Switch(Call_expr("_ort_barrier_me", numConstant(reason)),
	         Compound(
	           Block3(
	             Case(ZeroExpr(), Break()), 
	             Case(OneExpr(), Goto(Symbol(label))),
	             Case(numConstant(2), Goto(Symbol(mabel)))
	           )
	         )
	       );
}


void xform_barrier(aststmt *t)
{
	aststmt parent = (*t)->parent, v;

	v = ompdir_commented((*t)->u.omp->directive);
	ast_free(*t);
	*t = BlockList(v, BarrierCall(OMPBAR_EXPLICIT));
	(*t)->parent = parent;
}


void xform_taskwait(aststmt *t)
{
	aststmt parent = (*t)->parent, v;

	v = ompdir_commented((*t)->u.omp->directive);
	ast_free(*t);
	*t = BlockList(v, FuncCallStmt("_ort_taskwait", ZeroExpr()));
	(*t)->parent = parent;
}


void xform_taskyield(aststmt *t)
{
	aststmt parent = (*t)->parent, v;

	v = ompdir_commented((*t)->u.omp->directive);
	ast_free(*t);
	*t = v;            /* Just ignore it for now */
	(*t)->parent = parent;
}

/* See below for comments regarding the following func, which declares 
 * fully all private-scope vars.
 */
static
void declare_private_dataclause_vars(ompclause t)
{
	if (t == NULL) return;
	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			declare_private_dataclause_vars(t->u.list.next);
		assert((t = t->u.list.elem) != NULL);
	}
	switch (t->type)
	{
		case OCPRIVATE:
		case OCFIRSTPRIVATE:
		case OCLASTPRIVATE:
		case OCCOPYPRIVATE:
		case OCCOPYIN:
			ast_declare_varlist_vars(t->u.varlist, t->type, t->subtype);
			break;
		case OCREDUCTION:
		case OCMAP:
			ast_declare_xlist_vars(t);
			break;
		default:
			break;
	}
}


/**
 * We must first transform the body of the construct and then
 * do the transformation of the construct itself (i.e. create new code
 * based on the clauses).
 * BUT, FOR CONSTRUCTS WITH DATA CLAUSES:
 *   before transforming the body, we have to *declare* (fully)
 *   all the identifiers in the clauses that have *private* scope,
 *   so that in the body those variables have the correct visibility.
 *   (the others don't need any treatment as they have already been
 *    declared in previous scopes).
 *
 * Thus, we begin our new scope, declare them, transform the body
 * and finaly close the scope (ditching the declarations).
 * Note that the declarations, DO NOT PRODUCE ANY NEW CODE; they
 * are simply inserted in the symbol table for the correct visibility
 * of variables used in the body. The declarations are produced when
 * we transform the construct itself.
 * @param t the openmp statement
 */
void xform_ompcon_body(ompcon t)
{
	if (t->body->type == COMPOUND && t->body->body == NULL)
		return;      /* degenerate case */
	
	xc_validate_only_dataclause_vars(t->directive);  /* check for duplicates */
	scope_start(stab);

	/* If we are in a target data put __ort_denv in the symbol table, so that if
	 * there is any clause that uses outline in the body it will put __ort_denv
	 * in the struct it creates.
	 */
	if (is_target_related(t->type))
	{
		aststmt tmp = get_denv_var_decl(true);
		ast_declare_decllist_vars(tmp->u.declaration.spec, tmp->u.declaration.decl);
	}

	declare_private_dataclause_vars(t->directive->clauses);
	ast_stmt_xform(&(t->body));     /* First transform the body */
	scope_end(stab);
}


/**
 * Gets rid of an unsupported OpenMP directive and keeps only its body.
 * Also displays a warning.
 * 
 * @param t the unsupported OpenMP statement
 */
void xform_unsupported(aststmt *t)
{
	aststmt d;

	fprintf(stderr, "#pragma omp %s: currently unsupported; ignoring.\n", 
	                ompdirnames[(*t)->u.omp->type]);

	/* Get rid of the OMP directive */
	d = (*t);
	(*t) = (*t)->u.omp->body;
	(*t)->parent = d->parent;
}


STACK_TYPE_IMPLEMENT(ompdirs);
stack(ompdirs) ompdirectives;

static void _omp_xform_with_ruleset(aststmt *t, ompdirt_e type)
{
	ompdiritem_t con;
	int splcombtype;
	int iscombpar = 0;
	xfr_t *rule;
	
	if ((rule = codetarg_get_xformrule(xformingFor, type)) == NULL) 
	{
		fprintf(stderr, "[%s] no %s rules for directive `%s'; exiting.", 
		                __func__, codetarg_name(xformingFor), ompdirnames[type]);
		exit(1);
	}
	
	/* (1) Find if the construct must be split or combined and act accordingly */
	splcombtype = rule->action;
	switch (splcombtype)
	{
		case XFR_ACTION_COMBSPLIT:
			ccc_try_combining(t);
			ccc_try_splitting(t);
			break;
		case XFR_ACTION_SPLIT:
			if (type == DCPARFOR || type == DCPARSECTIONS) iscombpar = 1;
			ccc_try_splitting(t);
			break;
		case XFR_ACTION_COMBINE:
			ccc_try_combining(t);
			break;
		case XFR_ACTION_NONE:
		default:
			break;
	}
	
	/* (2) Initialize and push a stack item with info about the construct */
	con = smalloc(sizeof(struct ompdiritem_));
	con->dirtype = (*t)->u.omp->type;
	con->modulename = codetarg_name(xformingFor);
	con->prevdirtype = type;
	con->nparallel = 0;
	con->ismasterworker = 0;
	con->iscombpar = iscombpar;

	/* (3) Push the construct into the stack */
	stack_push(ompdirectives, con);

	/* (4) Get the rule for the RESULTED type, if the construct changed */
	if (splcombtype != XFR_ACTION_NONE)
		rule = codetarg_get_xformrule(xformingFor, (*t)->u.omp->type);

	/* (5) Call the transformation function */
	rule->xform_func(t);

	/* (6) Pop the construct from the stack */
	stack_pop(ompdirectives);
}


/**
 * Searches for an outer target construct in
 * the current construct stack
 *
 * @return The target stack element
 */
stackelem(ompdirs) find_target_parent()
{
	stackelem(ompdirs) e;
	for (e = ompdirectives->top; e; e = e->prev)
		if (e->data->dirtype == DCTARGET 
			|| e->data->dirtype == DCTARGETPARALLEL 
			|| e->data->dirtype == DCTARGETPARFOR)
			return e;
	return NULL;
}


/**
 * Searches for a perfectly nested parallel region 
 * 
 * @param t          The OpenMP construct, the body of which will be searched
 * @param dircontype The type of construct to search for
 * @return           The level of the discovered parallel region, otherwise 0
 */
int search_nested_construct(ompcon t, ompdirt_e dircontype)
{
	ompcon *tptr = &t;
	int level = 1;

	if (t == NULL) return 0;

	while ((*tptr)->type != dircontype)
	{
		if (dircontype == DCPARALLEL && (*tptr)->type == DCPARFOR) break;
		if ((*tptr == NULL) || ((*tptr)->body == NULL)) return 0;
		if ((*tptr)->body->type == COMPOUND) 
		{
			if ((*tptr)->body->body && (*tptr)->body->body->type == OMPSTMT)
			{
				tptr = &((*tptr)->body->body->u.omp);
				continue;
			}
		}
		if ((*tptr)->body->type != OMPSTMT) return 0;
		tptr = &((*tptr)->body->u.omp);
		level++;
	}

	return level;
}


void ast_omp_xform(aststmt *t)
{
	ompdirt_e type = (*t)->u.omp->type;

	/* First do a check on the clauses */
	xc_validate_clauses((*t)->u.omp->type, (*t)->u.omp->directive->clauses);

	if (!codetarg_get_xformrule(xformingForHOST, (*t)->u.omp->type))
	{
		fprintf(stderr, "WARNING: %s directive not implemented\n",
			ompdirnames[(*t)->u.omp->type]);
		ast_ompcon_xform(&((*t)->u.omp));
	}

	if (empty_bodied_omp(t))
		return;

	if (is_target_related(type))
		CHECKTARGET();
	
	/* If we are transforming for the host and we meet a #target region, then
	 * transform all N copies of this region, where N = # of different 
	 * code targets, according the the corresponding transformation rules.
	 */
	if (is_or_combines(type, DCTARGET) && xformingForHOST)
	{
		int ctn;
		aststmt copy;

		xformingTarget = &((*t)->u.omp);
		xformingKernelID = codetargs_kernel_getid((*t)->u.omp);

		/* First transform for the code targets */
		for (ctn = __codetarg_num-1; ctn > 0; ctn--)
		{
			xformingFor = ctn;
			copy = codetargs_kernel_getstmt((*t)->u.omp, ctn); 

			assert(copy != NULL);
			_omp_xform_with_ruleset(&copy, type);
			/* IMO we never need to access the transformed node, but just in case.. */
			codetargs_kernel_updatestmt((*t)->u.omp, ctn, copy);
		}
		
		/* Transform host LAST, because it changes the construct node */
		xformingFor = 0;
		_omp_xform_with_ruleset(t, type);  /* host */
		xformingTarget = NULL;
		xformingKernelID = -1;
	}
	else /* Either transforming any other construct or a copy of a #target */
		_omp_xform_with_ruleset(t, type);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     TRANSFORM THE AST (ITS OPENMP NODES)                      *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/**
 * Here we produce code for the declared variables and add it to the outlined
 * areas. This function is called just before *_ompi.c file is printed.
 */
static void _produce_decl_var_code()
{
	aststmt varinits = NULL, regstmts = NULL, structinit;
	kernel_t *k;

	decltarg_struct_code(&varinits, &regstmts, &structinit);
	for (k = __kernels; k - __kernels < __kernels_num; k++)
	{
		ast_stmt_in_place_append(k->rep_struct, ast_copy(structdecl));
		ast_stmt_in_place_append(k->rep_struct, ast_copy(structinit));
		free(k->decl_struct->left);  /* Free the "0" & replace by identifier */
		k->decl_struct->left = Identifier(declstructVar);
	}
	ast_free(structinit);
	if (varinits)
		bld_ortinits_add(varinits);
	bld_ortinits_add(regstmts);
}


void ast_xform(aststmt *tree)
{
	if (__has_target && analyzeKernels)
		cars_analyze_declared_funcs(*tree);   /* CARS analysis */
	decltarg_find_all_directives(*tree);    /* Get all #declare target ids */
	
	stack_init(ompdirs, &ompdirectives);
	set_init(fbstmts, &funcbodycp);
	ast_stmt_xform(tree);     /* Core */
	
	/* Add "ort_*_required" functions to constructor */
	add_ort_required_funcs();

#ifdef OMPI_REMOTE_OFFLOADING
	if (__has_omp  && enableOpenMP)
		bld_register_roff_snapshot();
#endif

#ifdef PORTABLE_BUILD
	bld_register_moduledir();
#endif

	bld_add_requirements();

	if (bundleKernels)
		codetargs_kernels_add_bubins_autoinits();

	bld_outfuncs_xform();        /* Transform & add the new thread/task funcs */
	bld_outfuncs_place();        /* Place them wisely */
	sgl_fix_sglvars();           /* Adds something to the tail */
	if (declvars_exist())
		_produce_decl_var_code();  /* Adds to globals and tail */
	bld_ctors_build();           /* Autoinits */
	bld_headtail_place(tree);    /* Finish up the tree */
	stack_free(ompdirectives);
}
