/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ast_types.c -- transformations related to usertypes, structs, declarations */

#include <fcntl.h>
#include <unistd.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "ast_types.h"
#include "ast_free.h"
#include "ast_xform.h"
#include "ast_copy.h"
#include "ast_print.h"
#include "ast_show.h"
#include "ast_traverse.h"
#include "symtab.h"
#include "str.h"
#include "sem.h"
#include "ompi.h"


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     STRUCTURES HANDLING                                       *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Things we do:
 *  (1) name all unnamed structures
 *  (2) all declarations/typedefs that include a full structure
 *      are broken into 2 statements, one for the structure definition and
 *      one for the simplified declaration/typedef.
 *  (3) multifields with user type specifiers are transformed to
 *      a list of single-field nodes.
 *  (4) all structures are recursively modified so as to not include
 *      any user types (i.e. all fields get substituted if needed).
 */


/* All unnamed structures get named here.
 */
static int nonamectr = 0;
static char _nns[16];
#define get_noname_struct() Symbol(_nns)
static symbol new_noname_struct()
{
	snprintf(_nns, 15, "_noname%d_", nonamectr++);
	return (get_noname_struct());
}


/* Break a multifield ("type f1, f2, ...;") into a series of single
 * fields and also substitute any user types. spec is assumed to
 * contain a usertype.
 */
static
astdecl multi2single_field(astspec spec, astdecl decl)
{
	if (decl)
		if (decl->type == DLIST && decl->subtype == DECL_decllist)
			return (StructfieldList(multi2single_field(spec, decl->u.next),
			                        multi2single_field(spec, decl->decl)));
	if (decl)
	{
		assert(decl->type == DECLARATOR || decl->type == DBIT);
		decl = ast_decl_copy(decl);
	}
	spec = ast_spec_copy(spec);
	ast_xt_barebones_substitute(&spec, &decl);       /* action: substitute */
	return (StructfieldDecl(spec, decl));
}


/**
 * Breaks all multifields of a structure (i.e. multiple fields declared in 
 * one field declration) into separate field declarations.
 * 
 * @param t t is assumed to point to the u.decl of a structure and upon
 *          return will point to the new u.decl.
 */
static
void ast_xt_break_multifield(astdecl *t)
{
	if (*t == NULL) return;
	if ((*t)->type == DLIST && (*t)->subtype == DECL_fieldlist)
	{
		ast_xt_break_multifield(&((*t)->u.next));
		ast_xt_break_multifield(&((*t)->decl));
	}
	else    /* Should be just a DSTRUCTFIELD */
	{
		astspec s;

		if ((*t)->type != DSTRUCTFIELD) return;

		/* Check if spec has another struct in it */
		if ((s = speclist_getspec((*t)->spec, SUE, 0)) != NULL &&
		    (s->subtype == SPEC_struct || s->subtype == SPEC_union))
			ast_xt_break_multifield(&(s->u.decl));

		//    if (speclist_getspec((*t)->spec, USERTYPE, 0) != NULL)
		{
			astdecl n = multi2single_field((*t)->spec, (*t)->decl);
			ast_decl_free(*t);
			*t = n;
		}
	}
}


/* s/u/e { ... };            leave it alone
 * s/u/e name { ... };       ditto
 * s/u/e name x;             ditto
 * s/u/e { ... } x;          break into "s/u/e noname { ... };
 *                                       s/u/e noname x;"
 * s/u/e name { ... } x;     break into "s/u/e name { ... };
 *                                       s/u/e name x;"
 *
 * We don't care for such transformations within struct/union fields or
 * function parameter lists since their scope is not of concern to us.
 *
 * If no breaking is performed, we also take care of inserting stuff in the 
 * symbol table. Notice that we only do one thing: we either break or
 * insert in the symbol table; thus if a break was done, the caller must
 * call us again on the new declaration node.
 */
void ast_xt_suedecl_xform(aststmt *d)
{
	aststmt n, parent;
	astspec s;
	int     su;       /* 1 if struct or union, 0 if enum */

	assert((*d)->type == DECLARATION || (*d)->type == FUNCDEF);
	if ((s = speclist_getspec((*d)->u.declaration.spec, SUE, 0)) == NULL) return;
	su = (s->subtype == SPEC_struct || s->subtype == SPEC_union);

	/* Break multidecl fields into single-field declarations (if needed),
	 * and substitute any user-defined types.
	 */
	if (su)
		ast_xt_break_multifield(&(s->u.decl));

	/* Check for plain definition of a SUE */
	if ((*d)->u.declaration.decl == NULL)
	{
		/* Put it in the symbol table */
		if (s->name != NULL) /* Nothing to be done for "struct { ...};" */
		{
			stentry e = symtab_put(stab, s->name, su ? SUNAME : ENUMNAME);
			e->spec = (*d)->u.declaration.spec;    /* Remember those */
			e->decl = NULL;
		}
		return;
	}
	
	if ((su && s->u.decl == NULL) || (!su && s->body == NULL))
		return;  /* Nothing to be done here */

	/* Alter the specs if unnamed struct: name it. */
	if (s->name == NULL)
		s->name = new_noname_struct();

	/* Now we need to create the 2-node BlockList */
	n = Declaration(ast_spec_copy(s), NULL);   /* The 1st node */
	if (su)                                    /* Ditch the fields */
	{
		ast_decl_free(s->u.decl);
		s->u.decl = NULL;                   /* The 2nd node is ready */
	}
	else                                  /* Ditch the enumerators */
	{
		ast_spec_free(s->body);
		s->body = NULL;                     /* The 2nd node is ready */
	}

	parent = (*d)->parent;                    /* Save the parent */
	*d = BlockList(n, *d);                    /* Do it */
	(*d)->parent = parent;
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     USER-TYPE SUBSTITUTIONS                                   *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* We ditch all user types by substituting with bare bones ones.
 */


/* We are given a **(ABS)DECLARATOR** (subst) and we shall substitute it in
 * place of the IDENTIFIER of the original declarator.
 */
static
astdecl ast_xt_substitute_decl(astdecl origdecl, astdecl subst)
{
	astdecl copy, id;

	assert(origdecl != NULL);
	assert(subst->type == DECLARATOR || subst->type == ABSDECLARATOR);

	id = decl_getidentifier(copy = ast_decl_copy(origdecl));
	if (!(subst->spec == NULL &&    /* if a DIDENT/DPAREN but not a pointer .. */
	      subst->decl != NULL && (subst->decl->type == DIDENT ||
	                              subst->decl->type == DPAREN ||
	                              subst->decl->type == DFUNC)))
	{
		subst = ParenDecl(subst);  /* Use ulgy parenthesis only if really needed */
		*id = *subst;
	}
	else
	{
		*id = *(subst->decl);          /* Cannot replace DIDENT with DECLARATOR! */
		free(subst->decl);
	}
	free(subst);
	return (copy);
}


/* Takes an astdecl node and substitutes any user types found in there,
 * which may occur basically in function params/structure fields/casts.
 */
void ast_xt_barebones_decl(astdecl d)
{
	if (d == NULL) return;
	switch (d->type)
	{
		case DIDENT:
		case DELLIPSIS:
			break;
		case DINIT:
		case DPAREN:
		case DECLARATOR:
		case ABSDECLARATOR:
		case DBIT:
		case DARRAY:
			if (d->u.expr)
				ast_expr_xform(&(d->u.expr));
			if (d->decl) /* Maybe abstract declarator */
				ast_xt_barebones_decl(d->decl);
			break;
		case DFUNC:
			if (d->decl)          /* Maybe abstract declarator */
				ast_xt_barebones_decl(d->decl);
			if (d->u.params)
				ast_xt_barebones_decl(d->u.params);
			break;
		case DPARAM:
		case DCASTTYPE:         /* Here we get some action */
			ast_xt_barebones_substitute(&(d->spec), &(d->decl));
			break;
		case DSTRUCTFIELD:
			/* This should never happen! */
			warning("[ast_xt_barebones_decl]: DSTRUCTFIELD (?!)\n");
			break;
		case DLIST:
			switch (d->subtype)
			{
				case DECL_idlist:
					break;              /* Nothing here */
				case DECL_decllist:
				case DECL_paramlist:
				case DECL_fieldlist:
					ast_xt_barebones_decl(d->u.next);
					ast_xt_barebones_decl(d->decl);
					break;
				default:
					warning("[ast_xt_barebones_decl]: list  b u g !!\n");
			}
			break;
		default:
			warning("[ast_xt_barebones_decl]: b u g !!\n");
	}
}


/**
 * Take a declarator and transform it into an abstract one (a plain
 * copy is produced if it is already an abstract declarator).
 * Through the recursion, orig may not be a DECLARATOR at some point.
 * Anyways, there are only 4 possibilities if you look at the grammar.
 * The original declarator is not touched (can be freed).
 * NOTE:
 *   If orig is just an identifier (no specs), then NULL is returned.
 * @param orig the original concrete declarator (left untouched)
 * @return     the new, abstract declarator
 */
astdecl ast_xt_concrete_to_abstract_declarator(astdecl orig)
{
	astdecl a = NULL, c = orig;
	int     isdor = 0;             /* 1 if c is DECLARATOR */

	if (orig == NULL) return NULL;
	if (c->type == ABSDECLARATOR) return (ast_decl_copy(orig));
	if (c->type == DECLARATOR) { isdor = 1; c = orig->decl; }
	if (c != NULL)
		switch (c->type)
		{
			case DIDENT:
				a = NULL;
				break;
			case DPAREN:
				assert(c->decl != NULL);
				a = (c->decl->type == DIDENT) ? NULL :
				      /*Added ParenDecl (Alex, Nov 14) */
				      ParenDecl(ast_xt_concrete_to_abstract_declarator(c->decl));
				break;
			case DARRAY:
				a = ArrayDecl(ast_xt_concrete_to_abstract_declarator(c->decl),
				              (c->spec != NULL && c->spec->type == SPEC &&
				               c->spec->subtype == SPEC_star) ?
				                 ast_spec_copy(c->spec) : NULL,
				              ast_expr_copy(c->u.expr));
				break;
			case DFUNC:
				a = FuncDecl(ast_xt_concrete_to_abstract_declarator(c->decl),
				             (c->u.params && c->u.params->type == DLIST &&
				              c->u.params->subtype == DECL_idlist) ?
				                NULL : ast_decl_copy(c->u.params));
				break;
			default:
				warning("[c2a declarator]: c->decl->type = %d%s BUG !?\n",
				        c->decl->type, isdor ? ", isdor=1" : ".");
		};
	if (isdor)
		a = (a || orig->spec) ? AbstractDeclarator(ast_spec_copy(orig->spec), a) 
		                      : NULL;    /* VVD, 3/2020 */
	return (a);
}


/**
 * Given an abstract declarator produce a concrete one using the second 
 * argument as the base declarator (normally condecl would be an identifier).
 * If it is already a concrete declarator, we just return a copy.
 * @param orig    the original abstract declarator (left untouched)
 * @param condecl what to use ast the base declarator (utilized)
 * @return        the new, concrete declarator
 */
astdecl ast_xt_abstract_to_concrete_declarator(astdecl orig, astdecl condecl)
{
	astdecl a = NULL, c = orig;
	bool    isdor = false;              /* true if c is ABSDECLARATOR */

	if (orig == NULL) return condecl;   /* Just return the concrete base */
	if (c->type == DECLARATOR) return (ast_decl_copy(orig));
	if (c->type == ABSDECLARATOR) { isdor = true; c = orig->decl; }
	if (c != NULL)
		switch (c->type)
		{
			case DIDENT:
				a = NULL;
				break;
			case DPAREN:
				assert(c->decl != NULL);
				a = (c->decl->type == DIDENT) ? NULL :
				     ParenDecl(ast_xt_abstract_to_concrete_declarator(c->decl,condecl));
				break;
			case DARRAY:
				a = ArrayDecl(ast_xt_abstract_to_concrete_declarator(c->decl, condecl),
				              (c->spec != NULL && c->spec->type == SPEC &&
				               c->spec->subtype == SPEC_star) ?
				                 ast_spec_copy(c->spec) : NULL,
				              ast_expr_copy(c->u.expr));
				break;
			case DFUNC:
				a = FuncDecl(ast_xt_abstract_to_concrete_declarator(c->decl, condecl),
				             (c->u.params && c->u.params->type == DLIST &&
				              c->u.params->subtype == DECL_idlist) ?
				                NULL : ast_decl_copy(c->u.params));
				break;
			default:
				warning("[a2c declarator]: c->decl->type = %d%s BUG !?\n",
				        c->decl->type, isdor ? ", isdor=1" : ".");
		}
	else 
		/* This must be the place to put the condecl */
		a = condecl;
	if (isdor)
		a = (a || orig->spec) ? Declarator(ast_spec_copy(orig->spec), a) 
		                      : NULL;    /* VVD, 3/2020 */
	return (a);
}


/**
 * Substitute the spec and decl parts of a declaration by their barebones
 * specifier and declarator elements, i.e. replace any user types with 
 * their original definitions, recursively till only basic C types are used.
 * Parameters spec and decl should point to the original ones; upon return they
 * point to the new ones that have been created through substitutions. The 
 * function frees up the original stuff.
 * This can be only called for a spec/decl thing, i.e. in exactly 4 cases:
 *   (1) declarations (includes typedefs)
 *   (2) struct fields
 *   (3) function parameters declaration
 *   (4) type casts
 * and obiously decl must be a (abs/init)declarator.
 * @param spec pointer to the the original spec; upon return original 
 *             spec will be the new one
 * @param decl pointer to the the original decl; upon return original 
 *             del will be the new one
 */
void ast_xt_barebones_substitute(astspec *spec, astdecl *decl)
{
	stentry e;
	astdecl child;
	astspec sp;
	bool    subst = false; /* True if something was substituted and we need to 
	                          update the symbol table (only used for typedefs!) */

	if (*decl != NULL)
	{
		assert((*decl)->type == DECLARATOR || (*decl)->type == ABSDECLARATOR ||
		       (*decl)->type == DINIT      || (*decl)->type == DBIT);
		if ((*decl)->type == DINIT)
			ast_expr_xform(&((*decl)->u.expr));
		/* Skip the initializer, if any */
		if ((*decl)->type == DINIT || (*decl)->type == DBIT)
			decl = &((*decl)->decl);
	}

	/* If it is a usertype, then substitute the specifier. */
	if ((sp = speclist_getspec(*spec, USERTYPE, 0)) != NULL)
	{
		/* The user type may not even be in the stab (!), and this happens I think
		 * only for the thread function argument. In this case, don't substitute.
		 */
		if ((e = symtab_get(stab, sp->name, TYPENAME)) != NULL)
			/* If built-in additional type (e.g. __builtin_va_list), forget it */
			if (e->spec != NULL)
			{
				subst = 1;
				if (sp == *spec)                  /* I.e. spec was exactly a USERTYPE */
				{
					ast_spec_free(*spec);
					*spec = ast_spec_copy_nosc_asis(e->spec);       /* way easy */
				}
				else
				{
					astspec tmp;
					*sp = *(tmp = ast_spec_copy_nosc_asis(e->spec));
					free(tmp);
				}
			};
	}
	else
		/* Check if spec is typeof, and substitute */
		if (*spec)
		{
			astspec newspec;
			astdecl newdecl;
				
			if ((*spec)->type == TYPEOFEXPR)
			{
				sem_expr_typeof((*spec)->u.expr, &newspec, &newdecl);
				/* Substitute if a concrete declarator exists (i.e. not a cast) 
				 * otherwise the newdecl from sem_expr_typeof() will be used.
				 */
				if (*decl != NULL && (*decl)->type != ABSDECLARATOR)
				{
					newdecl = ast_xt_abstract_to_concrete_declarator(newdecl,
					                                                 (*decl)->decl);
					free(*decl);      /* shallow free because we used (*decl)->decl */
					*decl = newdecl;
				}
				ast_spec_free(*spec);
				*spec = newspec;
				*decl = newdecl;
			}
			if ((*spec)->type == TYPEOFTYPE)
			{
				if (*decl != NULL && (*decl)->type != ABSDECLARATOR)
				{
					newspec = ast_spec_copy((*spec)->u.decl->spec);
					newdecl= ast_xt_abstract_to_concrete_declarator((*spec)->u.decl->decl,
				                                                  (*decl)->decl);
					free(*decl);      /* shallow free because we used (*decl)->decl */
					ast_spec_free(*spec);
				}
				else   /* For casts, use the ones within the cast... */
				{
					newspec = (*spec)->u.decl->spec;
					newdecl = (*spec)->u.decl->decl;
					free(*spec);
					free(*decl);
				}
				*spec = newspec;
				*decl = newdecl;
			}
		}
		else
			/* May need to recurse inside structs, so break multifields */
			if ((sp = speclist_getspec(*spec, SUE, 0)) != NULL)
				if (sp->subtype == SPEC_struct || sp->subtype == SPEC_union)
					ast_xt_break_multifield(&(sp->u.decl));

	/* If now there is no declarator (e.g. unnamed stucts, abstract pointers)
	 * we are done.
	 */
	if (*decl == NULL || (*decl)->decl == NULL)
	{
		if (subst)
		{
			/* Was abstract declarator obvioulsy */
			if (*decl != NULL && (*decl)->type == ABSDECLARATOR)  /* A lonely ptr */
				*decl = ast_xt_substitute_decl(e->decl, *decl);
			else
				*decl = ast_xt_concrete_to_abstract_declarator(e->decl);
			ast_xt_barebones_substitute(spec, decl);    /* recurse */
		}
		return;
	}

	/* At this point, *decl is a DECLARATOR/ABSDECLARATOR */
	child = (*decl)->decl;
	switch (child->type)
	{
		case DIDENT:
			if (subst)
				*decl = ast_xt_substitute_decl(e->decl, *decl);
			break;
		case DPAREN:
			ast_xt_barebones_decl(child->decl);
			if (subst)
				*decl = ast_xt_substitute_decl(e->decl, *decl);
			break;
		case DARRAY:
			/* This is the only place we don't follow C99 to its full extend;
			 * we cannot support varaible-length arrays! This is because
			 * we won't check the spec/expresion inside the square brackets
			 * of the array.
			 */
			if (child->u.expr)
				ast_expr_xform(&((*decl)->decl->u.expr));
			if (child->decl) /* Maybe abstract declarator */
				ast_xt_barebones_decl(child->decl);
			if (subst)
				*decl = ast_xt_substitute_decl(e->decl, *decl);
			break;
		case DINIT:
			warning("[ast_xt_barebones_substitute]: DINIT (!?)\n");
			break;
		case DBIT:
			ast_xt_barebones_decl(child->decl);
			if (subst)
				*decl = ast_xt_substitute_decl(e->decl, *decl);
			break;
		case DFUNC:             /* Maybe abstract declarator */
			if (child->decl)
				ast_xt_barebones_decl(child->decl);
			if (child->u.params)
				ast_xt_barebones_decl(child->u.params);
			if (subst)
				*decl = ast_xt_substitute_decl(e->decl, *decl);
			break;
		case DLIST:
			/* This can only occur when recursing, for normal declarations and
			 * struct fields.
			 */
			ast_xt_barebones_decl(child->u.next);
			ast_xt_barebones_decl(child->decl);
			break;
		case ABSDECLARATOR:
		case DECLARATOR:
			/* should not happen? */
			ast_xt_barebones_decl(child->decl);
			if (subst)
				*decl = ast_xt_substitute_decl(e->decl, *decl);
			break;
		default:
			warning("[ast_xt_barebones_substitute]: child->type = %d (!?)\n",
			        child->type);
	}

	if (subst)
		ast_xt_barebones_substitute(spec, decl);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     DECLARATION STATEMENT HANDLING                            *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* A list of retired typedefs */
static aststmt retiretree = NULL;

static
void ast_xt_retire(aststmt t)
{
	if (retiretree == NULL)
		retiretree = t;
	else
		retiretree = BlockList(retiretree, t);
}

void ast_xt_free_retired()
{
	if (retiretree)
		ast_stmt_free(retiretree);
}


aststmt multi2single_stmt(astspec spec, astdecl decl)
{
	if (decl->type == DLIST && decl->subtype == DECL_decllist)
		return BlockList(multi2single_stmt(spec, decl->u.next),
		                 multi2single_stmt(spec, decl->decl));
	assert(decl->type == DECLARATOR || decl->type == DINIT);
	return (Declaration(ast_spec_copy(spec), ast_decl_copy(decl)));
}


/* takes "type var1, var2, ...;" and produces "type var1; type var2; ...;"
 * where the above are declarations.
 */
static
void ast_xt_break_multidecl(aststmt *t)
{
	aststmt n =
	  multi2single_stmt((*t)->u.declaration.spec, (*t)->u.declaration.decl);
	n->parent = (*t)->parent;
	ast_stmt_free(*t);
	*t = n;
}


void ast_xt_declaration_xform(aststmt *t)
{
	assert((*t)->type == DECLARATION);
	/* specs; (i.e. no declaration) */
	if ((*t)->u.declaration.decl == NULL)
	{
		ast_xt_suedecl_xform(t); /* just inform symtab about SUE names (if any) */
		return;
	}

	ast_xt_suedecl_xform(t);   /* symtab & possibly break a complex struct decl */
	if ((*t)->type != DECLARATION)   /* it was xformed after all */
	{
		ast_stmt_xform(t);
		return;
	}

	/* If we have >1 vars and the specs have a user type or typeof, we must
	 * split the declaration into a list of seperate declarations, cause
	 * we cannot handle it otherwise.
	 */
	if ((*t)->u.declaration.decl->type == DLIST &&
	    (speclist_getspec((*t)->u.declaration.spec, USERTYPE, 0) ||
	     speclist_getspec((*t)->u.declaration.spec, TYPEOFEXPR, 0) ||
	     speclist_getspec((*t)->u.declaration.spec, TYPEOFTYPE, 0)))
	{
		ast_xt_break_multidecl(t);
		ast_stmt_xform(t);
		return;
	}

	/* We now have a list without usertype specs/typeof or a single variable;
	 * prepare & declare (after checking that we are not on a typedef).
	 */
	if (!speclist_getspec((*t)->u.declaration.spec, STCLASSSPEC, SPEC_typedef))
	{
		/* Be careful! Declare *after* any substitutions! */
		if ((*t)->u.declaration.decl->type == DLIST)
			ast_xt_barebones_decl((*t)->u.declaration.decl);
		else
			ast_xt_barebones_substitute(&((*t)->u.declaration.spec),
			                            &((*t)->u.declaration.decl));
		ast_declare_decllist_vars((*t)->u.declaration.spec,
		                          (*t)->u.declaration.decl);
		return;
	}

	/* We are left with a typedef, which we also insert into stab. */
	ast_declare_decllist_vars((*t)->u.declaration.spec, (*t)->u.declaration.decl);

	/* We replace the node with a comment BUT the original node still hangs
	 * around since there may follow declarations depending on it.
	 * We retire those things so as to free them at some point in the future.
	 */
	ast_xt_retire(*t);

	A_str_truncate();                  /* commentize */
	str_printf(strA(), "/* (l%d) ", (*t)->l);
	ast_stmt_print(strA(), *t);
	str_seek(strA(), A_str_tell() - 1); /* delete the \n */
	str_printf(strA(), " */\n");
	if (cppLineNo)
		str_printf(strA(), "# %d \"%s\"", (*t)->l, (*t)->file->name);
	{
		aststmt parent = (*t)->parent;
		*t = Verbatim(strdup(A_str_string()));
		(*t)->parent = parent;
	}
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     UTILITIES                                                 *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* The next 3 funcs are only called for function parameters;
 * they "reduce" array variables to pointers. The first function
 * is for a declaration list (old style) and the 3rd for
 * a parameter list.
 */


void ast_xt_dlist_array2pointer(aststmt d)
{
	if (d->type == STATEMENTLIST)
	{
		ast_xt_dlist_array2pointer(d->u.next);
		ast_xt_dlist_array2pointer(d->body);
		return;
	}
	assert(d->type == DECLARATION);
	if (d->u.declaration.decl)
		ast_xt_decl_array2pointer(d->u.declaration.decl);
}


/* This is only used by ast_xt_decl_array2pointer() and assumes d is a
 * declarator or a direct_declarator.
 */
static
void ast_xt_directdecl_array2pointer(astdecl d)
{
	switch (d->type)
	{
		case DECLARATOR:
			ast_xt_decl_array2pointer(d);
			break;
		/* direct_declarator cases */
		case DPAREN:         /* cannot have (id)[10] -- see parser.y */
			ast_xt_decl_array2pointer(d->decl);
			break;
		case DIDENT:         /* nothing to do here */
		case DFUNC:
			break;
		case DARRAY:
			if (d->decl->type != DIDENT)
				ast_xt_directdecl_array2pointer(d->decl);
			else   /* Got it */
			{
				astdecl t = ParenDecl(Declarator(Pointer(), d->decl));
				if (d->u.expr) ast_expr_free(d->u.expr);
				if (d->spec)   ast_spec_free(d->spec);
				*d = *t;
				free(t);
			}
			break;
		default:
			break;
	}
}


void ast_xt_decl_array2pointer(astdecl d)
{
	if (d == NULL) return; /* just in case */
	switch (d->type)
	{
		case DLIST:
			ast_xt_decl_array2pointer(d->u.next);
			ast_xt_decl_array2pointer(d->decl);
			break;
		case DINIT:
		case DPARAM:
			if (d->decl)
				ast_xt_decl_array2pointer(d->decl);
			break;
		case ABSDECLARATOR:        /* Should not occur in function definition */
		case DELLIPSIS:
			break;
		case DECLARATOR:
			ast_xt_directdecl_array2pointer(d->decl);
			/* The following removes redundant parentheses.
			 * However, it is wrong in that if inside the parenthesis there is
			 * a pointer, we must "propagate" this outside the parentheses;
			 * too tired to do it ...
			if (d->decl->type == DPAREN)
			{
			  astdecl tmp = d->decl;
			  *d = *d->decl->decl;
			  free(tmp);
			}
			 */
			break;
		default:
			warning("[ast_xt_decl_array2pointer]: !? BUG !? (type = %d):\n",
			        d->type);
			ast_decl_show_stderr(d);
	}
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *         SUE DEPENDECY GRAPH                                   *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

/* Uncomment to debug SUEDG construction */
//#define DEBUG_SUEDG

/* This is to be utilized mostly by codetargs so that only useful global
 * types / declarations get output to a kernel file.
 */


#define SUENODE_COLOR_UNION  1   /* it is a union, not a struct */
#define SUENODE_COLOR_ENUM   2   /* it is an enum, not a struct, not a union */
#define SUENODE_COLOR_USED   4   /* the SUE was used to declare something */
#define SUENODE_COLOR_GLOBAL 8   /* it is a global SUE */

typedef struct { 
    namespace ns;   /* namespace (basically SUNAME, ENUMNAME) */
    symbol name;    /* the name of the SUE */
    int color;      /* used to categorize SUE nodes */
  } suedgval_t;     /* dg node values */

static dgnode_t *suedg_add_node(dg_t *g, astspec s);


#ifdef DEBUG_SUEDG
/*
 * Visualize the SUE dependency graph
 *
 * Structs are circles, unions are hexagons, enums are parallelograms;
 * used nodes have thicker outlines;
 * globls are gray-filled.
 */
static char *shownode(void *val) 
{
	static char txt[128];
	suedgval_t *v = (suedgval_t *) val;
	char *fillcolor = v->color & SUENODE_COLOR_GLOBAL ? "lightgray" : "white";
	char *shape = (v->color & SUENODE_COLOR_ENUM) ? "parallelogram" :
	              (v->color & SUENODE_COLOR_UNION) ? "hexagon" : "circle";
	int penwidth = v->color & SUENODE_COLOR_USED ? 3 : 1;

	if (v->name == NULL)
		sprintf(txt, "::[shape=%s,penwidth=%d,style=filled,fillcolor=%.15s,"
		             "label=\"(anon)\"]", shape, penwidth, fillcolor);
	else
	sprintf(txt, "::[shape=%s,penwidth=%d,style=filled,fillcolor=%.15s,"
		             "label=\"%.10s\"]", shape, penwidth, fillcolor, v->name->name);
	return txt;
}


void suedg_savegv(dg_t *g, char *filename)
{
	int fd = open(filename, O_CREAT | O_TRUNC | O_WRONLY, 0644);
	int outbak = dup(2);

	if (fd > 0) { dup2(fd, 2); close(fd); }
	dg_visualize(g, shownode);
	dup2(outbak, 2);
	close(outbak);
}
#endif 


/* Adds new nodes and edges if the (struct) fields are structures */
static 
void _suedg_add_deps(dg_t *g, dgnode_t *n, astdecl d)
{
	if (d == NULL) return;
	if ((d)->type == DLIST && d->subtype == DECL_fieldlist)
	{
		_suedg_add_deps(g, n, d->u.next);
		_suedg_add_deps(g, n, d->decl);
	}
	else    /* Should be just a DSTRUCTFIELD */
		if (d->type == DSTRUCTFIELD)
		{
			/* Check if spec has another struct in it */
			astspec s = speclist_getspec(d->spec, SUE, 0);
			if (s != NULL && (s->subtype == SPEC_struct || s->subtype == SPEC_union))
				if (s->name)
					dg_add_edge(n, suedg_add_node(g, s));
		}
}


static 
bool cmpsue(suedgval_t *a, suedgval_t *b)
{
	return (a->ns == b->ns && a->name == b->name);
}


/**
 * @brief Adds a node to the SUE graph if it does not already exist, using
 *        the specifier of a declaration.
 *
 * @param g the SUE graph
 * @param s the whole AST spec of the struct/enum because it may 
 *          contain fields that are structs/unions
 * @return  the new node
 */
static 
dgnode_t *suedg_add_node(dg_t *g, astspec s)
{
	suedgval_t *v;
	dgnode_t *node;
	
	s = speclist_getspec(s, SUE, 0); /* it may have stor. class qual. */
	assert(s && s->name != NULL);
	v = smalloc(sizeof(suedgval_t));
	v->ns = (s->subtype == SPEC_enum) ? ENUMNAME : SUNAME;
	v->name = s->name;
	v->color = 0;
		
	dgvisopts.valcmp = (bool (*)(void*,void*)) cmpsue;
	if ((node = dg_search(g, v)) == NULL)   /* this is a new node */
	{
		dg_add_node(g, node = dg_alloc_node(v));
		if (s->subtype == SPEC_union)
			v->color |= SUENODE_COLOR_UNION;
		if (s->subtype == SPEC_enum)
			v->color |= SUENODE_COLOR_ENUM;
		if (s->subtype != SPEC_enum)
			/* Now add dependencies if it has fields that are structs */
			_suedg_add_deps(g, node, s->u.decl);
	}
	return node;
}


/**
 * @brief Adds a node to the SUE graph from a declaration.
 *        We are not interested in finding all variables that depend on a 
 *        SUE; we simply want to know which SUEs are used in the program.
 *        Thus, we only mark that a SUE is used, without going into the 
 *        declarator details.
 *
 * @param g  the SUE graph
 * @param s  the declaration specifier (u.declaration.spec)
 * @param d  the declaration declarator (u.declaration.decl)
 * @param cf the set of called functions (so as to check if a declared function
 *           is indeed called); if NULL, we assume all declared functions are 
 *           called and thus any SUEs in their prototype are marked as used.
 */
static 
void _add_node_from_declaration(dg_t *g, astspec s, astdecl d, set(cgfun) cf)
{
	dgnode_t *node = suedg_add_node(g, s);
	
	if (!d) return;
	
	/* TODO: for DINIT we may need to check the initializer for casts */
	if (d->type == DINIT || d->type == DECLARATOR)
		_add_node_from_declaration(g, s, d->decl, cf);
	else
		if (d->type == DLIST)
		{
			_add_node_from_declaration(g, s, d->u.next, cf);
			_add_node_from_declaration(g, s, d->decl, cf);
		}
		else
			if (d->type == DFUNC) /* We assume that the function is called */
			  ((suedgval_t *) (node->val))->color |= SUENODE_COLOR_USED;
			else                  /* A variable declaration */
				((suedgval_t *) (node->val))->color |= SUENODE_COLOR_USED;
}


/**
 * @brief Creates a new SUE dependency graph from the global SUE definitions;
 *        all nodes are colored by SUENODE_COLOR_GLOBAL.
 * @return the graph
 */
dg_t *suedg_graph_from_globals()
{
	dg_t *G = dg_new();
	dgnode_t *node;
	stentry e;

	for (e = stab->top; e; e = e->stacknext)
	{
		if (e->scopelevel > 0) continue;  /* skip until globals */
		if ((e->space == SUNAME || e->space == ENUMNAME) && e->spec && 
		    !speclist_getspec(e->spec, STCLASSSPEC, SPEC_typedef))
			if ((node = suedg_add_node(G, e->spec)) != NULL)
				((suedgval_t *) (node->val))->color |= SUENODE_COLOR_GLOBAL;
	}
	return G;
}


/*
 * SUE graph from AST
 */
 
typedef struct {
		dg_t *g;        /* the SUE graph */
		set(cgfun) cf;  /* set of called functions */
	} suedg_trstate_t;

static void declaration_c(aststmt t, void *state, int ignore)
{
	suedg_trstate_t *stt = (suedg_trstate_t *) state;
	astspec s;
	
	/* Check if we have a SUE spec and we are not on a typedef */
	if ((s = speclist_getspec(t->u.declaration.spec, SUE, 0)) != NULL && 
	    !speclist_getspec(t->u.declaration.spec, STCLASSSPEC, SPEC_typedef))
	  /* The spec must be named and either have fields or a var declared */
		if (s->name && (s->u.decl || t->u.declaration.decl))
			_add_node_from_declaration(stt->g, s, t->u.declaration.decl, stt->cf);
}

static void funcdef_c(aststmt t, void *state, int ignore)
{
	suedg_trstate_t *stt = (suedg_trstate_t *) state;
	astspec s;

	/* Return type */
	s = (t->u.declaration.spec) ?
	        speclist_getspec(t->u.declaration.spec, SUE, 0) : NULL;
	if (s && s->name) /* named struct/union/enum */
		_add_node_from_declaration(stt->g, s, t->u.declaration.decl, stt->cf);
}

static void dcasttype_c(astdecl t, void *state, int ignore)
{
	astspec s = speclist_getspec(t->spec, SUE, 0);
	
	if (s && s->name)
	{
		dgnode_t *node = suedg_add_node(((suedg_trstate_t *) state)->g, t->spec);
		((suedgval_t *) (node->val))->color |= SUENODE_COLOR_USED;
	}
}


/**
 * @brief Walks the given AST and adds SUE nodes to the given graph.
 * @param g the SUE dependency graph to add nodes to
 * @param t the subtree to walk
 *
 * @note It only acts on declaration nodes, function definition (to get the 
 *       return type), casts and function parameter declarations. 
 */
void suedg_add_from_ast(dg_t *g, aststmt t, set(cgfun) cf)
{
	suedg_trstate_t state = { g, cf };
	travopts_t trops;
	
	travopts_init_noop(&trops);
	trops.starg = &state;
	
	trops.stmtc.declaration_c = declaration_c;
	trops.stmtc.funcdef_c = funcdef_c;
	trops.declc.dcasttype_c = 
	trops.declc.dparam_c = dcasttype_c;

	ast_stmt_traverse(t, &trops);
}


/*
 * Get only the actually used global SUEs from the graph
 */
 
 
set(vars) usedsuenames;

static void _mark_node(dgnode_t *n)
{
	suedgval_t *v = (suedgval_t *) n->val;
	if (v->color & SUENODE_COLOR_GLOBAL)
		set_put_unique(usedsuenames, v->name);
}


/**
 * @brief Given a SUE dependency graph, determine the used SUEs plus all
 *        SUEs they depend on directly or indirectly. The returned set has
 *        the names of the SUEs in the correct order (i.e. if A depends on B,
 *        then B appears before A in the set) if the set elements are 
 *        accessed from first to last.
 * @param g the graph
 * @return the set of used SUE names in the correct order.
 */
set(vars) suedg_get_used_global_sues(dg_t *g)
{
	setelem(ptr2int) node;
	suedgval_t *v;
	
	dgvisopts.nodefunc = _mark_node;
	dgvisopts.edgefunc = NULL;
	dgvisopts.visorder = DG_TRAV_POSTORDER;
	set_init(vars, &usedsuenames);
	
	for (node = g->nodeset->first ; node; node = node->next)
	{
		v = (suedgval_t *) ((dgnode_t *) node->key)->val;
		if ((v->color & SUENODE_COLOR_USED) && !set_get(usedsuenames, v->name))
			dg_traverse_fromnode(node->key);   /* Get the whole subgraph */
	}
	
	return usedsuenames;
}


/**
 * @brief Given a SUE dependency graph, determine the used SUEs plus all
 *        SUEs they depend on directly or indirectly and create a statement 
 *        list with their declarations.
 * @param g the graph
 * @return a copy of the declarations
 */
aststmt suedg_declare_used_globals(dg_t *g)
{
	set(vars) used;
	setelem(vars) s;
	stentry e;
	aststmt all = NULL, d;

#ifdef DEBUG_SUEDG
	suedg_savegv(g, "_suedg.gv");
#endif

	used = suedg_get_used_global_sues(g);
	for (s = used->first; s; s = s->next)
	{
		if ((e = symtab_get(stab, s->key, SUNAME)) == NULL)
			e = symtab_get(stab, s->key, ENUMNAME);
		assert (e);
		if (e->spec)
		{
			d = Declaration(ast_spec_copy_nosc(e->spec), ast_decl_copy(e->decl));
			all = (all) ? BlockList(d, all) : d;
		}
	}
	return (all);
}
