/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ast_copy.c -- create a copy of an AST */

#include <stdio.h>
#include <string.h>
#include "ast_copy.h"
#include "assorted.h"

/* The _almost functions do the job; the others simply
 * copy the original location information (line,column,file)
 * of the node.
 */

astexpr   ast_expr_copy_almost(astexpr tree);
aststmt   ast_stmt_copy_almost(aststmt stmt);
astdecl   ast_decl_copy_almost(astdecl tree);
ompcon    ast_ompcon_copy_almost(ompcon tree);
ompclause ast_ompclause_copy(ompclause tree);
ompdir    ast_ompdir_copy(ompdir tree);
astspec   ast_spec_copy_nosc_asis(astspec tree);


static astexpr clone_expr(astexpr n)
{ astexpr s = smalloc(sizeof(struct astexpr_)); *s = *n; return (s); }
static astexpr lr_expr(astexpr n)
{
	astexpr s = smalloc(sizeof(struct astexpr_)); *s = *n;
	s->left = ast_expr_copy(n->left);
	s->right = ast_expr_copy(n->right);
	return (s);
}
static aststmt clone_stmt(aststmt n)
{ aststmt s = smalloc(sizeof(struct aststmt_)); *s = *n; return (s); }
static astdecl clone_decl(astdecl n)
{ astdecl s = smalloc(sizeof(struct astdecl_)); *s = *n; return (s); }
static astdecl sd_decl(astdecl n)
{
	astdecl s = smalloc(sizeof(struct astdecl_)); *s = *n;
	if (s->type == DECLARATOR && s->postspec)
		s->postspec = ast_spec_copy(n->postspec);
	s->spec = ast_spec_copy(n->spec);
	s->decl = ast_decl_copy(n->decl);
	return (s);
}
static astspec clone_spec(astspec n)
{ astspec s = smalloc(sizeof(struct astspec_)); *s = *n; return (s); }


aststmt ast_stmt_jump_copy(aststmt tree)
{
	if (tree == NULL) return (NULL);
	switch (tree->subtype)
	{
		case SBREAK:
		case SCONTINUE:
		case SGOTO:
			return (clone_stmt(tree));
		case SRETURN:
			return (Return(ast_expr_copy(tree->u.expr)));
		default:
			fprintf(stderr, "[ast_stmt_jump_copy]: b u g !!\n");
			return (NULL);
	}
}


aststmt ast_stmt_iteration_copy(aststmt tree)
{
	if (tree == NULL) return (NULL);
	switch (tree->subtype)
	{
		case SFOR:
			return (For(ast_stmt_copy(tree->u.iteration.init),
			            ast_expr_copy(tree->u.iteration.cond),
			            ast_expr_copy(tree->u.iteration.incr),
			            ast_stmt_copy(tree->body)));
		case SWHILE:
			return (While(ast_expr_copy(tree->u.iteration.cond),
			              ast_stmt_copy(tree->body)));
		case SDO:
			return (Do(ast_stmt_copy(tree->body),
			           ast_expr_copy(tree->u.iteration.cond)));
		default:
			fprintf(stderr, "[ast_stmt_iteration_copy]: b u g !!\n");
			return (NULL);
	}
}


aststmt ast_stmt_selection_copy(aststmt tree)
{
	if (tree == NULL) return (NULL);
	switch (tree->subtype)
	{
		case SSWITCH:
			return (Switch(ast_expr_copy(tree->u.selection.cond),
			               ast_stmt_copy(tree->body)));
		case SIF:
			return (If(ast_expr_copy(tree->u.selection.cond),
			           ast_stmt_copy(tree->body),
			           ast_stmt_copy(tree->u.selection.elsebody)));
		default:
			fprintf(stderr, "[ast_stmt_selection_copy]: b u g !!\n");
			return (NULL);
	}
}


aststmt ast_stmt_labeled_copy(aststmt tree)
{
	if (tree == NULL) return (NULL);
	switch (tree->subtype)
	{
		case SLABEL:
			return (Labeled(tree->u.label, ast_stmt_copy(tree->body)));
		case SCASE:
			return (Case(ast_expr_copy(tree->u.expr), ast_stmt_copy(tree->body)));
		case SDEFAULT:
			return (Default(ast_stmt_copy(tree->body)));
		default:
			fprintf(stderr, "[ast_stmt_labeled_copy]: b u g !!\n");
			return (NULL);
	}
}


asmop ast_asmop_copy_almost(asmop op)
{
	if (!op)
		return NULL;
	return (AsmOp(ast_expr_copy(op->symbolicname),
	              op->constraint ? strdup(op->constraint) : NULL,
	              ast_expr_copy(op->var),
	              op->op, op->next));
}


aststmt ast_stmt_copy_almost(aststmt tree)
{
	if (tree == NULL) return (NULL);
	switch (tree->type)
	{
		case JUMP:
			return (ast_stmt_jump_copy(tree));
		case ITERATION:
			return (ast_stmt_iteration_copy(tree));
		case SELECTION:
			return (ast_stmt_selection_copy(tree));
		case LABELED:
			return (ast_stmt_labeled_copy(tree));
		case EXPRESSION:
			return (Expression(ast_expr_copy(tree->u.expr)));
		case COMPOUND:
			return (Compound(ast_stmt_copy(tree->body)));
		case STATEMENTLIST:
			return (BlockList(ast_stmt_copy(tree->u.next),
			                  ast_stmt_copy(tree->body)));
		case DECLARATION:
			return (Declaration(ast_spec_copy(tree->u.declaration.spec),
			                    ast_decl_copy(tree->u.declaration.decl)));
		case FUNCDEF:
			return (FuncDef(ast_spec_copy(tree->u.declaration.spec),
			                ast_decl_copy(tree->u.declaration.decl),
			                ast_stmt_copy(tree->u.declaration.dlist),
			                ast_stmt_copy(tree->body)));
		case ASMSTMT:
			return (AsmStmt(tree->subtype,
			                ast_spec_copy(tree->u.assem->qualifiers),
											strdup(tree->u.assem->template),
											ast_asmop_copy(tree->u.assem->outs),
											ast_asmop_copy(tree->u.assem->ins),
											ast_expr_copy(tree->u.assem->clobbers),
											ast_expr_copy(tree->u.assem->labels)));
		case OMPSTMT:
		{
			aststmt t = OmpStmt(ast_ompcon_copy(tree->u.omp));
			t->u.omp->parent = t;  /* Needed */
			if (t->u.omp->body) t->u.omp->body->parent = t;
			return t;
		}
		case VERBATIM:
			return (Verbatim(strdup(tree->u.code)));
		case OX_STMT:
		{
			aststmt t = OmpixStmt(ast_oxcon_copy(tree->u.ox));
			t->u.omp->parent = t;  /* Needed */
			if (t->u.omp->body) t->u.omp->body->parent = t;
			return t;
		}
		default:
			fprintf(stderr, "[ast_stmt_copy]: b u g !!\n");
			return (NULL);
	}
}


astexpr ast_expr_copy_almost(astexpr tree)
{
	if (tree == NULL) return (NULL);
	switch (tree->type)
	{
		case IDENT:
		case DOTDES:
			return (clone_expr(tree));
		case CONSTVAL:
			return (Constant(strdup(tree->u.str)));
		case STRING:
			return (String(strdup(tree->u.str)));
		case FUNCCALL:
		case ARRAYIDX:
		case DOTFIELD:
		case PTRFIELD:
		case BRACEDINIT:
		case BOP:
		case PREOP:
		case POSTOP:
		case ASS:
		case DESIGNATED:
		case IDXDES:
		case COMMALIST:
		case SPACELIST:
			return (lr_expr(tree));
		case CASTEXPR:
			return (CastedExpr(ast_decl_copy(tree->u.dtype),
			                   ast_expr_copy(tree->left)));
		case CONDEXPR:
			return (ConditionalExpr(ast_expr_copy(tree->u.cond),
			                        ast_expr_copy(tree->left),
			                        ast_expr_copy(tree->right)));
		case UOP:
			if (tree->opid == UOP_sizeoftype)
				return (Sizeoftype(ast_decl_copy(tree->u.dtype)));
			else
				if (tree->opid == UOP_alignoftype)
					return (Alignoftype(ast_decl_copy(tree->u.dtype)));
				else
					if (tree->opid == UOP_typetrick)
						return (TypeTrick(ast_decl_copy(tree->u.dtype)));
					else
						return (lr_expr(tree));
		default:
			fprintf(stderr, "[ast_expr_copy]: b u g !!\n");
			return (NULL);
	}
}


static bool _must_skip(int st, int *skipset)
{
	int i;
	if (!skipset) return false;
	for (i = 0; skipset[i] != -1; i++)
		if (skipset[i] == st)
			return true;
	return false;
}

/* Copy by filtering out unwnanted specifiers.
 * If you do not want to filter out anything, pass NULL as the skipset.
 * Otherwise, skipset must be a (-1)-terminated array of unwanted specifilers.
 */
static astspec ast_spec_copy_filtered(astspec tree, int *skipset)
{
	astspec s;
	
	if (tree == NULL) return (NULL);
	switch (tree->type)
	{
		case STCLASSSPEC:
		case SPEC:
			return _must_skip(tree->subtype, skipset) ? NULL : clone_spec(tree);
		case USERTYPE:
			return (clone_spec(tree));
		case SUE:
			switch (tree->subtype)
			{
				case SPEC_enum:
					return (Enumdecl(tree->name, 
					                 ast_spec_copy(tree->body),
					                 ast_spec_copy(tree->sueattr)));
				case SPEC_struct:
				case SPEC_union:
					return (SUdecl(tree->subtype,
					               tree->name, ast_decl_copy(tree->u.decl),
					               ast_spec_copy(tree->sueattr)));
				default:
					fprintf(stderr, "[ast_spec_copy]: SUE b u g !!\n");
					return (NULL);
			}
		case ENUMERATOR:
			s = Enumerator(tree->name, ast_expr_copy(tree->u.expr));
			CopyPosInfo(s, tree);
			return s;
		case SPECLIST:
		{
			astspec next;

			s = ast_spec_copy_filtered(tree->body, skipset);
			next = ast_spec_copy_filtered(tree->u.next, skipset);
			if (s != NULL && next != NULL)
			{
				s = Specifierlist(tree->subtype, s, next);
				CopyPosInfo(s, tree);
				return s;
			}
			if (s != NULL)
				return (s);
			else
				return (next);
		}
		case ATTRSPEC:
			s = AttrSpec(tree->u.txt ? strdup(tree->u.txt) : NULL);
			if (s) CopyPosInfo(s, tree);
			return s;
		case ASMNAMESPEC:
			s = AsmNameSpec(tree->u.txt ? strdup(tree->u.txt) : NULL);
			if (s) CopyPosInfo(s, tree);
			return s;
		case TYPEOFEXPR:
			s = TypeofExpr(ast_expr_copy(tree->u.expr));
			if (s) CopyPosInfo(s, tree);
			return s;
		case TYPEOFTYPE:
			s = TypeofType(ast_decl_copy(tree->u.decl));
			if (s) CopyPosInfo(s, tree);
			return s;
		default:
			fprintf(stderr, "[ast_spec_copy_filtered]: b u g (%d)!!\n", tree->type);
			return (NULL);
	}
}


astdecl ast_decl_copy_almost(astdecl tree)
{
	if (tree == NULL) return (NULL);
	switch (tree->type)
	{
		case DIDENT:
		case DELLIPSIS:
			return (clone_decl(tree));
		case DPAREN:
			return (ParenDecl(ast_decl_copy(tree->decl)));
		case DARRAY:
			return (ArrayDecl(ast_decl_copy(tree->decl),
			                  ast_spec_copy(tree->spec),
			                  ast_expr_copy(tree->u.expr)));
		case DFUNC:      /* Maybe abstract declarator */
			return (FuncDecl(ast_decl_copy(tree->decl),
			                 ast_decl_copy(tree->u.params)));
		case DINIT:
			return (InitDecl(ast_decl_copy(tree->decl),
			                 ast_expr_copy(tree->u.expr)));
		case DECLARATOR:
		case ABSDECLARATOR:
		case DPARAM:
		case DSTRUCTFIELD:
		case DCASTTYPE:
			return (sd_decl(tree));
		case DBIT:
			return (BitDecl(ast_decl_copy(tree->decl),
			                ast_expr_copy(tree->u.expr)));
		case DLIST:
			return (Declanylist(tree->subtype,
			                    ast_decl_copy(tree->u.next),
			                    ast_decl_copy(tree->decl)));
		default:
			fprintf(stderr, "[ast_decl_copy]: b u g (type %d) !!\n", tree->type);
			return (NULL);
	}
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     OpenMP NODES                                              *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


static omparrdim ast_omparrdim_copy(omparrdim d)
{
	omparrdim new = NULL;
	
	if (d)
	{
		new = OmpArrDim(ast_expr_copy(d->lb), ast_expr_copy(d->len));
		new->next = ast_omparrdim_copy(d->next);
	}
	return (new);
}


ompxli ast_ompxli_copy(ompxli xl)
{
	ompxli new = NULL;
	
	if (xl)
	{
		if (xl->xlitype == OXLI_IDENT)
			new = PlainXLI(xl->id);
		else
			new = ArraySection(xl->id, ast_omparrdim_copy(xl->dim));
		new->next = ast_ompxli_copy(xl->next);
	}
	return (new);
}


ompclause ast_ompclause_copy_almost(ompclause t)
{
	ompclause new;
	
	if (t == NULL) return (NULL);
	switch (t->type)
	{
		case OCIF:
			new = IfClause(ast_expr_copy(t->u.expr));
			break;
		case OCFINAL:
			new = FinalClause(ast_expr_copy(t->u.expr));
			break;
		case OCNUMTHREADS:
			new = NumthreadsClause(ast_expr_copy(t->u.expr));
			break;
		case OCNUMTEAMS:
			new = NumteamsClause(ast_expr_copy(t->u.x2.expr1), 
			                     ast_expr_copy(t->u.x2.expr2));
			break;
		case OCTHREADLIMIT:
			new = ThreadlimitClause(ast_expr_copy(t->u.expr));
			break;
		case OCHINT:
			new = HintClause(ast_expr_copy(t->u.expr));
			break;
		case OCPRIORITY:
			new = PriorityClause(ast_expr_copy(t->u.expr));
			break;
		case OCDEVICE:
			new = DeviceClause(ast_expr_copy(t->u.expr));
			break;
		case OCDISTSCHEDULE:
			new = DistScheduleClause(t->subtype, ast_expr_copy(t->u.expr));
			break;
		case OCSCHEDULE:
			new = ScheduleClause(t->subtype, ast_expr_copy(t->u.expr));
			break;
		case OCDEFAULT:
			new = DefaultClause(t->subtype);
			break;
		case OCATOMICDEFAULTMEMORDER:
			new = AtomicDefaultMemOrderClause(t->subtype);
			break;
		case OCREDUCTION:
			new = ReductionClause(t->subtype, ast_ompxli_copy(t->u.xlist));
			break;
		case OCINREDUCTION:
			new = InReductionClause(t->subtype, ast_ompxli_copy(t->u.xlist));
			break;
		case OCTASKREDUCTION:
			new = TaskReductionClause(t->subtype, ast_ompxli_copy(t->u.xlist));
			break;
		case OCDEPEND:
			if (t->subtype == OC_sink)
			{
				new = DependClause(t->subtype, NULL);
				new->u.expr = ast_expr_copy(t->u.expr);
			}
			else
				new = DependClause(t->subtype, ast_ompxli_copy(t->u.xlist));
			break;
		case OCMAP:
			new = MapClause(t->subtype, ast_ompxli_copy(t->u.xlist));
			break;
		case OCTO:
		case OCFROM:
		case OCLINK:
			new = UpdateClause(t->type, ast_ompxli_copy(t->u.xlist));
			break;
		case OCNOWAIT:
		case OCORDERED:
		case OCUNTIED:
		case OCMERGEABLE:
		case OCPARALLEL:
		case OCSECTIONS:
		case OCFOR:
		case OCTASKGROUP:
		case OCDEFAULTMAP:
		case OCREVERSEOFFLOAD:
		case OCUNIFIEDADDRESS:
		case OCUNIFIEDSHAREDMEMORY:
		case OCDYNAMICALLOCATORS:
		case OCREAD:
		case OCWRITE:
		case OCUPDATE:
		case OCCAPTURE:
		case OCSEQCST:
		case OCACQREL:
		case OCACQUIRE:
		case OCRELEASE:
		case OCRELAXED:
		case OCCOMPARE:
		case OCWEAK:
		/* new-clause.sh:ast_copy_plain */
			new = PlainClause(t->type);
			break;
		case OCCOPYIN:
		case OCPRIVATE:
		case OCCOPYPRIVATE:
		case OCFIRSTPRIVATE:
		case OCLASTPRIVATE:
		case OCSHARED:
		case OCISDEVPTR:
		case OCUSEDEVPTR:
		case OCAUTO:
			new = VarlistClause(t->type, ast_decl_copy(t->u.varlist));
			break;
		case OCLIST:
			new = OmpClauseList(ast_ompclause_copy(t->u.list.next),
			                      ast_ompclause_copy(t->u.list.elem));
			break;
		case OCCOLLAPSE:
			new = CollapseClause(t->subtype);
			break;
		case OCORDEREDNUM:
			new = OrderedNumClause(t->subtype);
			break;
		case OCPROCBIND:
			new = ProcBindClause(t->subtype);
			break;
		case OCFAIL:
			new = FailClause(t->subtype);
			break;
		case OCTAG:
			new = TagClause(ast_expr_copy(t->u.expr));
			break;
		case OCFILTER:
			new = FilterClause(ast_expr_copy(t->u.expr));
			break;
		case OCAT:
			new = AtClause(t->subtype);
			break;
		case OCSEVERITY:
			new = SeverityClause(t->subtype);
			break;
		case OCMESSAGE:
			new = MessageClause(t->s->name);
			break;
		case OCEXT:
			new = OmpExtClause(t->s);
			break;
		/* new-clause.sh:ast_copy_expr */
		default:
			fprintf(stderr, "[ast_ompclause_copy_almost]: b u g !!\n");
			return (NULL);
	}
	
	/* Copy the modifiers, too */
	if (t->modifs)
		new->modifs = intvec_copy(t->modifs);
	return new;
}


ompdir ast_ompdir_copy_almost(ompdir t)
{
	if (t == NULL) return (NULL);
	switch (t->type)
	{
		case DCCRITICAL:
			return (OmpCriticalDirective(t->u.region,ast_ompclause_copy(t->clauses)));
		case DCTHREADPRIVATE:
			return (OmpThreadprivateDirective(ast_decl_copy(t->u.varlist)));
		case DCFLUSH:
			return (OmpFlushDirective(ast_decl_copy(t->u.varlist), 
			                          ast_ompclause_copy(t->clauses)));
		default:
			return (OmpDirective(t->type, ast_ompclause_copy(t->clauses)));
	}
}


ompcon ast_ompcon_copy_almost(ompcon t)
{
	if (t == NULL) return (NULL);
	return ( OmpConstruct(t->type,  /* v45 #declare target may NOT have a body */
	                      ast_ompdir_copy(t->directive),
	                      ast_stmt_copy(t->body)) );
}

astexpr ast_expr_copy(astexpr tree)
{
	astexpr t = ast_expr_copy_almost(tree);
	CopyPosInfo(t, tree);
	return (t);
}


asmop ast_asmop_copy(asmop tree)
{
	asmop t = ast_asmop_copy_almost(tree);
	return (t);
}


aststmt ast_stmt_copy(aststmt tree)
{
	aststmt t = ast_stmt_copy_almost(tree);
	if (t) t->parent = tree->parent;
	CopyPosInfo(t, tree);
	return (t);
}


astdecl ast_decl_copy(astdecl tree)
{
	astdecl t = ast_decl_copy_almost(tree);
	CopyPosInfo(t, tree);
	return (t);
}


astspec ast_spec_copy(astspec tree)
{
	return ast_spec_copy_filtered(tree, NULL);
}


/* Versions that do not copy storage class specifiers (_nosc) and type 
 * qualifiers (_nosc_notq); well, maybe we also need a version that keeps the 
 * const qualifier (?). 
 * The non-"asis" version adds an "int" specifier in case the specifier list 
 * becomes empty.
 */


astspec ast_spec_copy_nosc_asis(astspec tree)
{
	static int reject[] = { SPEC_typedef, SPEC_extern, SPEC_static, SPEC_auto, 
	                        SPEC_register, SPEC_const, -1 }; 
	return ast_spec_copy_filtered(tree, reject);
}


/* This adds an "int" type in case the specifier list becomes empty */
astspec ast_spec_copy_nosc(astspec tree)
{
	astspec t = ast_spec_copy_nosc_asis(tree);
	if (t == NULL)
	{
		t = Declspec(SPEC_int);
		CopyPosInfo(t, tree);
	}
	return (t);
}


astspec ast_spec_copy_nosc_notq_asis(astspec tree)
{
	static int reject[] = { SPEC_typedef,  SPEC_extern, SPEC_static, SPEC_auto, 
	                        SPEC_register, SPEC_const,  SPEC_const,  SPEC_restrict,
									        SPEC_volatile, SPEC_inline, -1 };
	return ast_spec_copy_filtered(tree, reject);
}


astspec ast_spec_copy_nosc_notq(astspec tree)
{
	astspec t = ast_spec_copy_nosc_notq_asis(tree);
	if (t == NULL)
	{
		t = Declspec(SPEC_int);
		CopyPosInfo(t, tree);
	}
	return (t);
}


ompclause ast_ompclause_copy(ompclause tree)
{
	ompclause t = ast_ompclause_copy_almost(tree);
	if (t) ast_ompclause_parent(tree->parent, t);
	CopyPosInfo(t, tree);
	return (t);
}


ompdir ast_ompdir_copy(ompdir tree)
{
	ompdir t = ast_ompdir_copy_almost(tree);
	if (t) t->parent = tree->parent;
	CopyPosInfo(t, tree);
	return (t);
}


ompcon  ast_ompcon_copy(ompcon tree)
{
	ompcon t = ast_ompcon_copy_almost(tree);
	if (t) t->parent = tree->parent;
	CopyPosInfo(t, tree);
	return (t);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     OMPi-EXTENSION NODES                                      *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


oxclause ast_oxclause_copy_almost(oxclause t)
{
	if (t == NULL) return (NULL);
	switch (t->type)
	{
		case OX_OCATALL:
		case OX_OCTIED:
		case OX_OCUNTIED:
		case OX_OCDETACHED:
		case OX_OCLOCAL:
		case OX_OCREMOTE:
			return (OmpixPlainClause(t->type));
		case OX_OCIN:
		case OX_OCOUT:
		case OX_OCINOUT:
			return (OmpixVarlistClause(t->type, ast_decl_copy(t->u.varlist)));
		case OX_OCREDUCE:
			return (OmpixReductionClause(t->operator, ast_decl_copy(t->u.varlist)));
		case OX_OCATNODE:
			return (OmpixAtnodeClause(t->u.expr));
		case OX_OCATWORKER:
			return (OmpixAtworkerClause(t->u.expr));
		case OX_OCIF:
			return (OmpixIfClause(t->u.expr));
		case OX_OCHINTS:
			return (OmpixHintsClause(t->u.expr));
		case OX_OCLIST:
			return (OmpixClauseList(ast_oxclause_copy(t->u.list.next),
			                        ast_oxclause_copy(t->u.list.elem)));
		case OX_OCSTART:
			return (OmpixStartClause(t->u.expr));
		case OX_OCSTRIDE:
			return (OmpixStrideClause(t->u.expr));
		case OX_OCSCOPE:
			return (OmpixScopeClause(t->u.value));
		case OX_OCTAG:
			return (OmpixClause(OX_OCTAG, NULL, t->u.expr));
	}
	return (NULL);
}


oxdir ast_oxdir_copy_almost(oxdir t)
{
	if (t == NULL) return (NULL);
	if (t->type == OX_DCPROCEXT)
	{
		oxdir d = OmpixDirective(OX_DCPROCEXT, NULL);
		d->varlist = ast_decl_copy(t->varlist);
		return d;
	}
	return (OmpixDirective(t->type, ast_oxclause_copy(t->clauses)));
}


oxcon ast_oxcon_copy_almost(oxcon t)
{
	if (t == NULL) return (NULL);
	return (OmpixConstruct(t->type,
	                       ast_oxdir_copy(t->directive),
	                       ast_stmt_copy(t->body)));
}


oxclause ast_oxclause_copy(oxclause tree)
{
	oxclause t = ast_oxclause_copy_almost(tree);
	if (t) ast_oxclause_parent(tree->parent, t);
	CopyPosInfo(t, tree);
	return (t);
}


oxdir  ast_oxdir_copy(oxdir tree)
{
	oxdir t = ast_oxdir_copy_almost(tree);
	if (t) t->parent = tree->parent;
	CopyPosInfo(t, tree);
	return (t);
}


oxcon  ast_oxcon_copy(oxcon tree)
{
	oxcon t = ast_oxcon_copy_almost(tree);
	if (t) t->parent = tree->parent;
	CopyPosInfo(t, tree);
	return (t);
}
