/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* ast_assorted.c - Assorted functions and utilities related to the AST */

#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>
#include <string.h>
#include "ast_copy.h"
#include "ast_show.h"
#include "ast_assorted.h"
#include "ast_print.h"
#include "ompi.h"
#include "str.h"


/**
 * @brief Given the u.decl of a struct/union, find the given field declaration
 * 
 * @param d  the u.decl of the spec of a struct or union
 * @param f  the field symbol we need to find
 * @return the field declaration
 */
astdecl _sufield(astdecl d, symbol f)
{
	astdecl r;
	
	if (d == NULL) return NULL;
	if (d->type == DLIST && d->subtype == DECL_fieldlist)
	{
		if ((r = _sufield(d->u.next, f)) != NULL)
			return r;
		return _sufield(d->decl, f);
	}
	else    /* Should be just a DSTRUCTFIELD */
	{
		astdecl r;

		if (d->type != DSTRUCTFIELD) return NULL;
		r = decl_getidentifier(d->decl);
		return (r && r->u.id == f) ? d : NULL;
	}
}


/**
 * @brief Get the filed of a struct or union
 * 
 * @param s the name of the struct or union
 * @param f the name of the field
 * @return the structfield or NULL if s or f are unknown
 */
astdecl get_sufield(symbol s, symbol f)
{
	stentry e = symtab_get(stab, s, SUNAME);
	astspec sp;
	
	/* Sanity checks that the struct or union exists */
	if (!e) return NULL;
	if (!(sp = e->spec)) return NULL;
	if ((sp = speclist_getspec(sp, SUE, 0)) == NULL) return NULL;
	if (sp->subtype != SPEC_struct && sp->subtype != SPEC_union) return NULL;
	
	return _sufield(sp->u.decl, f);
}


aststmt linepragma(int line, symbol file)
{
	return ((cppLineNo) ? verbit("# %d \"%s\"", line, file->name) : verbit(""));
}


/* This simply prints the directive into a string and encloses it
 * in comments, adding also the source line number.
 * It returns a verbatim node with the comment.
 */
aststmt ompdir_commented(ompdir d)
{
	static str bf = NULL;
	aststmt    st;

	if (bf == NULL) bf = Strnew();
	str_printf(bf, "/* (l%d) ", d->l);
	ast_ompdir_print(bf, d);
	str_seek(bf, str_tell(bf)-1);      /* get rid of the \n */
	str_printf(bf, " */");

	st = Verbatim(strdup(str_string(bf)));
	if (cppLineNo) 
	  st = BlockList(st, linepragma(d->l, d->file));
	  
	str_truncate(bf);
	return (st);
}


/** 
 * Compares two expression trees and returns true if they are identical
 */
bool same_expr(astexpr e, astexpr f)
{
	if (!e && !f) return true;
	if (!e || !f) return false;
	if (e->type != f->type) return false;
	switch (e->type)
	{
		case IDENT:
		case DOTDES:
			return (e->u.sym == f->u.sym);
		case CONSTVAL:
		case STRING:
			return (strcmp(e->u.str, f->u.str) == 0);
		case FUNCCALL:
		case ARRAYIDX:
		case DESIGNATED:
		case SPACELIST:
		case COMMALIST:
			return same_expr(e->left, f->left) && 
			       same_expr(e->right, f->right);
		case PREOP:
		case POSTOP:
		case IDXDES:
		case BRACEDINIT:
			return same_expr(e->left, f->left); 
		case DOTFIELD:
		case PTRFIELD:
			return e->u.sym == f->u.sym && same_expr(e->left, f->left);
		case CONDEXPR:
			return same_expr(e->u.cond, f->u.cond) && 
			       same_expr(e->left, f->left) && 
			       same_expr(e->right, f->right);
		case BOP:
		case ASS:
			return e->opid == f->opid && 
			       same_expr(e->left, f->left) && 
			       same_expr(e->right, f->right);
		case CASTEXPR:
			return same_decl(e->u.dtype, f->u.dtype) &&
			       same_expr(e->left, f->left);
		case UOP:
			if (e->opid != f->opid) return false;
			if (e->opid == UOP_sizeoftype || e->opid == UOP_alignoftype || 
			    e->opid == UOP_typetrick)
				return same_decl(e->u.dtype, f->u.dtype);
			else 
				return same_expr(e->left, f->left);
		default:
			fprintf(stderr, "[same_expr]: b u g !!\n");
			return false;
	}
}


/** 
 * Compares two specifier trees and returns true if they are identical
 */
bool same_spec(astspec s, astspec t)
{
	if (!s && !t) return true;
	if (!s || !t) return false;
	if (s->type != t->type) return false;
	
	switch (s->type)
	{
		case SPEC:
		case STCLASSSPEC:
			return s->subtype == t->subtype;
		case USERTYPE:
			return s->name == t->name;
		case SUE:
			if (s->subtype != t->subtype) return false;
			switch (s->subtype)
			{
				case SPEC_enum:
					return s->name == t->name &&
					       same_spec(s->body, t->body) &&
					       same_spec(s->sueattr, t->sueattr);
				case SPEC_struct:
				case SPEC_union:
				default:  /* should normally emit an error */
					return s->name == t->name &&
					       same_decl(s->u.decl, t->u.decl) &&
					       same_spec(s->sueattr, t->sueattr);
			}
			break;
		case ENUMERATOR:
			return s->name == t->name && same_expr(s->u.expr, t->u.expr);
		case SPECLIST:
			return s->subtype == t->subtype &&
			       same_spec(s->body, t->body) &&
			       same_spec(s->u.next, t->u.next);
		case ATTRSPEC:
		case ASMNAMESPEC:
			return (strcmp(s->u.txt, t->u.txt) == 0);
		case TYPEOFEXPR:
			return (same_expr(s->u.expr, t->u.expr));
		case TYPEOFTYPE:
			return (same_decl(s->u.decl, t->u.decl));
		default:
			fprintf(stderr, "[same_spec]: b u g !!\n");
			return false;
	}
}


/** 
 * Compares two declarator trees and returns true if they are identical
 */
bool same_decl(astdecl d, astdecl e)
{
	if (!d && !e) return true;
	if (!d || !e) return false;
	switch (d->type)
	{
		case DIDENT:
			return d->u.id == e->u.id;
		case DPAREN:
			return same_decl(d->decl, e->decl);
		case DARRAY:
			return same_decl(d->decl, e->decl) &&
			       same_spec(d->spec, e->spec) &&
			       same_expr(d->u.expr, e->u.expr);
		case DFUNC:      /* Maybe abstract declarator */
			return same_decl(d->decl, e->decl) &&
			       same_decl(d->u.params, e->u.params);
		case DINIT:
		case DBIT:
			return same_decl(d->decl, e->decl) &&
			       same_expr(d->u.expr, e->u.expr);
		case DECLARATOR:
			if (d->type == DECLARATOR && !same_spec(d->postspec, e->postspec))
				return false;
			/* else fall through */
		case ABSDECLARATOR:
		case DPARAM:
		case DSTRUCTFIELD:
		case DCASTTYPE:
			return same_decl(d->decl, e->decl) &&
			       same_spec(d->spec, e->spec);
		case DELLIPSIS:
			return true;
		case DLIST:
			return d->subtype == e->subtype &&
			       same_decl(d->u.next, e->u.next) &&
			       same_decl(d->decl, e->decl);
		default:
			fprintf(stderr, "[same_decl]: b u g (%d) !!\n", d->type);
			return false;
	}
}


/** 
 * Compares xlists and returns true if they are identical
 */
bool same_xlist(ompxli x, ompxli y)
{
	omparrdim d, e;
	
	for ( ; ; x = x->next, y = y->next)
	{
		if (!x && !y) return true;
		if (!x || !y) return false;
		if (x->xlitype != y->xlitype) return false;
		if (x->id != y->id) return false;
		/* compare dimensions */
		for (d = x->dim, e = y->dim; d && e; d = d->next, e = e->next)
			if (!same_expr(d->lb, e->lb) || !same_expr(d->len, e->len))
				return false;
		if (d != e) return false;
	} 
	return true;
}


/* Given a (valid) BOP, it returns the corresponding assignment operator 
 * (e.g. BOP_add -> ASS_add).
 */
int bop2assop(int bop)
{
	switch (bop) 
	{
		case BOP_add:  return ASS_add;
		case BOP_sub:  return ASS_sub;
		case BOP_mul:  return ASS_mul;
		case BOP_div:  return ASS_div;
		case BOP_mod:  return ASS_mod;
		case BOP_band: return ASS_and;
		case BOP_xor:  return ASS_xor;
		case BOP_bor:  return ASS_or ;
		case BOP_shl:  return ASS_shl;
		case BOP_shr:  return ASS_shr;
		default: break;
	}
	return ASS_eq;
}


/* Given a (valid) assignment operator, it returns the corresponding BOP 
 * (e.g. ASS_add -> BOP_add).
 */
int assop2bop(int assop)
{
	switch (assop)
	{
		case ASS_add: return BOP_add;
		case ASS_sub: return BOP_sub;
		case ASS_mul: return BOP_mul;
		case ASS_div: return BOP_div;
		case ASS_mod: return BOP_mod;
		case ASS_shl: return BOP_shl;
		case ASS_shr: return BOP_shr;
		case ASS_and: return BOP_band;
		case ASS_xor: return BOP_xor;
		case ASS_or:  return BOP_bor;
		default:
			break;
	}
	return -1;
}
