/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* AST - the Abstract Syntax Tree */

/*
 * 2009/05/11:
 *   added AUTO schedule type
 * 2009/05/03:
 *   added ATNODE ompix clause
 */

#include <stdlib.h>
#include <stdarg.h>
#include <assert.h>
#include <string.h>
#include "ast.h"
#include "ast_copy.h"
#include "ast_show.h"
#include "ompi.h"
#include "autoscope.h"
#include "set.h"
#include "ast_free.h"
#include "assorted.h"
#include "ast_xform.h"

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     EXRESSION NODES                                           *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


char *UOP_symbols[12] =
{
	"&", "*", "-", "~", "!", "sizeof", "sizeof", "_Alignof", "++", "--", "(", "\0"
},
*BOP_symbols[19] =
{
	"<<", ">>", "<=", ">=", "==", "!=", "&&", "||", "&", "|", "^",
	"+", "-", "<", ">", "*", "/", "%", "cast"
},
*ASS_symbols[11] =
{
	"=", "*=", "/=", "%=", "+=", "-=", "<<=", ">>=", "&=", "^=", "|="
};


astexpr Astexpr(enum exprtype type, astexpr left, astexpr right)
{
	astexpr n = smalloc(sizeof(struct astexpr_));
	n->type   = type;
	n->left   = left;
	n->right  = right;
	n->opid   = 0;
	n->l      = sc_original_line();
	n->c      = sc_column();
	n->file   = Symbol(sc_original_file());
	return (n);
}

astexpr Identifier(symbol s)
{
	astexpr n = Astexpr(IDENT, NULL, NULL);
	n->u.sym = s;
	return (n);
}

astexpr Constant(char *format, ...)
{
	static char str[1024];
	astexpr n = Astexpr(CONSTVAL, NULL, NULL);
	va_list ap;

	va_start(ap, format);
	vsnprintf(str, 1023, format, ap);
	va_end(ap);
	n->u.str = strdup(str);
	return n;
}


astexpr numConstant(int n)
{
	static char numstr[64];
	snprintf(numstr, 63, "%d", (n >= 0) ? n : -n);
	return n >= 0 ? Constant(strdup(numstr)) : 
	                UnaryOperator(UOP_neg, Constant(strdup(numstr)));
}

astexpr String(char *s)
{
	astexpr n = Astexpr(STRING, NULL, NULL);
	n->u.str = s;
	return (n);
}

astexpr DotField(astexpr e, symbol s)
{
	astexpr n = Astexpr(DOTFIELD, e, NULL);
	n->u.sym = s;
	return (n);
}

astexpr PtrField(astexpr e, symbol s)
{
	astexpr n = Astexpr(PTRFIELD, e, NULL);
	n->u.sym = s;
	return (n);
}

astexpr Operator(enum exprtype type, int opid, astexpr left, astexpr right)
{
	astexpr n = Astexpr(type, left, right);
	n->opid = opid;
	return (n);
}

astexpr ConditionalExpr(astexpr cond, astexpr t, astexpr f)
{
	astexpr n = Astexpr(CONDEXPR, t, f);
	n->u.cond = cond;
	return (n);
}

astexpr DotDesignator(symbol s)
{
	astexpr n = Astexpr(DOTDES, NULL, NULL);
	n->u.sym = s;
	return (n);
}

astexpr CastedExpr(astdecl d, astexpr e)
{
	astexpr n = Astexpr(CASTEXPR, e, NULL);
	n->u.dtype = d;
	return (n);
}

astexpr Sizeoftype(astdecl d)
{
	astexpr n = UnaryOperator(UOP_sizeoftype, NULL);
	n->u.dtype = d;
	return (n);
}


astexpr Alignoftype(astdecl d)
{
	astexpr n = UnaryOperator(UOP_alignoftype, NULL);
	n->u.dtype = d;
	return (n);
}


astexpr TypeTrick(astdecl d)
{
	astexpr n = UnaryOperator(UOP_typetrick, NULL);
	n->u.dtype = d;
	return (n);
}


/* Find the number of elements in a list (1 if not a COMMA/SPACE LIST) */
int expr_list_cardinality(astexpr expr)
{
	if (expr->type != COMMALIST && expr->type != SPACELIST)
		return (1);
	return ( expr_list_cardinality(expr->left) + 
	         expr_list_cardinality(expr->right) );
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     DECLARATION NODES                                         *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/*
 * SPECIFIER NODES
 */

char *SPEC_symbols[28] =
{
	NULL, "typedef", "extern", "static", "auto", "register", "void",
	"char", "short", "int", "long", "float", "double", "signed",
	"unsigned", "_Bool", "_Complex", "_Imaginary", "struct",
	"union", "enum", "const", "restrict", "volatile", "inline",
	"*", "speclistR", "speclistL"
};


astspec Specifier(enum spectype type, int subtype, symbol name, astspec body)
{
	astspec d  = smalloc(sizeof(struct astspec_));
	d->type    = type;
	d->subtype = subtype;
	d->name    = name;
	d->body    = body;
	d->sueattr = NULL;
	d->u.next  = NULL;
	d->l       = sc_original_line();
	d->c       = sc_column();
	d->file    = Symbol(sc_original_file());
	return (d);
}


astspec Enumerator(symbol name, astexpr expr)
{
	astspec d  = Specifier(ENUMERATOR, 0, name, NULL);
	d->u.expr = expr;
	return (d);
}


astspec Specifierlist(int type, astspec e, astspec l)
{
	astspec d  = Specifier(SPECLIST, type, NULL, e);
	d->u.next = l;
	return (d);
}


astspec SUdecl(int type, symbol sym, astdecl decl, astspec attr)
{
	astspec d = Specifier(SUE, type, sym, NULL);
	d->sueattr = attr;
	d->u.decl = decl;
	return (d);
}


astspec Enumdecl(symbol sym, astspec body, astspec attr)
{
	astspec d = Specifier(SUE, SPEC_enum, sym, body);
	d->sueattr = attr;
	return (d);
}


/* For GCC-like attributes */
astspec AttrSpec(char *s)
{
	astspec d = Specifier(ATTRSPEC, 0, NULL, NULL);
	d->u.txt = s;
	return (d);
}


/* Used for function declarations that need to specify a different name for 
 * the assembler (see GGC docs "Controlling Names Used in Assembler Code
 */
astspec AsmNameSpec(char *s)
{
	astspec d = Specifier(ASMNAMESPEC, 0, NULL, NULL);
	d->u.txt = s;
	return (d);
}


astspec TypeofExpr(astexpr e)
{
	astspec d = Specifier(TYPEOFEXPR, 0, NULL, NULL);
	d->u.expr = e;
	return (d);
}


astspec TypeofType(astdecl e)
{
	astspec d = Specifier(TYPEOFTYPE, 0, NULL, NULL);
	d->u.decl = e;
	return (d);
}


/*
 * DECLARATOR NODES
 */


astdecl Decl(enum decltype type, int subtype, astdecl decl, astspec spec)
{
	astdecl d   = smalloc(sizeof(struct astdecl_));
	d->type     = type;
	d->subtype  = subtype;
	d->decl     = decl;
	d->spec     = spec;
	d->postspec = NULL;
	d->u.next   = NULL;
	d->l        = sc_original_line();
	d->c        = sc_column();
	d->file     = Symbol(sc_original_file());
	return (d);
}


astdecl IdentifierDecl(symbol s)
{
	astdecl d = Decl(DIDENT, 0, NULL, NULL);
	d->u.id   = s;
	return (d);
}


astdecl ArrayDecl(astdecl decl, astspec s, astexpr e)
{
	astdecl d = Decl(DARRAY, 0, decl, s);
	d->u.expr = e;
	return (d);
}


astdecl FuncDecl(astdecl decl, astdecl p)
{
	astdecl d = Decl(DFUNC, 0, decl, NULL);
	d->u.params = p;
	return (d);
}


astdecl InitDecl(astdecl decl, astexpr e)
{
	astdecl d = Decl(DINIT, 0, decl, NULL);
	d->u.expr = e;
	return (d);
}


astdecl BitDecl(astdecl decl, astexpr e)
{
	astdecl d = Decl(DBIT, 0, decl, NULL);
	d->u.expr = e;
	return (d);
}


astdecl Declanylist(int subtype, astdecl l, astdecl e)
{
	astdecl d = Decl(DLIST, subtype, e, NULL);
	d->u.next = l;
	return (d);
}


/* Get the type of identifier declared - can be:
 *   - a scalar one
 *   - an array
 *   - a function
 * d is assumed to be a declarator node. As such it only has a ptr and
 * a direct_declarator child.
 */
int decl_getkind(astdecl d)
{
	assert(d->type == DECLARATOR);
	if (decl_ispointer(d)) return (DIDENT);   /* pointers are scalar */
	d = d->decl;                              /* direct_declarator */

	switch (d->type)
	{
		case DPAREN:
			return (decl_getkind(d->decl));
		case DFUNC:
			return (DFUNC);
		case DARRAY:
			return (DARRAY);
		case DIDENT:
			return (DIDENT);
		case DECLARATOR:       /* Should not happen normally */
			return (decl_getkind(d));
		default:
			exit_error(1, "[decl_getkind]: unexpected declarator type %d\n", d->type);
	}
	return (0);
}


/* Determine whether the declarator is a pointer. Here we are based on the
 * fact that the parser has removed redundant parenthesis, i.e.
 * (IDENT) has been converted to IDENT. Thus we can only have a pointer
 * if we are in a situation like ...(*IDENT)...; i.e. an identifier
 * declarator with pointer specifier, or ...(*)... for an abstract declarator.
 */
int decl_ispointer(astdecl d)
{
	if (d->type == DECLARATOR || d->type == ABSDECLARATOR)
	{
		if (d->spec != NULL  &&
		    (d->decl == NULL || (d->decl->type == DIDENT &&
		                       speclist_getspec(d->spec, SPEC, SPEC_star) != NULL)))
			return (1);
	}
	if (d->type == DIDENT)
		return (0);
	else
		if (d->type == DLIST)  /* Should be DECL_decllist */
			return (decl_ispointer(d->u.next));
		else
			return (decl_ispointer(d->decl));
}


/* Determine whether the declarator is a function returning a pointer. 
 * It ia assumed that this comes from a declaration statement or a function
 * definition statement.
 * We keep recursing on the decl field until we hit a DFUNC.
 */
int func_returnspointer(astdecl d)
{
	if (d == NULL) return 0;
	
	if (d->type == DECLARATOR)
		if (d->decl && d->decl->type == DFUNC)
			return (d->spec && speclist_getspec(d->spec, SPEC, SPEC_star));
	if (d->type == DIDENT)
		return (0);
	else
		if (d->type == DLIST)  /* Should never happen */
			return (func_returnspointer(d->u.next));
		else
			return (func_returnspointer(d->decl));
}


/* Get the identifier name of the declarator.
 * d is assumed to be the declarator part of a declaration (top-level).
 * It will crash if given an ABSDECLARATOR with no identifier!
 */
astdecl decl_getidentifier(astdecl d)
{
	if (d->type == DIDENT)
		return (d);
	else
		if (d->type == DLIST)  /* Should be DECL_decllist */
			return (decl_getidentifier(d->u.next));
		else
			return (decl_getidentifier(d->decl));
}


/* Finds the identifier and replaces it with a pointer to it.
 * Returns the declarator itself.
 */
astdecl decl_topointer(astdecl decl)
{
	astdecl newdecl, id = decl_getidentifier(decl);

	newdecl = ParenDecl(Declarator(Pointer(), ast_decl_copy(id)));
	*id = *newdecl;
	free(newdecl);
	return (decl);
}


/* Finds the identifier and replaces its name.
 * Returns the declarator itself.
 */
astdecl decl_rename(astdecl decl, symbol newname)
{
	astdecl newid, id = decl_getidentifier(decl);

	newid = IdentifierDecl(newname);
	*id = *newid;
	free(newid);
	return (decl);
}


/* Checks whether the speclist includes the given SPEC type */
astspec speclist_getspec(astspec s, int type, int subtype)
{
	if (s == NULL) return (NULL);
	if (s->type == SPECLIST)
	{
		astspec p;
		if ((p = speclist_getspec(s->body, type, subtype)) != NULL) return (p);
		return (speclist_getspec(s->u.next, type, subtype));
	}
	if (s->type != type) return (NULL);
	if (type != SPEC && type != STCLASSSPEC) return (s);
	return ((s->subtype == subtype) ? s : NULL);
}


/* Removes the given SPEC type from the speclist (once and if present).
 * Returns true if a removal was actually made.
 * We assume the list is indeed a list and not a tree...
 */
static int speclist_delspec(astspec *s, int type, int subtype)
{
	astspec curr = *s, prev = NULL;
	
	if (*s == NULL) return (0);
	for ( ; curr && curr->type == SPECLIST; curr = curr->u.next, prev = curr)
	{
		if (curr->body == NULL) continue;  /* Should never be NULL */
		if (curr->body->type == type && ((type != SPEC && type != STCLASSSPEC) || 
		                                 (curr->body->subtype == subtype)))
		{
			if (prev == NULL)                /* Found it */
				*s = curr->u.next;
			else
				prev->u.next = curr->u.next;
			ast_spec_free(curr);
			return (1);
		}
	}
	if (curr && curr->type != SPECLIST)        /* Last element */
	{
		if (curr->type == type && ((type != SPEC && type != STCLASSSPEC) || 
		                                 (curr->subtype == subtype)))
		{
			if (prev == NULL)                /* Found it */
				*s = NULL;
			else
				prev->u.next = NULL;
			ast_spec_free(curr);
			return (1);
		}
	}
	return (0);
}


/* Take a pointer declarator and transform it to an array of the given size;
 * obviously this works on non-abstract declarators.
 * Returns NULL if not a pointer declarator.
 */
void decl_ptr2arr(astdecl d, astexpr size)
{
	if (d == NULL) 
		return;              /* When recursing from ABSDECLARATOR */
	if (d->type == DECLARATOR)
	{
		if (d->spec != NULL  && (d->decl->type == DIDENT &&
		                         speclist_getspec(d->spec, SPEC,SPEC_star) != NULL))
		{
			astdecl ident = d->decl;
			
			speclist_delspec(&(d->spec), SPEC, SPEC_star);  /* Remove the star */
			d->decl = ArrayDecl(ident, NULL, size);
			return;
		}
	}
	if (d->type == DIDENT)   /* Not a pointer after all */
		return;
	else
		if (d->type == DLIST)  /* Should be DECL_decllist */
			decl_ptr2arr(d->u.next, size);
		else
			decl_ptr2arr(d->decl, size);
}


/* This takes an array declarator and reduces it to a simple pointer
 */
void decl_arr2ptr(astdecl d)
{
	if (!d)
		return;
	switch (d->type)
	{
		case DECLARATOR:
			decl_arr2ptr(d->decl);
			break;
		case DPAREN:         /* cannot have (id)[10] -- see parser.y */
			decl_arr2ptr(d->decl);
			break;
		case DARRAY:
			if (d->decl->type != DIDENT)
				decl_arr2ptr(d->decl);
			else   /* Got it */
			{
				astdecl t = ParenDecl(Declarator(Pointer(), d->decl));
				if (d->u.expr) ast_expr_free(d->u.expr);
				if (d->spec)   ast_spec_free(d->spec);
				*d = *t;
				free(t);
			}
			break;
		default:
			break;
	}
}


/* Find the number of top-level elements in an initializer (DINIT) */
int decl_initializer_cardinality(astdecl decl)
{
	astexpr expr = decl->u.expr;
	
	if (decl->type != DINIT)    /* No initializer */
		return (0);              
	if (expr->type != BRACEDINIT)     /* No braces */
		return (1);
	return ( expr_list_cardinality(expr->left) );
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     STATEMENT NODES                                           *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


aststmt Statement(enum stmttype type, int subtype, aststmt body)
{
	aststmt s  = smalloc(sizeof(struct aststmt_));
	if (body) body->parent = s;
	s->type    = type;
	s->subtype = subtype;
	s->body    = body;
	s->parent  = NULL;
	s->l       = sc_original_line();
	s->c       = sc_column();
	s->file    = Symbol(sc_original_file());
	return (s);
}


aststmt Goto(symbol s)
{
	aststmt n = Statement(JUMP, SGOTO, NULL);
	n->u.label = s;
	return (n);
}


aststmt Jumpstatement(int subtype, astexpr expr)
{
	aststmt s = Statement(JUMP, subtype, NULL);
	s->u.expr = expr;
	return (s);
}


aststmt Iterationstatement(int subtype,
                           aststmt init, astexpr cond, astexpr incr, aststmt body)
{
	aststmt n = Statement(ITERATION, subtype, body);
	if (body) body->parent = n; 
	if (init) init->parent = n;
	n->u.iteration.init = init;   /* Maybe declaration or expression */
	n->u.iteration.cond = cond;
	n->u.iteration.incr = incr;
	return (n);
}


aststmt Selectionstatement(int subtype,
                           astexpr cond, aststmt body, aststmt elsebody)
{
	aststmt n = Statement(SELECTION, subtype, body);
	if (body) body->parent = n;
	if (elsebody) elsebody->parent = n;
	n->u.selection.cond = cond;
	n->u.selection.elsebody = elsebody;
	return (n);
}


aststmt LabeledStatement(int subtype, symbol l, astexpr e, aststmt st)
{
	aststmt s = Statement(LABELED, subtype, st);
	if (st) st->parent = s;
	if (subtype == SLABEL)
		s->u.label = l;
	else
		s->u.expr = e;
	return (s);
}


aststmt Expression(astexpr expr)
{
	aststmt s = Statement(EXPRESSION, 0, NULL);
	s->u.expr = expr;
	return (s);
}


aststmt Declaration(astspec spec, astdecl decl)
{
	aststmt s = Statement(DECLARATION, 0, NULL);
	s->u.declaration.spec = spec;
	s->u.declaration.decl = decl;
	return (s);
}


aststmt BlockList(aststmt l, aststmt st)
{
	aststmt s = Statement(STATEMENTLIST, 0, st);
	if (st) st->parent = s;
	if (l) l->parent = s;
	s->u.next = l;
	return (s);
}


aststmt FuncDef(astspec spec, astdecl decl, aststmt dlist, aststmt body)
{
	aststmt s = Statement(FUNCDEF, 0, body);
	if (body) body->parent = s;
	if (dlist) dlist->parent = s;
	s->u.declaration.spec  = spec;
	s->u.declaration.decl  = decl;
	s->u.declaration.dlist = dlist;

	/***************** Agelos ***************************************/
	/* We store the funcdef node for interprocedural data flow analysis purposes.
	* **************************************************************/
	autoscope_userfunc_add(decl->decl->decl->u.id, ast_stmt_copy(body));

	return (s);
}


aststmt OmpStmt(ompcon omp)
{
	aststmt s = Statement(OMPSTMT, 0, NULL);
	if (omp) { omp->parent = s; if (omp->body) omp->body->parent = s; }
	s->u.omp = omp;
	return (s);
}


aststmt OmpixStmt(oxcon ox)
{
	aststmt s = Statement(OX_STMT, 0, NULL);
	if (ox) ox->parent = s;
	s->u.ox = ox;
	return (s);
}


/* This is used only by the transformation routines,
 * to produce code that is outputed verbatim (as is).
 */
aststmt Verbatim(char *code)
{
	aststmt s = Statement(VERBATIM, 0, NULL);
	s->u.code = code;
	return (s);
}


aststmt verbit(char *format, ...)
{
	static char str[1024];
	va_list ap;
	va_start(ap, format);
	vsnprintf(str, 1023, format, ap);
	va_end(ap);
	return (Verbatim(strdup(str)));
}


asmop AsmOp(astexpr id, char *con, astexpr var, asmop op, asmop nxt)
{
	asmop s = smalloc(sizeof(struct asmop_));
	s->symbolicname = id;
	s->constraint = con;
	s->var = var;
	s->next = op;
	s->op = nxt;
	return (s);
}


asmnode Asmnode(astspec qual, char *tpl, 
                asmop out, asmop in, astexpr clob, astexpr labs)
{
	asmnode s = smalloc(sizeof(struct asmnode_));
	s->qualifiers = qual;
	s->template   = tpl;
	s->clobbers   = clob;
	s->labels     = labs;
	s->ins        = in;
	s->outs       = out;
	return (s);
}


/* Inline assembly (asm) statements */
aststmt AsmStmt(int subtype, astspec qual, char *tpl, 
                asmop out, asmop in, astexpr clob, astexpr labs)
{
	aststmt s = Statement(ASMSTMT, subtype, NULL);
	s->u.assem = Asmnode(qual, tpl, out, in, clob, labs);
	return (s);
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     OPENMP NODES                                              *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

char *clausesubs[] =  { "static", "dynamic", "guided", "runtime",
                        "shared", "none", "+", "*", "-", "&", "|",
                        "^", "&&", "||", "affinity", "auto",
                        /* OpenMP 3.1 */
                        "min", "max",
                        /* OpenMP 4.0 */
                        /* Deprecated @ 5.1 "master", */ 
                        "close", "spread",
                        "alloc", "to", "from", "tofrom",
                        "in", "out", "inout",
                        /* OpenMP 4.5 */
                        "source", "sink", "release", "delete",
                        /* OpenMP 5.0 */
                        /* OpenMP 5.1 */
                        "primary", "seq_cst", "acq_rel", "relaxed", "acquire",
                        "compilation", "execution", "fatal", "warning",
                        NULL
                      };
char *clausemods[]  = { NULL, "always", "monotonic", "nonmonotonic", "simd",
                        "task", "default", "inscan", "directive-name", 
                        "reduction-id", "map-type",
                        NULL
                      };
char *ompdirnames[] = { NULL, "parallel", "for", "sections", "section",
                        "single", "parallel for", "parallel sections",
                        "for", "master", "critical", "atomic",
                        "ordered", "barrier", "flush", "threadprivate",
                        "task", "taskwait",     /* OpenMP 3.0 */
                        "taskyield",            /* OpenMP 3.1 */
                        /* OpenMP 4.0 */
                        "simd", "distribute", "distribute simd",
                        "distribute parallel for", "for simd",
                        "parallel for simd", "distribute parallel for simd",
                        "target", "target data", "target update",
                        "declare target", "cancel", "cancellation point",
                        "taskgroup", "teams", "target teams",
                        "teams distribute", "teams distribute simd",
                        "target teams distribute", 
                        "target teams distribute simd",
                        "teams distribute parallel for", 
                        "target teams distribute parallel for",
                        "teams distribute parallel for simd", 
                        "target teams distribute parallel for simd",
                        /* OpenMP 4.5 */
                        "target enter data", "target exit data",
                        "target parallel", "target parallel for", 
                        /* OpenMP 5.0 */
                        "requires",
                        /* OpenMP 5.1 */
                        "masked", "error",
                        /* new-directive.sh:dirnames */
                        NULL
                      };


/* (OpenMP 4.0) Extended lists of variables or array sections */
ompxli OmpXLItem(enum ompxli_type type, symbol id, omparrdim dim)
{
	ompxli x   = smalloc(sizeof(struct ompxli_));
	x->xlitype = type;
	x->id      = id;
	x->dim     = dim;
	x->next    = NULL;
	x->l       = sc_original_line();
	x->c       = sc_column();
	x->file    = Symbol(sc_original_file());
	return (x);
}


omparrdim OmpArrDim(astexpr lb, astexpr len)
{
	omparrdim d = smalloc(sizeof(struct omparrdim_));
	d->lb       = lb;
	d->len      = len;
	d->next     = NULL;

	return d;
}


ompclause
OmpClause(ompclt_e type, ompclsubt_e subtype, astexpr expr, astdecl varlist)
{
	ompclause c    = smalloc(sizeof(struct ompclause_));
	c->parent      = NULL;
	c->type        = type;
	c->subtype     = subtype;
	c->modifs      = NULL;
	if (varlist == NULL)
		c->u.expr    = expr;
	else
		c->u.varlist = varlist;
	c->l           = sc_original_line();
	c->c           = sc_column();
	c->file        = Symbol(sc_original_file());
	return (c);
}


ompclause 
MessageClause(char *msg)
{
	ompclause c = OmpClause(OCMESSAGE, 0, NULL, NULL);
	c->s = Symbol(msg);     /* Save string as a symbol */
	return c;
}


/* This is a clause that begins with ext_ */
ompclause
OmpExtClause(symbol s)
{
	ompclause c = OmpClause(OCEXT, OC_DontCare, NULL, NULL);
	c->s = s;
	return (c);
}


/* New clause type (5.1) bearing double expressions */
ompclause
OmpX2Clause(ompclt_e type, ompclsubt_e subtype, astexpr expr1,
            astexpr expr2)
{
	ompclause c = OmpClause(type, subtype, NULL, NULL);
	c->u.x2.expr1  = expr1;
	c->u.x2.expr2  = expr2;
	return (c);
}


ompclause 
OmpXlistClause(ompclt_e type, ompclsubt_e subtype, ompxli xlist)
{
	ompclause c = OmpClause(type, subtype, NULL, NULL);
	c->u.xlist = xlist;
	return (c);
}


ompclause OmpClauseList(ompclause next, ompclause elem)
{
	ompclause c    = OmpClause(OCLIST, 0, NULL, NULL);
	c->u.list.elem = elem;
	c->u.list.next = next;
	return (c);
}



ompdir OmpDirective(ompdirt_e type, ompclause cla)
{
	ompdir d = smalloc(sizeof(struct ompdir_));
	if (cla)     /* parentize all clauses */
		ast_ompclause_parent(d, cla); 
	d->parent    = NULL;
	d->type      = type;
	d->clauses   = cla;
	d->u.varlist = NULL;
	d->l         = sc_original_line() - 1; /* Cause of the \n at the end */
	d->c         = sc_column();
	d->file      = Symbol(sc_original_file());
	return (d);
}


ompdir OmpCriticalDirective(symbol r, ompclause cla)
{
	ompdir d = OmpDirective(DCCRITICAL, cla);
	d->u.region = r;
	return (d);
}


ompdir OmpFlushDirective(astdecl a, ompclause cla)
{
	ompdir d = OmpDirective(DCFLUSH, cla);
	d->u.varlist = a;
	return (d);
}


ompdir OmpThreadprivateDirective(astdecl a)
{
	ompdir d = OmpDirective(DCTHREADPRIVATE, NULL);
	d->u.varlist = a;
	return (d);
}


ompcon OmpConstruct(ompdirt_e type, ompdir dir, aststmt body)
{
	ompcon c = smalloc(sizeof(struct ompcon_));
	if (dir) dir->parent = c; 
	c->parent    = NULL;
	c->type      = type;
	c->directive = dir;
	c->body      = body;
	c->l         = dir ? dir->l : sc_original_line();
	c->c         = dir ? dir->c : sc_column();
	c->file      = dir ? dir->file : Symbol(sc_original_file());
	return (c);
}


ompclauseinfo_t ompclauseinfo[OCLASTCLAUSE+1];
#define OMPCLAUSE(id, name, props, compvior, mods, noargs) \
         ompclauseinfo[id] = (ompclauseinfo_t) { name, props, compvior, mods, noargs }
#define MODIFS_DEFAULT _modifiers(0)

/* All clause modifiers *except* the directive-name one */
static intvec_t _modifiers(int n, ...) 
{
	va_list args;
	int i;
	intvec_t mod = intvec_new(n+1);
	
	mod[0] = OCM_directivename; /* all clauses accept this modifier */
	va_start(args, n);
	for (i = 0; i < n; i++)
		mod[i] = va_arg(args, int);
	va_end(args);
	return mod;
}


/**
 * Adds a modifier (as long as it is not OCM_none) and returns the clause itself
 */
ompclause add_modifier(ompclause c, ompclmod_e mod)
{
	if (mod == OCM_none) return c;
	if (!intvec_contains(c->modifs, mod))
		c->modifs = intvec_append(c->modifs, 1, mod);
	return c;
}


/**
 * Returns the directive-name modifier (if any) of the clause
 * @param c the clause
 * @return the directive, it c contains a directive name modifier, or DCNONE.
 */
ompdirt_e get_dirname_modifier(ompclause c)
{
	int i;
	if (c && c->modifs)
		for (i =  intvec_len(c->modifs)-1; i >= 0; i--)
			if (IsDirnameModifier(c->modifs[i]))
				return UnpackModifier(c->modifs[i]);
	return DCNONE;
}


void ast_initomp()
{
	OMPCLAUSE(OCNOWAIT, "nowait", CMSP_UNIQUE, APPLY_OUTER, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCIF, "if", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCNUMTHREADS, "num_threads", CMSP_UNIQUE, APPLY_ALL, 
	          MODIFS_DEFAULT, false);
	OMPCLAUSE(OCORDERED, "ordered", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCSCHEDULE, "schedule", CMSP_UNIQUE, APPLY_ALL, 
	          _modifiers(3, OCM_monotonic, OCM_nonmonotonic, OCM_simd), false);
	OMPCLAUSE(OCCOPYIN, "copyin", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCPRIVATE, "private",  CMSP_DEFAULT, APPLY_INNER, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCCOPYPRIVATE, "copyprivate", CMSP_DEFAULT, APPLY_ALL, 
	          MODIFS_DEFAULT, false);
	OMPCLAUSE(OCFIRSTPRIVATE, "firstprivate", CMSP_DEFAULT, APPLY_ALL, 
	          MODIFS_DEFAULT, false);
	OMPCLAUSE(OCLASTPRIVATE, "lastprivate", CMSP_DEFAULT, APPLY_ALL, 
	          MODIFS_DEFAULT, false);
	OMPCLAUSE(OCSHARED,"shared", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCDEFAULT, "default", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCREDUCTION, "reduction", CMSP_DEFAULT, APPLY_ALL, 
	          _modifiers(3, OCM_default, OCM_inscan, OCM_task), false);
	OMPCLAUSE(OCLIST, "<list>", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCFIRSTLASTPRIVATE, "first/lastprivate", CMSP_DEFAULT, 
	          APPLY_ALL, MODIFS_DEFAULT, false);

	/* OpenMP 3.0 */
	OMPCLAUSE(OCUNTIED, "untied", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCCOLLAPSE, "collapse", CMSP_UNIQUE, APPLY_ONCE, MODIFS_DEFAULT, false);
 
	/* OpenMP 3.1 */
	OMPCLAUSE(OCFINAL, "final", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCMERGEABLE, "mergeable", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCREAD, "read", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCWRITE, "write", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCUPDATE, "update", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCCAPTURE, "capture", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCSEQCST, "seq_cst", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
 
	/* OpenMP 4.0 */
	OMPCLAUSE(OCPROCBIND, "proc_bind", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCMAP, "map", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCDEVICE, "device", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCTO, "to", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCFROM, "from", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCPARALLEL, "parallel", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCSECTIONS, "sections", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCFOR,"for", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCTASKGROUP, "taskgroup", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCDEPEND, "depend", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCNUMTEAMS, "num_teams", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCTHREADLIMIT, "thread_limit", CMSP_UNIQUE, APPLY_ALL, 
	          MODIFS_DEFAULT, false);
	OMPCLAUSE(OCDISTSCHEDULE, "dist_schedule", CMSP_UNIQUE, APPLY_ALL, 
	          MODIFS_DEFAULT, false);
 
	/* OpenMP 4.5 */
	OMPCLAUSE(OCHINT, "hint", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCPRIORITY, "priority", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCISDEVPTR, "is_device_ptr", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCUSEDEVPTR, "use_device_ptr", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCTHREADS, "threads", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCLINK,"link", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCDEFAULTMAP,"defaultmap(tofrom:scalar)", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCORDEREDNUM, "ordered", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
 
	/* OpenMP 5.0 */
	OMPCLAUSE(OCINREDUCTION, "in_reduction", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCTASKREDUCTION, "task_reduction", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCREVERSEOFFLOAD, "reverse_offload", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCUNIFIEDADDRESS, "unified_address", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCUNIFIEDSHAREDMEMORY, "unified_shared_memory", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCATOMICDEFAULTMEMORDER,"atomic_default_mem_order", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCDYNAMICALLOCATORS, "dynamic_allocators", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCEXT, "ext_", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCACQREL, "acq_rel", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCACQUIRE, "acquire", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCRELEASE, "release", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCRELAXED, "relaxed", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
 
	/* OpenMP 5.1 */
	OMPCLAUSE(OCFILTER, "filter", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCCOMPARE, "compare", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCWEAK, "weak", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, true);
	OMPCLAUSE(OCFAIL, "fail", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCAT, "at", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCSEVERITY, "severity", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCMESSAGE, "message", CMSP_UNIQUE, APPLY_ALL, MODIFS_DEFAULT, false);
 
	/* OMPi extensions */
	OMPCLAUSE(OCAUTO, "auto", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	OMPCLAUSE(OCTAG, "tag", CMSP_DEFAULT, APPLY_ALL, MODIFS_DEFAULT, false);
	
	/* new-clause.sh:clausenames */
}


/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     OMPi-EXTENSION NODES                                      *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


char *oxclausenames[20] = { NULL, "IN", "OUT", "INOUT", "<list>", "reduction",
                            "atnode", "atnode(*)", "detached", "tied",
                            "untied", "stride", "start", "scope",
                            "atworker", "if", "atnode(here)", "atnode(remote)",
                            "hints", "tag"
                          };
char *oxdirnames[7] = { NULL, "taskdef", "task", "tasksync", "taskschedule",
                        "procshared", "tag"
                      };


oxclause OmpixClause(oxclt_e type, astdecl varlist, astexpr expr)
{
	oxclause c     = smalloc(sizeof(struct oxclause_));
	c->parent      = NULL;
	c->type        = type;
	if (expr == NULL)
		c->u.varlist = varlist;
	else
		c->u.expr    = expr;
	c->l           = sc_original_line();
	c->c           = sc_column();
	c->file        = Symbol(sc_original_file());
	return (c);
}


oxclause OmpixClauseList(oxclause next, oxclause elem)
{
	oxclause c     = OmpixClause(OX_OCLIST, NULL, NULL);
	c->u.list.elem = elem;
	c->u.list.next = next;
	return (c);
}


oxclause OmpixReductionClause(int op, astdecl varlist)
{
	oxclause c = OmpixClause(OX_OCREDUCE, varlist, NULL);
	c->operator = op;
	return (c);
}


oxclause OmpixScopeClause(int scope)
{
	oxclause c = OmpixClause(OX_OCSCOPE, NULL, NULL);
	c->u.value = scope;
	return (c);
}


oxdir OmpixDirective(enum oxdircontype type, oxclause cla)
{
	oxdir d = smalloc(sizeof(struct oxdir_));
	if (cla)  /* parentize all clauses */
		ast_oxclause_parent(d, cla);
	d->parent  = NULL;
	d->type    = type;
	d->clauses = cla;
	d->varlist = NULL;
	d->l       = sc_original_line() - 1; /* Cause of the \n at the end */
	d->c       = sc_column();
	d->file    = Symbol(sc_original_file());
	return (d);
}


oxcon OmpixConstruct(enum oxdircontype type, oxdir dir, aststmt body)
{
	oxcon c = smalloc(sizeof(struct oxcon_));
	if (dir) dir->parent = c;
	c->parent    = NULL;
	c->type      = type;
	c->directive = dir;
	c->body      = body;
	c->callback  = NULL;
	c->l         = sc_original_line();
	c->c         = sc_column();
	c->file      = Symbol(sc_original_file());
	return (c);
}


oxcon OmpixTaskdef(oxdir dir, aststmt body, aststmt callbackblock)
{
	oxcon c = OmpixConstruct(OX_DCTASKDEF, dir, body);
	c->callback = callbackblock;
	return (c);
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                               *
 *     UTILITY FUNCTIONS                                         *
 *                                                               *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* Parentize:
 *    register the parent statement of every statement node.
 * No parents within expressions or declarators/specifiers.
 * For an openmp statement O, the construct's parent and the
 * construct's body parent is node O. The directive's parent is
 * the construct and the clauses' parent is the directive.
 */


void ast_ompclause_parent(ompdir parent, ompclause t)
{
	if (!t) return;
	if (t->type == OCLIST)
	{
		if (t->u.list.next != NULL)
			ast_ompclause_parent(parent, t->u.list.next);
		t->parent = parent;
		assert((t = t->u.list.elem) != NULL);
	}
	t->parent = parent;
}


void ast_ompdir_parent(ompcon parent, ompdir t)
{
	if (!t) return;
	if (t->clauses)
		ast_ompclause_parent(t, t->clauses);
	t->parent = parent;
}


void ast_ompcon_parent(aststmt parent, ompcon t)
{
	if (!t) return;
	ast_ompdir_parent(t, t->directive);
	if (t->body)     /* barrier & flush don't have a body */
		ast_stmt_parent(parent, t->body);
	t->parent = parent;
}


void ast_oxclause_parent(oxdir parent, oxclause t)
{
	if (!t) return;
	if (t->type == OX_OCLIST)
	{
		if (t->u.list.next != NULL)
			ast_oxclause_parent(parent, t->u.list.next);
		assert((t = t->u.list.elem) != NULL);
	}
	t->parent = parent;
}


void ast_oxdir_parent(oxcon parent, oxdir t)
{
	if (!t) return;
	if (t->clauses)
		ast_oxclause_parent(t, t->clauses);
	t->parent = parent;
}


void ast_oxcon_parent(aststmt parent, oxcon t)
{
	if (!t) return;
	ast_oxdir_parent(t, t->directive);
	if (t->body)     /* barrier & flush don't have a body */
		ast_stmt_parent(parent, t->body);
	t->parent = parent;
}


void ast_stmt_parent(aststmt parent, aststmt t)
{
	switch (t->type)
	{
		case JUMP:
			break;
		case ITERATION:
			if (t->subtype == SFOR)
				if (t->u.iteration.init != NULL)
					ast_stmt_parent(t, t->u.iteration.init);
			ast_stmt_parent(t, t->body);
			break;
		case SELECTION:
			if (t->subtype == SIF && t->u.selection.elsebody)
				ast_stmt_parent(t, t->u.selection.elsebody);
			ast_stmt_parent(t, t->body);
			break;
		case LABELED:
			ast_stmt_parent(t, t->body);
			break;
		case EXPRESSION:
			break;
		case COMPOUND:
			if (t->body)
				ast_stmt_parent(t, t->body);
			break;
		case STATEMENTLIST:
			ast_stmt_parent(t, t->u.next);
			ast_stmt_parent(t, t->body);
			break;
		case DECLARATION:
			break;
		case FUNCDEF:
			if (t->u.declaration.dlist)
				ast_stmt_parent(t, t->u.declaration.dlist);
			ast_stmt_parent(t, t->body);    /* always non-NULL */
			break;
		case ASMSTMT:
			break;
		case OMPSTMT:
			/* The parent of the construct and its body is this node here */
			ast_ompcon_parent(t, t->u.omp);
			break;
		case VERBATIM:
			break;
		case OX_STMT:
			/* The parent of the construct and its body is this node here */
			ast_oxcon_parent(t, t->u.ox);
			break;
		default:
			fprintf(stderr, "[ast_stmt_parent]: b u g (type = %d)!!\n", t->type);
	}
	t->parent = parent;
}


void ast_parentize(aststmt tree)
{
	ast_stmt_parent(tree, tree);
}


/* Just parentize the immediate descenants */
void ast_parentize_shallow(aststmt t)
{
	if (t->body)
		t->body->parent = t;
	switch (t->type)
	{
		case ITERATION:
			if (t->subtype == SFOR)
				if (t->u.iteration.init != NULL)
					t->u.iteration.init->parent = t;
			break;
		case SELECTION:
			if (t->subtype == SIF && t->u.selection.elsebody)
				t->u.selection.elsebody->parent = t;
			break;
		case STATEMENTLIST:
			if (t->u.next) t->u.next->parent = t;
			break;
		case FUNCDEF:
			if (t->u.declaration.dlist)
				t->u.declaration.dlist->parent = t;
			break;
		case OMPSTMT:
			t->u.omp->parent = t;
			if (t->u.omp->body) t->u.omp->body->parent = t;
			break;
		case OX_STMT:
			t->u.ox->parent = t;
			if (t->u.ox->body) t->u.ox->body->parent = t;
			break;
		default:
			break;
	}
}


/**
 * Replace a statement with another statement in-place; the other statement's
 * node is left untouched but may be optionally freed.
 * 
 * @param s         The old statement
 * @param with      The statement to replace with
 * @param freewith  If true, the "with" node will be freed
 */
void ast_stmt_in_place_replace(aststmt s, aststmt with, bool freewith)
{
	aststmt parent = s->parent;
	*s = *with;
	s->parent = parent;
	ast_parentize_shallow(s);
	if (freewith) free(with);
}


/**
 * Migrate (the contents of) a statement to a new statement; the original one 
 * remains as-is, in-place, and obviously ends up with invalid descendants; it 
 * should thus be modified subsequently. This function is only needed as an 
 * intermediate step to more complex operations, so as to avoid messing with 
 * node allocations and explicit parentizing. 
 * NOTE: If you already have a node N you want to migrate to, then simply use 
 *       ast_stmt_in_place_replace(N, s, false), i.e. replace N by the given 
 *       statement but without freeing the given statement.
 *
 * @param s  The old statement
 * @return   The new statement
 */
aststmt ast_stmt_in_place_migrate(aststmt s)
{
	aststmt to = smalloc(sizeof(struct aststmt_));
	*to = *s;
	ast_parentize_shallow(to);
	return to;
}


/**
 * Add a statement before another statement in-place; this is actually an
 * in-place migration followed by an in-place replacement with a blocklist.
 *
 * @param where The old statement
 * @param what  The statement you want to insert
 */
void ast_stmt_in_place_prepend(aststmt where, aststmt what)
{
	aststmt cp = smalloc(sizeof(struct aststmt_));

	*cp = *where;                 // Copy the original

	where->type = STATEMENTLIST;  // Make it a block list
	where->subtype = 0;

	where->body = cp;             // First the original one
	where->u.next = what;         // Then the new one

	where->parent = cp->parent;   // Parentize
	cp->parent = where;
	what->parent = where;
}


/**
 * Add a statement after another statement in-place; this is actually an
 * in-place migration followed by an in-place replacement with a blocklist.
 *
 * @param where The old statement
 * @param what  The statement you want to insert
 */
void ast_stmt_in_place_append(aststmt where, aststmt what)
{
	aststmt cp = smalloc(sizeof(struct aststmt_));

	*cp = *where;                 // Copy the original

	where->type = STATEMENTLIST;  // Make it a block list
	where->subtype = 0;

	where->body = what;           // First the new one
	where->u.next = cp;           // Then the original one

	where->parent = cp->parent;   // Parentize
	cp->parent = where;
	what->parent = where;
}


/* Given a statement, we return the function it belongs to.
 */
aststmt ast_get_enclosing_function(aststmt t)
{
	for (; t != NULL && t->type != FUNCDEF; t = t->parent)
		;     /* Go up the tree till we hit our current FUNCDEF */
	return (t);
}


/* Prepends a specifier to a declaration or function definition statement */
void ast_stmt_declordef_addspec(aststmt orig, astspec spec)
{
	if (orig->type != DECLARATION && orig->type != FUNCDEF)
		warning("[ast_stmt_declordef_addspec]: (bug) expected a declaration or "
		        "function definition\n\tbut got a node of type %d instead.\n", 
		        orig->type);
	else
		orig->u.declaration.spec = (orig->u.declaration.spec) ? 
			Speclist_right(spec, orig->u.declaration.spec) : spec;
}


/* Prepends a specifier to a declaration */
void ast_decl_addspec(astdecl orig, astspec spec)
{
	if (orig->type != DCASTTYPE && orig->type != DPARAM && orig->type != DSTRUCTFIELD)
		warning("[ast_decl_addspec]: (bug) expected a declaration\n\t "
		        "but got a node of type %d instead.\n", 
		        orig->type);
	else
		orig->spec = Speclist_right(spec, orig->spec);
}


/* Finds the first non-declaration node or NULL if none
 */
static
aststmt first_nondeclaration(aststmt tree)
{
	aststmt p;

	if (!tree || tree->type == DECLARATION)
		return (NULL);
	if (tree->type != STATEMENTLIST)
		return (tree);
	if ((p = first_nondeclaration(tree->u.next)) != NULL)
		return (p);
	else
		return (first_nondeclaration(tree->body));
}


/* Inserts a statement after the declaration section in a compound
 */
void ast_compound_insert_statement(aststmt tree, aststmt t)
{
	aststmt p;

	if (tree->type != COMPOUND)
		return;
	if ((p = first_nondeclaration(tree->body)) == NULL)  /* NULL or only decls */
	{
		if (tree->body == NULL)
		{
			tree->body = t;
			t->parent = tree;
		}
		else
		{
			tree->body = BlockList(tree->body, t);
			tree->body->parent = tree;
		}
		return;
	}
	tree = Statement(p->type, p->subtype, p->body);
	*tree = *p;
	tree->parent = p;
	t->parent = p;
	p->type = STATEMENTLIST;
	p->subtype = 0;
	p->body = tree;
	p->u.next = t;
}


/* Given a statement, return the closest enclosing OpenMP construct of the
 * given type (or of any type if type is 0); returns NULL if none found.
 */
ompcon ast_get_enclosing_ompcon(aststmt t, ompdirt_e type)
{
	for (; t != NULL && t->type != FUNCDEF; t = t->parent)
		if (t->type == OMPSTMT)
		{
			/* While "sections" is a construct, "section" is not */
			if (type == 0 && t->u.omp->type == DCSECTION)
				continue;

			if (type == 0 || t->u.omp->type == type)
				return (t->u.omp);
		};
	return (NULL);
}


/* LINEARIZE:
 * convert a STATEMENTLIST-based tree to an equivallent one
 * where every STATEMENTLIST node has a STATEMENTLIST as a body and
 * a STATEMENT as u.next (except the last one, of course).
 * This form allows easy splitting of the statement list.
 */

SET_TYPE_DEFINE(linearize, aststmt, char, 1031)
SET_TYPE_IMPLEMENT(linearize)

static set(linearize) progstmts;
static void progorder(aststmt tree, int dofree)
{
	if (tree->type != STATEMENTLIST)
		set_put(progstmts, tree);   /* visited in program order */
	else
	{
		progorder(tree->u.next, 1);
		progorder(tree->body, 1);
		if (dofree)
			free(tree);
	}
}


void ast_linearize(aststmt tree)
{
	aststmt s, parent;
	setelem(linearize) e;

	if (tree == NULL || tree->type != STATEMENTLIST)
		return;

	if (tree->type == COMPOUND) 
		tree = tree->body;

	parent = tree->parent;
	set_init(linearize, &progstmts);
	progorder(tree, 0);
	e = progstmts->last;
	if (set_size(progstmts) == 1)   /* Just 1 statement */
	{
		*tree = *(e->key);
		free(e->key);
	}
	else  /* Two or more elements - form a rightward spine */
	{
		s = BlockList(e->prev->key, e->key);
		e = e->prev;
		while ((e = e->prev) != NULL)
			s = BlockList(e->key, s);
		*tree = *s;
		free(e);
	}
	tree->parent = parent;
	set_drain(progstmts);
}


/* Display all OpenMP-related clauses or directives that contain the given
 * substring
 */
void ast_search_omp(char *substr)
{
	int i, header;

	for (header = 0, i = DCNONE; i < DCLASTDIR; i++)
		if (ompdirnames[i] && strstr(ompdirnames[i], substr))
		{
			if (!header)
			{
				printf("--- OpenMP directives ---\n");
				header++;
			}
			printf("  %s\n", ompdirnames[i]);
		};
	
	for (header = 0, i = OCNOCLAUSE+1; i < OCLASTCLAUSE; i++)
		if (ompclauseinfo[i].name && strstr(ompclauseinfo[i].name, substr))
		{
			if (!header)
			{
				printf("--- OpenMP clauses ---\n");
				header++;
			}
			printf("  %s\n", ompclauseinfo[i].name);
		};
	
	for (header = 0, i = OC_DontCare+1; i < OC_lastclsubt; i++)
		if (clausesubs[i] && strstr(clausesubs[i], substr))
		{
			if (!header)
			{
				printf("--- OpenMP clause arguments ---\n");
				header++;
			}
			printf("  %s\n", clausesubs[i]);
		};
	
	for (header = 0, i = OCM_none+1; i < OCM_lastclmod; i++)
		if (clausemods[i] && strstr(clausemods[i], substr))
		{
			if (!header)
			{
				printf("--- OpenMP clause modifiers ---\n");
				header++;
			}
			printf("  %s\n", clausemods[i]);
		};
}


/**
 * @brief Returns an integer vector with the constituent elements of a 
 *        (compound) directive
 * @param dir the directive
 * @return the vector or NULL if dir is not compund after all 
 */
intvec_t constituent_directives(ompdirt_e dir)
{
	switch (dir)
	{
		case DCPARSECTIONS:
			return intvec_new_withelems(2, DCPARALLEL, DCSECTIONS);
		case DCPARFOR:
			return intvec_new_withelems(2, DCPARALLEL, DCFOR);
		case DCPARFORSIMD:
			return intvec_new_withelems(3, DCPARALLEL, DCFOR, DCSIMD);
		case DCDISTPARFOR:
			return intvec_new_withelems(3, DCDISTRIBUTE, DCPARALLEL, DCFOR);
		case DCDISTPARFORSIMD:
			return intvec_new_withelems(4, DCDISTRIBUTE, DCPARALLEL, DCFOR, DCSIMD);
		case DCTEAMSDISTPARFOR:
			return intvec_new_withelems(4, DCTEAMS, DCDISTRIBUTE, DCPARALLEL, DCFOR);
		case DCTARGETTEAMSDISTPARFOR:
			return intvec_new_withelems(5, DCTARGET, DCTEAMS, DCDISTRIBUTE,DCPARALLEL, 
			                               DCFOR);
		case DCTEAMSDISTPARFORSIMD:
			return intvec_new_withelems(5, DCTEAMS, DCDISTRIBUTE, DCPARALLEL, DCFOR, 
			                               DCSIMD);
		case DCTARGETTEAMSDISTPARFORSIMD:
			return intvec_new_withelems(6, DCTARGET, DCTEAMS, DCDISTRIBUTE,DCPARALLEL, 
			                               DCFOR, DCSIMD);
		case DCTARGETPARALLEL:
			return intvec_new_withelems(2, DCTARGET, DCPARALLEL);
		case DCTARGETPARFOR:
			return intvec_new_withelems(3, DCTARGET, DCPARALLEL, DCFOR);
		case DCTEAMSDIST:
			return intvec_new_withelems(2, DCTEAMS, DCDISTRIBUTE);
		case DCTEAMSDISTSIMD:
			return intvec_new_withelems(3, DCTEAMS, DCDISTRIBUTE, DCSIMD);
		case DCTARGETTEAMS:
			return intvec_new_withelems(2, DCTARGET, DCTEAMS);
		case DCTARGETTEAMSDIST:
			return intvec_new_withelems(3, DCTARGET, DCTEAMS, DCDISTRIBUTE);
		case DCTARGETTEAMSDISTSIMD:
			return intvec_new_withelems(4, DCTARGET, DCTEAMS, DCDISTRIBUTE, DCSIMD);
		default:
			break;
	}
	return NULL;   /* it is not combined after all */
}


static bool _dirs_contain(ompdirt_e what, int ndirs, ...)
{
	va_list d;
	bool    isin = false;
	
	va_start(d, ndirs);
	for (; ndirs >= 0; ndirs--)
		if (what == va_arg(d, ompdirt_e))
			isin = true;
	va_end(d);
	return isin;
}


/**
 * @brief Given a plain or combined construct type, check it it is or contains 
 *        a particular construct.
 * @param combined  the given (combined) construct type
 * @param what      the construct type we look for
 * @return true if what is equal to or included in combined, false otherwise.
 */
bool is_or_combines(ompdirt_e combined, ompdirt_e what)
{
	switch (combined)
	{
		case DCPARSECTIONS:
			return _dirs_contain(what, 2, DCPARALLEL, DCSECTIONS);
		case DCPARFOR:
			return _dirs_contain(what, 2, DCPARALLEL, DCFOR);
		case DCPARFORSIMD:
			return _dirs_contain(what, 3, DCPARALLEL, DCFOR, DCSIMD);
		case DCDISTPARFOR:
			return _dirs_contain(what, 3, DCDISTRIBUTE, DCPARALLEL, DCFOR);
		case DCDISTPARFORSIMD:
			return _dirs_contain(what, 4, DCDISTRIBUTE, DCPARALLEL, DCFOR, DCSIMD);
		case DCTEAMSDISTPARFOR:
			return _dirs_contain(what, 4, DCTEAMS, DCDISTRIBUTE, DCPARALLEL, DCFOR);
		case DCTARGETTEAMSDISTPARFOR:
			return _dirs_contain(what, 5, DCTARGET, DCTEAMS, DCDISTRIBUTE,DCPARALLEL, 
			                               DCFOR);
		case DCTEAMSDISTPARFORSIMD:
			return _dirs_contain(what, 5, DCTEAMS, DCDISTRIBUTE, DCPARALLEL, DCFOR, 
			                               DCSIMD);
		case DCTARGETTEAMSDISTPARFORSIMD:
			return _dirs_contain(what, 6, DCTARGET, DCTEAMS, DCDISTRIBUTE,DCPARALLEL, 
			                               DCFOR, DCSIMD);
		case DCTARGETPARALLEL:
			return _dirs_contain(what, 2, DCTARGET, DCPARALLEL);
		case DCTARGETPARFOR:
			return _dirs_contain(what, 3, DCTARGET, DCPARALLEL, DCFOR);
		case DCTEAMSDIST:
			return _dirs_contain(what, 2, DCTEAMS, DCDISTRIBUTE);
		case DCTEAMSDISTSIMD:
			return _dirs_contain(what, 3, DCTEAMS, DCDISTRIBUTE, DCSIMD);
		case DCTARGETTEAMS:
			return _dirs_contain(what, 2, DCTARGET, DCTEAMS);
		case DCTARGETTEAMSDIST:
			return _dirs_contain(what, 3, DCTARGET, DCTEAMS, DCDISTRIBUTE);
		case DCTARGETTEAMSDISTSIMD:
			return _dirs_contain(what, 4, DCTARGET, DCTEAMS, DCDISTRIBUTE, DCSIMD);
		default:
			return (combined == what);
	}
}
