/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#ifndef __STDDEFS_H__
#define __STDDEFS_H__

#include "config.h"

/* Boolean */
#ifdef HAVE_STDBOOL_H
	#include <stdbool.h>
#else
	#ifndef HAVE__BOOL
		#ifdef __cplusplus
			typedef bool _Bool;
		#else
			#define _Bool signed char
		#endif
	#endif
	#define bool _Bool
	#define false 0
	#define true 1
	#define __bool_true_false_are_defined 1
#endif

#if defined (HAVE_ATOMIC_FAA)
#if defined(__cpu_arm)
	static inline int _faa(volatile int *ptr, int value)
	{
		const int __res__ = __sync_fetch_and_add(ptr, value);
		NON_TSO_FENCE;
		return __res__;
	}

	static inline 
	unsigned long _faaul(volatile unsigned long *ptr, unsigned long value)
	{
		const unsigned long __res__ = __sync_fetch_and_add(ptr, value);
		NON_TSO_FENCE;
		return __res__;
	}
#else
	#define _faa __sync_fetch_and_add
	#define _faaul _faa
#endif
#endif


#if defined (HAVE_ATOMIC_CAS)
#if defined(__cpu_arm)
	static inline int _cas(volatile int *ptr, int oldval, int newval)
	{
		const int __res__ = __sync_bool_compare_and_swap(ptr, oldval, newval);
		NON_TSO_FENCE;
		return __res__;
	}

	static inline unsigned long _casul(volatile unsigned long *ptr, 
	                                   unsigned long oldval, unsigned long newval)
	{
		const unsigned long __res__=__sync_bool_compare_and_swap(ptr,oldval,newval);
		NON_TSO_FENCE;
		return __res__;
	}
#else
	#define _cas __sync_bool_compare_and_swap
	#define _casul _cas
#endif
#endif

/* Integer/pointer sizes */
#ifdef HAVE_STDINT_H
	#include <stdint.h>
#else
	#ifndef HAVE_UINTPTR_T
	/* Heuristic */
		#if SIZEOF_CHAR_P == SIZEOF_INT
			typedef unsigned int uintptr_t;
		#else
			typedef unsigned long uintptr_t;
		#endif
	#endif
#endif
#ifndef HAVE_SIZE_T
	/* Heuristic */
	#if SIZEOF_CHAR_P == SIZEOF_INT
		typedef unsigned int size_t;
	#else
		typedef unsigned long size_t;
	#endif
#endif

typedef unsigned long u_long;
#define _MIN(X,Y) (((X) > (Y)) ? (Y) : (X))

#ifdef HAVE_UNSIGNED_LONG_LONG_INT
	typedef unsigned long long u_long_long;
	#define P2_21M1 (u_long_long) 2097151

	static inline u_long_long _ull_encode3(unsigned long x,
	                                       unsigned long y, unsigned long z)
	{
		return (_MIN(z,P2_21M1) << 42)| (_MIN(y,P2_21M1) << 21)| (_MIN(x,P2_21M1));
	}
	static inline void _ull_decode3(u_long_long n, unsigned long *x, 
		                            unsigned long *y, unsigned long *z)
	{
		*x = _MIN(n & P2_21M1, P2_21M1);
		*y = _MIN((n >> 21) & P2_21M1, P2_21M1);
		*z = _MIN((n >> 42) & P2_21M1, P2_21M1);
	}
#else
	#define _ull_encode3(x,y,z)
	#define _ull_decode3(n,x,y,z)
#endif

#define P2_16M1 65535

/* Encodes 2 unsigned shorts to an integer (x: bits 1-16, y: 17-32)*/
static inline unsigned int _uint_encode2(unsigned short x, unsigned short y)
{
	return (_MIN(y, P2_16M1) << 16) | (_MIN(x, P2_16M1));
}

/* Decodes an integer containing 2 shorts (encoded with _uint_encode2) */
static inline 
void _uint_decode2(unsigned int n, unsigned short *x, unsigned short *y)
{
	*x = n & P2_16M1;
	*y = (n >> 16);
}

/* OMPi standard device aliases -- only used/output by the compiler */
#define HOSTDEV_ALIAS -1 /* Alias to specify the host device */
#define DFLTDEV_ALIAS -2 /* Alias to specify the default device */

/* Loop schedule types */
#define FOR_SCHED_NONE          0
#define FOR_SCHED_AUTO          0
#define FOR_SCHED_STATIC        0
#define FOR_SCHED_STATIC_CHUNK  1
#define FOR_SCHED_DYNAMIC       2
#define FOR_SCHED_GUIDED        3
#define FOR_SCHED_RUNTIME       4
#define FOR_SCHED_AFFINITY      5
#define FOR_CLAUSE2SCHED(c,ch) \
	(((c) == OC_static && !(ch)) ? FOR_SCHED_STATIC : \
	(((c) == OC_static && (ch)) ? FOR_SCHED_STATIC_CHUNK : \
	 ((c) == OC_dynamic ? FOR_SCHED_DYNAMIC : \
	 ((c) == OC_guided ? FOR_SCHED_GUIDED : \
	 ((c) == OC_runtime ? FOR_SCHED_RUNTIME : \
	 ((c) == OC_affinity ? FOR_SCHED_AFFINITY : \
	 ((c) == OC_auto ? FOR_SCHED_AUTO : FOR_SCHED_NONE)))))))

/* Mapping types */
#define MAP_TYPE_IGNORE	      0  /* ignore */
#define MAP_TYPE_TO           1  /* to */
#define MAP_TYPE_FROM         2  /* from */
#define MAP_TYPE_TOFROM       3  /* tofrom */
#define MAP_TYPE_ALLOC        4  /* alloc only, no data movement */
#define MAP_TYPE_ZLAS         5  /* zero-length arrsec */
#define MAP_TYPE_DDENV_TOFROM 6  /* undefined (ddenv/tofrom) */

#ifdef PATH_MAX
	#define PATHSIZE PATH_MAX
#else
	#define PATHSIZE 4096
#endif

#ifndef PORTABLE_BUILD
    #define InstallPath PKG_INSTALL_PREFIX
    #define LibDir PKG_LIBDIR_ABSPATH
#else
    extern char InstallPath[PATHSIZE], LibDir[PATHSIZE];
#endif

/*
 * KERNEL BUNDLING
 */
typedef enum { BUNDLE_NONE = 0 /* do not change this */,
               BUNDLE_BINS, BUNDLE_SRCS 
             } bundling_e;

#endif /* __STDDEFS_H__ */
