/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* rt_common.h -- runtime facilities common among host and devices */

#ifndef __RT_COMMON_H__
#define __RT_COMMON_H__

/* ORT_DEBUG is the global macro that activates debugging of selected areas
 * (using |-separated flags like DBG_ORT, DBG_TASKS, etc)
 */
#define DBG_ORT          1
#define DBG_TASKS        2
#define DBG_BARRIERS     4
#define DBG_DEVICES      8
#define DBG_ROFF_PRIMARY 16
#define DBG_ROFF_WORKERS 32
#define DBG_ROFF_MEMORY  64
#define DBG_ROFF_NODEMGR 128
/*
#define ORT_DEBUG (DBG_ORT | DBG_TASKS | DBG_BARRIERS | DBG_DEVICES \
     | DBG_ROFF_PRIMARY | DBG_ROFF_WORKERS | DBG_ROFF_MEMORY | DBG_ROFF_NODEMGR)
*/


/* DBGPRN is a handy macro to print to stderr when debugging is active.
 * Independently of ORT_DEBUG, *at the top* of a file (before #include-ing
 * this one), one can force or block the macro using:
 *   #define DBGPRN_FORCE
 *   #define DBGPRN_BLOCK
 * It can also be activated only for certain areas using:
 *   #define DBGPRN_FILTER <DBG_* flags>
 */
#if defined(DBGPRN_BLOCK) || (!defined(ORT_DEBUG) && !defined(DBGPRN_FORCE))
	#define DBGPRN(s) 
#else
  #if defined(DBGPRN_FORCE)
	  #define DBGPRN(s) fprintf s
  #else
    #if defined(DBGPRN_FILTER) && ((ORT_DEBUG & DBGPRN_FILTER) == 0)
      #define DBGPRN(s)
    #else
	    #define DBGPRN(s) fprintf s
    #endif
	#endif
#endif

/* Finally, for fine-tuning, we have 5 levels of information
 * (0 is the most laconic one, 4 is the most detailed).
 * At the top of your file do:
 *  #define ORT_DEBUG_LEVEL <debugging-level>
 *
 * DBGPRNi() prints if debugging level is >= i
 * DBGPRN0() is equivallent to DBGPRN().
 */
#define DBGPPRN0 DBGPRN
#if defined(ORT_DEBUG_LEVEL) && ORT_DEBUG_LEVEL >= 1
  #define DBGPRN1(s) DBGPRN(s)
#else
  #define DBGPRN1(s)
#endif
#if defined(ORT_DEBUG_LEVEL) && ORT_DEBUG_LEVEL >= 2
  #define DBGPRN2(s) DBGPRN(s)
#else
  #define DBGPRN2(s)
#endif
#if defined(ORT_DEBUG_LEVEL) && ORT_DEBUG_LEVEL >= 3
  #define DBGPRN3(s) DBGPRN(s)
#else
  #define DBGPRN3(s)
#endif
#if defined(ORT_DEBUG_LEVEL) && ORT_DEBUG_LEVEL >= 4
  #define DBGPRN4(s) DBGPRN(s)
#else
  #define DBGPRN4(s)
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include "str.h"
#include "omp.h"
#include "stddefs.h"

#ifndef PORTABLE_BUILD
	#define ModuleDir PKG_LIBDIR_ABSPATH "/devices"
#else
	extern char *ModuleDir;
#endif

typedef enum is_mapped_
{
	NOT_MAPPED = 0,  
	MAPPED_DEVDATA = 1,
	MAPPED_DECLDATA = 2
} is_mapped_e;

typedef enum num_args_
{
	ARGS_NUMDECL = 0, ARGS_NUMFIP = 1, ARGS_NUMMAPPED = 2
} num_args_e;

/* LOCK TYPES 
 */
#define ORT_LOCK_NORMAL 0
#define ORT_LOCK_NEST   1
#define ORT_LOCK_SPIN   2

typedef struct
{
	omp_sched_t schedule;
	unsigned long chunksize;
	int         first_chunk;
	int         mean;
	double      sigma;
	double      alpha;
	int        *time_table;
} xsched_data_t;


/* ICVs (internal control variables -- OpenMP V.4.0) 
 */

/* Per-task ICVs */
typedef struct
{
	int         dynamic;
	int         nested;
	omp_sched_t rtschedule;     /* For runtime schedules */
	unsigned long rtchunk;      /* ditto */
	int         nthreads;       /* default # threads for a team */
	/* OpenMP 4.0 */
	int         threadlimit;
	omp_proc_bind_t proc_bind;
	int         def_ompdevid;   /* the default device ID */
	void       *cur_de;         /* current device data environment */
	
	xsched_data_t xsched;
} ort_task_icvs_t;

/* Global ICVs */
typedef struct
{
	int         dynamic;        /* For the initial task */
	int         nested;
	omp_sched_t rtschedule;     /* For runtime schedules */
	int         rtchunk;        /* ditto */
	int         nthreads;       /* default # threads for a team */

	int         ncpus;          /* Global */
	size_t      stacksize;
	int         waitpolicy;
	int         threadlimit;
	int         levellimit;

	/* OpenMP 4.0 */
	omp_proc_bind_t proc_bind;
	int         cancel;
	int         def_ompdevid;
	/* OpenMP 4.5 */
	int         max_task_prio;
	/* OpenMP 5.0 */
	int         display_affinity;   /* Global */
	char       *affinity_format;    /* per device */
	int         targetoffload;      /* Global */
	/* OpenMP 5.1 */
	int         nteams;             /* per device */
	int         teams_thread_limit; /* per device */
	/* OMPi-specific */
	xsched_data_t xsched;
} ort_icvs_t;

/* PLACES
 */
#define numplaces(p)   ((p) ? (p)[0][0] : 0)
#define placelen(p,i)  ((p) ? (((i) >= 0) && ((i) < (p)[0][0]) ? \
                           ( (p)[(i)+1] ? (p)[(i)+1][0] : 0 ) : 0) : 0)
int  **places_dup(int **from);
void   places_free(int **place);

/* THREAD/TASK/KERNEL (TTK) Functions
 * (for MPI-enabled OMPi)
 */

/* Handy type */
typedef void *(*ttkfunc_t)(void *);

typedef struct  {
	ttkfunc_t func;     /* function pointer */
	char      *name;    /* function name */
} ttkfuncdata_t;

typedef struct {
	ttkfuncdata_t *table;
	int           size;            /* used space */
	int           capacity;        /* total allocated space */
} ttkfunctable_t;

extern int is_userprog_portable;

typedef enum {
	DEVICE_UNINITIALIZED = 0,
	DEVICE_INITIALIZED = 1
} devicestatus_e;


/* Generic functions */
extern void ttkfunctable_add(ttkfunctable_t *t, ttkfunc_t func, char *name);
extern int  ttkfunctable_findbyptr(ttkfunctable_t *t, ttkfunc_t func);
extern int  ttkfunctable_findbyname(ttkfunctable_t *t, char *name);
extern ttkfunc_t ttkfunctable_getptr(ttkfunctable_t *t, int index);

/* Kernel-specific functions */
extern void _ort_kernfunc_register(char *name, ttkfunc_t funcptr);
extern int ort_kernfunc_findbyname(char *name);
extern ttkfunc_t ort_kernfunc_getptr(int id);
extern void ort_kernfunc_cleanup(void);

/* Memory mapping (host) */
extern void *ort_mapped_alloc(size_t size);
extern void ort_mapped_free(void *addr);
extern size_t ort_mapped_get_size(void *mapped);

/* OpenMP requirements */
void _ort_set_requirements(unsigned int num_requirements, ...);
bool ort_feature_is_required(char *f);

/* Kernel file bundling
 * For a user application <userprog>.c, the kernels file naming scheme is:
 *     <userprog>_d<XX>_<modulename>.<extesion>,
 * where XX is the index of the kernel, e.g. foo_d02_opencl.out.
 */
typedef struct {
		int           id;         /* the kernel index */
		char          *module;    /* the name of the module */
		char          *progname;  /* the user application without the .c */
		char          *filename;  /* the full kernel file name with its extension */
		char          *cars;      /* a string containing CARS metrics */
		unsigned char *data;      /* pointer @ beggining of bundled byte stream */
		unsigned int  size;       /* the size (#bytes) of the bundled binary */
	} bubin_t;

/* Compiler call */
extern void _ort_bundle_binfile(int kid, char *mod, const unsigned char *data, 
                                const unsigned int size, char *filename, char *progfname,
                                char *cars);
extern void _ort_set_bundling_type(int bundling_type);

/* Interface */
extern bundling_e ort_bundling_type();
extern bubin_t   *ort_bubins_search(char *filename);
extern int        ort_bubins_unbundle(char *filename);
extern void       ort_bubins_unbundle_and_compile(char *filename);

#endif /* __RT_COMMON_H__ */
