/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* rt_common.c -- common functionalities between the host and the devices */

#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include "rt_common.h"
#include "config.h"
#include "stddefs.h"
#include "assorted.h"
#include "stdarg.h"


/* PLACES
 *
 * Array of pointers; each one points to a place (i.e. an array of ints).
 * places[0][0] gives the total number of places;
 * places[i][0] (i > 0) gives the length of the i-th place array.
 */


int  **places_dup(int **from)
{
	int i;
	int num_of_places = numplaces(from);
	int **c_places;


	if(num_of_places <=0)
		return NULL;

	c_places = smalloc(num_of_places*sizeof(int **));

	c_places[0] = smalloc(sizeof(int));
	c_places[0][0] = from[0][0];

	for(i=1; i<=num_of_places; i++)
	{
		c_places[i] = malloc((placelen(from,i-1)+1)*sizeof(int));
		memcpy(c_places[i], from[i], (placelen(from,i-1)+1)*sizeof(int));
	}

	return c_places;
}


void  places_free(int **place)
{
	int n = numplaces(place) - 1;

	if(place == NULL)
		return;
	for (; n >=0; n--)
		if (placelen(place, n))
			free(place[n + 1]);
	free(place);
}


/* THREAD/TASK/KERNEL (TTK) Functions
 *
 * When using a cluster (MPI enabled) all nodes get a copy of the user's 
 * executable which contains all thread/task/kernel (TTK) functions. However, 
 * because of potential differences in architecture or memory layouts, 
 * the memory address at which each kernel function resides might be 
 * different among nodes. To solve this problem, each node creates a table 
 * that stores each TTK function's name and pointer; it is guaranteed that 
 * all TTK functions are stored in the same order, so that the **index** in 
 * the table is the same among all nodes. Consequently, nodes need only 
 * know the index in order to call the correct function; thus transmitters
 * communicate only the table indices, while receivers use the received
 * indices to get the function pointer from their own table.
 */


/**
 * @brief Dynamically adds a new function to a TTK function table
 * @param t    the function table
 * @param func pointer to the new function
 * @param name the function name (not dup'ed; can be NULL)
 */
void ttkfunctable_add(ttkfunctable_t *t, ttkfunc_t func, char *name)
{
	if (t->size >= t->capacity)
	{
		t->capacity += 10;       /* Increase (arbitrarily) by 10 entries */
		t->table = realloc(t->table, t->capacity * sizeof(ttkfuncdata_t));
		if (!t->table)
		{
			fprintf(stderr, "thread/task/kernel functable realloc failure.\n");
			exit(1);
		}
	}
	t->table[t->size].func = func;
	t->table[t->size++].name = name;   /* Not dupping anything */
}


/**
 * @brief Finds a TTK function by its function pointer
 * 
 * @param t    the TTK function table
 * @param func the function pointer
 * @return     the index in the table
 */
int ttkfunctable_findbyptr(ttkfunctable_t *t, ttkfunc_t func)
{
	int i;

	for (i = 0; i < t->size; ++i)
		if (t->table[i].func == func)
			return i;
	return -1;
}


/**
 * @brief Finds a TTK function by its function name
 * 
 * @param t    the TTK function table
 * @param func the function name
 * @return     the index in the table
 */
int ttkfunctable_findbyname(ttkfunctable_t *t, char *name)
{
	int i;

	for (i = 0; i < t->size; ++i)
		if (strncmp(t->table[i].name, name, strlen(name)) == 0)
			return i;
	return -1;
}


/**
 * @brief Returns a TTK function pointer given its table index
 * 
 * @param t    the TTK function table
 * @param func the index in the table
 * @return     the function pointer
 */
ttkfunc_t ttkfunctable_getptr(ttkfunctable_t *t, int index)
{
	if (index < 0 || index >= t->size)
		return NULL;
	else
		return t->table[index].func;
}

static ttkfunctable_t allkernfuncs;

void _ort_kernfunc_register(char *name, ttkfunc_t func)
{
	ttkfunctable_add(&allkernfuncs, func, name);
}

int ort_kernfunc_findbyname(char *name)
{
	return ttkfunctable_findbyname(&allkernfuncs, name);
}

ttkfunc_t ort_kernfunc_getptr(int fid)
{
	return ttkfunctable_getptr(&allkernfuncs, fid);
}

void ort_kernfunc_cleanup(void)
{
	if (allkernfuncs.table)
	{
		free(allkernfuncs.table);
		allkernfuncs.table = NULL;
	}
}


/* MEMORY MAPPING (host)
 *
 * Only needed for devdata/decldata structs 
 */

void *ort_mapped_alloc(size_t size)
{
	void *mapped = smalloc(size + sizeof(size_t));  /* Store the size */
	*((size_t *) mapped) = size;
	return ( mapped + sizeof(size_t) );  /* handled @ offload time */
}


void ort_mapped_free(void *addr)
{
	free(addr - sizeof(size_t));
}


size_t ort_mapped_get_size(void *mapped)
{
	return *((int *) (mapped - sizeof(size_t)));
}


/* OPENMP REQUIREMENTS
 */

/* All requirements from '#pragma omp requires', stored as strings */
static char **omp_requirements;
static unsigned int _num_requirements = 0;

/* Register OpenMP requirements (called in *_ompi.c) */
void _ort_set_requirements(unsigned int num_requirements, ...)
{
	int i;
	va_list args;

	omp_requirements = smalloc(num_requirements * sizeof(char *));
	_num_requirements = num_requirements;

	va_start(args, num_requirements);
	for (i = 0; i < num_requirements; i++)
		omp_requirements[i] = va_arg(args, char *);
	va_end(args);
}


/* Check if a feature is marked as required in the user program */
bool ort_feature_is_required(char *f)
{
	unsigned int i;
	for (i = 0; i < _num_requirements; i++)
		if (strcmp(omp_requirements[i], f) == 0)
			return true;
	return false;
}


/* KERNEL FILE BUNDLING/UNBUNDLING 
 *
 * Structures and functions to handle bundled binaries (bubins), i.e. 
 * kernel files embedded within the host executable. We provide some inactive 
 * facilities for dealing with the sources of the bundled kernels but basically 
 * we bundle compiled binaries.
 */
 
typedef struct {
		bubin_t     *table;
		unsigned int size;
	} bbundle_t;
static bbundle_t allbubins = { NULL, 0 };
static bundling_e bundlingType = BUNDLE_NONE;

/*
 * Returns BUNDLE_NONE if no kernel bundling is active, BUNDLE_BINS if 
 * kernel binaries are bundled or BUNDLE_SRCS if kernel sources are bundled.
 * For now the latter is not functional.
 */
bundling_e ort_bundling_type()
{
	return bundlingType;
}

/* Sets the bundling type; called by compiler-generated code */
void _ort_set_bundling_type(int bundling_type)
{
	bundlingType = bundling_type;
}


/*
 * Adds a binary to a bundle
 */
static void bubins_add(bbundle_t *b, int bid, char *mod, const unsigned char *data, 
                       const unsigned int size, char *filename, char *progname,
                       char *cars)
{
	if ((b->size % 16) == 0)
		b->table = (b->table == NULL) ? 
		                smalloc(16 * sizeof(bubin_t)) : 
		                srealloc(b->table, (b->size + 16) * sizeof(bubin_t));

	b->table[b->size].id = bid;
	b->table[b->size].module = mod;
	b->table[b->size].data = (unsigned char *) data;
	b->table[b->size].size = (unsigned int) size;
	b->table[b->size].filename = filename;
	b->table[b->size].progname = progname;
	b->table[b->size].cars = cars;
	b->size++;
}


/*
 * Searches a bundle for a specific binary by filename
 */
static bubin_t *bubins_search(bbundle_t *b, char *filename)
{
	int i;
	if (!b || !b->table) return NULL;

	for (i = 0; i < b->size; i++)
		if (strncmp(b->table[i].filename, filename, strlen(filename)) == 0)
			return &(b->table[i]);
	
	return NULL;
}


bubin_t *ort_bubins_search(char *filename)
{
	return bubins_search(&allbubins, filename);
}


/* Unbundles a bundled binary */
static int bubins_binary_unbundle(bubin_t *bubin)
{
	int fd;
	ssize_t bytes_written;

	if (bubin == NULL)
	{
		fprintf(stderr, "[bubins_binary_unbundle]: NULL bubin.\n");
		return 1;
	}

	if ((fd = open(bubin->filename, O_WRONLY | O_CREAT | O_TRUNC, 0644)) < 0)
	{
		perror("[bubins_binary_unbundle]");
		return 2;
	}

	bytes_written = write(fd, bubin->data, (size_t) bubin->size);
	if (bytes_written == -1 || ((size_t) bytes_written != bubin->size))
	{
		fprintf(stderr, "[bubins_binary_unbundle]: write failed.\n");
		close(fd);
		return 3;
	}

	close(fd);
	return 0;
}


int ort_bubins_unbundle(char *filename)
{
	bubin_t *entry = ort_bubins_search(filename);
	
	if (!entry)
	{
		fprintf(stderr, "[%s]: could not unbundle '%s'.\n", __func__, filename);
		return 1;
	}
	return bubins_binary_unbundle(entry);
}


/* Unbundles and compiles a binary using the corresponding MakeKernel 
 * recipe 
 * 
 * TODO: this should not rely on ompicc. The compilation logic should
 * be integrated into the runtime. One idea is to generate all makefiles
 * during compile time (as usual), and just run them here.
 */
static void bubins_unbundle_and_compile(bubin_t *bubin)
{
	int ret;
	str command = Strnew();
	
	if (bubins_binary_unbundle(bubin))
		exit (1);

	str_printf(command, "%s --devs=%s --kernel=%d %s", 
	                    OmpiCcName, bubin->module, bubin->id, bubin->progname);

	if ((ret = system(str_string(command))) != 0)
	{
		perror("bubins_unbundle_and_compile");
		exit(EXIT_FAILURE);
	}

	str_free(command);
}


void ort_bubins_unbundle_and_compile(char *filename)
{
	bubin_t *entry = ort_bubins_search(filename);
	if (!entry)
	{
		fprintf(stderr, "Could not retrieve bubin '%s'; exiting.\n", filename);
		exit(EXIT_FAILURE);
	}

	bubins_unbundle_and_compile(entry);
}


/* Adds a binary to the bundle; called by compiler-generated code */
void _ort_bundle_binfile(int kid, char *mod, const unsigned char *data, 
                         const unsigned int size, char *filename, char *progfname,
                         char *cars)
{
	bubins_add(&allbubins, kid, mod, data, size, filename, progfname, cars);
}
