/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <pwd.h>
#include <string.h>
#include <stdbool.h>
#include <unistd.h>
#include "str.h"
#include "assorted.h"
#include "roff_config.h"

#ifdef OMPI_REMOTE_OFFLOADING

static bool userprog_portable = false;
static bool print_ssh_hosts = false;

modules_t allmodules[] = {
	MODULE(cpu), 
	MODULE(cuda), 
	MODULE(opencl),
	MODULE(vulkan),
	MODULE(proc2), 
	MODULE(proc2l),
/* new-module.sh:roffallmodules */
	LASTMODULE
};
roff_config_t roff_config;

/* This is never the case when compiling _setup_remote_nodes.c */
#ifdef PORTABLE_BUILD

// When there is a portable build, follow the steps in ompicc.c
// to find the correct path the configure file should be at.
static void get_ompi_lib_path(char *path)
{
#ifdef HAVE_READLINK
	char buffer[PATHSIZE];
	ssize_t len = readlink("/proc/self/exe", buffer, PATHSIZE - 1);

	if (len == -1)
	{
		fprintf(stderr, "PARSER ERROR: couldn't retrieve installation path using readlink.\n");
		exit(EXIT_FAILURE);
	}
	else
		if (len == PATHSIZE - 1)
		{
			fprintf(stderr, "PARSER ERROR: path to %s too long.\n", PKG_NAME);
			exit(EXIT_FAILURE);
		}
		else
			buffer[len] = '\0';

	get_path(buffer, path, PATHSIZE);
	if (strcmp(path + strlen(path) - 4, "bin/"))
	{
		fprintf(stderr, "PARSER ERROR: invalid installation path for a portable build.\n");
		exit(EXIT_FAILURE);
	}
	path[strlen(path)-4] = 0;
	strcat(path, PKG_LIBDIR);
#else
	fprintf(stderr, "PARSER ERROR: cannot determine OMPi installation directory\n");
	exit(EXIT_FAILURE);
#endif
}

#endif

static const char *get_config_path()
{
	char *location_path, *config_path;
	size_t path_size;

#if defined(REMOTE_OFFLOADING_SETUP_MODE)
	location_path = getenv("HOME");
#elif defined(PORTABLE_BUILD)
	location_path = smalloc(PATHSIZE * sizeof(char));
	get_ompi_lib_path(location_path);
#else
	location_path = PKG_LIBDIR_ABSPATH;
#endif

	path_size = strlen(location_path) + strlen(ROFFCONF_CONF_FILE) + 1;
	config_path = calloc(path_size, sizeof(char));
	snprintf(config_path, path_size, "%s%s", location_path, ROFFCONF_CONF_FILE);

#if defined(PORTABLE_BUILD) && !defined(REMOTE_OFFLOADING_SETUP_MODE)
	free(location_path);
#endif

	return config_path;
}

static char *load_config_file(const char *filename, size_t *config_size)
{
	char *buffer = NULL;
	size_t byte_count;
	FILE *f = fopen(filename, "r");
	if (!f)
	{
		fprintf(stderr,
		        "Config file `%s` not found or could not be opened for reading.\n",
		        filename);
		return NULL;
	}

	fseek(f, 0L, SEEK_END);
	byte_count = ftell(f) + 1;
	fseek(f, 0L, SEEK_SET);

	buffer = (char *) calloc(byte_count + 1, sizeof(char));
	if (fread(buffer, sizeof(char), byte_count, f) <= 0)
	{
		fprintf(stderr, "[roff_config] load_config_file: error: fread failed; exiting.\n");
		fclose(f);
		exit(1);
	}

	fclose(f);

	if (config_size)
		*config_size = byte_count;

	return buffer;
}


/*
 * Creates an encoded snapshot of .ompi_remote_devices.
 * Used during compile time.
 */

char *roff_config_encode(char *snapshot_name)
{
	size_t config_size;
	const char *config_path = get_config_path();
	char *buffer = load_config_file(config_path, &config_size),
	     *roff_config_hex = NULL;

	if (!buffer)
	{
		perror("roff_config_encode");
		exit(EXIT_FAILURE);
	}

	roff_config_hex = encode_text_tohexarray_withdecl(buffer, snapshot_name, config_size,
	                                                  ENCODE_STATIC, PRINT_SIZE);

	free(buffer);

	return roff_config_hex;
}

/* Some parsing utilities. */
static int is_node_duplicate(const char *hostname, unsigned int port)
{
	int i;

	for (i = 0; i < roff_config.num_nodes; i++)
	{
		if ((strcmp(roff_config.nodes[i].name, hostname) == 0) && (roff_config.nodes[i].port == port))
		{
#ifdef DEBUG
			printf("Trying to add duplicate node '%s', omitting.\n", hostname);
#endif
			return 1;
		}
	}
	return 0;
}


static bool is_module_disabled(walk_t* walk, bool ignore)
{
	if (*(walk->p) == '~')
	{
		walk->p++;
		return (!ignore) ? true : false;
	}
	return false;
}


static bool is_valid_identifier_char(walk_t *walk) 
{
	return isalpha(*(walk->p)) || isdigit(*(walk->p)) || *(walk->p) == '.';
}


static void eat_whitespace(walk_t *walk)
{
	while (isspace(*(walk->p))) walk->p++;
}


static void eat_duplicate_decl(walk_t* walk)
{
	while (*(walk->p) != '}') walk->p++;
	// Closing bracket encountered, eating one more character.
	walk->p++;
	// If there's a comma, eat it and continue parsing.
	if (*(walk->p) == ',') walk->p++;
}


static void eat_comment_lines(walk_t* walk)
{
	eat_whitespace(walk);
	while (*(walk->p) == '#')
	{
		while (*(walk->p) != '\n') walk->p++;
		eat_whitespace(walk);
	}
}


static int accept_single_char(walk_t *walk, char ch)
{
	eat_whitespace(walk);
	if (*(walk->p) == ch) { walk->p++; return 1; }
	return 0;
}



static int expect_single_char(walk_t *walk, char ch)
{
	eat_whitespace(walk);
	if (*(walk->p) == ch)
	{
		walk->p++;
		return 1;
	}

	fprintf(stderr,
	        "PARSER ERROR: expected character '%c', '%c' encountered instead.\n", ch,
	        *walk->p);
	exit(EXIT_FAILURE);
}

static int expect_int(walk_t *walk)
{
	char num[32] = {0};
	size_t index = 0;

	eat_whitespace(walk);
	while (isdigit(*walk->p) && index < 32)
	{
		num[index] = *walk->p;
		walk->p++;
		index++;
	}

	int ret = atoi(num);

	return ret;
}


static int expect_string(walk_t *walk, char *key)
{
	// Reset the string buffer.
	size_t index = 0;
	memset(key, 0, 64);

	eat_whitespace(walk);
	while (is_valid_identifier_char(walk) && index < ROFFCONF_MAXNAME - 1)
	{
		key[index] = *walk->p;
		walk->p++;
		index++;
	}

	// Check if first character is a letter or digit.
	// Otherwise, it's a symbol, hence not a hostname or capability string.
	if (isalpha(key[0]) || isdigit(key[0]))
	{
		return 1;
	}

	// Unreachable if name is valid...
	fprintf(stderr,
	        "PARSER ERROR: expected valid name string (hostname or capability), encountered '%s' instead.\n",
	        key);
	exit(EXIT_FAILURE);
}


static int accept_strings(walk_t *walk, char **keys, unsigned int *ports, size_t *written)
{
	size_t index, it = 0;

	while (it < ROFFCONF_MAX_INL_DECL)
	{
		index = 0;
		keys[it] = (char*) calloc(1, ROFFCONF_MAXNAME);

		/* Check if identifier only contains letters, numbers and dots. */
		eat_whitespace(walk);
		while (is_valid_identifier_char(walk) && index < ROFFCONF_MAXNAME - 1)
			keys[it][index++] = *(walk->p)++;

		keys[it][index] = '\0';

		if (*(walk->p) == ':')
		{
			walk->p++;  // skip ':'
			ports[it] = (unsigned int) expect_int(walk);

			if (ports[it] < 1 || ports[it] > 65535)
			{
				fprintf(stderr,
						"PARSER ERROR: Bad port %d.\n", ports[it]);
				exit(EXIT_FAILURE);
			}
		}

		/* We can have multiple identifiers per line, so check for that. */
		if (!accept_single_char(walk, ',')) { *written = it + 1; return 1; }

		/* The last accepted comma could be a trailing one following the
		 * last identifier, so check if walk is pointing at a bracket or identifier.
		 */
		if (*walk->p == '{') { *written = it + 1; return 1; }

		it += 1; // no manual walk->p++ here!
	}
	return 0;
}


/* Adds a new node to the configuration and returns its ID */
static roff_config_status_t node_new(char *hostname, unsigned int port)
{
	int new_node_id;
	roff_config_node_t *new_node;
	roff_config_status_t status;

	if (!roff_config.initialized)
	{
		status.return_code = roff_config_status_fail;
		status.id = -1;
		return status;
	}

	if (is_node_duplicate(hostname, port))
	{
		status.return_code = roff_config_status_duplicate;
		status.id = -1;
		return status;
	}

	if (roff_config.num_nodes == roff_config.node_capacity)
	{
		roff_config.node_capacity += 32;
		roff_config.nodes = realloc(roff_config.nodes, roff_config.node_capacity * sizeof(roff_config_node_t));
		if (roff_config.nodes == NULL)
		{
			perror("node_new():");
			exit(EXIT_FAILURE);
		}
	}

	/* Print SSH hosts only if a node uses a port other than the default. */
	if ((port != 22) && (!print_ssh_hosts))
		print_ssh_hosts = true;

	/* Find the next available slot in the configuration */
	new_node = &(roff_config.nodes[new_node_id = (roff_config.num_nodes++)]);
	new_node->name = strndup(hostname, strlen(hostname));
	new_node->port = port;
	new_node->id = new_node_id;
	new_node->has_cpu_module = 0;
	new_node->num_modules = 0;
	new_node->num_modnames = 0;
	new_node->total_num_devices = 0;
	new_node->first_remote_devid = roff_config.num_devices;

	status.return_code = roff_config_status_success;
	status.id = new_node_id;
	return status;
}

/* Adds a module to an existing node */
static void node_add_module(int node_id, char *module_name, int num_devices, bool is_disabled)
{
	int new_pos;
	roff_config_node_t *node = &(roff_config.nodes[node_id]);
	roff_config_module_t *new_module;

	if (!roff_config.initialized) return;
	if (is_disabled) return;

	if (node->num_modules == ROFFCONF_MAXMODS)
	{
		fprintf(stderr, "node_add_module(): module array for node %d (%s) is full.\n",
		                node_id, node->name);
		return;
	}

	new_module = &(node->modules[new_pos = (node->num_modules++)]);
	new_module->name = strndup(module_name, strlen(module_name));
	new_module->num_devices = num_devices;
	new_module->node = node;
	new_module->is_disabled = is_disabled;

	if (IS_CPU_MODULE(module_name))
	{
		node->has_cpu_module = 1;
		roff_config.has_cpu_module = 1;
	}
	else
	{
		node->module_names[node->num_modnames++] = new_module->name;
		if (!is_substr(str_string(roff_config.modstr), new_module->name))
		{
			str_printf(roff_config.modstr, "%s ", new_module->name);

			if (roff_config.nuniquemodules <= 15)
				roff_config.uniquemodnames[roff_config.nuniquemodules++] = strdup(new_module->name);
			else
				fprintf(stderr, "[node_add_module] warning: modulenames array is full\n");
		}
	}

	node->total_num_devices += num_devices;

	roff_config.num_devices += num_devices;
	if (!is_disabled)
		roff_config.num_enabled_devices += num_devices;

	roff_config.num_modules++;
	if (!is_disabled)
		roff_config.num_enabled_modules++;
}

static void node_maybe_duplicate(int id, char **names, unsigned int *ports, size_t *count) 
{
	size_t mods, num_devices, index = 1;
	char *mod_name;
	roff_config_node_t *og_node;
	roff_config_status_t status;

	if (*count > 1)
	{
		og_node = &(roff_config.nodes[id]);

		for (; index < *count; index++)
		{
			if (!strcmp(names[index], "")) 
				continue;

			status = node_new(names[index], ports[index]);
			if (status.return_code == roff_config_status_success) 
			{
				for (mods = 0; mods < og_node->num_modules; mods++) 
				{
					mod_name = og_node->modules[mods].name;
					num_devices = og_node->modules[mods].num_devices;
					node_add_module(status.id, mod_name, num_devices, false);
				}
			}
					
		}
	}

	for (index = 0; index < *count; index++)
		free(names[index]);

	*count = 0;
}


static void configure(char *roff_config_text, bool ignore_disabled_modules)
{
	size_t count = 0, device_count = 0;
	int i;
	walk_t walk;
	char* names[ROFFCONF_MAX_INL_DECL] = {0};
	unsigned int ports[ROFFCONF_MAX_INL_DECL];
	char key[ROFFCONF_MAXNAME] = {0};
	char *buffer = roff_config_text;
	bool module_is_disabled = false;
	roff_config_status_t status;

	for (i = 0; i < ROFFCONF_MAX_INL_DECL; ++i)
		ports[i] = 22;

	if (!buffer)
	{
		perror("load_config_file()");
		exit(EXIT_FAILURE);
	}

	walk.buffer = buffer;
	walk.p = (char*)&buffer[0];

	/* Parsing starts here. */
	eat_comment_lines(&walk);
	while (accept_strings(&walk, names, ports, &count))
	{
		if (!strcmp(names[0], "")) break;
		
		status = node_new(names[0], ports[0]);
		switch(status.return_code)
		{
			case roff_config_status_success:
				/* Mandatory opening bracket after node name. */
				expect_single_char(&walk, '{');
				eat_comment_lines(&walk);

				/* Each node declaration ends on closing bracket. */
				while (!accept_single_char(&walk, '}'))
				{
					/* Parse module name and check if it's disabled. */
					module_is_disabled = is_module_disabled(&walk, ignore_disabled_modules);
					expect_string(&walk, key);
					expect_single_char(&walk, ':');
					
					/* Parse device number for the last parsed module name. */
					device_count = expect_int(&walk);
					node_add_module(status.id, key, device_count, module_is_disabled);

					/* Line cleanup (eat optional trailing comma and comments). */
					accept_single_char(&walk, ',');
					eat_comment_lines(&walk);
				}
				/* Line cleanup (eat optional trailing comma and comments). */
				accept_single_char(&walk, ',');
				eat_comment_lines(&walk);
				break;

			case roff_config_status_duplicate:
				/* Node already declared in previous scope, so we can skip over it. */
				eat_duplicate_decl(&walk);
				break;

			default:
				return;
		}
		/* If there was more than one identifier for this 'object' create the other nodes. */
		node_maybe_duplicate(status.id, names, ports, &count);
	}
}


roff_config_node_t *roff_config_get_node_from_hostname(char *hostname)
{
	roff_config_node_t *node;
	for (node = roff_config.nodes; node != NULL; node++)
		if (!strncmp(node->name, hostname, strlen(hostname)))
			return node;
	return (roff_config_node_t *) NULL;
}


/*
 * Returns a string containing all modules of a node, space-separated
 */
char *roff_config_get_node_modulestr(char *hostname)
{
	int i;
	str modulenames = Strnew();
	char *result = NULL;
	roff_config_node_t *node = roff_config_get_node_from_hostname(hostname);

	if (node == NULL)
		return NULL;

	for (i = 0; i < node->num_modules; ++i){
		str_printf(modulenames, "%s %d",
		           node->modules[i].name, node->modules[i].num_devices);
		if (i < node->num_modules - 1)
			str_printf(modulenames, " ");
	}

	result = strdup(str_string(modulenames));
	str_free(modulenames);

	return result; /* should be freed */
}


/* 
 * Produces a non-freeable string with the node names, separated by commas.
 */
char *roff_config_get_node_names(void)
{
	str names = Strnew();
	int i;

	/* concatenate the names */
	for (i = 0; i < roff_config.num_nodes; ++i)
	{
		str_printf(names, "%s", roff_config.nodes[i].name);
		if (i < roff_config.num_nodes - 1)
			str_printf(names, ",");
	}

	return str_string(names);
}


bool roff_config_should_print_ssh_hosts(void)
{
	return print_ssh_hosts;
}

/*
 * Prints the SSH configuration for the remote nodes
 * in a format suitable for ~/.ssh/config
 */
void roff_config_print_ssh_hosts(void)
{
	int i;
	struct passwd *p = getpwuid(getuid());

	/* concatenate the names */
	for (i = 0; i < roff_config.num_nodes; ++i)
	{
		if (roff_config.nodes[i].port == 22)
			continue;
		str host = Str("Host ");
		str_printf(host, "%s\n", roff_config.nodes[i].name);
		str_printf(host, "\tHostName %s\n", roff_config.nodes[i].name);
		str_printf(host, "\tPort %d\n", roff_config.nodes[i].port);
		if (p->pw_name != NULL)
			str_printf(host, "\tUser %s\n", p->pw_name);

		fprintf(stderr, "%s\n", str_string(host));
		
		str_free(host);
	}
}


/*
 * Iterates through the configured remote nodes and calls a
 * handle function for each visited module
 */
void roff_config_iterate(void (*handle)(roff_config_module_t *, void *), void *info)
{
	int i, j;
	if (!roff_config.initialized) return;

	for (i = 0; i < roff_config.num_nodes; i++)
		for (j = 0; j < roff_config.nodes[i].num_modules; j++)
			(handle)(&(roff_config.nodes[i].modules[j]), info);
}


/*
 * Initializes the remote configuration
 */
void roff_config_initialize(bool ignore_disabled_modules, bool is_userprog_portable)
{
	if (roff_config.initialized) return;
	const char *config_path = get_config_path();
	char *roff_config_text = load_config_file(config_path, NULL);

	userprog_portable = is_userprog_portable;
	roff_config.node_capacity = 32;
	roff_config.nodes = malloc(roff_config.node_capacity * sizeof(roff_config_node_t));
	if (roff_config.nodes == NULL)
	{
		perror("roff_config_initialize():");
		exit(EXIT_FAILURE);
	}

	roff_config.initialized = 1;
	roff_config.finalized = 0;
	roff_config.num_devices = 0;
	roff_config.has_cpu_module = 0;
	roff_config.num_modules = 0;
	roff_config.num_enabled_devices = 0;
	roff_config.num_enabled_modules = 0;
	roff_config.num_nodes = 0;
	roff_config.nuniquemodules = 0;
	roff_config.uniquemodnames = (char**) malloc(16 * sizeof(char*));

	if (!roff_config.modstr)
		roff_config.modstr = Strnew();

	configure(roff_config_text, ignore_disabled_modules);

	free(roff_config_text);
	free((void*)config_path);
}


/*
 * Initializes roff_config from the ompi_remote_devices embedded
 * in *_ompi.c, by first decoding it. Used during runtime.
 */
void roff_config_initialize_from_hex(char *roff_config_hex,
                                     unsigned long size,
                                     bool ignore_disabled_modules,
									 bool is_userprog_portable)
{
	if (roff_config.initialized) return;

	userprog_portable = is_userprog_portable;
	roff_config.node_capacity = 32;
	roff_config.nodes = malloc(roff_config.node_capacity * sizeof(roff_config_node_t));
	if (roff_config.nodes == NULL)
	{
		perror("roff_config_initialize():");
		exit(EXIT_FAILURE);
	}

	roff_config.initialized = 1;
	roff_config.finalized = 0;
	roff_config.num_devices = 0;
	roff_config.num_modules = 0;
	roff_config.num_enabled_devices = 0;
	roff_config.num_enabled_modules = 0;
	roff_config.num_nodes = 0;
	roff_config.nuniquemodules = 0;
	roff_config.uniquemodnames = (char**) malloc(16 * sizeof(char*));

	if (!roff_config.modstr)
		roff_config.modstr = Strnew();

	configure(roff_config_hex, ignore_disabled_modules);
}


/*
 * Finalizes the remote configuration
 */
void roff_config_finalize(void)
{
	int i, j;
	if (roff_config.finalized) return;

	for (i = 0; i < roff_config.num_nodes; i++)
	{
		for (j = 0; j < roff_config.nodes[i].num_modules; j++)
			free(roff_config.nodes[i].modules[j].name);
		free(roff_config.nodes[i].name);
	}

	for (i = 0; i < roff_config.nuniquemodules; i++)
		free(roff_config.uniquemodnames[i]);
	free(roff_config.uniquemodnames);

	str_free(roff_config.modstr);
	free(roff_config.nodes);
	roff_config.nodes = NULL;

	roff_config.initialized = 0;
	roff_config.finalized = 1;
}

#endif /* OMPI_REMOTE_OFFLOADING */
