/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* dg.c */

#include <stdio.h>
#include "dg.h"

/* 
 * A BAREBONES, GENERIC DIRECTED GRAPH:
 * this implements a collection of nodes that have directed edges between them, 
 * mostly for contructing dependency graphs.
 * The graph may be disconnected and this is why the nodes are also kept
 * in a separate set.
 * Traversal/search can be either within the node set or starting from a node
 * and following the edges.
 * To use it, just attach any type of values to the nodes 
 */

/* 
 * Nodes and edges
 */
 
SET_TYPE_IMPLEMENT(ptr2int)

dg_t *dg_new()
{
	dg_t *g = ssmalloc(sizeof(dg_t));
	g->G = NULL;
	g->nodeset = set_new(ptr2int);
	return g;
}


dgnode_t *dg_alloc_node(void *val)
{
	dgnode_t *t = ssmalloc(sizeof(dgnode_t));
	
	t->val = val;
	t->nsucc = 0;
	t->succ = NULL;
	return t;
}


void dg_add_node(dg_t *g, dgnode_t *n)
{
	set_put(g->nodeset, n);    /* Add to nodeset; the element value is ignored */
}


void dg_add_edge(dgnode_t *from, dgnode_t *to)
{
	if (from->nsucc % 4 == 0)  /* alloc/realloc */
		from->succ = (from->nsucc == 0) ? ssmalloc(4*sizeof(dgnode_t *)) :
		                    realloc(from->succ, (4+from->nsucc)*sizeof(dgnode_t *));
	from->succ[from->nsucc++] = to;
}


/* 
 * Search and traversal
 */
 
 
static set(ptr2int) visited, previsited;
dgvisopts_t dgvisopts;


/* Traverse starting at node n; keeps already visited info. */
void dg_traverse_continue(dgnode_t *n)
{
	int i;
	
	if (!n || set_get(visited, n))
		return;
		
	/* In postorder we need to mark as previsited to avoid infinite recursion */
	if (dgvisopts.visorder & DG_TRAV_POSTORDER)
	{
		if (set_get(previsited, n))    /* Break infinite recursion */
		{
			set_put(visited, n);
			if (dgvisopts.nodefunc) 
				(*dgvisopts.nodefunc)(n);
			return;
		}
		set_put(previsited, n);
	}
	
	if (dgvisopts.visorder & DG_TRAV_PREORDER)
	{
		set_put(visited, n);
		if (dgvisopts.nodefunc) 
			(*dgvisopts.nodefunc)(n);
		
	}
	for (i = 0; i < n->nsucc; i++)
	{
		if (dgvisopts.visorder & DG_TRAV_PREORDER && dgvisopts.edgefunc) 
			(*dgvisopts.edgefunc)(n, n->succ[i]);
		dg_traverse_continue(n->succ[i]);  /* depth-first */
		if (dgvisopts.visorder & DG_TRAV_POSTORDER && dgvisopts.edgefunc) 
			(*dgvisopts.edgefunc)(n,n->succ[i]);
	}
	if (dgvisopts.visorder & DG_TRAV_POSTORDER)
	{
		set_put(visited, n);
		if (dgvisopts.nodefunc) 
			(*dgvisopts.nodefunc)(n);
	}
}


/* Traverse starting at node n afresh (with clear visited info) */
void dg_traverse_fromnode(dgnode_t *n)
{
	set_init(ptr2int, &visited);
	set_init(ptr2int, &previsited);
	dg_traverse_continue(n);
}


/* Traverse whole graph (even if disconnected) */
void dg_traverse(dg_t *graph)
{
	setelem(ptr2int) node;
	
	set_init(ptr2int, &visited);
	set_init(ptr2int, &previsited);
	for (node = graph->nodeset->first ; node; node = node->next)
		dg_traverse_continue(node->key);
}


/* Search starting at node n; keeps already visited info. */
dgnode_t *dg_search_continue(dgnode_t *n, void *val)
{
	int i;
	dgnode_t *found;
	
	if (!n || !dgvisopts.valcmp || set_get(visited, n)) return NULL;
	set_put(visited, n);
	if ((*dgvisopts.valcmp)(n->val, val)) 
		return (n);
	for (i = 0; i < n->nsucc; i++)
		if ((found = dg_search_continue(n->succ[i], val)) != NULL)
			return found;
	return NULL;
}


/* Search starting at node n afresh (with clear visited info) */
dgnode_t *dg_search_fromnode(dg_t *graph, void *val)
{
	set_init(ptr2int, &visited);
	return dg_search_continue(graph->G, val);
}


/* Search whole graph (even if disconnected); simply searchs the node list */
dgnode_t *dg_search(dg_t *graph, void *val)
{
	setelem(ptr2int) e;
	
	if (!dgvisopts.valcmp) return NULL;
	for (e = graph->nodeset->first; e; e = e->next)
		if ((*dgvisopts.valcmp)(((dgnode_t*) e->key)->val, val)) 
			return (e->key);
	return NULL;
}


/*
 * Visualization
 */

static char *(*vlabelfunc)(void *val);


static void _gv_edge(dgnode_t *from, dgnode_t *to)
{
	if (to == NULL) return;
	fprintf(stderr, "    \"%p\" -> \"%p\";\n", from, to);
}


static void _gv_node(dgnode_t *n)
{
	char *s = vlabelfunc ? (*vlabelfunc)(n->val) : "-";
	
	if (*s == ':' && *(s+1) == ':')  /* hook to allow arbitrary formatting */
		fprintf(stderr, "  \"%p\" %s;\n", n, s+2);
	else
		fprintf(stderr, "  \"%p\" [label=\"%.10s\"];\n", n, s);
}


/* Output for graphviz (dot); shows only what a node reaches. */
void dg_visualize_fromnode(dgnode_t *n, char *(*labelfunc)(void *))
{
	vlabelfunc = labelfunc;
	fprintf(stderr, "digraph G {\n");
	fprintf(stderr, "  node [shape=circle];\n");
	dgvisopts.nodefunc = _gv_node;
	dgvisopts.edgefunc = _gv_edge;
	dgvisopts.visorder = DG_TRAV_POSTORDER;
	vlabelfunc = labelfunc;
	dg_traverse_continue(n);
	fprintf(stderr, "}\n");
}


/* Output for graphviz (dot); all nodes shown (even disconnected ones) */
void dg_visualize(dg_t *graph, char *(*labelfunc)(void *))
{
	setelem(ptr2int) e;
	dgnode_t *node;
	int i;
	
	vlabelfunc = labelfunc;
	fprintf(stderr, "digraph G {\n");
	fprintf(stderr, "  node [shape=circle];\n");
	if (graph)
		for (e = graph->nodeset->first; e; e = e->next)  /* output all nodes */
		{
			_gv_node(node = e->key);
			for (i = 0; i < node->nsucc; i++)
				_gv_edge(node, node->succ[i]);
		}
	fprintf(stderr, "}\n");
}
