/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* assorted.h */

#ifndef __ASSORTED_H__
#define __ASSORTED_H__

#include <stdbool.h>
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>

/*
 * INTVEC
 * A simple, dynamic int vector type, for storing a collection of integers.
 * Elements are accessed like normal arrays, by their index, starting at 0.
 */
typedef int* intvec_t;
typedef void intvec_iterfunc_t(int *elem, bool isfirst, bool islast, void *arg);
extern intvec_t intvec_new(unsigned int initsize);
extern intvec_t intvec_new_withelems(unsigned int initsize, ...);
extern int      intvec_len(intvec_t v);
extern intvec_t intvec_resize(intvec_t v, unsigned int newsize);
extern intvec_t intvec_append(intvec_t v, unsigned int nelems, ...);
extern intvec_t intvec_copy(intvec_t v);
extern intvec_t intvec_remove_all(intvec_t v, int val);
extern void     intvec_free(intvec_t v);
extern int     *intvec_search(intvec_t v, int val);
#define         intvec_contains(v, val) (intvec_search(v, val) != NULL)
extern void     intvec_iterate(intvec_t v, intvec_iterfunc_t f, void *arg);

/* 
 * JOBS -- simple functions for doing work in parallel (using processes)
 * 
 * Available macros:
 * 
 *   JOB_START(jobname, nprocs, worksize): 
 *     Starts a job of `worksize' iterations, named `jobname', that will be executed by 
 *     `nprocs' processes.
 * 
 *   JOB_LOOP(jobname, itername): 
 *     Forms a loop for `jobname', using an iteration variable named `itername',
 *     with each process receiving a chunk of a total of `worksize' iterations.
 * 
 *   JOB_FINISH(jobname, exitstatus): 
 *     Kills a job named `jobname', with every child exiting with `exitstatus'.
 */

enum execstatus_ { JOB_STOPPED = 0, JOB_RUNNING, JOB_KILLED };

typedef struct jobcb_
{
	int myid;
	pid_t pid;
	pid_t parent_pid;
	int nprocs;
	int worksize;
	enum execstatus_ status;
	unsigned long fi, li;
} jobcb_t;

#define JOB_START(jobname,nprocs,worksize) \
	jobcb_t * jobname = job_new(nprocs, worksize); \
	if (!jobname)                                  \
	{                                              \
		fprintf(stderr, "JOB_START: %s: error: invalid worksize\n", \
		                #jobname);                 \
		exit(EXIT_FAILURE);                        \
	}                                              \
	job_run(jobname);

#define JOB_LOOP(jobname,itername) \
	unsigned long itername = 0; \
	if (job_get_chunk(jobname)) \
		for (itername = jobname->fi; itername < jobname->li; itername++)

#define JOB_FINISH(jobname,exitstatus) job_kill(jobname, exitstatus)

extern int  job_get_chunk(jobcb_t *job);
extern jobcb_t *job_new(int nprocs, int worksize);
extern void job_run(jobcb_t *job);
extern int  job_kill(jobcb_t *job, int exitstatus);

/* 
 * ENCODING/DECODING 
 */

enum encstaticflag_ {
	ENCODE_NOT_STATIC = 0, ENCODE_STATIC
};

enum encprintflag_ {
	DONT_PRINT_SIZE = 0, PRINT_SIZE
};

extern char *create_string_decl(char *buffer, char *array_name, unsigned long nbytes);
extern char *encode_text_tohexarray_withdecl(char *buffer, char *array_name, unsigned long nbytes,
                                             bool static_decl, bool print_size);
extern char *encode_binary_tohexarray(unsigned char *buffer, char *array_name, unsigned long nbytes);

extern int is_substr(char *str, char *substr);
extern char *sanitize_str(char *str, bool repl);
extern bool contains_word(const char *str, const char *word) ;

extern void get_path(char *executable, char *path, size_t size);
extern bool is_path(const char *str);


/* malloc() with failure check */
static inline void *smalloc(int size)
{
	void *m;
	if ((m = (void *) malloc(size)) == NULL)
	{
		fprintf(stderr, "[smalloc]: memory allocation failed\n");
		exit(1);
	}
	return (m);
}

/* calloc() with failure check */
static inline void *scalloc(int n, int size)
{
	void *m;
	if ((m = (void *) calloc(n, size)) == NULL)
	{
		fprintf(stderr, "[scalloc]: memory allocation failed\n");
		exit(1);
	}
	return (m);
}

/* realloc() with failure check */
static inline void *srealloc(void *ptr, int size)
{
	void *m;
	if ((m = (void *) realloc(ptr, size)) == NULL)
	{
		fprintf(stderr, "[srealloc]: memory reallocation failed\n");
		exit(1);
	}
	return (m);
}

// extern void *smalloc(int size);
// extern void *scalloc(int n, int size);

#endif /* __ASSORTED_H__ */
