/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* assorted.c -- various utilities */

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdbool.h>
#include <unistd.h>
#include <stdarg.h>
#include <sys/wait.h>
#include <sys/types.h>
#include "str.h"
#include "assorted.h"


/* 
 * INTVEC
 * A very simple (dynamic) integer vector; metadata are stored at the beginning
 * and the actual data start after the metadata; the actual data is returned 
 * as the starting address of vector.
 * Elements are accessed like normal arrays, by their index, starting at 0;
 * care must be taken to not access elements outside the vector's size. 
 * To expand the vector with new elements, use the intvec_append() function
 * [not implemented yet :-)]. Functions that expand/shrink a vector may 
 * return a new vector address.
 */

#define VECNELEMS(v) ((v)[-1])


/**
 * @brief Creates a new int vector with the given initial size.
 * @param initsize the initially allocated size of the vector
 * @return a new int vector
 */
intvec_t intvec_new(unsigned int initsize)
{
	int *v;
	
	if (initsize <= 0) initsize = 1;   /* at least 1 */
	v = (int *) smalloc((initsize+1) * sizeof(int));
	v[0] = initsize;     /* metadata (size) */
	return (intvec_t) (v+1);
}


/**
 * @brief Creates a new int vector with the given initial size and elements.
 * @param initsize the initially allocated size of the vector
 * @param ...      the elements (initisize in number)
 * @return a new int vector
 */
intvec_t intvec_new_withelems(unsigned int initsize, ...)
{
	int *v, i;
	va_list args;
	
	if ((v = intvec_new(initsize)) == NULL) return NULL;
	va_start(args, initsize);
	for (i = 0; i < initsize; i++)
		v[i] = va_arg(args, int);
	va_end(args);
	return v;
}


/**
 * Returns the number of elements (length) of the vector.
 */
int intvec_len(intvec_t v)
{
	return (v ? VECNELEMS(v) : 0);
}


intvec_t intvec_resize(intvec_t v, unsigned int newsize)
{
	if (newsize <= 0 || (v && newsize == VECNELEMS(v))) return v; 
	v =  v ? (int *) realloc(v-1, (newsize+1) * sizeof(int)) : 
	         (int *) smalloc((newsize+1) * sizeof(int));
	v[0] = newsize;     /* metadata (new size) */
	return (intvec_t) (v+1);
}


intvec_t intvec_append(intvec_t v, unsigned int nelems, ...)
{
	int i, size;
	va_list args;
	
	if (nelems <= 0) return v;
	size = v ? VECNELEMS(v) : 0;
	v = intvec_resize(v, size + nelems);
	va_start(args, nelems);
	for (i = 0; i < nelems; i++)
		v[size + i] = va_arg(args, int);
	va_end(args);
	return v;
}


/**
 * Copies the given vector.
 */
intvec_t intvec_copy(intvec_t v)
{
 	intvec_t new;
	int n = v ? VECNELEMS(v) : 0;

	if (n <= 0) return NULL;  /* empty vector */
	new = intvec_new(n);
	memcpy(new, v, n * sizeof(int));
	return new;
}


/**
 * Frees the memory allocated for the vector.
 */
void intvec_free(intvec_t v)
{
	if (v)
		free(v-1);
}


/**
 * @brief Searches for a specific value in the int vector.
 * @param v    the vector
 * @param val  the value to search for
 * @return     pointer to the element of the vector that contains the value,
 *             or NULL if not found
 */
int *intvec_search(intvec_t v, int val)
{
	int i;
	if (v)
		for (i = VECNELEMS(v); i >= 0; i--)
			if (v[i] == val)
				return &v[i];
	return NULL;
}


void intvec_iterate(intvec_t v, intvec_iterfunc_t func, void *arg)
{
	if (v) 
	{
		int i, n = VECNELEMS(v);
		if (n > 0)
			for (i = 0; i < n; i++)
				func(&v[i], i == 0, i == n-1, arg);
	}
}


/**
 * @brief Removes all elements with value val and resizes vector appropriately
 * @param v the vector
 * @param val the value to remove completely
 * @return the new vector
 */
intvec_t intvec_remove_all(intvec_t v, int val)
{
	int size = VECNELEMS(v), removed = 0, i;
	
	if (v == NULL) return NULL;
	
	for (i = size; i >= 0; i--)
		if (v[i] == val)
		{
			if (i != size-1-removed)
				memmove(v+i, v+i+1, size-1-removed-i);
			removed++;
		};
	return intvec_resize(v, size-removed);
}


/* 
 * JOBS
 */

/**
 * @brief Assigns a chunk of job iterations, statically, to a calling
 *        process.
 * 
 * @param job the job
 * @return 1 if any iterations are left
 */
int job_get_chunk(jobcb_t *job)
{
	int chunksize, rem, numsiblings = job->nprocs;
	int myid = job->myid;

	if (numsiblings == 1) /* I am the only one */
	{
		job->fi = 0;
		job->li = job->worksize;
		return (job->fi != job->li);
	}

	chunksize = job->worksize / numsiblings;
	rem = job->worksize % numsiblings;
	if (rem) chunksize++;

	if (myid < rem || rem == 0)
	{
		job->fi = myid * chunksize;
		job->li = job->fi + chunksize;
	}
	else
	{
		job->fi = rem * chunksize + (myid - rem) * (chunksize - 1);
		job->li = job->fi + (chunksize - 1);
	}

	return (job->fi != job->li);
}


/** 
 * @brief Creates a job of `worksize' iterations that will be executed by 
 *        `nprocs' processes.
 * 
 * @param nprocs   the number of processes that will participate
 * @param worksize the size of the work to be done
 * @return a new job
 */
jobcb_t *job_new(int nprocs, int worksize)
{
	jobcb_t *job;
	int actual_nprocs;

	if (worksize == 0)
		return NULL;
		
	/* Correct the actual # of jobs */
	if ((worksize < nprocs) || (nprocs == 0))
		actual_nprocs = worksize;
	else if (nprocs < 0) /* Requested procs are not configured */
		actual_nprocs = 1;
	else
		actual_nprocs = nprocs;
	
	job = smalloc(sizeof(jobcb_t));
	job->status = JOB_STOPPED;
	job->myid = 0;
	job->pid = -1;
	job->nprocs = actual_nprocs;
	job->worksize = worksize;
	job->fi = job->li = 0;

	return job;
}


/**
 * @brief Run a job
 * 
 * @param job the job to be run
 */
void job_run(jobcb_t *job)
{
	int j;
	pid_t newpid = (pid_t) -1;

	if (job->status != JOB_STOPPED) 
		return;

	job->status = JOB_RUNNING;
	job->parent_pid = getpid();

	for (j = 1; j < job->nprocs; j++)
	{
		if ((newpid = fork()) == 0)
		{
			job->myid = j;
			job->pid = getpid();
			break;
		}
	}
}


/**
 * @brief Kill a running job (called by all processes)
 * 
 * @param job    the job to be killed
 * @param status the status a child will exit with
 * @return 1 if at least one child exited with status != 0,
 *         0 if all children exited with status = 0,
 *        -1 if the job is not currently being executed.
 */
int job_kill(jobcb_t *job, int status)
{
	int childst = 0, any_failed = 0;

	if (job->status != JOB_RUNNING) 
		return -1;

	/* All processes exit, except for the parent */
	if (getpid() != job->parent_pid)
		exit(status);
	
    /* Parent: reap all children and accumulate statuses */
	while ((job->nprocs > 1) && (wait(&childst) > 0))
		if (WIFEXITED(childst) && WEXITSTATUS(childst) != 0)
    		any_failed = 1;

    job->status = JOB_KILLED;
    free(job);

    return (any_failed);
}


/* Creates an string declaration, initialized at the buffer's contents
 * Note: currently unused 
 */
char *create_string_decl(char *buffer, char *array_name, unsigned long nbytes)
{
	str s = Strnew();
	char *text = NULL;
	
	str_printf(s, "char *%s = \"%s\";\n", array_name, buffer);
	str_printf(s, "unsigned long %s_size = %lu;\n", array_name, nbytes);
	  
	text = strdup(str_string(s));
	str_free(s);
	
	return text;
}


/* 
 * ENCODING/DECODING
 */

static 
void dump_hex_values_tostr(void *src, str dst, unsigned long nbytes, bool is_unsigned)
{
	unsigned long i;

	for (i = 0; i < nbytes; i++)
	{
		str_printf(dst, "0x%02X", is_unsigned 
		          ? ((unsigned char *) src)[i] 
		          : (unsigned char) ((char *) src)[i]
		);

		if (i < nbytes - 1)
			str_printf(dst, ",");
		else    
			str_printf(dst, "\n");
	}
}


/* Encodes a buffer to a hex array and dumps its C declaration */
char *encode_text_tohexarray_withdecl(char *buffer, char *array_name, unsigned long nbytes,
                                      bool static_decl, bool print_size)
{
	str s = Strnew();
	char *hex = NULL;
	
	/* [static] char <filename>[<size>] = { */
	str_printf(s, "%schar %s[%lu] = {", 
	              static_decl ? "static ": "", array_name, nbytes);

	/* <byte1>, <byte2>, ... */
	dump_hex_values_tostr(buffer, s, nbytes, false);

	/* }; */
	str_printf(s, "};\n"); 

	if (print_size)
	{
		/* unsigned long <filename>_size = <size>; */
		str_printf(s, "unsigned long %s_size = %lu;\n", array_name, nbytes);
	}
	  
	hex = strdup(str_string(s));
	str_free(s);
	
	return hex;
}


/* Encodes a buffer to a hex array and dumps its C declaration */
char *encode_binary_tohexarray(unsigned char *buffer, char *array_name, unsigned long nbytes)
{
	str s = Strnew();
	char *hex = NULL;
	
	/* { */
	str_printf(s, "{");

	/* <byte1>, <byte2>, ... */
	dump_hex_values_tostr(buffer, s, nbytes, true);

	/* } */
	str_printf(s, "}");
	  
	hex = strdup(str_string(s));
	str_free(s);
	
	return hex;
}


int is_substr(char *str, char *substr) 
{
	size_t start, end;
    char *ptr; 
	int noletterbefore, noletterafter;

	if ((str == NULL) || (substr == NULL))
		return 0;

	ptr = strstr(str, substr);
    if (ptr == NULL) 
        return 0;
    
    start = ptr - str;
    end = start + strlen(substr);
    
	noletterbefore = !isalnum(*(ptr-1));
	noletterafter = !isalnum(*(ptr+strlen(substr)));

    return ((start == 0 || noletterbefore) && (end == strlen(str) || noletterafter));
}


/* Removes all special symbols from a string, or replaces them with '_'
 * if repl==true, keeping only alphas and digits 
 */
char *sanitize_str(char *str, bool repl)
{
	size_t len = strlen(str);
	char *buf = smalloc(len + 1);
	int i, k = 0;

	for (i = 0; i < len; i++)
		if (isalpha(str[i]) || isdigit(str[i]) || str[i] == '_')
			buf[k++] = str[i];
		else if (repl)
			buf[k++] = '_';

	buf[k] = '\0';
	return buf;
}

static bool is_substring_at_position(const char *str, const char *sub, int pos) 
{
    int i = 0;
    while (sub[i] != '\0') 
	{
        if (str[pos + i] != sub[i])
            return false;
        i++;
    }
    return true;
}


/* Checks if a word is contained in a string */
bool contains_word(const char *str, const char *word) 
{
	int i, word_len, str_len;
    if (str == NULL || word == NULL || strlen(word) == 0)
        return false;

    word_len = strlen(word);
    str_len = strlen(str);

    for (i = 0; i <= str_len - word_len; i++) 
        if (is_substring_at_position(str, word, i)) 
		{
            bool is_start_word = (i == 0) || !isalpha(str[i - 1]);
            bool is_end_word = (i + word_len == str_len) || !isalpha(str[i + word_len]);
            if (is_start_word && is_end_word)
                return true;
        }

    return false;
}


bool is_path(const char *str) 
{
    if (strchr(str, '/'))
        return true;
    return false;
}


void get_path(char *executable, char *path, size_t size)
{
	int i;

	memset(path, '\0', size);

	for (i = strlen(executable); i >= 0; i--)
	{
		if (executable[i] == '/')
		{
			strncpy(path, executable, i + 1);
			path[i + 1] = '\0';
			break;
		}
	}
}


#if 0

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 *  MACHINERY TO HANDLE COMMON USES OF ENVIRONMENTAL VARIABLES       *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* We need to know how to react if a given environemntal variable is set 
 * by the user. The most common usage patterns include setting an integer or 
 * a string program variable based on the value of the environmental variable.
 * We thus need to store a pointer to the affected program variable and, if 
 * the environmental variable is set, to assign a value accordingly.
 *
 * For cases where the value of the environmental variable does not directly 
 * give the value for the program variable, we need to have a set of cases,
 * whereby for each possible value of the environmental variable we assign
 * a corresponding integer value to the (integer) program variable.
 *
 * To be able to handle any other case, we also provide a callback function 
 * that can do anything with the value of the environmental variable. If set,
 * this function will also be called in the event that the environmental 
 * variable value does not fall into any of the given integer cases.
 */
typedef struct {
	char *envname;            /* The name of the evnironmental variable */
	int  *ivarptr;            /* Pointer to the int program variable to set */
	char **svarptr;           /* Pointer to the char* program variable to set */
	struct _intcases {        /* Cases for the int program variable values */
		char *envvalue;           /* A value of the environmental variable */
		int  intvalue;            /* Corresponding int value for the program var. */
	} *intcases;              /* BE CAREFUL: funcptr must come next! */
	void (*funcptr)(char*, char*);  /* Callback func with the env. var. value */
} envhandler_t;

#define HANDLE_ENV(name, how) { name, how }  /* Define action for an env var */
#define SETVAR_INT(p) p, NULL, NULL, NULL    /* Set an integer progr. var */
#define SETVAR_STR(s) NULL, s, NULL, NULL    /* Set a string progr. var */
#define CALLFUNC(f) NULL, NULL, NULL, f      /* Call a function */
                                             /* Cases for an int progr. var. */
#define SETVAR_INT_CASES(p,c) p, NULL, (struct _intcases []) { c
#define INTCASE(s, v)         { s, v },
#define INTCASE_END           { NULL } }     /* No more cases */
#define INTCASE_END_OTHER(f)  { NULL } }, f  /* Ditto but call func if needed */
#define HANDLE_ENV_END        { NULL }       /* No more env. vars */

/* Usage example for an env. variable "MYENV" 
 * envhandler_t myenvs[] = {
 *   HANDLE_ENV("MYENV1", // Set myint1 to the integer value of the env var
 *              SETVAR_INT(&myint1)),
 *   HANDLE_ENV("MYENV2", // Set mystr to point to the value of the env var
 *              SETVAR_STR(&mystr)),
 *   HANDLE_ENV("MYENV3", // Call myfunc(n,v) to do whatever, where n is the
 *              CALLFUNC(myfunc))  // name and v is the value of the env var
 *   HANDLE_ENV("MYENV4", // Depending on the value of MYENV4, give myint2
 *     SETVAR_INT_CASES(&myint2,   // appropriate integer value
 *                      INTCASE("small",  1)
 *                      INTCASE("medium", 2)
 *                      INTCASE("large",  3)
 *                      INTCASE_END)           // no more cases
 *   ),
 *   HANDLE_ENV("MYENV5", // If MYENV5 does not have an acceptable value, 
 *     SETVAR_INT_CASES(&myint2,   // then call myfunc(n,v)
 *                      INTCASE("no",  0)
 *                      INTCASE("yes", 1)
 *                      INTCASE_END_OTHER(myfunc)) // callback @ all other cases
 *   ),
 *   HANDLE_ENV_END  // no more environmental variables to handle
 * };
 */

void envvars_handle(envhandler_t *vars)
{
	char *s;
	
	for (; vars->envname != NULL; vars++)
	{
		if ((s = getenv(vars->envname)) != NULL)
		{
			if (vars->ivarptr)
			{
				if (vars->intcases == NULL)
					*(vars->ivarptr) = atoi(s);
				else
				{
					int i;
					for (i = 0; vars->intcases[i].envvalue; i++)
						if (strcmp(vars->intcases[i].envvalue, s) == 0)
						{
							*(vars->ivarptr) = vars->intcases[i].intvalue;
							break;
						};
					/* Call function if no legal value */
					if (!vars->intcases[i].envvalue && vars->funcptr)
							vars->funcptr(vars->envname, s);
				}
			}
			else
				if (vars->svarptr)
					*(vars->svarptr) = s;
				else
					if (vars->funcptr)
						vars->funcptr(vars->envname, s);
		};
	}
}

/* From the opencl module */
envhandler_t opencl_envs[] = {
		/* OMPI_MODULE_OPENCL_XLIMIT: warn (default) | nowarn | exit | exitsilent */
		HANDLE_ENV("OMPI_MODULE_OPENCL_XLIMIT", 
			SETVAR_INT_CASES((int *) &xceed_limits_behavior,
		                   INTCASE("warn", XLB_WARN)
		                   INTCASE("nowarn", XLB_NOWARN)
		                   INTCASE("exit", XLB_EXIT)
		                   INTCASE("exitsilent", XLB_EXITMUTE)
		                   INTCASE_END)
		),
		HANDLE_ENV_END
	};

#endif
