/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/


// #define DBGPRN_FORCE
// #define DBGPRN_BLOCK
#define DBGPRN_FILTER DBG_RDEV_NODEMGR

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include "ort_prive.h"

#ifdef OMPI_REMOTE_OFFLOADING

#include "remotedev/rdev.h"
#include "remotedev/node_manager.h"
#include "remotedev/memory.h"
#include "remotedev/workers.h"
#include "remotedev/workercmds.h"
#include "remotedev/rdev_prive.h"
#include "rdev_config.h"
#include "../../rt_common.h"


#define DBGTERMINAL "/usr/bin/xterm" /* xterm used for debugging */

enum noderoletype node_role = ROLE_PRIMARY; /* default role */

static int *user_argc;
static char ***user_argv;

static void ***devinfos; /* Used by the primary node to store all devinfos */
void *comminfo;

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 *                                                                   *
 * NODE HANDLING                                                     *
 *                                                                   *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */


/* Returns the number of configured remote nodes */
int rdev_man_get_num_nodes(void)
{
	return rdev_config.num_nodes;
}


char *rdev_man_get_nodename(int node_id)
{
	return rdev_config.nodes[node_id].name;
}


void rdev_man_finalize(void)
{
	int i, num_nodes = rdev_man_get_num_nodes();
	
	rdev_config_finalize();
	
	for (i = 0; i < num_nodes; ++i)
		free(devinfos[i]);
	free(devinfos);
	devinfos = NULL;
	
	DBGPRN((stderr, "[remotedev] closing; rdev_man_finalize was called\n"));
}


/* Returns the ID of the calling node */
int rdev_man_get_my_id(void)
{
	return Comm_Get_id(comminfo);
}


/* This is called by the host and the other nodes (when spawned)
 * from _ort_init().
 */
void rdev_man_prepare_node(int *argc, char ***argv)
{   
	DBGPRN((stderr, "[remotedev] rdev_man_prepare_node: initializing comm layer (pid = %d)\n", 
	                getpid()));

	comminfo = Comm_Init(argc, argv);
	
	if (comminfo == NULL)
		ort_error(1, "[remotedev] rdev_man_prepare_node: communication layer could not be initialized; exiting.\n");

	if (node_role == ROLE_WORKER)
		rdev_worker_init();
}


/* The primary node initially retrieves the number of devices installed 
 * at each worker node, from the configuration. Then, from each node, 
 * it receives an array containing the devinfos of all its local devices.
 */
static void receive_all_devinfos(void)
{
	int i, j, tabidx = 0, numdevs, num_nodes = rdev_man_get_num_nodes();
	int device_id_in_node, device_id;
	rdevinfo_t *item;
	
	/* Receive devinfos from all nodes and add them to the map */
	for (i = 0; i < num_nodes; i++)
	{
		numdevs = rdev_config.nodes[i].total_num_devices;
		devinfos[i] = (void**) smalloc(numdevs * sizeof(void*));
		Comm_Recv(comminfo, i+1, COMM_BYTE, devinfos[i], numdevs*sizeof(void*), 
		          COMM_ANY_TAG, NULL);
		/* Add the received devinfos to the devinfo table.
		 * For each device, we also store the intranode device ID,
		 * the intramodule device ID and the node ID.
		 */
		 
		/* Device order in .ompi_remote_devices must be equal to the
		 * actual device order in the worker node. The CPU device must be 
		 * detected and always be placed at 0.
		 */
		for (device_id_in_node = 0, j = 0; j < rdev_config.nodes[i].num_modules; j++)
		{
			for (device_id = 0; device_id < rdev_config.nodes[i].modules[j].num_devices; 
			     device_id_in_node++, device_id++, tabidx++)
			{
				item = &(devinfotab[tabidx]);
				item->device_id_in_node = device_id_in_node;
				item->device_id_in_module = device_id;
				item->global_device_id = tabidx + ort->num_local_devices;
				item->node_id = i + 1;
				item->devinfo = devinfos[i][device_id_in_node];
				item->modulename = rdev_config.nodes[i].modules[j].name;
				item->alloc_table = rdev_alloctab_new(item->global_device_id);
			}
		}
	}
}


/* This one is called only from the host, during hm_initialize().
 * It spawns the workers and returns the merged communicator.
 */
int rdev_man_create_workers(void)
{
	int num_nodes;

	DBGPRN((stderr, "[remotedev] primary creating nodes (pid = %d)\n", getpid()));

	if ((num_nodes = rdev_man_get_num_nodes()) == 0)
		return 0;

	devinfos = (void ***) smalloc(num_nodes * sizeof(void**));
	Comm_Spawn(comminfo, num_nodes);

	/* The primary node receives all devinfos from all workers */
	receive_all_devinfos();

	return 1;
}

#undef DEBUG_NODE_MANAGER
#endif /* OMPI_REMOTE_OFFLOADING */
