/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

/* This is the pthreads threading library for OMPi
 */

#ifndef __EE_H__
#define __EE_H__

#include "config.h"

#define EE_TYPE_PROCESS
#define AVOID_OMPI_DEFAULT_TASKS

#ifdef HAVE_SPINLOCKS
	/* Needed to get spin locks, before including <pthread.h> */
	#define _XOPEN_SOURCE 600
#endif

#include <unistd.h>
#include <pthread.h>
#include "../sysdeps.h"


/* The threading library should provide the following types:
 *    oprc_key_t
 *    oprc_lock_t
 */

typedef union {
	struct {
		int   val;             /* for user-defined locks */
		int   count;           /* # a nested lock is locked */
		int   owner;           /* lock owner for nested locks*/
		int   type;            /* normal/spin/nested */
	} lock;
	char padding[CACHE_LINE];
} oprc_lock_t;


/*
 * The threading library should provide the functions that follow.
 */

/* Base functions */

extern void  oprc_shm_free(int *p);
extern void  oprc_shmalloc(void **p, size_t size, int *memid);
extern int   oprc_shm_alloc(size_t size);
extern void *oprc_shm_ptr(int memid);
extern int   oprc_pid(void);
extern int   oprc_initialize(int *argc, char ***argv,
                             ort_icvs_t *icv, ort_caps_t *cap);
extern void  oprc_finalize(int exitvalue);
extern int   oprc_request(int numthreads, int level, int oversubscribe);
extern void  oprc_create(int numthreads, int level, void *arg,
                         void **info);
extern void  oprc_waitall(void **info);
extern int   oprc_bindme(int **places, int pindex);
extern int   oprc_getselfid(void);
extern void *oprc_getself(unsigned int *size);
/* DON'T put parentheses! */
#define oprc_yield sched_yield

/* Thread-specifics */
#ifdef TLS_KEYWORD
	#define USE_TLS  /* This is only for ort.c */
#endif
#define oprc_key_t            pthread_key_t
#define oprc_key_create(a,b)  pthread_key_create(a,b)
#define oprc_getspecific(a)   pthread_getspecific(a)
#define oprc_setspecific(a,b) pthread_setspecific(a,b)

/* Locks */
extern void *oprc_init_lock(oprc_lock_t *lock, int type);
extern int  oprc_destroy_lock(oprc_lock_t *lock);
extern int  oprc_set_lock(oprc_lock_t *lock);
extern int  oprc_unset_lock(oprc_lock_t *lock);
extern int  oprc_test_lock(oprc_lock_t *lock);

typedef struct taskgroup_s
{
	struct taskgroup_s *parent;
	volatile int       is_canceled;     /* Cancel flag for taskgroup */
} taskgroup_t;

/* Tasks */
typedef struct Node
{
	void *(*func)(void *);
	void *funcarg; /* Task function argument */
	int num_children; /* taskcreated */
	struct Node *parent;
	struct Node *next;
	taskgroup_t *taskgroup; /* OpenMP 4.0 */
	int rtid; /* Special task id (nested loop to task) */
	int isfinal;
	int inherit_task_node;
	ort_task_icvs_t icvs; /* OpenMP3.0 */
} ort_task_node_t;

extern void oprc_start_implicit_task(void **eeinfo, void *parent_eeinfo);
extern void oprc_new_task(int final, int untied, 
                          void *(*func)(void *arg), void *arg);
extern void oprc_taskwait(int how, void *info, int thread_num);
extern void oprc_set_currtask(void **eeinfo, ort_task_node_t *tw);
extern ort_task_node_t *oprc_get_currtask(void **eeinfo, int thread_num);
extern void *oprc_taskenv_alloc(int size, void *task_func);
extern void oprc_taskenv_free(void *arg);

/* The following functions will NOT get called as long as you compile user
 * programs with --taskopt=size option, but you will need to provide dummy
 * implementations for them to keep the compiler happy. 
 */
extern void *oprc_task_immediate_start(int final);
extern void oprc_task_immediate_end(void *new_node);

#endif
