/*
  OMPi OpenMP Compiler
  == Copyright since 2001 the OMPi Team
  == Dept. of Computer Science & Engineering, University of Ioannina

  This file is part of OMPi.

  OMPi is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  OMPi is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with OMPi; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/


#include <stdarg.h>
#include <stdlib.h>
#include <stdbool.h>
#include <stdio.h>
#include "globals.h"
#include "locks.h"
#include "worksharing.h"
#include "parallel.h"
#include "barrier.h"
#include "tasks.h"

__SHAREDQLFR ort_workshare_t workshare;

__DEVQLFR 
void _ort_fence(void) 
{
	__threadfence();
}

__DEVQLFR 
int ws_enter_workshare_region(int wstype, int nowait, int hasordered,
	                int nsections)
{
	ort_workshare_t *r;
	int             notfirst = 1;

	r = &(workshare);
	
	if (omp_get_thread_num() == 0) /* I am the first to enter */
	{
		if (wstype == _OMP_SECTIONS) /* Initialize */
        {
			r->sectionsleft = nsections;
            r->wslock.in_area = true;
            r->wslock.mutex = 0;
            r->wslock.mutex_init = 1;
        }
		else if (wstype == _OMP_FOR)
		{
			r->forloopiter = 0; /* The very 1st iteration to be given away */
			r->forlooplock.in_area = false;
			r->forlooplock.mutex = 0;
			r->forlooplock.mutex_init = 1;
		}

		r->left = 0; /* None left yet */
		r->empty = notfirst = 0;
	}

	dev_barrier_wait(BARRIER_SLOT_1, dev_get_parallel_active_threads());

	return (!notfirst);
}

/* Returns the # of threads that have not yet left the region.
 * A zero means that I am the last thread to leave.
 */
__DEVQLFR 
int ws_leave_workshare_region(int wstype)
{
	ort_workshare_t *r;

	r = &(workshare);

	if (omp_get_thread_num() == 0)
	{
		r->empty = 1; /* The region is now empty */
		r->left = 0;
		
		if (wstype == _OMP_SECTIONS) /* Initialize */
        {
            r->wslock.in_area = false;
        }
		else if (wstype == _OMP_FOR) /* blocking for */
		{
			r->forloopiter = 0; /* Prepare for next use */
			r->forlooplock.in_area = false;
			r->forlooplock.mutex = 0;
			r->forlooplock.mutex_init = 1;
		}
	}

	dev_barrier_wait(BARRIER_SLOT_1, dev_get_parallel_active_threads());

	return (omp_get_thread_num());
}

#if SINGLE_IMPLEMENTATION == SINGLE_LIGHT
__DEVQLFR
int _ort_mysingle(int nowait)
{
	if (omp_get_thread_num() == 0)
	{
		workshare.insingle = true;
		return 1;
	}
	return 0;
}

__DEVQLFR
void _ort_leaving_single(void)
{
	if (omp_get_thread_num() == 0)
		workshare.insingle = false;
	dev_barrier_wait(BARRIER_SLOT_1, dev_get_parallel_active_threads());
	return;
}

#else
__DEVQLFR
int _ort_mysingle(int nowait)
{
	if (omp_get_num_threads() == 1)
		return (1);
	else
		return (ws_enter_workshare_region(_OMP_SINGLE, nowait, 0, 0));
}

__DEVQLFR
void _ort_leaving_single(void)
{
	if (omp_get_num_threads() != 1)
		ws_leave_workshare_region(_OMP_SINGLE);
}
#endif


__DEVQLFR 
void _ort_entering_for(int nowait, int hasordered) 
{
    if (omp_get_num_threads() == 1)
		return;

	ws_enter_workshare_region(_OMP_FOR, nowait, hasordered, 0);
}

__DEVQLFR 
int _ort_leaving_for(void) 
{
	if (omp_get_num_threads() == 1)
		return (0);
	else
		return (ws_leave_workshare_region(_OMP_FOR));
}
